
#include <math.h>

void sleep_ms(int ms); // Not in the standard library, unfortunately

#define NUM_DIST 128
#define A_BRK 14.3 /* Braking decelaration, as a positive number in m/s^2 */
#define V_MAX 15   /* Maximum speed, in m/s */
#define DELTA_T (1.0/30.0) /* Length of one tick in s, here with frequency of 30 Hz  */
#define SAFETY_DIST 130 /* Size of vehicle circumference, plus any safety margin */

enum mode_t {STATE_INITIAL, STATE_CRUISING, STATE_BRAKING}; 

struct state_t { 
  enum mode_t mode;
  double x;
  double y;
  double v;
  double phi;
  double omega;
  double a;
  };

struct input_t {
  double a_in;
  double w_in;
  double L[NUM_DIST]; 
};

double read_acceleration();
double read_steering();
void set_acceleration(double a);
void set_steering(double w);

void read_sensors(double dist[]);

double dist_traveled(double a, double v) 
{
  return ((a/2* DELTA_T+ v)* DELTA_T);
}
double polar_to_cart_x(double r, double phi) { return r* cos(phi); }
double polar_to_cart_y(double r, double phi) { return r* sin(phi); }

double brk_dist(double v)
{
  return (v*v/(2* A_BRK));
}

double min_dist(double dist[], int num_dist)
{
  int i;
  double c;

  c= 0;
  for (i= 0; i< num_dist; i++) {
    if (dist[i] < c) { c= dist[i]; }
  }
  return c;
}


struct state_t state =  {STATE_INITIAL, 0, 0, 0, 0, 0, 0};

void tick(struct input_t i)
{

  if (state.mode == STATE_INITIAL) {
    if (i.a_in > 0) {
      state.a= i.a_in;
      state.omega= i.w_in;
      state.mode= STATE_CRUISING;
    } else {
      state.a= 0; state.omega= 0;
    }
  } else if (state.mode == STATE_CRUISING) {
    if (SAFETY_DIST+ brk_dist(state.v+ state.a* DELTA_T)+
          dist_traveled(state.a, state.v)>= min_dist(i.L, NUM_DIST)) {
      /* start braking */
      state.a= -A_BRK;
      state.omega= 0;
      state.mode= STATE_BRAKING;
    } else {
      state.a= i.a_in;
      state.omega= i.w_in;
    }
  } else if (state.mode == STATE_BRAKING) {
    if (state.v == 0) {
      state.a= 0;
      state.omega= 0;
      state.mode= STATE_INITIAL;
    } else {
       state.a= -A_BRK;
       state.omega= 0;
    }
  }
  /* Order of assignments is important here */
  state.x += polar_to_cart_x(dist_traveled(state.a, state.v), state.phi);
  state.y += polar_to_cart_y(dist_traveled(state.a, state.v), state.phi);
  state.v = fmax(0, fmin(state.v+ state.a* DELTA_T, V_MAX));
  state.phi += state.omega* DELTA_T;
}

void loop()
{
  struct input_t i;

  while (1) {
    i.a_in= read_acceleration();
    i.w_in= read_steering();
    read_sensors(i.L);
    tick(i); 
    set_acceleration(state.a);
    set_steering(state.phi);
    sleep_ms(DELTA_T); // Let physics take its course.
  }

}
