module Queue(Qu, empty, enq, first, deq) where

import Data.List(intercalate)

-- A Queue is given by two lists
{-- q --}
data Qu alpha = Qu [alpha] [alpha]
{-- end --}

-- The empty Q
{-- empty --}
empty :: Qu alpha
empty = Qu [] []
{-- end --}

-- Enqueue
{-- enq --}
enq   :: alpha-> Qu alpha-> Qu alpha
enq x (Qu xs ys) = queue xs (x:ys)  
{-- end --}

-- Front of Q
{-- first --}
first :: Qu alpha-> alpha  
first (Qu [] _)     = error "Queue: first of empty Q"
first (Qu (x:xs) _) = x
{-- end --}

-- Dequeue
{-- deq --}
deq   :: Qu alpha-> Qu alpha
deq (Qu [] _ )     = error "Queue: deq of empty Q"
deq (Qu (_:xs) ys) = queue xs ys
{-- end --}

-- Check and restore invariant (smart constructor)
{-- queue --}
queue :: [alpha]-> [alpha]-> Qu alpha
queue [] ys = Qu (reverse ys) []
queue xs ys = Qu xs ys
{-- end --}


-- Equality for queues.
-- Two queues are equal iff their content is the same.
{-- eq --}
instance Eq alpha=> Eq (Qu alpha) where
  Qu xs1 ys1 == Qu xs2 ys2 =
    xs1 ++ reverse ys1 == xs2 ++ reverse ys2
{-- end --}

-- Show (needed with quickcheck to show failures)
-- Displays front of queue to the right, as on the slides
instance Show a=> Show (Qu a) where
  show (Qu xs ys) =
    "<"++ intercalate "->" (map show ys)
     ++ "->" ++ intercalate "->" (map show $ reverse xs) ++ ">"
