module QSort where

import System.CPUTime 
import System.Random
import Control.DeepSeq

import Data.List(partition)

-- Very short with list comprehension.
-- Also surprisingly efficient.
{-- qsort1 --}
qsort1 :: Ord alpha=> [alpha]-> [alpha]
qsort1 [] = []
qsort1 xs@(x:_) = qsort1 [y | y<- xs, y< x ]++ 
                         [x0| x0<- xs, x0 == x ]++
                  qsort1 [z | z<- xs,  z> x ]
{-- end --}

-- Lookes more efficient with partition -- only one recursion
qsort2 :: Ord alpha=> [alpha]-> [alpha]
qsort2 [] = []
qsort2 (x:xs) =
  let (leq, gt) = partition (x >) xs
  in  qsort2 leq ++ x: qsort2 gt 

-- Uses only one recursion and a custom 3-way splitting.
-- Fastest, in particular when compiled with optimization.
qsort3 :: Ord alpha=> [alpha]-> [alpha]
qsort3 [] = []
qsort3 (x:xs) =
  let (le, eq, gt) = split3 x xs [] [] []
  in  qsort3 le ++ (x: eq) ++ qsort3 gt

-- Split a list into every less, equal or greater as efficiently as possible.
split3 :: Ord alpha=> alpha -> [alpha]
                            -> [alpha]-> [alpha]-> [alpha]
                            -> ([alpha], [alpha], [alpha])
split3 x [] lt eq gt = (reverse lt, eq, reverse gt)
split3 x (y:ys) y1 y2 y3 =
   case y `compare` x of 
    LT -> split3 x ys (y:y1) y2 y3
    EQ -> split3 x ys y1 (y:y2) y3
    GT -> split3 x ys y1 y2 (y:y3)

-- Like qsort1, but list comprehension spelt out with filter, and
--   about as efficient
qsort4 ::  Ord alpha=> [alpha]-> [alpha]
qsort4 [] = []
qsort4 (x:xs) =
  qsort4 (filter (x>) xs) ++ (x: filter (x==) xs) ++ qsort4 (filter (x<) xs)

-- Again using filter, but not three-way splitting (like qsort2).
qsort5 :: Ord alpha => [alpha]-> [alpha]
qsort5 [] = []
qsort5 (x:xs) = qsort5 (filter (x>) xs) ++ (x: qsort5 (filter (x<=) xs))


{- Very simple timing -}

timeIt label num sortfun = do 
  ts <- sequence $ replicate num $ do
    ls <- sequence (replicate 100000 (randomIO :: IO Int))
    t0 <- getCPUTime
    t1 <- sortfun ls `deepseq` getCPUTime
    return $ fromIntegral (t1- t0)* 1e-9
  putStrLn $ label ++ " : " ++ show (sum ts/fromIntegral num) ++ " ms"

main = do 
  let runs = 20
  putStrLn $ "Results, averaged over "++ show runs ++ " runs:"
  timeIt "qsort1" runs qsort1
  timeIt "qsort2" runs qsort2
  timeIt "qsort3" runs qsort3
--  timeIt "qsort4" runs qsort4
--  timeIt "qsort5" runs qsort5
