module Labyrinth where

-- Modelling Labyrinths: a maze of twisty little passages, all alike
import Data.Maybe(fromJust,fromMaybe)
-- import Debug.Trace(trace)

-- The passageways in the labyrinth (nodes in the graph):
data Lab alpha = Dead alpha
           | Pass alpha (Lab alpha)
           | TJnc alpha (Lab alpha) (Lab alpha)

-- A path in the labyrinth (essentially, a list of node identifiers):
type Path alpha = [alpha]

-- A traversal: a successful path, or a failure:
type Trav alpha = Maybe [alpha]

-- Selector for the identity of a node
{-- node_id --}
nid :: Lab alpha -> alpha
nid (Dead i) = i
nid (Pass i _) = i
nid (TJnc i _ _)= i
{-- end --}

-- Add a node to a traversal, if successful:
{-- cons --}
cons :: alpha -> Trav alpha -> Trav alpha
cons _ Nothing      = Nothing
cons i (Just is) = Just (i: is)
{-- end --}

-- Add two traversals, disregarding first if it fails:
{-- select --}
select :: Trav alpha -> Trav alpha-> Trav alpha
select Nothing t = t
select t    _ = t
{-- end --}

-- Traversal of a cycle-free labyrinth to a given node.
{-- traverse_1 --}
traverse_1 :: (Show alpha, Eq alpha)=> alpha-> Lab alpha-> Trav alpha
traverse_1 t l 
  | nid l == t = Just [nid l]
  | otherwise = case l of
    Dead _ -> Nothing
    Pass i n -> cons i (traverse_1 t n)
    TJnc i n m -> cons i (select (traverse_1 t n) 
                                 (traverse_1 t m))
{-- end --}

-- Traversal of a labyrinth with cycles to a given node.
{-- traverse_2 --}
traverse_2 :: Eq alpha=> alpha-> Lab alpha-> Trav alpha
traverse_2 t l = trav_2 l [] where
  trav_2 l p
    | nid l == t = Just (reverse (nid l: p))
    | elem (nid l) p = Nothing
    | otherwise = case l of
      Dead _ -> Nothing
      Pass i n -> trav_2 n (i: p)
      TJnc i n m -> select (trav_2 n (i: p)) (trav_2 m (i: p))
{-- end --}


-- A nicer show instance
instance (Eq alpha, Show alpha)=> Show (Lab alpha) where
  show l = draw [l] [] where
    draw [] _ = ""
    draw (n:ns) visited
      | nid n `elem` visited = draw ns visited
      | otherwise = case n of 
        Dead i -> show i ++ " -->\n" ++ draw ns (i: visited)
        Pass i l -> show i ++ " --> "++ show (nid l)++ "\n" ++
                      draw (l:ns) (i: visited)
        TJnc i l1 l2 -> show i ++ " --> "++ show (nid l1) ++ " "++ show (nid l2)++ "\n"++
                      draw (l1: l2: ns) (i: visited)


-- Reading labyrinths

-- Construct a labyrinth
{-- makeLab --}
makeLab :: (Read alpha, Eq alpha, Show alpha) => [(String, [String])]-> alpha-> Lab alpha
makeLab vs id =
  let mk_lab map [] = []
      mk_lab map ((s, ts):rest) =
        let src = read s
            get v = fromJust (lookup (read v) map)   
            l = case ts of
                  [] -> Dead src         
                  [v] -> Pass src (get v)
                  [v1, v2] -> TJnc src (get v1) (get v2)
                  _ -> error ("Too many edges from "++ show src++ ": "++ show ts)
        in  (src, l): mk_lab map rest
      map = mk_lab map vs
  in fromMaybe (error ("Undefined label: "++ show id)) (lookup id map)
{-- end --}

-- Parse from string
{-- parseString --}
parseString :: String-> [(String, [String])]
parseString = map parseLine . lines 
{-- end --}

-- Parse a single line of the form "<lab> --> <lab>*"
{-- parseLine --}
parseLine :: String-> (String, [String])
parseLine str = case words str of
  [] -> error "parseLine: empty line"
  [_] -> error ("parseLine: line too short, "++ show str)
  hd :ar :tl | ar /= "-->" -> error ("parseLine: expected -->, got "++ ar)
             | otherwise -> (hd, tl)
{-- end --}

-- Read a whole labyrinth from a string
{-- readLab --}
readLab :: (Read alpha, Eq alpha, Show alpha)=> String-> alpha-> Lab alpha
readLab = makeLab . parseString
{-- end --}
