module Colours where

data Colour = RGB Int Int Int
            deriving (Show, Eq)

red :: Colour -> Int
red (RGB r _ _) = r 

green :: Colour -> Int
green (RGB _ g _) = g

blue :: Colour -> Int
blue (RGB _ _ b) = b

{-
-- Definition des Datentypen mit Selektoren:
data Colour = RGB { red :: Int, green :: Int, blue :: Int }
     	    deriving (Eq, Show) 
-}
-- Some example colours

cyan :: Colour
cyan = RGB 0 255 255       -- 0x00FFFF

magenta :: Colour
magenta = RGB 255 0 255    -- 0xFF00FF

yellow :: Colour
yellow = RGB 255 255 0     -- 0xFFFF00

violet :: Colour
violet = RGB 238 130 238   -- 0xEE82EE

salmon :: Colour
salmon = RGB 250 128 114   -- 0xFA8072

goldenrod :: Colour
goldenrod = RGB 218 165 32 -- 0xDAA520

-- More example functions:

{-- brightness --}
brightness :: Colour-> Float
brightness (RGB r g b) = fromIntegral (max (max r g) b) / 255
{-- end --}

{-- adjust --}
adjust :: Colour-> Float-> Colour
adjust (RGB r g b) f = RGB (conv r) (conv g) (conv b) where
  conv colour = min (round (fromIntegral colour* f)) 255
{-- end --}

-- With error checking:
darken1 :: Colour-> Float-> Colour
darken1 (RGB r g b) f 
   | f < 0     = RGB r g b
   | otherwise = RGB (conv r f) (conv r g) (conv r b) where
       conv colour factor = round (max (fromIntegral r* f) 255)

-- Notice how we construct RGB r g b which is exactly the value we have
-- been given. To avoid  this, we can name the whole pattern (binding it
-- to the name c):
darken2 :: Colour-> Float-> Colour
darken2 c@(RGB r g b) f 
   | f < 0     = c
   | otherwise = RGB (conv r f) (conv r g) (conv r b) where
       conv colour factor = round (max (fromIntegral r* f) 255)

