{-# LANGUAGE OverloadedStrings, QuasiQuotes, TemplateHaskell #-}
module View where

import Data.String
import Data.Text.Lazy hiding (head, span)
import Control.Exception(try)
import Text.Read(readMaybe)
import Text.Blaze
import Text.Blaze.Html
import Text.Blaze.Html5 hiding (map, label)
import Text.Blaze.Html5.Attributes hiding (title, form, item, span)
import Prelude hiding (div, id, head, span, drop, length)
import qualified Prelude (div)
import Control.Monad

import Shop.Backend

-- import Types

page :: String -> Html -> Html
page t c = docTypeHtml $ do
    head $ do
        title (toHtml t)
        link ! rel "stylesheet" ! type_ "text/css" ! href "/static/css/bob.css"
    body c

homePage :: Text-> [(Posten, Int)]-> Html
homePage einkauf ps = page "Home" $ do
  h1 "Willkommen bei Bob's Delikatessen-Handlung"
  p "Heute im Angebot:"
  div ! class_ "indented" $ table $ forM_ ps $ \ (Posten a m, p)-> 
    tr $ do
      td (showHtml a)
      td (showEuro p)
      td (toHtml (showMenge a m))
  p $ a ! href (lazyTextValue einkauf) $ text "Come in and find out."
    

errorPage :: String -> Html
errorPage e = page "Error" $ do
  h1 "ERROR."
  p $ do toHtml $ "An error has occured: " ++ e
         text "Please accept out apologies."

artikelListe :: [(Posten, Int)]-> Int-> Html
artikelListe as summe =
  div ! class_ "boxed" ! class_ "greyed" $ table $ do
    forM_ as $ \ (Posten a m, p) ->
      tr $ do td $ showHtml a
              td ! class_ "zahl" $ toHtml $ showMenge a m
              td ! class_ "zahl" $ showEuro p
    tr $ do td "Summe"
            td ""
            td ! class_ "summe" $ showEuro summe
  
shoppingPage :: String-> String-> [Text]-> [(Posten, Int)]-> Int-> [Posten]-> Html
shoppingPage post kasse errors ekwg summe angebot = page "Einkaufen" $ do
  h1 "Willkommen im Einkaufsparadies"
  div ! id "shopping-errors" $ ul $ forM_ errors $ \e -> li $ toHtml e
  h2 "Ihr Einkaufswagen"
  artikelListe ekwg summe
  h2 "Das Angebot" 
  form ! action (fromString post) ! method "POST" ! class_ "indented" $ do
    table $ do
      forM_ angebot $ \ (Posten a v) -> tr $ do
        td $ span $ toHtml (artikelName a)
        let nmid = lazyTextValue $ artikelId a
        td $ input ! class_ "input-menge" ! type_ "text" ! id nmid ! name nmid 
        td $ span $ toHtml (showMenge' (mengeneinheit a))
        td $ if v < mindestvorrat a then
          span ! class_ "warning" $ toHtml $ "Nur noch "++ showMenge a v ++ " vorrätig!"
          else text ""
    p $ do button ! type_ "submit" $ text "In den Einkaufswagen"
           button ! type_ "reset" ! class_ "right-button" $ text "Abbrechen"
  p $ a ! href (fromString kasse) $ button $ text "Zur Kasse"


checkoutPage :: String-> String-> [(Posten, Int)]-> Int-> Html
checkoutPage buy abort ekwg summe = page "Kasse" $ do
  h1 "Kasse 1"
  h2 "Ihr Kassenzettel"
  artikelListe ekwg summe
  a ! href (fromString buy) $ button $ text "Zahlen"
  a ! href (fromString abort) $ button  ! class_ "right-button" $ text "Abbruch"

thankYouPage :: Text-> Html
thankYouPage msg = page "Thank You" $ do
  h1 "Vielen Dank!"
  div $ lazyText msg
  div $ text "Besuchen Sie uns bald wieder."
  a ! href "/" $ text "Zurück."

artikelName :: Artikel-> String
artikelName (Apfel Boskoop)= "Apfel (Boskoop)"
artikelName (Apfel CoxOrange)= "Apfel (Cox Orange)"
artikelName (Apfel GrannySmith) = "Apfel (Granny Smith)"
artikelName Eier = "Eggses"
artikelName (Kaese Gouda) = "Gouda"
artikelName (Kaese Appenzeller) = "Appenzeller"
artikelName Schinken = "Schinken"
artikelName Salami = "Salami"
artikelName (Milch Bio) = "Bio-Milch"
artikelName (Milch Konv) = "Milch"

artikelIdPrefix = "aid-"

artikelId :: Artikel-> Text
-- artikelId a = artikelIdPrefix ++ (show $ fromEnum a)
artikelId a = artikelIdPrefix `append` (pack $ show $ fromEnum a)

-- Used to retrieve the article from the submitted form data
idToArtikel :: Text-> Maybe Artikel
idToArtikel t = do
  str <- stripPrefix artikelIdPrefix t
  id <- readMaybe (unpack str)
  if minBound <= id && id <= maxBound
    then Just (toEnum id)
    else Nothing
 

error_text :: Error a-> Text
error_text (EkwgExists a) = "Existiert schon"
error_text (EkwgNotFound _) =  "Ihr Einkauf wurde abgebrochen"
error_text (AtMostAvailable a m) =
  pack $ artikelName a++ ": höchstens "++ showMenge a m++ " verfügbar."



showMenge :: Artikel-> Int-> String
showMenge a i = show i ++ " " ++ showMenge' (mengeneinheit a)

showMenge' :: Menge-> String
showMenge' Stueck = "Stk"
showMenge' Gramm  = "g."
showMenge' Liter  = "l"

showEuro :: Int-> Html
showEuro i = do
  toHtml ("€ " :: String)
  toHtml (show (i `Prelude.div` 100) ++ "." ++
          show ((i `Prelude.div` 10) `mod` 10) ++
          show (i `mod` 10))

class ShowHtml a where
  showHtml :: a-> Html

instance ShowHtml Artikel where
  showHtml = toHtml . artikelName

instance ShowHtml Posten where
  showHtml (Posten a m) = toHtml $ artikelName a ++ " " ++ showMenge a m
