{- # LANGUAGE OverloadedStrings, QuasiQuotes, TemplateHaskell # -}
module Shop.Backend(
   module Shop.Artikel
 , Shop
 , Res(..)
 , Error(..)
 , initial
 , angebot
 , anliefern
 , neuerEkwg
 , imEkwg
 , inDenEkwg
 , ekwgBezahlt
 , ekwgZurueck
 ) where

import qualified Data.Map as M
import Data.Maybe(fromJust)

import Shop.Artikel
import qualified Shop.Lager as L
-- import Shop.Einkaufswagen

type Ekwg = [Posten]

data Shop a = Shop { lager :: L.Lager
                   , ekwg  :: M.Map a Ekwg
                   }

-- Mgliche Fehler
data Error a = EkwgExists a         -- Ekwg existiert schon
             | EkwgNotFound a       -- Kein Ekwg gefunden
             | AtMostAvailable Artikel Int -- Hchstverfgbarkeit, wurde berschritten

type Res a b = Either (Error a) b

fromList :: [(Artikel, Int)] -> L.Lager
fromList =
  foldr (\ (a, m)-> L.einlagern (L.Posten a m)) L.leeresLager

initial :: Shop a
initial = Shop (fromList is) M.empty where
  is = [ (Apfel Boskoop, 100) 
       , (Schinken, 5000)
       , (Milch Bio, 60)
       , (Apfel Boskoop, 400)
       , (Milch Konv, 100)
       , (Salami, 500)
       ]

-- Alle vorraetigen Artikel
angebot :: Shop a -> Res a [Posten]
angebot = return . L.bestand . lager 

anliefern :: Artikel-> Int-> Shop a-> Res a (Shop a)
anliefern a m s = return $ s{lager = L.einlagern (Posten a m) (lager s)}

belegt :: Ord a=> a-> Shop a-> Bool
belegt eid s = M.notMember eid (ekwg s)

neuerEkwg :: Ord a=> a-> Shop a-> Res a (Shop a)
neuerEkwg eid s
  | M.member eid (ekwg s) = Left $ EkwgExists eid
  | otherwise = return $ s { ekwg = M.insert eid [] (ekwg s) }

get_ekwg :: Ord a=> a-> Shop a-> Res a Ekwg
get_ekwg eid s =
  case M.lookup eid (ekwg s) of
    Just ek -> return ek
    Nothing -> Left $ EkwgNotFound eid

imEkwg :: Ord a=> a-> Shop a-> Res a ([(Posten, Int)], Int)
imEkwg eid s = do
  ek <- get_ekwg eid s
  let as = map (\p-> (p, preis p)) ek
  return $ (as, sum $ map snd as)

inDenEkwg :: Ord a=> a-> Artikel-> Int-> Shop a-> Res a (Shop a)
inDenEkwg eid a m s = do
  ekw <- get_ekwg eid s
  let p = Posten a m
  let l = L.verfuegbar a (lager s)
  if m <= l then (return $ Shop { lager= L.entnehmen p (lager s)
                                , ekwg = M.insert eid (p:ekw) (ekwg s)
                                })
     else Left $ AtMostAvailable a l

ekwgBezahlt :: Ord a=> a-> Shop a-> Res a (Shop a)
ekwgBezahlt eid s = return $ s {ekwg = M.delete eid (ekwg s)}

ekwgZurueck :: Ord a=> a-> Shop a-> Res a (Shop a)
ekwgZurueck eid s = do
  ekw <- get_ekwg eid s
  return $ Shop { lager= foldr L.einlagern (lager s) ekw
                , ekwg = M.delete eid (ekwg s)
                }
