{-# LANGUAGE OverloadedStrings #-}
module Controller where

import Control.Concurrent.MVar
import Control.Monad
import Control.Monad.IO.Class (liftIO)
import Data.UUID
import System.Random(randomIO)
import Text.Read(readMaybe)
import Data.Maybe(isJust)

import qualified Text.JSON as J

import Text.Blaze.Html hiding (text)
import Text.Blaze.Html.Renderer.Text

import Web.Scotty
import Data.Text.Lazy(Text,pack,unpack)

import Shop.Backend(Artikel,Posten(..))
import qualified Shop.Backend as Shop
import View

type EID   = UUID
type M     = Shop.Shop EID
type Res a = Either (Shop.Error EID) a

-- Finish action with the error page
exit :: String-> ActionM ()
exit e = html $ renderHtml $ errorPage e -- ; finish ??

-- Run a function which changes the state of the shop.
-- May return an explicit error
shop :: MVar M -> (M-> Res M)-> ActionM (Maybe (Shop.Error EID))
shop mv act = do
  s<- liftIO $ takeMVar mv
  case act s of
    Right s' -> do liftIO $ putMVar mv s'
                   return Nothing
    Left  e  -> do liftIO $ putMVar mv s
                   return $ Just e -- exit (error_text e)

-- Run a function which merely reads the state of the shop
shop' :: MVar M-> (M-> Res a)-> ActionM a
shop' mv act = do
  s<- liftIO $ readMVar mv
  case act s of
    Right a -> return a
    Left  e -> raise (error_text e)

parseEID :: Text-> ActionM EID
parseEID nm = do
  estr <- param nm
  case fromString estr of
    Just uuid-> return uuid
    Nothing  -> raise $ pack $ "Invalid EID "++ estr

-- Extract one parameter if present
extractParam :: Read a=> Text-> ActionM (Maybe a)
extractParam nm =
  (do v<- param nm; return $ readMaybe v)
  `rescue` \_ -> return Nothing

-- Extract all article-id parameters; return those which are present.
parseArtikelParams :: ActionM [(Artikel, Int)]
parseArtikelParams = do
  paraVals <- forM Shop.alleArtikel $ \art-> do 
    val <- extractParam (artikelId art)
    -- liftIO $ putStrLn $ "Extracting "++ show (artikelId art)++ " = " ++ show val
    return (art, val)
  return [(a, v) | (a, Just v) <- paraVals]

-- Liste von Posten in Einkaufskorb
einkaufen :: MVar M-> EID-> [(Artikel, Int)]-> ActionM [Text]
einkaufen _ _ [] = return []
einkaufen sv eid ((a, m):as) = do
  eh <- shop sv $ Shop.inDenEkwg eid a m
  case eh of
    Nothing -> einkaufen sv eid as
    Just e@(Shop.AtMostAvailable _ _) -> do
      r <- einkaufen sv eid as
      return $ error_text e : r
    Just e -> raise (error_text e)


controller :: MVar M -> ScottyM ()
controller sv = do 
  get "/" $ do
    let p2a (Posten a _) = let m1 = Shop.einheitsmenge a
                               q  = Posten a m1
                           in  (q, Shop.preis q)
    ps <- shop' sv $ Shop.angebot
    html $ renderHtml $ homePage "/einkauf/neu/" (map p2a ps)
  get "/einkauf/neu/" $ do
    eid <- liftIO $ randomIO
    s2 <- shop sv $ Shop.neuerEkwg eid
    redirect $ pack ("/einkauf/" ++ toString eid)
  get "/einkauf/:eid" $ do
    eid <- parseEID "eid"
    (ekwg, total) <- shop' sv $ Shop.imEkwg eid
    as <- shop' sv $ Shop.angebot
    html $ renderHtml $ shoppingPage ("/einkauf/"++ toString eid)
                                     ("/einkauf/" ++ toString eid++ "/kasse")
                                     [] ekwg total as
  post "/einkauf/:eid" $ do
    eid <- parseEID "eid"
    ams <- parseArtikelParams
    errs <- einkaufen sv eid ams
    (ekwg, total) <- shop' sv $ Shop.imEkwg eid
    as <- shop' sv $ Shop.angebot
    html $ renderHtml $ shoppingPage ("/einkauf/" ++ toString eid)
                                     ("/einkauf/" ++ toString eid++ "/kasse")
                                     errs ekwg total as
    -- would be better to redirect to get above... !
  get "/einkauf/:eid/kasse" $ do
    eid <- parseEID "eid"
    (ekwg, total) <- shop' sv $ Shop.imEkwg eid
    html $ renderHtml $
      checkoutPage ("/einkauf/"++ toString eid++ "/kaufen")
                   ("/einkauf/"++ toString eid++ "/abbruch")
                   ekwg total
  get "/einkauf/:eid/kaufen" $ do
    eid <- parseEID "eid"
    shop sv $ Shop.ekwgBezahlt eid
    html $ renderHtml $ thankYouPage "Vielen Dank für Ihren Einkauf!" 
  get "/einkauf/:eid/abbruch" $ do
    eid <- parseEID "eid"
    shop sv $ Shop.ekwgZurueck eid
    html $ renderHtml $ thankYouPage "Vielen Dank für Ihren Besuch."
  post "/lieferung" $ (do
    ams <- parseArtikelParams
    shop sv $ \s-> foldM (\s0 (a, m) -> Shop.anliefern a m s0) s ams
    text "OK")
    `rescue`
    \e -> text $ mappend "ERROR: " e
  get "/lager" $ do
    as <- shop' sv $ Shop.angebot
    -- [Posten] --> [(String, Int)] --> JSON
    text $ pack $ J.encode $ J.toJSObject $
      map ( \ (Posten a m)-> (unpack $ artikelId a, m)) as
