module QSort where

import Data.List(partition)

-- Very short with list comprehension.
-- Also surprisingly efficient.
qsort1 :: Ord alpha=> [alpha]-> [alpha]
qsort1 [] = []
qsort1 xs@(x:_) = qsort1 [y | y<- xs, y< x ]++ 
                         [x0| x0<- xs, x0 == x ]++
                  qsort1 [z | z<- xs,  z> x ]

-- Lookes more efficient with partition -- only one recursion
-- But keeps less and equal in one list, as opposed to above, and hence is
-- *very* inefficient
qsort2 :: Ord alpha=> [alpha]-> [alpha]
qsort2 [] = []
qsort2 (x:xs) =
  let (leq, gt) = partition (x >) xs
  in  qsort2 leq ++ x: qsort2 gt 

-- Uses only one recursion and a custom 3-way splitting, but is about
--  as efficient as qsort1 (slightly faster if compiled)
qsort3 :: Ord alpha=> [alpha]-> [alpha]
qsort3 [] = []
qsort3 (x:xs) =
  let (le, eq, gt) = split3 x xs [] [] []
  in  qsort3 le ++ (x: eq) ++ qsort3 gt

-- Split a list into every less, equal or greater as efficiently as possible.
split3 :: Ord alpha=> alpha -> [alpha]
                            -> [alpha]-> [alpha]-> [alpha]
                            -> ([alpha], [alpha], [alpha])
split3 x [] lt eq gt = (reverse lt, eq, reverse gt)
split3 x (y:ys) y1 y2 y3 =
   case y `compare` x of 
    LT -> split3 x ys (y:y1) y2 y3
    EQ -> split3 x ys y1 (y:y2) y3
    GT -> split3 x ys y1 y2 (y:y3)

-- Like qsort1, but list comprehension spelt out with filter, and
--   about as efficient
qsort4 ::  Ord alpha=> [alpha]-> [alpha]
qsort4 [] = []
qsort4 (x:xs) =
  qsort4 (filter (x>) xs) ++ (x: filter (x==) xs) ++ qsort4 (filter (x<) xs)

-- Again using filter, but not three-way splitting.
--   Like qsort2, *very* inefficient. 
qsort5 :: Ord alpha => [alpha]-> [alpha]
qsort5 [] = []
qsort5 (x:xs) = qsort5 (filter (x>) xs) ++ (x: qsort5 (filter (x<=) xs))


{- Efficiency was "measured" very roughly with ghci:


-}
