" Vim filetype plugin file
" Language: C and C++ (common stuff)
" Maintainer:   Gabriel Zachmann (www.gabrielzachmann.org)
" Last Change:  2002 Aug 13

augroup c
	au!

	" Read the c template automagically when starting a new c file;
	" where CLASSNAME is replaced by actual name.
	autocmd BufNewFile *.c		r <sfile>:p:h/template-file.c
	autocmd BufNewFile *.h		r <sfile>:p:h/template-class.h|exec "%s/CLASSNAME/".expand("%:t:r")."/g"
	autocmd BufNewFile *.cpp	r <sfile>:p:h/template-class.cpp|exec "%s/CLASSNAME/".expand("%:t:r")."/g"

	autocmd VimEnter *.cpp,*.CC,*.cc,*.cxx,*.c,*.l,*.y	call s:Start_Cscope()

augroup end


"   Set errorformat and makeprg
if $HOSTTYPE == "iris4d"
	compiler CC
elseif $HOSTTYPE =~ ".*-linux"
	compiler gcc
endif

"  Options for C editing
setl formatoptions+=croq formatoptions-=t
setl cindent cinoptions+=t0,(0,p0,g0,h0 cinwords+={

"  Search path for include_search and tags
let s:basedir = substitute( expand("%:p"), '^\(.*/work/[^/]*\).*$', '\1', "" )
let &l:path = ".," . s:basedir . "/**" . ",/usr/include/**,/home/work/Zach/include/**"
let &l:tags = s:basedir . "/CollDet/tags," . "/home/work/Zach/OpenSG/tags," . "./tags,tags,../tags"

"  Abbreviations of often typed C structs
iab pr printf(
iab pu puts(
iab fpr fprintf(stderr,
iab uint unsigned int

"  Comment out a highlighted block with #if0
vmap <buffer> <F7> <ESC>'>o#endif<ESC>'<O#if 0<ESC>j0

"	g-^] is cooler
nmap <buffer> <C-]>	g<C-]>

"  Function keys for Quickfix
nmap <buffer> <F5>	:make<CR>
nmap <buffer> <F6>	:cn<CR>

"  Menu entries

"  generates list of functions
nmenu Misc.List\ all\ functions			:call ListAllFunctions()<CR>
"  jumps to one of the functions of the above list
nmenu Misc.Jump\ to\ function			:call JumpToFunction( input("\rfunction nr.: ") )<CR>

"  show a man page inside vim.
nmenu &Man.&Word\ under\ Cursor<Tab>K	:call <SID>ShowPage(expand("<cword>"),"man")<CR>
nmenu &Man.&Input						:call <SID>ShowPage(input("man: "),"man")<CR>

"  menu for compiling
amenu Tools.Make\ clean					:make clean<CR>


"
"  CVS support
"
" from ideas from http://www.multimania.com/phic/vim/rc (rcs only).
"

nmenu CVS.Update\ this\ file		:update<CR>:!cvs update % <CR>:e<CR>
nmenu CVS.Show\ log					:call <SID>ShowPage( expand("%:t"), "cvs log" )<CR>
nmenu CVS.Show\ diff				:update<CR>:!cvsgdiff % <CR>

" make syntax highlighting for log pages permanent, so we can come back
augroup c
autocmd! BufNewFile,BufReadPost cvs\ log\ *
autocmd  BufNewFile,BufReadPost cvs\ log\ * 	:runtime my/rlog.vim
augroup end


"
" Show a man page inside vim.
" Includes syntax highlighting,
" and when you type 'K' on a word like printf(2) (e.g., in the "see also"
" section) then man will search only that section.
"
" Ideas from http://www.multimania.com/phic/vim/functions.vim.
" Modified by me (GZ).
" I'm afraid you need to type <cr> after 'man' has found the man page :(
" (and don't be alarmed if it says "1 buffer unloaded" :)
" ~/.gvimrc contains some autocmd's to add the man page to the Man-menu.
"

nmap <buffer> K :call <SID>ShowPage(expand("<cword>"),"man")<CR>




"
"   Functions
"

" safeguard
if !exists("*s:Start_Cscope")



" Init cscope database.
" Assumes that it is called at the VimEnter event.
function s:Start_Cscope()
	if ! has("cscope")
		return
	endif

	" find cscope database;
	" it's in the base directory of each project
	if filereadable( s:basedir . "cscope.db" )
		let cscopedir = s:basedir
	else
		echohl WarningMsg
		echo "No cscope database found"
		echohl None
		return
	endif

	execute "cscope add " . cscopedir . "cscope.db"
	" 'cscope add' must come first
	set cscopeprg="/sgi/bin/cscope -d -P " . cscopedir

	set cscopetagorder=0 cscopeverbose
	"set cscopetag

	" cscope menu
	nmenu &Cscope.Go\ to\ Tag<Tab>Ctrl+]						g<C-]>
	nmenu &Cscope.Where\ is\ this\ symbol\ &used				:cs find s <C-R>=expand("<cword>")<CR><CR>
	nmenu &Cscope.Where\ is\ this\ symbol\ &defined				:cs find g <C-R>=expand("<cword>")<CR><CR>
	nmenu &Cscope.Find\ &assignments\ to\ this\ variable		:cs find t <C-R>=expand("<cword>")<CR><CR>
	nmenu &Cscope.Find\ &callers\ of\ this\ function			:cs find c <C-R>=expand("<cword>")<CR><CR>
	nmenu &Cscope.Find\ &functions\ called\ by\ this\ function	:cs find d <C-R>=expand("<cword>")<CR><CR>
	nmenu &Cscope.Find\ files\ &including\ this\ file			:cs find i <C-R>=expand("<cword>")<CR><CR>
	nmenu &Cscope.&Add\ database								:execute "cs add " . input("Filename: ")<CR>
	nmenu &Cscope.&Show\ databases								:execute "cs show"<CR>

endfunction



" Functions: ListAllFunctions, JumpToFunction
" Usage:
"   Call the former to list all C/C++ functions in current buffer;
"   call the latter, to jump to particular function (by number).
"   JumpToFunction(0) jumps to function # v:count,
"   JumpToFunction(x) jumps to function # x.
"
"   It should work as well for other languages if the source code
"   adheres to the following assumption:
"                   # <- empty line
"   foo( ....
"   {               # single '{' at the beginning of the line
"
" Authors:
"   Gabriel Zachmann <zach@cs.uni-bonn.de> (quick-n-dirty attempt)
"   Marius Gedminas <mgedmin@takas.lt> (quick-n-dirty try to improve it)
" Bugs:
"   annoying but harmless error message
"   forgets cursor column (how do I do `z in ex?)
"   what highlight group does the [I vim command use?

function s:Pad(str, len)
	return strpart( "                                              ",
\					1, a:len - strlen(a:str) )
\		   . a:str
endfunc

function! s:ListAllFunctions()

    let w = &wrapscan
    setl nowrapscan          " don't let // wrap around

    mark z                  " remember cursor pos
    1                       " go to first line in buffer
    let v:errmsg = ""
    let cnt = 0

    while 1
        /^{\s*$/            " search for next function body start

        if v:errmsg != ""
            break
        endif

        let cnt = cnt + 1
        let b = line(".")
        ?^\s*$?             " go back to line before function declaration
        +                   " go to first line of function declaration

		if getline(".") =~ 'typedef' || getline(".") =~ '='
			execute b
			+				" skip typedef's and array-of-string init's
			continue
		endif

        echon  s:Pad(cnt, 3) ": "
        echohl MoreMsg
        echon  s:Pad(line("."), 4)
        echohl None
        echon  " " getline(".") "\n"
        +
        while line(".") < b " print lines
            echon  "         "
            echon  " " getline(".") "\n"
            +
        endwhile

        +
    endwhile

    'z                      " restore cursor pos
    let &l:wrapscan = w       " restore option variable
endfunc


function s:JumpToFunction( nr )

    let w = &wrapscan
    setl nowrapscan          " don't let // wrap around

    mark z                  " remember cursor pos
    1                       " go to first line in buffer
    let v:errmsg = ""
    let cnt = 0
	let nr = a:nr
    if nr < 1
		let nr = v:count
		if nr < 1
			let nr = 1
		endif
    endif

    while 1
        /^{\s*$/            " search for next function body start

        if v:errmsg != ""
            'z              " restore cursor pos
            break
        endif

        let cnt = cnt + 1
        if cnt == nr
            ?^\s*$?             " go back to line before function declaration
            +                   " go to first line of function declaration
            break
        endif
    endwhile

    let &l:wrapscan = w       " restore option variable
endfunction



"
" Unload the buffer whose name contains "cmd " (space!)
"

function s:HidePage(cmd)
	let i = 1
	while i <= bufnr("$")
		if bufname(i) =~ a:cmd." "
		"&& bufwinnr(bufname(i))>0 oder so aehnlich, erst ab vim-5.4
			execute "bunload!" i
		endif
		let i = i + 1
	endwhile
endfunction


"
" Execute 'cmd' and open a new buffer for it.
" 'whichpage' will be appended to 'cmd' to construct a unique buffer name.
" If the buffer is already in the list, 'cmd' ist not executed but that
" buffer is loaded and a window opened for it.
" A new buffer will be put in /tmp/$USER/.
" Special code is provided if cmd="man".
"

function s:ShowPage(whichpage, cmd)
	" modify the shortmess option:
	" don't give the "ATTENTION" message when an existing swap file is found.
	let save_shortmess = &shortmess
	setl shortmess+=A

	call s:HidePage(a:cmd)

	if a:whichpage =~ "(*)" && a:cmd == "man"
		" extract section
		let section = substitute( a:whichpage, "[^(]*(\\(.*\\))$", "\\L\\1", "")
		" on SGI, 'man 3s printf' works, but 'man 3S printf' doesn't, grmbl
		let whichpage = substitute( a:whichpage, "(.*)$", "", "" )
	else
		let whichpage = a:whichpage
		let section = ""
	endif

	" construct unique buffer name
	let bufnam = escape( a:cmd . ' ' . whichpage, ' ' )

	" if the buffer already exists, go to it
	let i =1
	while i <= bufnr("$")
		if bufname(i) == bufnam."$"
			let &shortmess = save_shortmess
			execute "sb" i
			return
		endif
		let i = i + 1
	endwhile

	" there's no buffer "cmd"."whichpage" yet; create a new one
	if exists("$USER")
		let tmpdir = "/tmp/" . $USER . "/"
		if ! isdirectory(tmpdir)
			execute "!mkdir" tmpdir
			if ! isdirectory(tmpdir)
				echo "Failed to create" tmpdir "!"
				return
			endif
		endif
	else
		let tmpdir = "/tmp/"
	endif
	execute 'new ' . tmpdir . bufnam

	" read the man page in ('col -b' is there in case of cmd="man")
	execute 'r!' a:cmd section whichpage '| col -b'
	" write it so we can come back later without re-reading it
	w

	" set some options
	setl ts=8
	setl nomodified readonly
	"set iskeyword+=(,)

	" go to the beginning of the buffer
	execute "normal 1G"

	" restore the shortmess option
	let &l:shortmess = save_shortmess
endfunction


" end of function guard
endif

