#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>

#include "igc.h"
#include "ascii.h"
#include "parse.h"
#define GLOBALS
#include "numbers.h"


#define RCNAME "/.igcrc"


#define ATRC 1
#define NOTATRC 2
#define BOTH (ATRC | NOTATRC)

char login[20] = "";
char password[20] = "";

char friend[MAXFRIENDS][MAXNAME];	/* friends list */
int nfriends = 0;

message mesg;			/* structure for getting parsed info from
				 * server */

char prefix[21];		/* Text prefix for each outgoing line */
int xcur = -1, ycur = -1;	/* Cursor locations on board */
int boardon = 0;		/* Is the board being displayed? */
int boardmode = 0;		/* Are we on the board? */
int beepcount = 1;		/* Number of beeps to send */
int saybeep = 0;		/* Beep on say strings? */
int tellbeep = 0;		/* Beep on tell strings? */
int friendfilter = 0;		/* Filter login messages for friends? */
int displayintro = 1;		/* Display intro before log in */

int ingame = -1;		/* Game which is displayed, -1 for none */
int observing = 0;		/* Are we observing a game? */
int justpeeked = 0;		/* Fixup flag for game number after peek */
int needrestore = 0;		/* Need to restore board? (After status) */
int bschange = 0;		/* boardsize changed?  Used with needrestore */
extern int boardsize;

char local[1000], *loc;		/* Buffer for characters typed by local user */

struct {
   char white[21], wrank[10], black[21], brank[10];
   int hcap;
   float komi;
   int bbyo, wbyo, wtime, btime;
}  curgame;			/* Information about current game */


int (*whosort) ();
int (*gamessort) ();

#define MAXSITES 10

struct {
   char name[50];
   char site[100];
   int port;
}  sitetable[MAXSITES] =
{
   { "igs", "igs.joyjoy.net", 6969 },
   { "nngs", "nngs.cosmic.org", 9696 },  
   { "", "", 6969 },
   { "", "", 6969 },
   { "", "", 6969 },
   { "", "", 6969 },
   { "", "", 6969 },
   { "", "", 6969 },
   { "", "", 6969 },
   { "", "", 6969 }
};

int do_log = 0;



void myexit( int ex)
{
    if (boardon)
	endAscii();
    else
	setecho(1);
    if ( log )
	fclose( log );
    exit(ex);
}


void handleterm()
{
   close_connection();
   myexit(0);
}


char *getpassword(repeat)
int repeat;
{
   char trash[20];
   if (repeat || !strlen(password)) {
      printf("Password: ");
      for (;;) {
	 fgets(password, sizeof(password), stdin);
	 putchar('\n');
	 if (feof(stdin))
	    handleterm();
	 if (password[strlen(password) - 1] != '\n') {
	    password[strlen(password) - 1] = '\n';
	    do {
	       fgets(trash, sizeof(trash), stdin);
	    } while (trash[strlen(trash) - 1] != '\n');
	 }
	 if (strstr(password, "helpigc"))
	    printf("You are entering your password for the Internet Go Server.  You will be\nprompted twice for your password if you are creating a new account.  Enter\nthe same password twice in this case.\nPassword: ");
	 else if (*password=='\n')
	    printf("Empty password not allowed.\nPassword: ");
	 else
	    break;
      }
   }
   return password;
}


char *getloginname(repeat)
int repeat;
{
   char trash[20];
   if (repeat == 1)
      printf("Login failed.\n");
   if (repeat || !strlen(login)) {
      printf("Login: ");
      for (;;) {
	 fgets(login, sizeof(login), stdin);
	 if (feof(stdin))
	    handleterm();
	 if (login[strlen(login) - 1] != '\n') {
	    login[strlen(login) - 1] = '\n';
	    do {
	       fgets(trash, sizeof(trash), stdin);
	    } while (trash[strlen(trash) - 1] != '\n');
	 }
	 if (strstr(login, "helpigc"))
	    printf("You are logging into the Internet Go server.  Type a login name now.\nLogin: ");
	 else if (*login == '\n')
	    printf("Blank names not allowed.\nLogin: ");
	 else
	    break;
      }
   } else
      printf("Logging on as %s", login);
   return login;
}



void dobeep()
{
   char ch;
   for (ch = beepcount; ch--;)
      putchar(7);
}


void readrc()
{
   struct passwd *pw;
   char rcfilename[1000];
   FILE *rcfile;
   char line[1000], saveline[1000];

   pw = getpwuid(geteuid());
   strcpy(rcfilename, pw->pw_dir);
   strcat(rcfilename, RCNAME);
   if (rcfile = fopen(rcfilename, "rt")) {
      while (fgets(line, 990, rcfile)) {
	 if (*line == '#')
	    continue;
	 strcpy(saveline, line);
	 if (!process(line, 0, ATRC)) {
	    if (saveline[strlen(saveline) - 1] == '\n')
	       saveline[strlen(saveline) - 1] = 0;
	    printf("%%Unknown command '%s' in .igcrc file\n", saveline);
	 }
      }
   }
}



void dopassword(first)
char *first;
{
   strncpy(password, first, 18);
   strcat(password, "\n");
   password[20] = 0;
}


void dologin(first)
char *first;
{
   strncpy(login, first, 18);
   strcat(login, "\n");
   login[20] = 0;
}


void putstr( char *s, int append_newline )
{
    if (boardon)
    {
	addstring(s);
	if ( append_newline )
	    addchar('\n');
    }
    else
    {
	if ( append_newline )
	    puts(s);
	else
	    fputs( s, stdout );
    }
}




void handletstp()
{
   signal(SIGTSTP, handletstp);
   if (boardon)
      suspend();
   kill(getpid(), SIGSTOP);
   if (boardon) {
      unsuspend();
      if (boardmode)
	 boardrefresh();
      else
	 serverrefresh();
   }
}


int startgame(n)
int n;
{
   char str[100], str2[30];
   int ret, gotmove;

   gotmove = 0;

   sprintf(str, "games %d\n", n);
   sendstr(str);
   do {
      do {
	 ret = getmessage(&mesg, 0);
	 if (ret < 0)
	    handleterm();
      }
      while (!ret);
      if (mesg.id == MOVE) {
	 if (mesg.movecount)
	    putstr("%Premature move.  Restart game.", 1 );
	 else
	    gotmove = 1;
      }
   }
   while (mesg.id != GAMES);
   if (mesg.gamecount != 1 || mesg.gamelist[0].gnum != n)
      return -1;
   if (mesg.gamelist[0].bsize > 19) {
      putstr("%Boardsize too large!", 1 );
      return -1;
   }
   strcpy(curgame.white, mesg.gamelist[0].white);
   strcpy(curgame.black, mesg.gamelist[0].black);
   strcpy(curgame.brank, mesg.gamelist[0].brank);
   strcpy(curgame.wrank, mesg.gamelist[0].wrank);
   curgame.hcap = mesg.gamelist[0].hcap;
   curgame.komi = mesg.gamelist[0].komi;
   if (observing) {
      int remove=0;
      addstring("Removing observe\n");
      sendstr("unobserve\n");
      do {
	 do {
	    ret = getmessage(&mesg, 0);
	    if (ret < 0)
	       handleterm();
	 }
	 while (!ret);
	 if (mesg.id==INFO && !strncmp(mesg.text,"Removing",8))
	    remove=1;
      }
      while (mesg.id != PROMPT || !remove);
      observing = 0;
   }
   ingame = n;
   needrestore = 0;
   initboard(mesg.gamelist[0].bsize);
   displaygamenumber(ingame);
   xcur = ycur = mesg.gamelist[0].bsize / 2;
   sprintf(str, "%s (%s)", curgame.black, curgame.brank);
   sprintf(str2, "%s (%s)", curgame.white, curgame.wrank);
   boardtitle(str, str2);
   if (gotmove)
      highlight(0, 0, 0, 0, mesg.btime, mesg.bbyo, mesg.wtime, mesg.wbyo);
   showkomi(curgame.komi);
   boardon = 1;
   return 0;
}


void getmoves(n)
int n;
{
   char str[20];

   sprintf(str, "moves %d\n", n);
   sendstr(str);
}


void unobserve()
{
   sendstr("unobserve\n");
}


int observegame(n)
int n;
{
   int ret;
   char str[20];
   if (n < 0)
      return 1;
   if (startgame(n))
      return 1;
   sprintf(str, "observe %d\n", n);
   sendstr(str);
   observing = 1;
   do {
      do {
	 ret = getmessage(&mesg, 0);
	 if (ret < 0)
	    handleterm();
      }
      while (!ret);
      if ((mesg.id == INFO) && !strncmp(mesg.text, "Removing", 8))
	 putstr("%fatal sync error.  Restart igc!", 1 );
   }
   while (mesg.id != MOVE && mesg.id != UNDO);
   getmoves(n);
   return 0;
}


void setgame(newgame)
int newgame;
{
   if (newgame != ingame) {
      ingame = newgame;
      displaygamenumber(ingame);
   }
}


int peekgame(n)
int n;
{
   if (startgame(n))
      return 1;
   getmoves(n);
   justpeeked = 1;
   setgame(-1);
   return 0;
}

void loadgame(name)
char *name;
{
   char str[100];
   int ret;
   sprintf(str, "load %s\n", name);
   sendstr(str);
   do {
      ret = getmessage(&mesg, 0);
      if (ret < 0)
	 handleterm();
   } while ((!ret) ||
	    (mesg.id != MOVE && mesg.id != ERROR && mesg.id != PROMPT));
   if (mesg.id == ERROR)
   {
       putstr( "! ", 0 );
      putstr(mesg.text, 1 );
   }
   else if (mesg.id == PROMPT)
      putstr("%Load error!", 1 );
   else {
      if (!startgame(mesg.gamenum))
	 getmoves(mesg.gamenum);
   }
}

/* Caution: if you change the help msg, then you also 
 * have to adjust the number of lines in function help() below! */
char *helpmsg = "                   Server Window\n\
unobserve      Stops observing.\n\
peek <n>       Looks at the board for game n\n\
restore        Restores active game (after look or status)\n\
noboard        Goes back to full screen mode\n\
clear          Clears the window on the right side of the screen\n\
[no]inverse    Switches board to/from inverse \n\
[no]invedge    Turns on/off inversed edges\n\
[no]piecemarks Turns on/off piece marks\n\
[no]edgemarks  Turns on/off edge marks\n\
whosort {none|rank|rating|game|name|idle} Chooses sorting order for who command\n\
gamessort {none|number|obcount} Chooses sorting order for games command\n\
user <name>    Checks output from last who for <name>\n\
chars <list>   Sets board characters in client\n\
border <char>  Sets the border character\n\
beeps <n>      Sets number of beeps to use\n\
[no]saybeep    Turns on/off beeps when a say is received\n\
[no]tellbeep   Turns on/off beeps when a tell is received\n\
goto <n>       Goes to a particular move in the current game.\n\
[no]filter     Turn filtering of shouts on/off\n\
[un]friend     Add or remove a name from the friends list\n\
prefix         Sets prefix for all outgoing text you type\n\
               Start commands with '!' to avoid using the prefix\n\
\n\
ESC switches between windows         ^L  refreshes the screen\n\
\n\
                   Board Window\n\
Use number keys (12346789), emacs keys (^P ^N ^F ^B) or rogue keys\n\
   (hjklbnyu) to move the cursor.\n\
Press '0', return, or space to move (send position of cursor)\n\
Press '.' or '>' to move forward through game, ',' or '<' to move backwards.\n\
  '.' and ',' move by single moves.  '>' and '<' move by sets of ten moves.\n\
Press 's' to go to start of game, 'e' to go to end.\n\
Press 'f' or '/' to find the a particular move in a game by location.\n\
Press 'c' to clear the current move mark.";


void help()
{
    bigmess(33, helpmsg);
}



void printusage( int x )
{
    fprintf(stderr,"igc [options] [shorthand hostname | full hostname [port]]\n");

    exit( x );
}


void doargs(argc, argv)
int argc;
char *argv[];
{
   int i, j;

   for (i = 1; i < argc; i++)
   {
       if ( argv[i][0] != '-' ||
	    (argv[i][0] == '-' && argv[i][1] == '-')
	  )
       {
	   if ( argv[i][0] == '-' && argv[i][1] == '-' )
	       i ++ ;
	   break;
       }

       switch ( argv[i][1] )
       {
	   case 'h':
	       printusage(0);
	       break;

	   case 'l':
	       do_log = 1;
	       break;

	   default:
	       fprintf(stderr,"unknown option -%c!\n", argv[i][1] );
	       printusage(1);
      }
   }

   if ( i < argc )
   {
       if ( ! strchr(argv[i], '.') )
       {
	   printf("shorthand site %s = ", argv[i] );

	   for (j = 0; j < 10; j ++)
	       if (!strcmp(argv[i], sitetable[j].name))
	       {
		   sethost(sitetable[j].site);
		   setport(sitetable[j].port);

		   fputs( sitetable[j].site, stdout );
		   printf(":%d\n", sitetable[j].site );

		   if ( i < argc-1 )
		       fprintf(stderr, "Port specified with shorthand site "
			       "name!\n(ignoring)\n");
		   return;
	       }

	   fputs("\nunknown shorthand hostname!\n",stderr);
	   exit(1);
       }

       /* explicit host */
       strcpy(login, "");
       strcpy(password, "");

       sethost(argv[i]);
      if ( i < argc-1 )
      {
	 if (atoi(argv[i+1]))
	    setport(atoi(argv[i+1]));
	 else
	 {
	    fprintf(stderr, "Illegal port specification (%s)!\n\n", argv[i+1] );
	    exit(1);
	 }
      }
   }
}


void weave()
{
   char *copy[MAXWHO];
   int i, num;
   num = mesg.whocount / 2 + (mesg.whocount & 1);
   for (i = 0; i < num * 2; i++)
      copy[i] = mesg.who[i];
   for (i = 0; i < num; i++) {
      mesg.who[2 * i] = copy[i];
      mesg.who[2 * i + 1] = copy[num + i];
   }
}


#define NAMECOL 12
#define IDLECOL 25
#define RANKCOL 32
#define RATECOL 33
#define FLAGCOL 2
#define GAMECOL 10


int namesort(left, right)
char *left, *right;
{
   return strcasecmp(*(char **) left + NAMECOL, *(char **) right + NAMECOL);
}


int gamesort(left, right)
char *left, *right;
{
   int i;
   char *a, *b;
   a = *(char **) left;
   b = *(char **) right;
   if (a[GAMECOL] != '-' && b[GAMECOL] != '-') {
      i = strncmp(a + GAMECOL - 2, b + GAMECOL - 2, 3);
      return i ? i : namesort(left, right);
   }
   if (a[GAMECOL] != '-')
      return -1;
   if (b[GAMECOL] != '-')
      return 1;
   if (a[FLAGCOL] != b[FLAGCOL]) {
      if (a[FLAGCOL] == 'X')
	 return -1;
      if (b[FLAGCOL] == 'X')
	 return 1;
      if (a[FLAGCOL] == '!')
	 return 1;
      if (b[FLAGCOL] == '!')
	 return -1;
   }
   return ranksort(left, right);
}


int ratingsort(left, right)
char *left, *right;
{
   char *a, *b;
   a = *(char **) left;
   b = *(char **) right;
   if (a[RATECOL] == b[RATECOL])
      return ranksort(left, right);
   if (a[RATECOL] == '*')
      return -1;
   return 1;
}



int ranksort(left, right)
char *left, *right;
{
   int i;
   char *a, *b;
   a = *(char **) left;
   b = *(char **) right;

if (a[GAMECOL]==' ' && b[GAMECOL]==' ') return namesort(left,right);
if (a[GAMECOL]==' ') return -1;
if (b[GAMECOL]==' ') return 1;

   if (a[RANKCOL] != b[RANKCOL]) {
      if (a[RANKCOL] == '?')
	 return 1;
      if (b[RANKCOL] == '?')
	 return -1;
      if (a[RANKCOL] == 'R')
	 return 1;
      if (b[RANKCOL] == 'R')
	 return -1;
      if (a[RANKCOL] == 'k')
	 return 1;
      if (b[RANKCOL] == 'k')
	 return -1;
      if (a[RANKCOL] == 'd')
	 return 1;
      if (b[RANKCOL] == 'd')
	 return -1;

      if (a[RANKCOL] == 'p')  /* pros are above everyone else */
	 return -1;
      if (b[RANKCOL] == 'p')
	 return 1;

      return namesort(left, right); /* handle weird ranks like MCM93 */
   }
   if (a[RANKCOL] == 'R' || a[RANKCOL] == '?')
      return namesort(left, right);
   i = atoi(a + RANKCOL - 2) - atoi(b + RANKCOL - 2);
   if (a[RANKCOL] == 'k')
      i = -i;
   return i ? -i : namesort(left, right);
}



/*******************
 * idlesort  decides whether a or b has been idle longer
 *******************/

int idlesort(left, right)
char *left, *right;
{
   int i;
   char *a, *b;

   a = *(char **) left;
   b = *(char **) right;

   /* first check the units letter (h,m,s) assume that we don't need to worry
    * about d. */
   if (a[IDLECOL] != b[IDLECOL]) {
      if (a[IDLECOL] == 'h')
	 return -1;
      if (b[IDLECOL] == 'h')
	 return 1;
      if (a[IDLECOL] == 'm')
	 return -1;
      if (b[IDLECOL] == 'm')
	 return 1;
   }
   i = atoi(b + IDLECOL - 2) - atoi(a + IDLECOL - 2);
   return i ? i : namesort(left, right);
}


int gamenumsort(a, b)
struct gametype *a;
struct gametype *b;
{
   if (a->gnum < b->gnum)
      return -1;
   if (a->gnum > b->gnum)
      return 1;
   return 0;
}

int gamesobsort(a, b)
struct gametype *a;
struct gametype *b;
{
   if (a->obcount > b->obcount)
      return -1;
   if (a->obcount < b->obcount)
      return 1;
   return 0;
}



int choosegamessort(first)
char *first;
{
   if (!first)
      return 1;
   if (!strcmp(first, "number")) {
      gamessort = gamenumsort;
      return 0;
   }
   if (!strcmp(first, "none")) {
      gamessort = NULL;
      return 0;
   }
   if (!strcmp(first, "obcount")) {
      gamessort = gamesobsort;
      return 0;
   }
   return 1;
}


int choosesort(first)
char *first;
{
   if (!first)
      return 1;
   if (!strcmp(first, "name")) {
      whosort = namesort;
      return 0;
   }
   if (!strcmp(first, "rank")) {
      whosort = ranksort;
      return 0;
   }
   if (!strcmp(first, "game")) {
      whosort = gamesort;
      return 0;
   }
   if (!strcmp(first, "none")) {
      whosort = 0;
      return 0;
   }
   if (!strcmp(first, "idle")) {
      whosort = idlesort;
      return 0;
   }
   if (!strcmp(first, "rating")) {
      whosort = ratingsort;
      return 0;
   }
   return 1;
}


void restoregame()
{
   char str[30], str2[30];

   if (needrestore) {
      if (bschange)
	 initboard(bschange);
      restoremaxmoves();
      endgame();
      needrestore = 0;
      sprintf(str, "%s (%s)", curgame.black, curgame.brank);
      sprintf(str2, "%s (%s)", curgame.white, curgame.wrank);
      boardtitle(str, str2);
      highlight(0, 0, 0, -1, curgame.btime, curgame.bbyo,
		curgame.wtime, curgame.wbyo);
      displaygamenumber(ingame);
   }
}


void dofilter(first)
int first;
{
   friendfilter = first;
}



void dofriend(who)
char *who;
{
   int i;
   char buff[60];

   if (!who)
   {
      putstr("%Current list of friends:", 1 );
      for (i = 0; i < nfriends; i++)
	 putstr(friend[i], 1 );
   }
   else
   {
      if (nfriends >= MAXFRIENDS)
	 putstr("%Friend list is full!", 1 );
      else if (-1 != find_friend(who)) {
	 sprintf(buff, "%%Player %s already in friend list.", who);
	 putstr(buff, 1 );
      } else {
	 strcpy(friend[nfriends], who);
	 nfriends++;
	 sprintf(buff, "%%Friend %s added.", who);
	 putstr(buff, 1 );
      }
   }
}


int isfriend(text)
char *text;
{
   int i, length;

   if (*text != '{')
      return 1;
   text++;
   for (i = 0; i < nfriends; i++) {
      length = strlen(friend[i]);
      if (!strncmp(friend[i], text, length) && text[length] == ' ')
	 return 1;
   }
   return 0;
}



int find_friend(fr)
char *fr;
{
   char i;

   for (i = 0; i < nfriends && strcmp(friend[i], fr); i++);
   return i == nfriends ? -1 : i;
}



void dounfriend(who)
char *who;
{
   int i;

   if (!who)
      putstr("%unfriend requires 1 parameter!", 1 );
   else if (-1 != (i = find_friend(who))) {
      for (; i < nfriends; i++)
	 strcpy(friend[i], friend[i + 1]);
      nfriends--;
      putstr("%Friend removed.", 1 );
   } else
      putstr("%Friend not in list!", 1 );
}


void switchmode()
{
   boardmode = !boardmode;
   if (boardmode)
      setcursor(xcur, ycur);
}


static char *nomoves = "%no moves recorded!";

void doboardmode(ch)
char ch;
{
   char movestr[10];
   switch (ch) {
      case '^':
	 switchmode();
	 clearserver();
	 switchmode();
	 break;
      case ',':
	 restoregame();
	 if (backward(1))
	    putstr(nomoves, 1 );
	 break;
      case '.':
	 restoregame();
	 if (forward(1))
	    putstr(nomoves, 1 );
	 break;
      case '<':
	 restoregame();
	 if (backward(10))
	    putstr(nomoves, 1 );
	 break;
      case '>':
	 restoregame();
	 if (forward(10))
	    putstr(nomoves, 1 );
	 break;
      case 's':
	 restoregame();
	 if (beginninggame())
	    putstr(nomoves, 1 );
	 break;
      case 'e':
	 restoregame();
	 if (endgame())
	    putstr(nomoves, 1 );
	 break;
      case 'f':
      case '/':
	 restoregame();
	 if (search(xcur, ycur))
	    putstr("%No move found at that point!", 1 );
	 break;

      case '1':
      case 'b':
	 if (++ycur == boardsize)
	    ycur = 0;
	 if (!xcur--)
	    xcur = boardsize - 1;
	 break;
      case '3':
      case 'n':
	 if (++ycur == boardsize)
	    ycur = 0;
	 if (++xcur == boardsize)
	    xcur = 0;
	 break;
      case '7':
      case 'y':
	 if (!xcur--)
	    xcur = boardsize - 1;
	 if (!ycur--)
	    ycur = boardsize - 1;
	 break;
      case '9':
      case 'u':
	 if (!ycur--)
	    ycur = boardsize - 1;
	 if (++xcur == boardsize)
	    xcur = 0;
	 break;
      case 'B' - 64:
      case '4':
      case 'h':

	 if (!xcur--)
	    xcur = boardsize - 1;
	 break;
      case 'P' - 64:
      case '8':
      case 'k':
	 if (!ycur--)
	    ycur = boardsize - 1;
	 break;
      case 'N' - 64:
      case '2':
      case 'j':
	 if (++ycur == boardsize)
	    ycur = 0;
	 break;
      case 'F' - 64:
      case '6':
      case 'l':
	 if (++xcur == boardsize)
	    xcur = 0;
	 break;
      case 'c':
	 unmark();
	 break;
      case '\n':
      case '\r':
      case ' ':
      case '0':
	 sprintf(movestr, "%c%d\n", xcur + 'A' + (xcur + 'A' >= 'I' ? 1 : 0),
		 boardsize - ycur);
	 sendstr(movestr);
	 break;
   }
   setcursor(xcur, ycur);
}


void doservermode(ch)
char ch;
{
    static int notfirst = 0;

   *loc = ch;
   *(loc + 1) = 0;
   if (ch < ' ' && ch != ('U' - 64) && ch != '\b' && ch != '\n')
      return;
   if (boardon) {
      if (*loc == '\b' || *loc == '\177') {
	 if (strlen(local) > 1) {
	    deletechar(strlen(local) - 2);
	    loc -= 2;
	 } else
	    loc--;
      } else if (*loc == 'U' - 64) {
	 loc = local;
	 clrline();
	 return;
      } else if (strlen(local) >= linesize()) {
	 if (!notfirst)
	    sendstr(prefix);
	 sendstr(local);
	 loc = local;
	 notfirst = 1;
	 clrline();
	 putstr(local, 1 );
	 return;
      } else
	 addcharline(strlen(local) - 1, *loc);
   }
   if (*loc == '\n') {
      loc++;
      *loc = 0;
      if (boardon) {
	 addstring(local);
      }
      if (notfirst)
	 sendstr(local);
      else if (local[0] == '!')
	 process(local + 1, 0, NOTATRC);
      else if (strlen(prefix)) {
	 sendstr(prefix);
	 sendstr(local);
      } else
	 process(local, 1, NOTATRC);
      if (boardon)
	 clrline();
      notfirst = 0;
      loc = local;
   } else
      loc++;
}




void main(argc, argv)
int argc;
char *argv[];
{
   char ch;
   int ret;
   puts("igc - Ascii Client version 0.751 by Adrian Mariano");
   puts("      (with little bug fixes / changes by Gabriel Zachmann)");
   puts("      type 'helpigc' to get client help, 'help' for server help\n");
   strcpy(prefix, "");
   whosort = namesort;
   gamessort = gamenumsort;
   readrc();
   sethost(sitetable[0].site);
   setport(sitetable[0].port);
   doargs(argc, argv);
   loc = local;
   signal(SIGINT, handleterm);
   signal(SIGTERM, handleterm);
   signal(SIGHUP, handleterm);
   signal(SIGTSTP, handletstp);
   if (open_connection())
      handleterm();
   puts("Connection established.");
   mesg.whocount = 0;
   {
      char *whoptr;
      for (whoptr = &(mesg.whodata[0][0]), ret = 0; ret < MAXWHO; ret++, whoptr += 40)
	 mesg.who[ret] = whoptr;
   }
   initparser();

   do
   {

      if (boardon)
	 if (boardmode)
	    boardrefresh();
	 else
	    serverrefresh();

      ret = getmessage(&mesg, 1);

      if (ret < 0 && ret != KEY)
      {
	 if (boardon) {
	    endAscii();
	    boardon = 0;
	 }
	 puts("Connection closed by server.");
	 if ( log )
	     fputs("Connection closed by server.\n",log);
	 myexit(0);
      }

      if (ret > 0)
      {

	 switch (mesg.id)
	 {

	    case TOOMANY:
	       fprintf(stderr, "Too many players.  Sorry.\n");
	       myexit(0);
	       break;

	    case QUITMESG:
	       if (boardon) {
		  endAscii();
		  boardon = 0;
	       }
	       puts(mesg.text);
	       break;

	    case ONSERVER:
	       break;

	    case BEEP:
	       if (mesg.beep)
		  dobeep();
	       break;

	    case MOVE:
	       if (!boardon)
		  puts("%Error: isolated move received!");
	       else {
		  int i;
		  restoregame();
		  sethighlight(0);
		  for (i = 0; i < mesg.movecount - 1; i++)
		     makemove(mesg.moves[i].x, mesg.moves[i].y,
			      mesg.moves[i].movenum,
			      mesg.moves[i].color, mesg.btime,
			      mesg.bbyo, mesg.wtime, mesg.wbyo);
		  sethighlight(1);
		  if (mesg.movecount)
		     makemove(mesg.moves[i].x, mesg.moves[i].y,
			      mesg.moves[i].movenum,
			      mesg.moves[i].color, mesg.btime,
			      mesg.bbyo, mesg.wtime, mesg.wbyo);
		  curgame.bbyo = mesg.bbyo;
		  curgame.btime = mesg.btime;
		  curgame.wtime = mesg.wtime;
		  curgame.wbyo = mesg.wbyo;

		  setcursor(xcur, ycur);
		  if (!justpeeked)
		     setgame(mesg.gamenum);
		  justpeeked = 0;
	       }
	       break;

	    case UNDO:
	       restoregame();
	       if (!boardon)
		  puts("%Error: isolated undo received");
	       else {
		  setgame(mesg.gamenum);
		  undo();
		  setcursor(xcur, ycur);
	       }
	       break;

	    case SCOREUNDO:
	       restoregame();
	       if (endgame())
		  putstr("%score undo error!", 1 );
	       else
		  putstr("Scoring undone.", 1 );
	       break;

	    case LOAD:
	       if (!startgame(mesg.gamenum))
		  getmoves(mesg.gamenum);
	       break;

	    case MATCH:
	       startgame(mesg.gamenum);
	       break;

	    case REMOVE:
	       restoregame();
	       removeGroup(mesg.x, mesg.y);
	       setcursor(xcur, ycur);
	       break;

	    case SCORE_M:
	       {
		  char str[50];
		  sprintf(str, "  %s (B): %g\n  %s (W): %g",
			  mesg.black, mesg.bscore, mesg.white, mesg.wscore);
		  putstr(str, 1 );
	       }
	       break;

	    case STARTSCORE:
	       putstr("Remove dead groups to score.", 1 );
	       putstr(mesg.text, 1 );
	       break;

	    case STATUS:
	    case LOOK_M:
	       {
		  int pris[2];
		  char str[30], str2[30];

		  if (mesg.boardsize > 19)
		     putstr("%Boardsize too big!", 1 );
		  else {
		     needrestore = 1;
		     bschange = 0;
		     savemaxmoves();
		     if (!boardon || boardsize != mesg.boardsize) {
			if (!boardon)
			   needrestore = 0;
			else
			   bschange = boardsize;
			initboard(mesg.boardsize);
			xcur = ycur = mesg.boardsize / 2;
			boardon = 1;
		     }
		     pris[0] = mesg.bcap;
		     pris[1] = mesg.wcap;
		     sprintf(str, "%s (%s)", mesg.black, mesg.brank);
		     sprintf(str2, "%s (%s)", mesg.white, mesg.wrank);
		     boardtitle(str, str2);
		     highlight(0, 0, 0, -1, mesg.btime, mesg.bbyo,
			       mesg.wtime, mesg.wbyo);
		     drawpris(pris);
		     showboard(mesg.board);
		     displaygamenumber(-2);
		  }
	       }
	       break;

	    case KIBITZ:
	       {
		  char s[3000];
		  sprintf(s, "%s: %s", mesg.kibitzer, mesg.kibitz);
		  putstr(s, 1 );
	       }
	       break;

	    case WHO:
	       {
		  char out[80 * MAXWHO];
		  int i;

		  if (whosort) {
		     qsort(mesg.who, mesg.whocount, sizeof(char *), whosort);
		     weave();
		  }
		  /* print legend */
		  strcpy( out,
			  "Q = quiet on, S = shout off, X = open off, ! = looking on\n" );
		  strcat(out, mesg.whofirst);
		  for (i = 0; i < mesg.whocount; i++) {
		     strcat(out, i & 1 ? " | " : "\n");
		     strcat(out, mesg.who[i]);
		  }
		  strcat(out, "\n");
		  strcat(out, mesg.wholast);
		  bigmess(mesg.lines, out);
	       }
	       break;

	    case GAMES:
	       {
		  char out[80 * MAXGAMES];
		  int i;

		  if (gamessort)
		     qsort(mesg.gamelist, mesg.gamecount, sizeof(struct gametype), gamessort);
		  strcpy(out, mesg.gamefirst);
		  strcat(out, "\n");
		  for (i = 0; i < mesg.gamecount; i++)
		     strcat(out, mesg.gamelist[i].gametext);
		  bigmess(mesg.gamecount + 1, out);
	       }
	       break;

	    case STORED:
	       if (!strlen(mesg.text))
		  putstr("No stored games!", 1 );
	       else
		  putstr(mesg.text, 1 );
	       break;

	    case CHKOMI:
	       curgame.komi = mesg.mykomi;
	       showkomi(curgame.komi);

	    case VERSION:
	       putstr( mesg.text, 1 );
	       break;

	    case INFO:
	       if ( strlen(mesg.text) )
	       {
		   if (strstr(mesg.text, "Removing")) {
		      observing = 0;
		      setgame(-1);
		      putstr(mesg.text, 1 );
		   }
		   else if ( !strncmp( mesg.text,
				       "Games is titled:", 16) )
		   {
		       break;
		   }
		   else
		   {
		       putstr( "> ", 0 );
		       putstr(mesg.text, 1 );
		       /*bigmess(mesg.lines, mesg.text);*/
		   }
	       }
	       break;

	    case ERROR:
	       if ( strlen(mesg.text) )
	       {
		   putstr( "! ", 0 );
		   putstr(mesg.text, 1 );
	       }
	       break;

	    case PROMPT:
	       if (ingame != -1 && mesg.prompttype == 5) {
		  setgame(-1);
		  observing = 0;
	       }
	       break;

	    case ENDGAMEINFO:
	       if (observing && mesg.gamenum == ingame) {
		  setgame(-1);
		  observing = 0;
	       }
	       putstr(mesg.text, 1 );
	       break;

	    case ENDGAME:
	       if (observing && mesg.gamenum == ingame) {
		  setgame(-1);
		  observing = 0;
	       }
	       putstr(mesg.text, 1 );
	       break;

	    case SHOUT:
	       if (!friendfilter || isfriend(mesg.text))
	       {
		   /* remove ! */
		   int l = strlen(mesg.text);
		   char *c;
		   if ( *mesg.text == '!' )
		   {
		       l -- ;
		       memmove( mesg.text, mesg.text+1, l );
		       if ( c = strchr(mesg.text, '!') )
		       {
			   memmove( c, c+1, l-(c-mesg.text) );
			   l -- ;
		       }
		   }

		   putstr(mesg.text, 1 );
	       }
	       break;

	    case SAY:
	       if (saybeep)
		  dobeep();
	       putstr(mesg.text, 1 );
	       break;

	    case TELL:
	       if (tellbeep)
		  dobeep();
	       putstr(mesg.text, 1 );
	       break;

	    case INTRO:
	       if (displayintro)
		  putstr(mesg.text, 1 );
	       break;

	    case HELP:
	       if (strlen(mesg.text))
		  bigmess(mesg.lines, mesg.text);
	       break;

	    case UNKNOWN:
	       /* seems to never contain a mesg
		* should not happen at all, should it? */
	       break;

	    default:

	      if ( mesg.id >= MAXMSGTYP || mesg.id < 0 )
	      {
		  char m[1000];
		  if ( log )
		      fprintf(log,"!!! message type (%d) out of range (0..%d)!\n",
			      mesg.id, MAXMSGTYP );
		  sprintf( m, "message type %d out of range!", mesg.id );
		  putstr( m, 1 );
	      }
	      else
	      {
		  if ( log )
		      fprintf(log,"message %s received.\n",
			      mesgtypestr[mesg.id] );
		  putstr( mesgtypestr[mesg.id], 0 );
		  putstr( ": ", 1 );
	      }

	       if ( mesg.id == 0 )
		   break;

	       if (strlen(mesg.text))
		  bigmess(mesg.lines, mesg.text);
	       break;
	 }
      }

      if (ret == KEY) {
	 ret = read(fileno(stdin), &ch, 1);
	 if (ch == '\r')
	    ch = '\n';
	 if (ch == 'L' - 64 && boardon)
	    redraw();
	 else if (ch == 27 && boardon)
	    switchmode();
	 else if (boardmode)
	    doboardmode(ch);
	 else
	    doservermode(ch);
      }
   }
   while (1);

   handleterm();
}





void dopeek(first)
char *first;
{
   if (first) {
      if (!observing && ingame != -1)
	 putstr("%Can't observe while playing!", 1 );
      else if (observing && ingame == atoi(first))
	 unobserve();
      else if (peekgame(atoi(first)))
	 putstr("%No such game!", 1 );
   }
}


void doobserve(first)
char *first;
{
   if (first) {
      if (!observing && ingame != -1)
	 putstr("%Can't observe while playing!", 1 );
      else if (observing && ingame == atoi(first))
	 unobserve();
      else if (observegame(atoi(first)))
	 putstr("%No such game!", 1 );
   }
}


void dounobserve()
{
   if (observing)
      unobserve();
   else
      putstr("%Not observing!", 1 );
}


void doload(first)
char *first;
{
   if (first)
      loadgame(first);
}


void dochars(first)
char *first;
{
   if (first) {
      setchars(first);
      if (boardon) {
	 reinitboard(boardsize);
	 endgame();
      }
   }
}


void dorestore()
{
   if (!needrestore)
      putstr("%no game to restore!", 1 );
   restoregame();
}


void dowhosort(first)
char *first;
{
   if (choosesort(first))
      putstr("%Unknown who sorting method!", 1 );
}


void dogamessort(first)
char *first;
{
   if (choosegamessort(first))
      putstr("%Unknown games sorting method!", 1 );
}



void doborder(first)
char *first;
{
   if (!first)
      setborder(' ');
   else
      setborder(*first);
   if (boardon) {
      reinitboard(boardsize);
      endgame();
   }
}


void dointro(first)
int first;
{
   displayintro = first;
}


void doinverse(first)
int first;
{
   setinverse(first);
   if (boardon) {
      reinitboard(boardsize);
      endgame();
   }
}


void doedgemarks(first)
int first;
{
   setedgemark(first);
   if (boardon) {
      reinitboard(boardsize);
      endgame();
   }
}

void dopiecemarks(first)
int first;
{
   setpiecemark(first);
   if (boardon) {
      reinitboard(boardsize);
      endgame();
   }
}



void doinvedge(first)
int first;
{
   setinverseborder(first);
   if (boardon) {
      reinitboard(boardsize);
      endgame();
   }
}


void doprefix(first)
char *first;
{
   if (!first)
      strcpy(prefix, "");
   else {
      strncpy(prefix, first, 19);
      prefix[20] = 0;
      for(first=prefix;*first;first++) if (*first=='_') *first=' ';
   }

   if (boardon)
      showprefix(prefix);
   else
      printf("Prefix '%s'\n", prefix);
}

void douser(first)
char *first;
{
   int i, found;

   if (!first)
      putstr("%No user specified!", 1 );
   else {
      found = 0;
      for (i = 0; i < mesg.whocount; i++) {
	 if (!strncmp(mesg.who[i] + NAMECOL, first, strlen(first)) &&
	     mesg.who[i][NAMECOL + strlen(first)] == ' ') {
	    putstr(mesg.who[i], 1 );
	    found = 1;
	 }
      }
      if (!found)
	 putstr("%User not found in last who!", 1 );
   }
}


void dosaybeep(first)
int first;
{
   saybeep = first;
}


void dotellbeep(first)
int first;
{
   tellbeep = first;
}


void dobeeps(first)
char *first;
{
   if (first)
      beepcount = atoi(first);
}

void doclear()
{
   if (!boardon)
      putstr("%Cannot clear!", 1 );
   else
      clearserver();
}

void donoboard()
{
   if (boardon) {
      if (ingame != -1) {
	 putstr("Board in use.  Cannot remove!", 1 );
      } else {
	 boardon = 0;
	 boardmode = 0;
	 endAscii();
      }
   } else
      putstr("%No board to remove!", 1 );
}


void dogoto(first)
char *first;
{
   if (boardon) {
      first = strtok(NULL, " \t");
      if (first)
	 construct(atoi(first));
   } else
      putstr("%no board!", 1 );
}


/* Warning:  This function makes calls to strtok which assume that we've
 * already started parsing the input line. */

void dosite(first)
char *first;
{
   int num;
   char *next;

   num = atoi(first);
   if (num < 0 || num >= MAXSITES) {
      printf("%%invalid site number: %d\n", num);
      return;
   }
   next = strtok(NULL, " \t\n=");
   if (!next) {
      printf("%%invalid or missing site name in 'site %d'\n", num);
      return;
   }
   strcpy(sitetable[num].name, next);
   next = strtok(NULL, " \t\n=");
   if (!next) {
      printf("%%invalid or missing host name in 'site %d' command\n", num);
      return;
   }
   strcpy(sitetable[num].site, next);
   next = strtok(NULL, " \t\n=");
   if (next && atoi(next))
      sitetable[num].port = atoi(next);
}



struct commandtype {
   char *command;
   void (*action) ();
   int inrc;
   int toggle;
}  comlist[] =
{
   { "border", doborder, BOTH, 0 },
   { "chars", dochars, BOTH, 0 },
   { "clear", doclear, NOTATRC, 0 },
   { "beeps", dobeeps, BOTH, 0 },
   { "edgemarks", doedgemarks, BOTH, 1 },
   { "filter", dofilter, BOTH, 1}, 
   { "friend", dofriend, BOTH, 0 }, 
   { "gamessort", dogamessort, BOTH, 0 },
   { "goto", dogoto, NOTATRC, 0 },
   { "helpigc", help, NOTATRC, 0 },
   { "intro", dointro, ATRC, 1 }, 
   { "invedge", doinvedge, BOTH, 1 },
   { "inverse", doinverse, BOTH, 1 },
   { "load", doload, NOTATRC, 0 },
   { "login", dologin, ATRC, 0 },
   { "moves", dopeek, NOTATRC, 0 },
   { "noboard", donoboard, NOTATRC, 0 },
   { "ob", doobserve, NOTATRC, 0 },
   { "obs", doobserve, NOTATRC, 0 },
   { "obse", doobserve, NOTATRC, 0 },
   { "obser", doobserve, NOTATRC, 0 },
   { "observ", doobserve, NOTATRC, 0 },
   { "observe", doobserve, NOTATRC, 0 },
   { "password", dopassword, ATRC, 0 },
   { "peek", dopeek, NOTATRC, 0 },
   { "piecemarks", dopiecemarks, BOTH, 1 },
   { "prefix", doprefix, NOTATRC, 0 },
   { "restore", dorestore, NOTATRC, 0 },
   { "saybeep", dosaybeep, BOTH, 1 },
   { "site", dosite, ATRC, 0}, 
   { "whosort", dowhosort, BOTH, 0 },
   { "tellbeep", dotellbeep, BOTH, 1 },
   { "unfriend", dounfriend, BOTH, 0 }, 
   { "unob", dounobserve, NOTATRC, 0 },
   { "unobserve", dounobserve, NOTATRC, 0 },
   { "user", douser, NOTATRC, 0 }
};


/* Returns 1 if the command was parsed locally.  Returns 0 if the command was
 * passed to the server, or not parsed locally. */

int process(comm, dopre, state)
char *comm;
int dopre;
int state;
{
   char *first, *second;
   char save[1000];
   int i;

   strcpy(save, comm);
   first = strtok(comm, " \n\t=");
   second = strtok(NULL, " \n\t=");
   if (!first)
      return 1;
   for (i = 0; i < sizeof(comlist) / sizeof(struct commandtype); i++) {
      if ((state & comlist[i].inrc) &&
	  !strcmp(first, comlist[i].command)) {
	 if (comlist[i].toggle)
	    comlist[i].action(1);
	 else
	    comlist[i].action(second);
	 return 1;
      }
   }
   if (!strncmp(first, "no", 2)) {
      first += 2;
      for (i = 0; i < sizeof(comlist) / sizeof(struct commandtype); i++) {
	 if ((state & comlist[i].inrc) && comlist[i].toggle &&
	     !strcmp(first, comlist[i].command)) {
	    comlist[i].action(0);
	    return 1;
	 }
      }
   }
   if (state == NOTATRC) {
      if (dopre)
	 sendstr(prefix);
      sendstr(save);
   }
   return 0;
}
