/***************************************************************************/
/* 		This code is part of Desktop Background changer		   */
/*		called ChBg						   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "config.h"
#include "absimg.h"
#include "gaccel.h"
#include "gprop.h"
#include "recurse.h"
#include "gtkmultirowsel.h"
#include "gtkselbox.h"
#include "gtkbgpixmap.h"

#include "pixmaps/stock_close.xpm"
#include "pixmaps/stock_stop.xpm"
#include "pixmaps/stock_ok.xpm"
#include "pixmaps/stock_apply.xpm"
#include "pixmaps/stock_restart.xpm"

#define THUMB_PAGE_SIZE 25
#define THUMB_SIZE	70

struct thumbpv_info {
	thumb_type	type;
	GtkWidget	*tl;
	GtkWidget	*swin;
	GtkWidget	*tab;
	GtkWidget	*pv;
	GtkWidget	*pmap;
	GtkWidget	*prevb;
	GtkWidget	*nextb;
	GtkWidget	*optb;
	GtkWidget	*combo;
	GtkWidget	*menu;
	GtkWidget	*view_menu;
	GdkWindow	*fullwin;
	GtkWidget	*progress;
	GtkWidget	*refreshb;
	gint		num;
	GList		*imglist;
	GList		*alllist;
	gint		curr;
	GList		*currimg;
	absimg_image_t	*curr_absimg;
	guint		curr_width;
	guint		curr_height;
	GSList		*dirlist;
	gboolean	frst;
	gboolean	frst_time;
	guint		fulltm;
	guint		normtm;
	GdkWindow	*root_window;
	guint		view_type;
	GtkWidget	*swinpv;
	guint		drag_start;
	gdouble		drag_x;
	gdouble		drag_y;
	guint		drag_type;
	GtkTooltips	*tt;
};

static void ThumbsDoPage(struct thumbpv_info *);

static void ThumbInfoInit(info)
struct thumbpv_info *info;
{
	info->type = THUMB_ALL;
	info->tl = NULL;
	info->swin = NULL;
	info->tab = NULL;
	info->pv = NULL;
	info->pmap = NULL;
	info->prevb = NULL;
	info->nextb = NULL;
	info->optb = NULL;
	info->combo = NULL;
	info->menu = NULL;
	info->view_menu = NULL;
	info->fullwin = NULL;
	info->num = 0;
	info->imglist = NULL;
	info->alllist = NULL;
	info->curr = 0;
	info->currimg = NULL;
	info->curr_absimg = NULL;
	info->curr_width = 0;
	info->curr_height = 0;
	info->dirlist = NULL;
	info->frst = TRUE;
	info->frst_time = TRUE;
	info->progress = NULL;
	info->refreshb = NULL;
	info->normtm = 0;
	info->fulltm = 0;
	info->root_window = GDK_ROOT_PARENT();
	info->view_type = 0;
	info->swinpv = NULL;
	info->drag_start = FALSE;
	info->drag_x = 0.0;
	info->drag_y = 0.0;
	info->drag_type = 0;
}

static void ThumbViewFullImg(info, fname)
struct thumbpv_info *info;
char *fname;
{
	absimg_image_t *img;
	prop_t prop;

	img = absimg_load(fname);

	if (!img)
	{
		gdk_beep();
		return;
	}

	memcpy(&prop, &cfg.properties, sizeof(prop_t));
	prop.type = RENDERT_SMART;

	if (info->fullwin)
		set_root_bg_real(&info->fullwin, img, fname, 
				 &prop, NULL, 0, 0, FALSE);

	absimg_destroy(img);
	gdk_flush();
	chbg_status_progress(0,1);
}

static int ThumbViewFullSlide(info)
struct thumbpv_info *info;
{
	if (info->currimg && info->currimg->next)
		info->currimg = info->currimg->next;
	else
		info->currimg = info->alllist;

	if (info->currimg)
	{
		picentry_t *pe;

		pe = (picentry_t *)info->currimg->data;
		ThumbViewFullImg(info, pe->name);
	}

	return TRUE;
}

static gint ThumbViewFullEvents(event, info)
GdkEvent  *event;
struct thumbpv_info *info;
{
	if (event->any.window == info->fullwin)
	{
	    switch (event->type)
	    {
		case GDK_VISIBILITY_NOTIFY:
		case GDK_CONFIGURE:
			gdk_window_raise(info->fullwin);
			break;
		case GDK_KEY_RELEASE:
		{
			GdkEventKey *kevent = (GdkEventKey *)event;
			switch(kevent->keyval)
			{
				case GDK_Q:
				case GDK_q:
				case GDK_Escape:
					gdk_keyboard_ungrab(GDK_CURRENT_TIME);
					gdk_pointer_ungrab(GDK_CURRENT_TIME);
	  				gdk_window_hide(info->fullwin);
					info->fullwin = NULL;
					if (info->fulltm)
					{
						gtk_timeout_remove(info->fulltm);
						info->fulltm = 0;
					}
				break;
				case GDK_S:
				case GDK_s:
					if (info->fulltm)
					{
						gtk_timeout_remove(info->fulltm);
						info->fulltm = 0;
					}
					else
					{
						int iv;

						if (!gprop_get_int("view-slide-interval", &iv))
							iv = 10;

						info->fulltm = gtk_timeout_add(iv * 1000,
							(GtkFunction) ThumbViewFullSlide, info);
					}
				break;
				case GDK_N:
				case GDK_n:
				case GDK_space:
				{
					if (info->currimg && info->currimg->next)
					{
						picentry_t *pe;

						info->currimg = info->currimg->next;
						pe = (picentry_t *)info->currimg->data;
						ThumbViewFullImg(info, pe->name);
					}
					else if (info->type == THUMB_PAGE)
					{
						if ((info->curr + THUMB_PAGE_SIZE) > info->num)
							gdk_beep();
						else
						{
							picentry_t *pe;

							info->curr += THUMB_PAGE_SIZE;
							ThumbsDoPage(info);

							info->currimg = info->imglist;
							pe = (picentry_t *)info->currimg->data;
							ThumbViewFullImg(info, pe->name);
						}
					}
					else
						gdk_beep();
				}
				break;
				case GDK_P:
				case GDK_p:
				case GDK_BackSpace:
				{
					if (info->currimg && info->currimg->prev)
					{
						picentry_t *pe;

						info->currimg = info->currimg->prev;
						pe = (picentry_t *)info->currimg->data;
						ThumbViewFullImg(info, pe->name);
					}
					else if (info->type == THUMB_PAGE)
					{
						if (!info->curr)
							gdk_beep();
						else
						{
							picentry_t *pe;

							info->curr -= THUMB_PAGE_SIZE;
							ThumbsDoPage(info);

							info->currimg = g_list_last(info->imglist);
							pe = (picentry_t *)info->currimg->data;
							ThumbViewFullImg(info, pe->name);
						}
					}
					else
						gdk_beep();
				}
				break;
			}
		}
		default:
		break;
	    }
	}
	else gtk_main_do_event(event);

	return TRUE;
}

static void ThumbViewFull(w, info, ppe)
GtkWidget *w;
struct thumbpv_info *info;
picentry_t *ppe;
{
	char *imgname;

	if (!info->fullwin)
	{
		info->fullwin = chbg_init_full_win(NULL, ThumbViewFullEvents, info);
	}

	if (w)
	{
		picentry_t *pe;

		pe = (picentry_t *)gtk_object_get_user_data(GTK_OBJECT(w));
		info->currimg = g_list_find(info->imglist, pe);
		imgname = pe->name;
	}
	else
		imgname = ppe->name;

	ThumbViewFullImg(info, imgname);
}

static void ThumbViewSetSize(info, sadd)
struct thumbpv_info *info;
double sadd;
{
	GdkPixmap *pixmap;
	GtkWidget *tpmap;

	if (!info->curr_absimg)
	{
		if (info->pmap)
		{
			gtk_widget_destroy(info->pmap);
			info->pmap = NULL;
		}
		return;
	}

	info->curr_width += (guint)(sadd * (double)info->curr_width);
	info->curr_height += (guint)(sadd * (double)info->curr_height);

	if (info->curr_width < 1) info->curr_width = 1;
	if (info->curr_height < 1) info->curr_height = 1;

	tpmap = gtk_bg_pixmap_new(NULL);

	if (absimg_have_alpha(info->curr_absimg))
	{
		GdkGC *gc;

		gc = gdk_gc_new(GDK_ROOT_PARENT());
		gtk_widget_ensure_style(tpmap);
		gdk_gc_set_foreground(gc, &(tpmap->style->bg[GTK_STATE_NORMAL]));

		pixmap = gdk_pixmap_new(GDK_ROOT_PARENT(),
			info->curr_width, info->curr_height,
			gdk_visual_get_best_depth());
		gdk_draw_rectangle(pixmap, gc, TRUE, 0, 0, info->curr_width, info->curr_height);
		absimg_render_alpha_to_pixmap(pixmap, info->curr_absimg, info->curr_width, info->curr_height);
		gdk_gc_unref(gc);
	}
	else
		pixmap = absimg_render(info->curr_absimg, info->curr_width, info->curr_height);

	if (info->pmap)
	{
		gtk_widget_destroy(info->pmap);
		info->pmap = NULL;
	}

	info->pmap = tpmap;
	gtk_bg_pixmap_set(GTK_BG_PIXMAP(info->pmap), pixmap);
	gtk_signal_connect(GTK_OBJECT(info->pmap), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &(info->pmap));

	if (info->view_type)
	{
		int w,rw,h,rh,wx,wy;

		gdk_window_get_size(GDK_ROOT_PARENT(), &rw, &rh);

		if (GTK_WIDGET_REALIZED(info->pv))
			gdk_window_get_position(info->pv->window, &wx, &wy);
		else
		{
			wx = 0;
			wy = 0;
		}

		rw -= wx;
		rh -= wy;

		w = MIN(info->curr_width + 5, rw - 30);
		h = MIN(info->curr_height + 5, rh - 30);

		gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(info->swinpv), info->pmap);
		gtk_window_set_default_size(GTK_WINDOW(info->pv), w, h);
	}
	else
		gtk_container_add(GTK_CONTAINER(info->pv), info->pmap);

	gtk_widget_show(info->pmap);

	absimg_destroy_pixmap(pixmap);
}

static void ThumbViewImg(info, fname)
struct thumbpv_info *info;
char *fname;
{
	char pom[2248];

	sprintf(pom, gettext("ChBg: picture %s"), fname);

	gtk_window_set_title(GTK_WINDOW(info->pv) , pom);

	if (info->curr_absimg)
	{
		absimg_destroy(info->curr_absimg);
	}

	info->curr_absimg = absimg_load(fname);
	if (info->curr_absimg)
	{
		info->curr_width = info->curr_absimg->rgb_width;
		info->curr_height = info->curr_absimg->rgb_height;
	}
	else
		gdk_beep();

	info->frst_time = TRUE;

	ThumbViewSetSize(info, 0.0);
}

static int ThumbViewSlide(info)
struct thumbpv_info *info;
{

	if (info->currimg && info->currimg->next)
		info->currimg = info->currimg->next;
	else
		info->currimg = info->alllist;

	if (info->currimg)
	{
		picentry_t *pe;

		pe = (picentry_t *)info->currimg->data;
		ThumbViewImg(info, pe->name);
	}

	return TRUE;
}

static void ThumbViewME_Close(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	gtk_widget_destroy(GTK_WIDGET(info->pv));
}

static void ThumbViewME_ToggleSlide(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	if (info->normtm)
	{
		gtk_timeout_remove(info->normtm);
		info->normtm = 0;
	}
	else
	{
		int iv;

		if (!gprop_get_int("view-slide-interval", &iv))
			iv = 10;

		info->normtm = gtk_timeout_add(iv * 1000,
			(GtkFunction) ThumbViewSlide, info);
	}
}

static void ThumbViewME_SizePlus10(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	ThumbViewSetSize(info, 0.10);
}

static void ThumbViewME_SizeMinus10(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	ThumbViewSetSize(info, -0.10);
}

static void ThumbViewME_SizeHalf(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	ThumbViewSetSize(info, -0.50);
}

static void ThumbViewME_SizeDouble(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	ThumbViewSetSize(info, 1.0);
}

static void ThumbViewME_SizeOrig(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	info->curr_width = info->curr_absimg->rgb_width;
	info->curr_height = info->curr_absimg->rgb_height;
	ThumbViewSetSize(info, 0.0);
}

static void ThumbViewME_FlipVert(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	absimg_flip_vertical(info->curr_absimg);
	ThumbViewSetSize(info, 0.0);
}

static void ThumbViewME_FlipHoriz(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	absimg_flip_horizontal(info->curr_absimg);
	ThumbViewSetSize(info, 0.0);
}

static void ThumbViewME_ImgNext(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	if (info->currimg && info->currimg->next)
	{
		picentry_t *pe;

		info->currimg = info->currimg->next;
		pe = (picentry_t *)info->currimg->data;
		ThumbViewImg(info, pe->name);
	}
	else if (info->type == THUMB_PAGE)
	{
		if ((info->curr + THUMB_PAGE_SIZE) > info->num)
			gdk_beep();
		else
		{
			picentry_t *pe;

			info->curr += THUMB_PAGE_SIZE;
			ThumbsDoPage(info);

			info->currimg = info->imglist;
			pe = (picentry_t *)info->currimg->data;
			ThumbViewImg(info, pe->name);
		}
	}
	else
		gdk_beep();
}
	
static void ThumbViewME_ImgPrev(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	if (info->currimg && info->currimg->prev)
	{
		picentry_t *pe;

		info->currimg = info->currimg->prev;
		pe = (picentry_t *)info->currimg->data;
		ThumbViewImg(info, pe->name);
	}
	else if (info->type == THUMB_PAGE)
	{
		if (!info->curr)
			gdk_beep();
		else
		{
			picentry_t *pe;

			info->curr -= THUMB_PAGE_SIZE;
			ThumbsDoPage(info);

			info->currimg = g_list_last(info->imglist);
			pe = (picentry_t *)info->currimg->data;
			ThumbViewImg(info, pe->name);
		}
	}
	else
		gdk_beep();
}

static void _ThumbOpen(cmd, fn)
char *cmd;
char *fn;
{
	char *c,*tc,*p;

	p = strstr(cmd, "%f");

	c = p ? g_strndup(cmd, p-cmd) : g_strdup(cmd);

	tc = c;
	c = g_strconcat(c , p ? "" : " ", fn, p ? p+2 : "", " &", NULL);
	g_free(tc);

	system(c);

	g_free(c);
}

static void ThumbOpen(cmd, fn)
char *cmd;
char *fn;
{
	char *afn;

	afn = g_strconcat("\'", fn, "\'", NULL);

	_ThumbOpen(cmd, afn);

	g_free(afn);
}

static void ThumbOpenMulti(cmd, list)
char *cmd;
GSList *list;
{
	char *allf = NULL, *ts;

 	while (list)
	{
		ts = allf;

		if (!allf)
			allf = g_strconcat("\'", (char *)list->data, "\' ", NULL);
		else
			allf = g_strconcat(allf, "\'", (char *)list->data, "\' ", NULL);

		g_free(ts);

		list = list->next;
	}

	_ThumbOpen(cmd, allf);

	g_free(allf);
}

static void ThumbViewME_Open(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	char *cmd, *fn;

	cmd = (char *) gtk_object_get_user_data(GTK_OBJECT(widget));

	fn = absimg_filename(info->curr_absimg);

	ThumbOpen(cmd, fn);
}

static void ThumbViewSetWalp(info, pe, image, type)
struct thumbpv_info *info;
picentry_t *pe;
absimg_image_t  *image;
rendering_type type;
{
	prop_t prop;

	if (!image)
	{
		gdk_beep();
		return;
	}

	if (pe->prop)
		memcpy(&prop, pe->prop, sizeof(prop_t));
	else
		memcpy(&prop, &cfg.properties, sizeof(prop_t));

	prop.type = type;

	set_root_bg_real(&info->root_window, image, pe->name, 
			 &prop, NULL, 0, 0, TRUE);

	gdk_flush();
	chbg_status_progress(0,1);
}

static void ThumbViewME_SetWalp(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	picentry_t pe;
	rendering_type type;

	type = (rendering_type)gtk_object_get_user_data(GTK_OBJECT(w));

	pe.prop = NULL;
	pe.name = NULL;

	ThumbViewSetWalp(info, &pe, info->curr_absimg, type);
}

static void ThumbViewScroll(info, x, y)
struct thumbpv_info *info;
gdouble x;
gdouble y;
{
	if (info->pmap)
	{
		GtkAdjustment *ha,*va;
		gfloat v,sv;

		ha = GTK_VIEWPORT(info->pmap->parent)->hadjustment;
		va = GTK_VIEWPORT(info->pmap->parent)->vadjustment;

		if (y)
		{
			sv = va->value;
			v =  va->value - y;
			if (v < 0) v = 0;
			if (v > (va->upper - va->page_size))
				v = va->upper - va->page_size;

			if (v != sv)
			{
				gtk_adjustment_set_value(va , v);
				gtk_adjustment_value_changed(va);
			}
		}
		if (x)
		{
			sv = ha->value;
			v =  ha->value - x;
			if (v < 0) v = 0;
			if (v > (ha->upper - ha->page_size)) v = ha->upper - ha->page_size;

			if (v != sv)
			{
				gtk_adjustment_set_value(ha , v);
				gtk_adjustment_value_changed(ha);
			}
		}
	}

	info->drag_x += x;
	info->drag_y += y;
}

static gint ThumbViewEvents(widget , event, info)
GtkWidget *widget;
GdkEvent  *event;
struct thumbpv_info *info;
{
	switch (event->type)
	{
		case GDK_MOTION_NOTIFY:
		{
			if ((info->drag_type == 0) &&
				(info->view_type == 1) && info->drag_start)
			{
				GdkEventMotion *mevent = (GdkEventMotion *)event;
				ThumbViewScroll(info, mevent->x - info->drag_x , mevent->y - info->drag_y);
			}
		}
		break;
		case GDK_CONFIGURE:
		{
			GdkEventConfigure *cevent = (GdkEventConfigure *)event;

			if ((info->view_type == 0) && !info->frst_time &&
			     (cevent->width != info->curr_width ||
			      cevent->height != info->curr_height))
			{
				info->curr_width = cevent->width;
				info->curr_height = cevent->height;
				ThumbViewSetSize(info, 0.0);
			}
			else if (info->frst_time)
				info->frst_time = FALSE;
                }
                break;
		case GDK_BUTTON_PRESS:
		{
			GdkEventButton *bevent = (GdkEventButton *) event;

			if (bevent->button == 3)
			{
				gtk_menu_popup(GTK_MENU(info->view_menu), 
					NULL, NULL, NULL, NULL, 3, bevent->time);
			}
			else if (bevent->button == 1)
			{
				info->drag_start = TRUE;
				info->drag_x = bevent->x;
				info->drag_y = bevent->y;

				gdk_pointer_grab(info->pv->window,
					FALSE,
					GDK_POINTER_MOTION_MASK |
					GDK_BUTTON_RELEASE_MASK,
					NULL, NULL, bevent->time);
			}
		}
		break;
		case GDK_BUTTON_RELEASE:
		{
			GdkEventButton *bevent = (GdkEventButton *) event;

			if ((info->drag_type == 1) &&
				(info->view_type == 1) && info->drag_start)
			{
				ThumbViewScroll(info, bevent->x - info->drag_x , bevent->y - info->drag_y);
			}

			info->drag_start = FALSE;
			gdk_pointer_ungrab(bevent->time);
		}
		break;
		case GDK_KEY_RELEASE:
		{
			GdkEventKey *kevent = (GdkEventKey *)event;
			switch(kevent->keyval)
			{
				case GDK_Q:
				case GDK_q:
				case GDK_Escape:
					ThumbViewME_Close(NULL, info);
				break;
				case GDK_S:
				case GDK_s:
					ThumbViewME_ToggleSlide(NULL, info);
				break;
				case GDK_plus:
				case GDK_KP_Add:
					ThumbViewSetSize(info, 0.10);
				break;
				case GDK_minus:
				case GDK_KP_Subtract:
					ThumbViewSetSize(info, -0.10);
				break;
				case GDK_D:
				case GDK_d:
					ThumbViewSetSize(info, 1.0);
				break;
				case GDK_H:
				case GDK_h:
					ThumbViewSetSize(info, -0.5);
				break;
				case GDK_O:
				case GDK_o:
					ThumbViewME_SizeOrig(NULL, info);
				break;
				case GDK_V:
				case GDK_v:
					ThumbViewME_FlipVert(NULL, info);
				break;
				case GDK_M:
				case GDK_m:
					ThumbViewME_FlipHoriz(NULL, info);
				break;
				case GDK_N:
				case GDK_n:
				case GDK_space:
					ThumbViewME_ImgNext(NULL, info);
				break;
				case GDK_P:
				case GDK_p:
				case GDK_BackSpace:
					ThumbViewME_ImgPrev(NULL, info);
				break;
			}
		}
		default:
		break;
	}
	return TRUE;
}

static void ThumbViewDestroy(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	if (info->normtm)
	{
		gtk_timeout_remove(info->normtm);
		info->normtm = 0;
	}

	if (info->curr_absimg)
	{
		absimg_destroy(info->curr_absimg);
		info->curr_absimg = NULL;
	}

	info->pv = NULL;
	info->pmap = NULL;
}

static void ThumbViewMenuOpenBuild(smenu, info)
GtkWidget *smenu;
struct thumbpv_info *info;
{
	int i;

        while (GTK_MENU_SHELL(smenu)->children)
        {
                gtk_widget_destroy(GTK_WIDGET(GTK_MENU_SHELL(smenu)->children->data));
        }

	for (i = 0; i < CHBG_EXTERNOPENER; i++)
	{
		char pom[1024];
		char *label = NULL, *cmd = NULL;

		sprintf(pom , "option-extop%02d-label", i);
		gprop_get_str(pom, &label);
		sprintf(pom , "option-extop%02d-cmd", i);
		gprop_get_str(pom, &cmd);

		if (cmd)
		{
			GtkWidget *mi;

			mi = gtk_menu_item_new_with_label(label ? label : "???");
			if (label)
			{
				sprintf(pom, "thumb_view/open-with/%s", label);
				gaccel_bind_widget(pom , "activate" , mi, NULL, guitl_menu_parent(GTK_MENU_SHELL(smenu)->parent_menu_shell));
			}
			gtk_object_set_user_data(GTK_OBJECT(mi), cmd);
			gtk_menu_append(GTK_MENU(smenu), mi);
			gtk_widget_show(mi);

			gtk_signal_connect(GTK_OBJECT(mi), "activate",
				GTK_SIGNAL_FUNC(ThumbViewME_Open), info);
		}
	}
}

static void ThumbViewMenu(info)
struct thumbpv_info *info;
{
	GtkWidget *mi, *smenu;

	info->view_menu = gtk_menu_new();
	guitl_menu_attach(info->view_menu, info->pv);
	gtk_widget_realize(info->view_menu);

	mi = gtk_menu_item_new_with_label(gettext("(n) Next"));
	gaccel_bind_widget("thumb_view/next" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_ImgNext) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("(p) Previous"));
	gaccel_bind_widget("thumb_view/prev" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_ImgPrev) , (gpointer)info);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);

	smenu = gtk_menu_new();
	gtk_signal_connect(GTK_OBJECT(smenu), "show",
		GTK_SIGNAL_FUNC(ThumbViewMenuOpenBuild), info);
	gtk_widget_realize(smenu);

	mi = gtk_menu_item_new_with_label(gettext("Open with"));
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), smenu);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);

	smenu = gtk_menu_new();
	gtk_widget_realize(smenu);

	mi = gtk_menu_item_new_with_label(gettext("Set wallpaper"));
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), smenu);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);

	mi = gtk_menu_item_new_with_label(gettext("Tiled"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_TILE);
	gaccel_bind_widget("thumb_view/wp/tiled" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SetWalp) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Tiled-mirrored"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_MIRROR);
	gaccel_bind_widget("thumb_view/wp/tiledmirrore" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SetWalp) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Maximized"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_MAXIMIZE);
	gaccel_bind_widget("thumb_view/wp/maximized" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SetWalp) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Centered"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_CENTER);
	gaccel_bind_widget("thumb_view/wp/centered" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SetWalp) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Smart scaled centered"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_SMART);
	gaccel_bind_widget("thumb_view/wp/smart" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SetWalp) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Center tiled"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_CENTER_TILE);
	gaccel_bind_widget("thumb_view/wp/center_tiled" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SetWalp) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Integer tiled"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_INT_TILE);
	gaccel_bind_widget("thumb_view/wp/int_tiled" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SetWalp) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Symetrical tiled"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_SYM_TILE);
	gaccel_bind_widget("thumb_view/wp/sym_tiled" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SetWalp) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Integer mirrored"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_INT_MIRROR);
	gaccel_bind_widget("thumb_view/wp/int_mirrored" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SetWalp) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Symetrical mirrored"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_SYM_MIRROR);
	gaccel_bind_widget("thumb_view/wp/sym_mirrored" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SetWalp) , (gpointer)info);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);

	mi = gtk_menu_item_new_with_label(gettext("(d) Size double"));
	gaccel_bind_widget("thumb_view/double" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SizeDouble) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("(h) Size half"));
	gaccel_bind_widget("thumb_view/half" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SizeHalf) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("(+) Size +10 %"));
	gaccel_bind_widget("thumb_view/plus10" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SizePlus10) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("(-) Size -10 %"));
	gaccel_bind_widget("thumb_view/minus10" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SizeMinus10) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("(o) Size original"));
	gaccel_bind_widget("thumb_view/orig" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_SizeOrig) , (gpointer)info);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);

	mi = gtk_menu_item_new_with_label(gettext("(m) Flip horizontal"));
	gaccel_bind_widget("thumb_view/horiz" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_FlipHoriz) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("(v) Flip vertical"));
	gaccel_bind_widget("thumb_view/vert" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_FlipVert) , (gpointer)info);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);

	mi = gtk_menu_item_new_with_label(gettext("(s) Toggle slideshow"));
	gaccel_bind_widget("thumb_view/slide" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_ToggleSlide) , (gpointer)info);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);

	mi = gtk_menu_item_new_with_label(gettext("(q) Close"));
	gaccel_bind_widget("thumb_view/close" , "activate" , mi, NULL, info->pv);
	gtk_menu_append(GTK_MENU(info->view_menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME_Close) , (gpointer)info);
}

static void ThumbView(w, info, ppe)
GtkWidget *w;
struct thumbpv_info *info;
picentry_t *ppe;
{
	char *imgname;

	if (!info->pv)
	{
		info->pv = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_policy(GTK_WINDOW(info->pv), TRUE, TRUE, TRUE);
		gtk_signal_connect(GTK_OBJECT(info->pv), "destroy",
			GTK_SIGNAL_FUNC(ThumbViewDestroy), info);
		gtk_widget_add_events(info->pv, GDK_KEY_RELEASE_MASK |
			GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
			GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK);
		gtk_signal_connect(GTK_OBJECT(info->pv), "key_release_event",
				GTK_SIGNAL_FUNC(ThumbViewEvents), info);
		gtk_signal_connect(GTK_OBJECT(info->pv), "configure_event",
				GTK_SIGNAL_FUNC(ThumbViewEvents), info);
		gtk_signal_connect(GTK_OBJECT(info->pv), "button_press_event",
				GTK_SIGNAL_FUNC(ThumbViewEvents), info);
		gtk_signal_connect(GTK_OBJECT(info->pv), "button_release_event",
				GTK_SIGNAL_FUNC(ThumbViewEvents), info);
		gtk_signal_connect(GTK_OBJECT(info->pv), "motion_notify_event",
				GTK_SIGNAL_FUNC(ThumbViewEvents), info);
		if (info->frst)
		{
			gtk_signal_connect(GTK_OBJECT(toplevel_window),
				"destroy", GTK_SIGNAL_FUNC(KillAll), &info->pv);
		}
		info->frst = FALSE;

		ThumbViewMenu(info);

		gprop_get_int("view_window_type", &info->view_type);
		gprop_get_int("view_drag_type", &info->drag_type);

		if (info->view_type)
		{
			info->swinpv = gtk_scrolled_window_new(NULL, NULL);
			gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(info->swinpv),
				GTK_POLICY_NEVER, GTK_POLICY_NEVER);
			gtk_widget_show(info->swinpv);
			gtk_container_add(GTK_CONTAINER(info->pv), info->swinpv);
		}
	}

	if (w)
	{
		picentry_t *pe;
		pe = (picentry_t *)gtk_object_get_user_data(GTK_OBJECT(w));

		info->currimg = g_list_find(info->imglist, pe);
		imgname = pe->name;
	}
	else
	{
		imgname = ppe->name;
	}

	ThumbViewImg(info, imgname);

	if (GTK_WIDGET_REALIZED(info->pv))
		gdk_window_raise(info->pv->window);
	gtk_widget_show(info->pv);
}

static gint ThumbTabBEvents(widget, event, info)
GtkWidget *widget;
GdkEvent  *event;
struct thumbpv_info *info;
{
	GdkEventButton *bevent;

	bevent = (GdkEventButton *) event;

	switch (event->type)
	{
		case GDK_BUTTON_PRESS:
			if (bevent->button == 3)
			{
				GList *ptr;
				int is_selected;

				is_selected = gtk_multirowsel_get_selected(GTK_MULTIROWSEL(info->tab)) != NULL;

				for (ptr = GTK_MENU_SHELL(info->menu)->children ; ptr ; ptr = ptr->next)
				{
					if (gtk_object_get_data(GTK_OBJECT(ptr->data), "senzible"))
						gtk_widget_set_sensitive(GTK_WIDGET(ptr->data), is_selected);
				}

				gtk_menu_popup (GTK_MENU(info->menu), 
					NULL, NULL, NULL, NULL, 3, bevent->time);
			}
		break;
		default:
		break;
	
	}
	return FALSE;
}

static gint ThumbBEvents(widget, event, info)
GtkWidget *widget;
GdkEvent  *event;
struct thumbpv_info *info;
{
	GdkEventButton *bevent;

	bevent = (GdkEventButton *) event;

	switch (event->type)
	{
		case GDK_2BUTTON_PRESS:
			{
				if (bevent->button == 1)
					ThumbView(widget, info, NULL);
				else if (bevent->button == 2)
					ThumbViewFull(widget, info, NULL);
			}
			break;
		default:
			break;
	}
	return FALSE;
}

static void ThumbsDo(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	GtkWidget *ptab;
	int num = 0,i;
	GList *ptr;
	GtkTooltips *tt;

	gtk_widget_set_sensitive(info->refreshb, FALSE);
	if (info->optb)
		gtk_widget_set_sensitive(info->optb, FALSE);
	if (info->combo)
		gtk_widget_set_sensitive(info->combo, FALSE);
	if (info->prevb)
		gtk_widget_set_sensitive(info->prevb, FALSE);
	if (info->nextb)
		gtk_widget_set_sensitive(info->nextb, FALSE);
	gtk_widget_show(info->progress);

	if (info->tab)
		gtk_widget_destroy(info->tab);

	if (!info->tt)
		info->tt = gtk_tooltips_new();
	tt = info->tt;

	num = g_list_length(info->imglist);

	info->tab = ptab = gtk_multirowsel_new(5);
	gtk_signal_connect(GTK_OBJECT(ptab), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &info->tab);
	gtk_signal_connect(GTK_OBJECT(ptab), "button_press_event",
		GTK_SIGNAL_FUNC(ThumbTabBEvents), info);

	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(info->swin), ptab);
	gtk_widget_show(ptab);

	i = 0;
	ptr = info->imglist;
	while (ptr)
	{
		GtkWidget *pmap,*frame;
		absimg_image_t *img;
		guint w,h;
		gfloat scale = 1.0;
		GdkPixmap *pixmap;
		picentry_t *pe = (picentry_t *) ptr->data;

		gtk_progress_bar_update(GTK_PROGRESS_BAR(info->progress), ((gfloat)i)/(gfloat)num);
		while (gtk_events_pending()) gtk_main_iteration();

		if (!info->tl)
			break;

		frame = gtk_selbox_new(TRUE, TRUE);
		gtk_container_add(GTK_CONTAINER(info->tab), frame);
		gtk_widget_show(frame);

		gtk_object_set_user_data(GTK_OBJECT(frame), pe);

		gtk_signal_connect(GTK_OBJECT(frame), "button_press_event",
			GTK_SIGNAL_FUNC(ThumbBEvents), info);

		img = absimg_load(pe->name);
		if (img)
		{
			char *p;
			struct stat estat;


			w = MIN(img->rgb_width, THUMB_SIZE);
			h = MIN(img->rgb_height, THUMB_SIZE);

			if (img->rgb_width > w)
				scale = (gfloat)w/(gfloat)img->rgb_width;
			if (img->rgb_height > h)
				scale = MIN((gfloat)h/(gfloat)img->rgb_height, scale);

			w = (guint)ceil(scale * img->rgb_width);
			h = (guint)ceil(scale * img->rgb_height);

			w = MIN(w, THUMB_SIZE);
			h = MIN(h, THUMB_SIZE);

			p = strrchr(pe->name, '/');
			if (p) p++;
			else p = pe->name;

			stat(pe->name, &estat);

			p = g_strdup_printf("Image:  %s\n"
					    "Dimensions:  %dx%d\n"
					    "Size:  %.1f kB",
					     p, img->rgb_width,
					     img->rgb_height, 
					     ((float)estat.st_size)/1024.0);
			gtk_tooltips_set_tip(tt, frame, p, NULL);
			g_free(p);

			gtk_object_set_data(GTK_OBJECT(frame),
				"pic_width", (gpointer)img->rgb_width);
			gtk_object_set_data(GTK_OBJECT(frame),
				"pic_height", (gpointer)img->rgb_height);
			gtk_object_set_data(GTK_OBJECT(frame),
				"pic_size", (gpointer)estat.st_size);

			if (absimg_have_alpha(img))
			{
				GdkGC *gc;

				gc = gdk_gc_new(GDK_ROOT_PARENT());
				gdk_gc_set_foreground(gc, &(frame->style->bg[GTK_STATE_NORMAL]));

				pixmap = gdk_pixmap_new(GDK_ROOT_PARENT(),
					w, h, gdk_visual_get_best_depth());
				gdk_draw_rectangle(pixmap, gc, TRUE, 0, 0, w, h);
				absimg_render_alpha_to_pixmap(pixmap, img, w, h);
				gdk_gc_unref(gc);
			}
			else
				pixmap = absimg_render(img, w, h);

			pmap = gtk_bg_pixmap_new(pixmap);
			gtk_container_add(GTK_CONTAINER(frame), pmap);
			gtk_widget_show(pmap);

			absimg_destroy_pixmap(pixmap);
			absimg_destroy(img);
		}
		else
			gtk_tooltips_set_tip(tt, frame, pe->name, NULL);

		i++;
		ptr = ptr->next;
	}
	if (info->tl)
	{
		gtk_widget_set_sensitive(info->refreshb, TRUE);
		if (info->optb)
			gtk_widget_set_sensitive(info->optb, TRUE);
		if (info->combo)
			gtk_widget_set_sensitive(info->combo, TRUE);
		if (info->prevb)
			gtk_widget_set_sensitive(info->prevb, TRUE);
		if (info->nextb)
			gtk_widget_set_sensitive(info->nextb, TRUE);
		gtk_widget_hide(info->progress);
	}
}

static void ThumbsListFree(info)
struct thumbpv_info *info;
{
	GList *ptr;
	GSList *sptr;

	ptr = info->alllist;
	while (ptr)
	{
		picentry_free((picentry_t *) ptr->data);
		ptr = g_list_remove_link(ptr, ptr);
	}
	info->alllist = NULL;

	if (info->imglist) 
	{
		g_list_free(info->imglist);
		info->imglist = NULL;
	}

	sptr = info->dirlist;
	while (sptr)
	{
		g_free(sptr->data);
		sptr = g_slist_remove_link(sptr, sptr);
	}
	info->dirlist = NULL;
}

static void ThumbsListInit(info)
struct thumbpv_info *info;
{
	int row;
	picentry_t *pe;
	prop_t *pprop;
	char *p;

	info->alllist = NULL;
	info->dirlist = NULL;

	for (row = 0 ; row < GTK_CLIST(picture_list)->rows ; row++)
	{
		gtk_clist_get_text(GTK_CLIST(picture_list) , row , 0 , &p);
		pe = picentry_new(g_strdup(p));

		pprop = gtk_clist_get_row_data(GTK_CLIST(picture_list) , row);
		if (pprop) pe->prop = prop_dup(pprop);

		info->alllist = g_list_append(info->alllist , (gpointer)pe);
	}
	
	info->num = g_list_length(info->alllist);

	recurse_dir(&(info->alllist), &(info->num), cfg.pattern, cfg.min_size,
		(recurse_getname_func)picentry_get_name,
		(recurse_newentry_func)picentry_clone,
		(recurse_free_func)picentry_free);
}

static void ThumbsDoRefresh(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	gtk_window_set_title(GTK_WINDOW(info->tl),
		gettext("Chbg: all thumbnails preview (wait ...)"));

	while (gtk_events_pending()) gtk_main_iteration();

	ThumbsListFree(info);
	ThumbsListInit(info);
	info->imglist = NULL;
	info->currimg = NULL;
	info->curr = 0;
	info->imglist = g_list_copy(info->alllist);

	ThumbsDo(NULL, info);

	if (info->tl)
		gtk_window_set_title(GTK_WINDOW(info->tl),
			gettext("Chbg: all thumbnails preview"));
}

static void ThumbWinClose(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	if (info->pv)
		gtk_widget_destroy(info->pv);

	ThumbsListFree(info);

	if (gtk_main_level() == 0)
		gtk_exit(0);

	if (cfg.thumb && 
	    (gtk_main_level() > 0))
	{
		gtk_main_quit();
	}
}

static void ThumbViewME(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	GtkWidget *sw = gtk_multirowsel_get_selected(GTK_MULTIROWSEL(info->tab));

	if (sw)
		ThumbView(sw, info);
}

static void ThumbViewFullME(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	GtkWidget *sw = gtk_multirowsel_get_selected(GTK_MULTIROWSEL(info->tab));

	if (sw)
		ThumbViewFull(sw, info, NULL);
}


static void ThumbViewWalpME(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	picentry_t *pe;
	absimg_image_t *image;
	rendering_type type;
	GtkWidget *sw = gtk_multirowsel_get_selected(GTK_MULTIROWSEL(info->tab));

	if (!sw)
		return;

	pe = (picentry_t *)gtk_object_get_user_data(GTK_OBJECT(sw));
	type = (rendering_type)gtk_object_get_user_data(GTK_OBJECT(w));

	image = absimg_load(pe->name);

	ThumbViewSetWalp(info, pe, image, type);

	absimg_destroy(image);
}

static void ThumbME_Open(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	picentry_t *pe;
	char *cmd;
	GSList *sel = gtk_multirowsel_get_selection(GTK_MULTIROWSEL(info->tab));
	GSList *sf = NULL,*ptr;

	cmd = (char *) gtk_object_get_user_data(GTK_OBJECT(widget));

	for (ptr = sel; ptr ; ptr = ptr->next)
	{
		pe = (picentry_t *)gtk_object_get_user_data(GTK_OBJECT(ptr->data));
		sf = g_slist_append(sf, pe->name);
	}

	ThumbOpenMulti(cmd, sf);

	while(sf) sf = g_slist_remove_link(sf, sf);

	gtk_multirowsel_kill_selection(sel);
}

static void ThumbNoRemovePic(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	gtk_selbox_set_selected(GTK_SELBOX(
		gtk_multirowsel_get_selected(GTK_MULTIROWSEL(info->tab))), FALSE);
/*
	KillWin(widget, gtk_widget_get_toplevel(widget));
*/
	gtk_main_quit();
}

static void ThumbRemovePic(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	int i;
	picentry_t *pe;
	GtkWidget *sw;
	GList *ptr;
	GSList *sptr;

	sw = gtk_multirowsel_get_selected(GTK_MULTIROWSEL(info->tab));

	pe = (picentry_t *)gtk_object_get_user_data(GTK_OBJECT(sw));

	if (unlink(pe->name))
	{
		gdk_beep();
	}
	else
	{
		for (i = GTK_CLIST(picture_list)->rows - 1; i >= 0 ; i--)
		{
			char *p;

			gtk_clist_get_text(GTK_CLIST(picture_list), i, 0, &p);
			if (!strcmp(p, pe->name))
			{
				gtk_clist_remove(GTK_CLIST(picture_list), i);
			}
		}

		gtk_selbox_set_selected(GTK_SELBOX(sw), FALSE);
		gtk_widget_destroy(sw);

		if (widget)
		{
			/*gtk_widget_destroy(gtk_widget_get_toplevel(widget));*/
			gtk_main_quit();
		}

		if ((ptr = g_list_find(info->imglist, pe)))
			info->imglist = g_list_remove_link(info->imglist, ptr);
		if ((sptr = g_slist_find(info->dirlist, pe)))
			info->dirlist = g_slist_remove_link(info->dirlist, sptr);
		if ((ptr = g_list_find(info->alllist, pe)))
			info->alllist = g_list_remove_link(info->alllist, ptr);

		if (pe->prop) g_free(pe->prop);
		g_free(pe->name);
		g_free(pe);
	}
}

static void ThumbME_Remove(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	picentry_t *pe;
	GtkWidget *dia = NULL, *button;
	GtkWidget *pix = NULL,*label = NULL;
	guint confirmrm;
	GSList *sel,*ptr;

	if (!gprop_get_bool("confirm_remove", &confirmrm))
		confirmrm = TRUE;

	sel = gtk_multirowsel_get_selection(GTK_MULTIROWSEL(info->tab));

	for (ptr = sel ; ptr ; ptr = ptr->next)
	{
		GtkWidget *sw,*box,*frame;

		sw = GTK_WIDGET(ptr->data);
		pe = (picentry_t *)gtk_object_get_user_data(GTK_OBJECT(sw));

		if (dia)
		{
			gtk_label_set(GTK_LABEL(label), pe->name);
			if (GTK_BIN(sw)->child)
				gtk_bg_pixmap_set(GTK_BG_PIXMAP(pix), GTK_BG_PIXMAP(GTK_BIN(sw)->child)->pixmap);
			else
				gtk_bg_pixmap_set(GTK_BG_PIXMAP(pix), NULL);
		}

		if (!dia && confirmrm)
		{
			dia = gtk_dialog_new();
			gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dia)->vbox), 5);
			gtk_window_set_title(GTK_WINDOW(dia), gettext("Remove picture ?"));
			gtk_window_position(GTK_WINDOW(dia), GTK_WIN_POS_MOUSE);
			gtk_window_set_modal(GTK_WINDOW(dia), TRUE);

			frame = gtk_frame_new(gettext("Do you realy want to remove this picture?"));
			gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dia)->vbox), frame, FALSE, FALSE, 2);
			gtk_widget_show(frame);

			box = gtk_hbox_new(FALSE, 2);
			gtk_container_add(GTK_CONTAINER(frame), box);
			gtk_widget_show(box);

			frame = gtk_frame_new(NULL);
			gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 2);
			gtk_widget_show(frame);

			if (GTK_BIN(sw)->child)
				pix = gtk_bg_pixmap_new(GTK_BG_PIXMAP(GTK_BIN(sw)->child)->pixmap);
			else
				pix = gtk_bg_pixmap_new(NULL);
			gtk_container_add(GTK_CONTAINER(frame), pix);
			gtk_widget_show(pix);

			label = gtk_label_new(pe->name);
			gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 2);
			gtk_widget_show(label);

			button = guitl_pixmap_button(stock_apply_xpm, gettext("Yes"));
			gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dia)->action_area), button, FALSE, FALSE, 1);
			gtk_widget_show(button);
			gtk_signal_connect(GTK_OBJECT(button), "clicked",
					GTK_SIGNAL_FUNC(ThumbRemovePic), info);

			button = guitl_pixmap_button(stock_close_xpm, gettext("No"));
			gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dia)->action_area), button, FALSE, FALSE, 1);
			gtk_widget_show(button);
			gtk_signal_connect(GTK_OBJECT(button), "clicked",
					GTK_SIGNAL_FUNC(ThumbNoRemovePic), info);

			gtk_widget_show(dia);

		}
		else if (!confirmrm)
		{
			ThumbRemovePic(NULL, info);
		}
		if (dia)
			gtk_main();
	}
	gtk_multirowsel_kill_selection(sel);
	if (dia)
		gtk_widget_destroy(dia);
}

static void ThumbRenameSetNext(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	GtkWidget *tl,*orig,*new,*pix,*sw;
	GSList *sel,*sel_list;

	tl = gtk_widget_get_toplevel(widget);
	pix = gtk_object_get_data(GTK_OBJECT(tl), "pix");
	orig = gtk_object_get_data(GTK_OBJECT(tl), "orig_entry");
	new = gtk_object_get_data(GTK_OBJECT(tl), "new_entry");
	sel = gtk_object_get_data(GTK_OBJECT(tl), "sel");
	sel_list = gtk_object_get_data(GTK_OBJECT(tl), "sel_list");

	if (!sel)
	{
		sel = sel_list;
	}
	else
	{
		sel = sel->next;
	}

	if (sel)
	{
		picentry_t *pe;
		char *p;

		sw = GTK_WIDGET(sel->data);
		pe = (picentry_t *)gtk_object_get_user_data(GTK_OBJECT(sw));

		if (GTK_BIN(sw)->child)
			gtk_bg_pixmap_set(GTK_BG_PIXMAP(pix), GTK_BG_PIXMAP(GTK_BIN(sw)->child)->pixmap);
		else
			gtk_bg_pixmap_set(GTK_BG_PIXMAP(pix), NULL);
			
		if ((p = strrchr(pe->name, '/')))
			p++;
		else
			p = pe->name;
		gtk_entry_set_text(GTK_ENTRY(orig), p);
		gtk_entry_set_text(GTK_ENTRY(new), p);
		gtk_entry_select_region(GTK_ENTRY(new), 0, strlen(p));
		gtk_widget_grab_focus(new);
	}

	gtk_object_set_data(GTK_OBJECT(tl), "sel", sel);

	if (!sel)
	{
		gtk_widget_destroy(tl);
	}
}

static void ThumbRename(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	GtkWidget *tl,*new,*orig,*sw;
	GSList *sel;

	tl = gtk_widget_get_toplevel(widget);
	new = gtk_object_get_data(GTK_OBJECT(tl), "new_entry");
	orig = gtk_object_get_data(GTK_OBJECT(tl), "orig_entry");
	sel = gtk_object_get_data(GTK_OBJECT(tl), "sel");

	if (sel)
	{
		picentry_t *pe;
		char *p,*p1;

		sw = GTK_WIDGET(sel->data);
		pe = (picentry_t *)gtk_object_get_user_data(GTK_OBJECT(sw));

		p = gtk_entry_get_text(GTK_ENTRY(new));
		p1 = gtk_entry_get_text(GTK_ENTRY(orig));

		if (strcmp(p1, p))
		{
			p1 = g_strndup(pe->name, strlen(pe->name) - strlen(p1));
			p = g_strconcat(p1, p, NULL);
			g_free(p1);

			if (!rename(pe->name, p))
			{
				int w,h,size,i;

				p1 = pe->name;
				pe->name = p;

				if ((p = strrchr(p, '/')))
					p++;
				else
					p = pe->name;

				w = (int) gtk_object_get_data(GTK_OBJECT(sw), "pic_width");
				h = (int) gtk_object_get_data(GTK_OBJECT(sw), "pic_height");
				size = (int) gtk_object_get_data(GTK_OBJECT(sw), "pic_size");
				p = g_strdup_printf("Image:  %s\n"
					"Dimensions:  %dx%d\n"
					"Size:  %.1f kB",
					p, w, h, 
					((float)size)/1024.0);
				gtk_tooltips_set_tip(info->tt, sw, p, NULL);
				g_free(p);

				for (i = GTK_CLIST(picture_list)->rows - 1; i >= 0 ; i--)
				{
					char *pom;

					gtk_clist_get_text(GTK_CLIST(picture_list), i, 0, &pom);
					if (!strcmp(p1, pom))
					{
						gtk_clist_set_text(GTK_CLIST(picture_list), i, 0, pe->name);
					}
				}

				g_free(p1);

				ThumbRenameSetNext(tl, info);
			}
			else
			{
				gdk_beep();
				g_free(p);
			}

		}
		else
		{
			ThumbRenameSetNext(tl, info);
		}
	}
	else
	{
		gdk_beep();
	}
}

static void ThumbME_Rename(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	GtkWidget *tl,*box,*entry,*pix,*btn,*bbox,*frame,*label;
	GSList *sel;

	sel = gtk_multirowsel_get_selection(GTK_MULTIROWSEL(info->tab));

	if (!sel)
	{
		gdk_beep();
		return;
	}

	tl = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_container_set_border_width(GTK_CONTAINER(tl), 4);
	gtk_window_set_title(GTK_WINDOW(tl), gettext("ChBg: rename image"));
	gtk_object_set_data_full(GTK_OBJECT(tl), "sel_list", sel,
		(GtkDestroyNotify)gtk_multirowsel_kill_selection);

	box = gtk_vbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(tl), box);
	gtk_widget_show(box);

	frame = gtk_frame_new(gettext("Image to rename"));
	gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);

	bbox = gtk_table_new(2, 3, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), bbox);
	gtk_widget_show(bbox);
	
	frame = gtk_frame_new(NULL);
	gtk_table_attach(GTK_TABLE(bbox), frame, 0, 2, 0, 1, GTK_SHRINK, GTK_SHRINK, 2, 2);
	gtk_widget_show(frame);

	pix = gtk_bg_pixmap_new(NULL);
	gtk_container_add(GTK_CONTAINER(frame), pix);
	gtk_widget_show(pix);
	gtk_object_set_data(GTK_OBJECT(tl), "pix", pix);

	label = gtk_label_new(gettext("Original name:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(bbox), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	gtk_widget_show(label);

	entry = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
	gtk_table_attach(GTK_TABLE(bbox), entry, 1, 2, 1, 2, GTK_FILL|GTK_EXPAND, GTK_FILL, 2, 2);
	gtk_widget_show(entry);
	gtk_object_set_data(GTK_OBJECT(tl), "orig_entry", entry);

	label = gtk_label_new(gettext("New name:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(bbox), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 2, 2);
	gtk_widget_show(label);

	entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(bbox), entry, 1, 2, 2, 3, GTK_FILL|GTK_EXPAND, GTK_FILL, 2, 2);
	gtk_widget_show(entry);
	gtk_object_set_data(GTK_OBJECT(tl), "new_entry", entry);
	gtk_signal_connect(GTK_OBJECT(entry), "activate",
		GTK_SIGNAL_FUNC(ThumbRename), info);

	bbox = gtk_hbutton_box_new();
	gtk_box_pack_start(GTK_BOX(box), bbox, FALSE, FALSE, 2);
	gtk_widget_show(bbox);

	btn = guitl_pixmap_button(stock_ok_xpm, gettext("Rename"));
	gtk_container_add(GTK_CONTAINER(bbox), btn);
	GTK_WIDGET_SET_FLAGS(btn, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(btn);
	gtk_widget_show(btn);
	gtk_signal_connect(GTK_OBJECT(btn), "clicked",
		GTK_SIGNAL_FUNC(ThumbRename), info);

	btn = guitl_pixmap_button(stock_stop_xpm, gettext("Don't rename"));
	gtk_container_add(GTK_CONTAINER(bbox), btn);
	GTK_WIDGET_SET_FLAGS(btn, GTK_CAN_DEFAULT);
	gtk_widget_show(btn);
	gtk_signal_connect(GTK_OBJECT(btn), "clicked",
		GTK_SIGNAL_FUNC(ThumbRenameSetNext), info);

	btn = guitl_pixmap_button(stock_close_xpm, gettext("Cancel"));
	gtk_container_add(GTK_CONTAINER(bbox), btn);
	GTK_WIDGET_SET_FLAGS(btn, GTK_CAN_DEFAULT);
	gtk_widget_show(btn);
	gtk_signal_connect(GTK_OBJECT(btn), "clicked",
		GTK_SIGNAL_FUNC(KillWin), tl);

	gtk_widget_show(tl);
	if (GTK_WIDGET_REALIZED(tl))
		gdk_window_raise(tl->window);

	ThumbRenameSetNext(tl, info);
}

static void ThumbME_Unselect(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	gtk_multirowsel_set_selected_all(GTK_MULTIROWSEL(info->tab), FALSE);
}

static void ThumbME_SelectAll(widget, info)
GtkWidget *widget;
struct thumbpv_info *info;
{
	gtk_multirowsel_set_selected_all(GTK_MULTIROWSEL(info->tab), TRUE);
}

static GSList *Thumb_get_selected_images(info)
struct thumbpv_info *info;
{
	GSList *sel = gtk_multirowsel_get_selection(GTK_MULTIROWSEL(info->tab));
	GSList *ret = NULL;
	GSList *ptr;

	for (ptr = sel; ptr; ptr = ptr->next)
	{
		picentry_t *pe;

		pe = (picentry_t *)gtk_object_get_user_data(GTK_OBJECT(ptr->data));
		ret = g_slist_append(ret, g_strdup(pe->name));
	}
	gtk_multirowsel_kill_selection(sel);
	return ret;
}

static void Thumb_ext_scale_image_dlg(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	GSList *sel = Thumb_get_selected_images(info);

	if (sel)
	{
		ext_scale_image_dlg(FALSE, sel);
	}
	else
		gdk_beep();
}

static void Thumb_ext_convert_image_dlg(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	GSList *sel = Thumb_get_selected_images(info);

	if (sel)
	{
		ext_convert_image_dlg(FALSE, sel);
	}
	else
		gdk_beep();
}

static void ThumbMenuOpenBuild(smenu, info)
GtkWidget *smenu;
struct thumbpv_info *info;
{
	int i;

        while (GTK_MENU_SHELL(smenu)->children)
        {
                gtk_widget_destroy(GTK_WIDGET(GTK_MENU_SHELL(smenu)->children->data));
        }

	for (i = 0; i < CHBG_EXTERNOPENER; i++)
	{
		char pom[1024];
		char *label = NULL, *cmd = NULL;

		sprintf(pom , "option-extop%02d-label", i);
		gprop_get_str(pom, &label);
		sprintf(pom , "option-extop%02d-cmd", i);
		gprop_get_str(pom, &cmd);

		if (cmd)
		{
			GtkWidget *mi;

			mi = gtk_menu_item_new_with_label(label ? label : "???");
			if (label)
			{
				sprintf(pom, "thumb/open-with/%s", label);
				gaccel_bind_widget(pom , "activate" , mi, NULL, guitl_menu_parent(GTK_MENU_SHELL(smenu)->parent_menu_shell));
			}
			gtk_object_set_user_data(GTK_OBJECT(mi), cmd);
			gtk_menu_append(GTK_MENU(smenu), mi);
			gtk_widget_show(mi);

			gtk_signal_connect(GTK_OBJECT(mi), "activate",
				GTK_SIGNAL_FUNC(ThumbME_Open), info);
		}
	}
}

static void ThumbMenu(info)
struct thumbpv_info *info;
{
	GtkWidget *mi, *smenu;

	info->menu = gtk_menu_new();
	guitl_menu_attach(info->menu, info->tl);
	gtk_widget_realize(info->menu);

	mi = gtk_menu_item_new_with_label(gettext("View ..."));
	gtk_object_set_data(GTK_OBJECT(mi), "senzible", (gpointer)TRUE);
	gaccel_bind_widget("thumb/view" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewME) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("View fullscreen ..."));
	gtk_object_set_data(GTK_OBJECT(mi), "senzible", (gpointer)TRUE);
	gaccel_bind_widget("thumb/viewfull" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewFullME) , (gpointer)info);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);

	mi = gtk_menu_item_new_with_label(gettext("Select all"));
	gaccel_bind_widget("thumb/select_all" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbME_SelectAll) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Unselect"));
	gaccel_bind_widget("thumb/uselect" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbME_Unselect) , (gpointer)info);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);


	mi = gtk_menu_item_new_with_label(gettext("Remove ..."));
	gtk_object_set_data(GTK_OBJECT(mi), "senzible", (gpointer)TRUE);
	gaccel_bind_widget("thumb/remove" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbME_Remove) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Rename ..."));
	gtk_object_set_data(GTK_OBJECT(mi), "senzible", (gpointer)TRUE);
	gaccel_bind_widget("thumb/rename" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbME_Rename) , (gpointer)info);


	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);

	smenu = gtk_menu_new();
	gtk_widget_realize(smenu);

	mi = gtk_menu_item_new_with_label(gettext("Set wallpaper"));
	gtk_object_set_data(GTK_OBJECT(mi), "senzible", (gpointer)TRUE);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), smenu);
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);

	mi = gtk_menu_item_new_with_label(gettext("Tiled"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_TILE);
	gaccel_bind_widget("thumb/wp/tiled" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewWalpME) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Tiled-mirrored"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_MIRROR);
	gaccel_bind_widget("thumb/wp/tiledmirrore" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewWalpME) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Maximized"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_MAXIMIZE);
	gaccel_bind_widget("thumb/wp/maximized" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewWalpME) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Centered"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_CENTER);
	gaccel_bind_widget("thumb/wp/centered" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewWalpME) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Smart scaled centered"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_SMART);
	gaccel_bind_widget("thumb/wp/smart" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewWalpME) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Center tiled"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_CENTER_TILE);
	gaccel_bind_widget("thumb/wp/center_tiled" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewWalpME) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Integer tiled"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_INT_TILE);
	gaccel_bind_widget("thumb/wp/int_tiled" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewWalpME) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Symetrical tiled"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_SYM_TILE);
	gaccel_bind_widget("thumb/wp/sym_tiled" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewWalpME) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Integer mirrored"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_INT_MIRROR);
	gaccel_bind_widget("thumb/wp/int_mirrored" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewWalpME) , (gpointer)info);

	mi = gtk_menu_item_new_with_label(gettext("Symetrical mirrored"));
	gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)RENDERT_SYM_MIRROR);
	gaccel_bind_widget("thumb/wp/sym_mirrored" , "activate" , mi, NULL, info->tl);
	gtk_menu_append(GTK_MENU(smenu) , mi);
	gtk_widget_show(mi);
	gtk_signal_connect(GTK_OBJECT(mi) , "activate" , 
		GTK_SIGNAL_FUNC(ThumbViewWalpME) , (gpointer)info);

	smenu = gtk_menu_new();
	gtk_signal_connect(GTK_OBJECT(smenu), "show",
		GTK_SIGNAL_FUNC(ThumbMenuOpenBuild), info);
	gtk_widget_realize(smenu);

	mi = gtk_menu_item_new_with_label(gettext("Open with"));
	gtk_object_set_data(GTK_OBJECT(mi), "senzible", (gpointer)TRUE);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), smenu);
	gtk_menu_append(GTK_MENU(info->menu) , mi);
	gtk_widget_show(mi);

	if (check_program("convert"))
	{
		smenu = gtk_menu_new();
		gtk_widget_realize(smenu);

		mi = gtk_menu_item_new_with_label(gettext("Tools"));
		gtk_object_set_data(GTK_OBJECT(mi), "senzible", (gpointer)TRUE);
		gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), smenu);
		gtk_menu_append(GTK_MENU(info->menu) , mi);
		gtk_widget_show(mi);

		mi = gtk_menu_item_new_with_label(gettext("Batch rescaling ..."));
		gtk_menu_append(GTK_MENU(smenu) , mi);
		gtk_widget_show(mi);
		gtk_signal_connect(GTK_OBJECT(mi), "activate",
			GTK_SIGNAL_FUNC(Thumb_ext_scale_image_dlg), info);

		mi = gtk_menu_item_new_with_label(gettext("Batch conversion ..."));
		gtk_menu_append(GTK_MENU(smenu) , mi);
		gtk_widget_show(mi);
		gtk_signal_connect(GTK_OBJECT(mi), "activate",
			GTK_SIGNAL_FUNC(Thumb_ext_convert_image_dlg), info);
	}
}

void ThumbPreview(w, data)
GtkWidget *w;
gpointer data;
{
	static GtkWidget *thumbpv = NULL;
	static int frst = TRUE;
	static struct thumbpv_info info = {0};

	if (!thumbpv)
	{
		GtkWidget *col, *button, *box, *swin;

		if (info.type == THUMB_NONE)
		{
			ThumbInfoInit(&info);
			info.type = THUMB_ALL;
		}

		info.tl = thumbpv = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title(GTK_WINDOW(thumbpv), gettext("Chbg: thumbnails preview (wait ...)"));
		gtk_signal_connect(GTK_OBJECT(thumbpv), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &thumbpv);
		gtk_signal_connect(GTK_OBJECT(thumbpv), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &info.tl);
		gtk_signal_connect(GTK_OBJECT(thumbpv), "destroy",
			GTK_SIGNAL_FUNC(ThumbWinClose), &info);

		ThumbMenu(&info);

		if (frst)
		{
			gtk_signal_connect(GTK_OBJECT(toplevel_window), "destroy",
				GTK_SIGNAL_FUNC(KillAll), &thumbpv);
			frst = FALSE;
		}

		col = gtk_vbox_new(FALSE, 2);
		gtk_container_add(GTK_CONTAINER(thumbpv), col);
		gtk_widget_show(col);

		info.swin = swin = gtk_scrolled_window_new(NULL, NULL);
		gtk_widget_set_usize(swin , 445 , 395);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
			GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
		gtk_widget_show(swin);
		gtk_container_add(GTK_CONTAINER(col), swin);

		info.progress = gtk_progress_bar_new();
		gtk_box_pack_start(GTK_BOX(col), info.progress, FALSE, FALSE, 2) ;

		box = gtk_hbutton_box_new();
		gtk_box_pack_start(GTK_BOX(col), box, FALSE, FALSE, 2);
		gtk_widget_show(box);

		button = guitl_pixmap_button(stock_close_xpm, gettext("Close"));
		gtk_box_pack_end(GTK_BOX(box) , button , FALSE , TRUE , 0);
		gtk_widget_show(button);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(KillWin), thumbpv);

		info.refreshb = button = guitl_pixmap_button(stock_restart_xpm, gettext("Refresh"));
		gtk_box_pack_end(GTK_BOX(box) , button , FALSE , TRUE , 0);
		gtk_widget_show(button);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(ThumbsDoRefresh), &info);

		gtk_widget_show(thumbpv);
		while (gtk_events_pending()) gtk_main_iteration();

		ThumbsDoRefresh(NULL, &info);
	}
	else
		gtk_widget_show(thumbpv);

	if (thumbpv && GTK_WIDGET_REALIZED(thumbpv))
		gdk_window_raise(thumbpv->window);
}


static void ThumbsDoPage(info)
struct thumbpv_info *info;
{
	GList *ptr;
	int i;

	gtk_window_set_title(GTK_WINDOW(info->tl),
		gettext("Chbg: page thumbnails preview (wait ...)"));

	while (gtk_events_pending()) gtk_main_iteration();

	if (info->imglist) 
	{
		g_list_free(info->imglist);
		info->imglist = NULL;
		info->currimg = NULL;
	}

	for (ptr = g_list_nth(info->alllist, info->curr), i = 0;
		ptr && i < THUMB_PAGE_SIZE;
		ptr = ptr->next, i++)
	{
		info->imglist = g_list_append(info->imglist, ptr->data);
	}

	ThumbsDo(NULL, info);

	if (info->tl)
	{
		if (g_list_nth(info->alllist, info->curr + THUMB_PAGE_SIZE))
			gtk_widget_set_sensitive(info->nextb, TRUE);
		else
			gtk_widget_set_sensitive(info->nextb, FALSE);

		if (info->curr > 0)
			gtk_widget_set_sensitive(info->prevb, TRUE);
		else
			gtk_widget_set_sensitive(info->prevb, FALSE);

		gtk_option_menu_set_history(GTK_OPTION_MENU(info->optb), info->curr/THUMB_PAGE_SIZE);

		gtk_window_set_title(GTK_WINDOW(info->tl),
			gettext("Chbg: page thumbnails preview"));
	}
}

static void ThumbsDoNextPage(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	info->curr += THUMB_PAGE_SIZE;
	ThumbsDoPage(info);
}

static void ThumbsDoPrevPage(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	if (info->curr > 0)
		info->curr -= THUMB_PAGE_SIZE;
	ThumbsDoPage(info);
}

static void ThumbsDoSetPage(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	info->curr = (int) gtk_object_get_user_data(GTK_OBJECT(w));
	ThumbsDoPage(info);
}


static void ThumbsDoFirstPage(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	GtkWidget *menu,*mi;
	int i;

	ThumbsListFree(info);
	ThumbsListInit(info);
	info->imglist = NULL;
	info->curr = 0;

	if (GTK_OPTION_MENU(info->optb)->menu)
		gtk_widget_destroy(GTK_OPTION_MENU(info->optb)->menu);

	menu = gtk_menu_new();
	for (i = 0; i < info->num ; i += THUMB_PAGE_SIZE)
	{
		char pom[100];

		sprintf(pom , "%d - %d", i , 
			(i + THUMB_PAGE_SIZE - 1) < info->num ?
				(i + THUMB_PAGE_SIZE - 1) :
				info->num - 1);

		mi = gtk_menu_item_new_with_label(pom);
		gtk_menu_append(GTK_MENU(menu), mi);
		gtk_widget_show(mi);

		gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)i);

		gtk_signal_connect(GTK_OBJECT(mi), "activate" ,
			GTK_SIGNAL_FUNC(ThumbsDoSetPage) ,(gpointer)info); 
	}

	gtk_option_menu_set_menu(GTK_OPTION_MENU(info->optb), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(info->optb), 0);
	gtk_widget_show(info->optb);

	ThumbsDoPage(info);
}

void ThumbPreviewPerPage(w, data)
GtkWidget *w;
gpointer data;
{
	static GtkWidget *thumbpv = NULL;
	static int frst = TRUE;
	static struct thumbpv_info info = {0};

	if (!thumbpv)
	{
		GtkWidget *col, *button, *box, *swin;

		if (info.type == THUMB_NONE)
		{
			ThumbInfoInit(&info);
			info.type = THUMB_PAGE;
		}

		info.tl = thumbpv = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title(GTK_WINDOW(thumbpv), gettext("Chbg: page thumbnails preview (wait ...)"));
		gtk_signal_connect(GTK_OBJECT(thumbpv), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &thumbpv);
		gtk_signal_connect(GTK_OBJECT(thumbpv), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &info.tl);
		gtk_signal_connect(GTK_OBJECT(thumbpv), "destroy",
			GTK_SIGNAL_FUNC(ThumbWinClose), &info);

		ThumbMenu(&info);

		if (frst)
		{
			gtk_signal_connect(GTK_OBJECT(toplevel_window), "destroy",
				GTK_SIGNAL_FUNC(KillAll), &thumbpv);
			frst = FALSE;
		}

		col = gtk_vbox_new(FALSE, 2);
		gtk_container_add(GTK_CONTAINER(thumbpv), col);
		gtk_widget_show(col);

		box = gtk_hbutton_box_new();
		gtk_box_pack_start(GTK_BOX(col), box, FALSE, FALSE, 2);
		gtk_widget_show(box);

		button = gtk_button_new_with_label(gettext("<< previous"));
		gtk_widget_set_sensitive(button, FALSE);
		gtk_box_pack_end(GTK_BOX(box) , button , FALSE , TRUE , 0);
		gtk_widget_show(button);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(ThumbsDoPrevPage), &info);
		info.prevb = button;

		button = gtk_option_menu_new();
		gtk_box_pack_end(GTK_BOX(box) , button , FALSE , TRUE , 0);
		info.optb = button;

		button = gtk_button_new_with_label(gettext("next >>"));
		gtk_widget_set_sensitive(button, FALSE);
		gtk_box_pack_end(GTK_BOX(box) , button , FALSE , TRUE , 0);
		gtk_widget_show(button);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(ThumbsDoNextPage), &info);
		info.nextb = button;

		info.swin = swin = gtk_scrolled_window_new(NULL, NULL);
		gtk_widget_set_usize(swin , 425 , 400);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
			GTK_POLICY_NEVER, GTK_POLICY_NEVER);
		gtk_widget_show(swin);
		gtk_container_add(GTK_CONTAINER(col), swin);

		info.progress = gtk_progress_bar_new();
		gtk_box_pack_start(GTK_BOX(col), info.progress, FALSE, FALSE, 2) ;

		box = gtk_hbutton_box_new();
		gtk_box_pack_start(GTK_BOX(col), box, FALSE, FALSE, 2);
		gtk_widget_show(box);

		button = guitl_pixmap_button(stock_close_xpm, gettext("Close"));
		gtk_box_pack_end(GTK_BOX(box) , button , FALSE , TRUE , 0);
		gtk_widget_show(button);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(KillWin), thumbpv);

		info.refreshb = button = guitl_pixmap_button(stock_restart_xpm, gettext("Refresh"));
		gtk_box_pack_end(GTK_BOX(box) , button , FALSE , TRUE , 0);
		gtk_widget_show(button);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(ThumbsDoFirstPage), &info);

		gtk_widget_show(thumbpv);
		while (gtk_events_pending()) gtk_main_iteration();

		ThumbsDoFirstPage(NULL, &info);
	}
	else
		gtk_widget_show(thumbpv);
	if (thumbpv && GTK_WIDGET_REALIZED(thumbpv))
		gdk_window_raise(thumbpv->window);
}

static void ThumbsDoDir(info, dir)
struct thumbpv_info *info;
char *dir;
{
	GList *ptr;

	gtk_window_set_title(GTK_WINDOW(info->tl),
		gettext("Chbg: dir thumbnails preview (wait ...)"));

	while (gtk_events_pending()) gtk_main_iteration();

	if (info->imglist) 
	{
		g_list_free(info->imglist);
		info->imglist = NULL;
		info->currimg = NULL;
	}

	for (ptr = info->alllist; dir && ptr; ptr = ptr->next)
	{
		char *p;
		picentry_t *pe = (picentry_t *)ptr->data;
		char pom[2048];

		strcpy(pom, pe->name);
		if ((p = strrchr(pom, '/')))
			*p = '\0';
		else
			pom[0] = '\0';

		if (!strcmp(dir, pom))
			info->imglist = g_list_append(info->imglist, ptr->data);
	}

	ThumbsDo(NULL, info);

	if (info->tl)
		gtk_window_set_title(GTK_WINDOW(info->tl),
			gettext("Chbg: dir thumbnails preview"));
}

static void ThumbsDoSetDir(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	char *dir = (char *) gtk_object_get_user_data(GTK_OBJECT(w));
	ThumbsDoDir(info, dir);
}


static void ThumbsDoFirstDir(w, info)
GtkWidget *w;
struct thumbpv_info *info;
{
	GtkWidget *menu,*mi;
	GList *ptr;
	GSList *sptr;

	ThumbsListFree(info);
	ThumbsListInit(info);
	info->imglist = NULL;

	for (ptr = info->alllist; ptr; ptr = ptr->next)
	{
		char pom[2048];
		char *p;

		strcpy(pom, ((picentry_t *)ptr->data)->name);

		p = strrchr(pom, '/');

		if (p) *p = '\0';
		else pom[0] = '\0';
		p = pom;

		if (!g_slist_find_custom(info->dirlist, p, (GCompareFunc)strcmp))
		{
			info->dirlist = g_slist_insert_sorted(info->dirlist,
				g_strdup(p), (GCompareFunc)strcmp);
		}
	}

	if (GTK_OPTION_MENU(info->combo)->menu)
		gtk_widget_destroy(GTK_OPTION_MENU(info->combo)->menu);

	menu = gtk_menu_new();
	for (sptr = info->dirlist; sptr ; sptr = sptr->next)
	{
		mi = gtk_menu_item_new_with_label(sptr->data ? (char *)sptr->data : "");
		gtk_menu_append(GTK_MENU(menu), mi);
		gtk_widget_show(mi);

		gtk_object_set_user_data(GTK_OBJECT(mi), (gpointer)sptr->data);

		gtk_signal_connect(GTK_OBJECT(mi), "activate" ,
			GTK_SIGNAL_FUNC(ThumbsDoSetDir) ,(gpointer)info); 
	}

	gtk_option_menu_set_menu(GTK_OPTION_MENU(info->combo), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(info->combo), 0);
	gtk_widget_show(info->combo);

	ThumbsDoDir(info, (char *)(info->dirlist ? info->dirlist->data : NULL));
}

void ThumbPreviewPerDir(w, data)
GtkWidget *w;
gpointer data;
{
	static GtkWidget *thumbpv = NULL;
	static int frst = TRUE;
	static struct thumbpv_info info = {0};

	if (!thumbpv)
	{
		GtkWidget *col, *button, *box, *swin;

		if (info.type == THUMB_NONE)
		{
			ThumbInfoInit(&info);
			info.type = THUMB_DIR;
		}

		info.tl = thumbpv = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title(GTK_WINDOW(thumbpv), gettext("Chbg: dir thumbnails preview (wait ...)"));
		gtk_signal_connect(GTK_OBJECT(thumbpv), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &thumbpv);
		gtk_signal_connect(GTK_OBJECT(thumbpv), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &info.tl);
		gtk_signal_connect(GTK_OBJECT(thumbpv), "destroy",
			GTK_SIGNAL_FUNC(ThumbWinClose), &info);

		ThumbMenu(&info);

		if (frst)
		{
			gtk_signal_connect(GTK_OBJECT(toplevel_window), "destroy",
				GTK_SIGNAL_FUNC(KillAll), &thumbpv);
			frst = FALSE;
		}

		col = gtk_vbox_new(FALSE, 2);
		gtk_container_add(GTK_CONTAINER(thumbpv), col);
		gtk_widget_show(col);

		button = gtk_option_menu_new();
		gtk_box_pack_start(GTK_BOX(col) , button , FALSE , TRUE , 0);
		info.combo = button;

		info.swin = swin = gtk_scrolled_window_new(NULL, NULL);
		gtk_widget_set_usize(swin , 445 , 395);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_widget_show(swin);
		gtk_container_add(GTK_CONTAINER(col), swin);

		info.progress = gtk_progress_bar_new();
		gtk_box_pack_start(GTK_BOX(col), info.progress, FALSE, FALSE, 2) ;

		box = gtk_hbutton_box_new();
		gtk_box_pack_start(GTK_BOX(col), box, FALSE, FALSE, 2);
		gtk_widget_show(box);

		button = guitl_pixmap_button(stock_close_xpm, gettext("Close"));
		gtk_box_pack_end(GTK_BOX(box) , button , FALSE , TRUE , 0);
		gtk_widget_show(button);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(KillWin), thumbpv);

		info.refreshb = button = guitl_pixmap_button(stock_restart_xpm, gettext("Refresh"));
		gtk_box_pack_end(GTK_BOX(box) , button , FALSE , TRUE , 0);
		gtk_widget_show(button);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(ThumbsDoFirstDir), &info);

		gtk_widget_show(thumbpv);
		while (gtk_events_pending()) gtk_main_iteration();

		ThumbsDoFirstDir(NULL, &info);
	}
	else
		gtk_widget_show(thumbpv);
	if (thumbpv && GTK_WIDGET_REALIZED(thumbpv))
		gdk_window_raise(thumbpv->window);
}

static int picture_find_by_name(pe, name)
picentry_t *pe;
char *name;
{
	return strncmp(pe->name, name, strlen(name));
}

void ThumbViewSA(filename)
char *filename;
{
	static struct thumbpv_info info = {0};

	if (info.type == THUMB_NONE)
	{
		ThumbInfoInit(&info);
		info.type = THUMB_SA;
	}

	ThumbsListFree(&info);
	ThumbsListInit(&info);
	info.imglist = NULL;
	info.currimg = NULL;
	info.curr = 0;
	info.imglist = g_list_copy(info.alllist);
	info.currimg = g_list_find_custom(info.imglist, filename, (GCompareFunc)picture_find_by_name);

	if (info.currimg)
		ThumbView(NULL, &info, info.currimg->data);
	else
		gdk_beep();
}

void ThumbViewFullSA(filename)
char *filename;
{
	static struct thumbpv_info info = {0};

	if (info.type == THUMB_NONE)
	{
		ThumbInfoInit(&info);
		info.type = THUMB_SA;
	}

	ThumbsListFree(&info);
	ThumbsListInit(&info);
	info.imglist = NULL;
	info.currimg = NULL;
	info.curr = 0;
	info.imglist = g_list_copy(info.alllist);
	info.currimg = g_list_find_custom(info.imglist, filename, (GCompareFunc)picture_find_by_name);

	if (info.currimg)
		ThumbViewFull(NULL, &info, info.currimg->data);
	else
		gdk_beep();
}

static void ThumbViewSinglePictureListInit(info, list)
struct thumbpv_info *info;
GtkWidget *list;
{
	int row;
	picentry_t *pe;
	char *p;

	info->alllist = NULL;
	info->dirlist = NULL;

	for (row = 1; row < GTK_CLIST(list)->rows; row++)
	{
		gtk_clist_get_text(GTK_CLIST(list) , row , 0 , &p);
		pe = picentry_new(g_strdup(p));

		info->alllist = g_list_append(info->alllist , (gpointer)pe);
	}
	
	info->num = g_list_length(info->alllist);
}

void ThumbViewPictureList(filename, list)
char *filename;
void *list;
{
	static struct thumbpv_info info = {0};

	if (info.type == THUMB_NONE)
	{
		ThumbInfoInit(&info);
		info.type = THUMB_SA;
	}

	ThumbsListFree(&info);
	ThumbViewSinglePictureListInit(&info, list);
	info.imglist = NULL;
	info.currimg = NULL;
	info.curr = 0;
	info.imglist = g_list_copy(info.alllist);
	info.currimg = g_list_find_custom(info.imglist, filename, (GCompareFunc)picture_find_by_name);

	if (info.currimg)
		ThumbView(NULL, &info, info.currimg->data);
	else
		gdk_beep();

}

