/***************************************************************************/
/* 		This code is part of Desktop Background changer		   */
/*		called ChBg						   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include "config.h"
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>

#include "gprop.h"
#include "gaccel.h"
#include "options.h"

cfg_param cfg;

void chbg_srand(unsigned int seed)
{
#ifdef HAVE_RANDOM
    srandom(seed);
#else
    srand(seed);
#endif
}

int chbg_rand()
{
#ifdef HAVE_RANDOM
    return random();
#else
    return rand();
#endif
}

char *get_1qstr(str)
char *str;
{
    static char *p = NULL;
    char *pom1 = NULL, *pom2 = NULL;
    int found;

    if (str) p = str;

    for ( ; p && *p && (!pom1 || !pom2) ; p++)
    {
	found = FALSE;

	if (*p == '\"')
	{
	    if ((p == str) || (*(p-1) != '\\'))
		found = TRUE;
	}

	if (!pom1 && found)
	    pom1 = p+1;
	else if (!pom2 && found)
	    pom2 = p;
    }
    if (pom1 && pom2)
    {
	*pom2 = '\0';
	return pom1;
    }
    else
    {
	p = NULL;
	return NULL;
    }
}

gint _atoi(str) 
gchar *str;
{
    gchar *__eptr__;
    long int rv;

    rv = strtol(str, (char **) &__eptr__, 0);
    if (*__eptr__ != '\0') errno = ERANGE;
    else errno = 0;

    return (gint)rv;
}

gfloat _atof(str) 
char *str;
{
    gchar *__eptr__;
    double rv;

    rv = strtod(str, (char **) &__eptr__);
    if (*__eptr__ != '\0') errno = ERANGE;
    else errno = 0;

    return (gfloat)rv;
}

static int parse_interval(str, start, end)
char *str;
gfloat *start;
gfloat *end;
{
    char *p;
    int rv = 0;

    str = g_strdup(str);

    strtok(str, "-");
    p = strtok(NULL, "");

    *start = _atof(str);

    if (!*start && errno == ERANGE)
	rv = -1;

    if (p)
    {
	*end = _atof(p);

	if (!*end && errno == ERANGE)
	    rv = -1;
    }
    else
	*end = 0.0;

    g_free(str);

    return rv;
}

char *simplify_path(inpath)
char *inpath;
{
    char *p, *path = NULL;
    char res[2048];
    int l;

    if (inpath[0] != '/')
    {
	if (!getcwd(res, sizeof(res)))
	    strcpy(res, "/");

	path = g_strconcat(res, "/" , inpath, NULL);
	p = path;
    }
    else
	p = inpath;

    res[0] = '\0';
    while (*p)
    {
	p += strspn(p, "/");
	l = strcspn(p, "/");

	if (strncmp(p, ".", l))
	{
	    if (!strncmp(p, "..", l))
	    {
		char *sl = strrchr(res, '/');

		if (sl)
		    *sl = '\0';
	    }
	    else
	    {
		strcat(res, "/");
		strncat(res, p, l);
	    }
	}
	p +=l;
    }
    if (!res[0])
	strcpy(res, "/");

    if (path)
	g_free(path);
    return g_strdup(res);	
}

int makealldirs(path)
char *path;
{
    char pom[PATH_MAX];
    char *p;

    pom[0]='\0';

    if (path)
    {
	p = path;

	while(*p)
	{
	    int ilen = strcspn(p, "/");

	    strcat(pom, "/");
	    strncat(pom, p, ilen);

	    p += strspn(p, "/");			
	    p += ilen;

	    if (access(pom,F_OK))
	    {
		if (mkdir(pom, S_IRWXU | S_IRGRP | S_IROTH | 
			  S_IXGRP | S_IXOTH))
		    return -1;
	    }
	}
    }
    return 0;
}

int check_program(prog)
char *prog;
{
    int rv = FALSE;
    char *p;

    p = g_getenv("PATH");
    if (p)
    {
	int i;
	char **vec = g_strsplit(p, ":", 100);

	for (i = 0; vec && vec[i]; i++)
	{
	    p = g_strconcat(vec[i], "/", prog, NULL);
	    rv = !access(p, X_OK);
	    g_free(p);
	    if (rv)
		break;
	}
	g_strfreev(vec);
    }

    return rv;
}

GSList *append_nr_to_list(lst, str)
GSList *lst;
char *str;
{
    int ilen;
    int i;
    char *p;

    while(*str)
    {
	str += strspn(str, ",");
	ilen = strcspn(str, ",");

	i = strtol(str, &p, 10);
	if (*p != ',' && *p != '\0')
	{
	    return NULL;
	}

	if (ilen && !g_slist_find(lst, (gpointer)i))
	    lst = g_slist_append(lst, (gpointer)i);

	str += ilen;
    }

    return lst;
}

prop_t *prop_dup(prop)
prop_t *prop;
{
    prop_t *rv;

    rv = g_memdup(prop, sizeof(prop_t));

    if (rv->tile)
	rv->tile = g_strdup(rv->tile);

    if (rv->banner)
	rv->banner = g_strdup(rv->banner);

    return rv;
}

prop_t *prop_copy(prop, sprop)
prop_t *prop;
prop_t *sprop;
{
    g_free(prop->tile);
    g_free(prop->banner);
    memcpy(prop, sprop, sizeof(prop_t));

    if (prop->tile)
	prop->tile = g_strdup(prop->tile);

    if (prop->banner)
	prop->banner = g_strdup(prop->banner);

    return prop;
}

void prop_free(prop)
prop_t *prop;
{
    g_free(prop->tile);
    g_free(prop->banner);
    g_free(prop);
}

char *picentry_get_name(pe)
picentry_t *pe;
{
    return pe->name;
}

picentry_t *picentry_clone(name, pe)
char *name;
picentry_t *pe;
{
    picentry_t *rv;

    rv = g_malloc(sizeof(picentry_t));

    rv->name = name;
    rv->prop = NULL;

    if ( pe && pe->prop)
	rv->prop = prop_dup(pe->prop);

    return rv;
}

picentry_t *picentry_new(name)
char *name;
{
    picentry_t *pe;

    pe = g_malloc(sizeof(picentry_t));

    pe->name = name;
    pe->prop = NULL;

    return pe;
}

picentry_t *picentry_dup(pe)
picentry_t *pe;
{
    picentry_t *rv;

    rv = g_malloc(sizeof(picentry_t));

    rv->name = g_strdup(pe->name);

    if (pe->prop)
	rv->prop = prop_dup(pe->prop);
    else
	rv->prop = NULL;

    return rv;
}

void picentry_free(pe)
picentry_t *pe;
{
    if (pe->prop)
	prop_free(pe->prop);
    g_free(pe->name);
    g_free(pe);
}

banner_prop_t *banner_prop_dup(bp)
banner_prop_t *bp;
{
    banner_prop_t *rv;

    rv = g_malloc(sizeof(banner_prop_t));

    rv->max_grow = bp->max_grow;
    rv->max_size = bp->max_size;
    rv->xpos = bp->xpos;
    rv->ypos = bp->ypos;
    rv->mode = bp->mode;

    return rv;
}

banner_prop_t *banner_prop_new()
{
    return banner_prop_dup(&cfg.properties.banner_prop);
}

void banner_prop_free(bp)
banner_prop_t *bp;
{
    g_free(bp);
}

char *banner_get_name(bn)
banner_t *bn;
{
    return bn->name;
}

banner_t *banner_clone(name, pe)
char *name;
banner_t *pe;
{
    banner_t *rv;

    rv = g_malloc(sizeof(banner_t));

    rv->name = name;
    rv->prop = NULL;

    if (pe->prop)
	rv->prop = banner_prop_dup(pe->prop);

    return rv;
}

banner_t *banner_new(name)
char *name;
{
    banner_t *rv;

    rv = g_malloc(sizeof(banner_t));

    rv->name = name;
    rv->prop = NULL;

    return rv;
}

void banner_free(bn)
banner_t *bn;
{
    g_free(bn->name);
    if (bn->prop)
	banner_prop_free(bn->prop);
    g_free(bn);
}

void cfg_usage(prg)
char *prg;
{
    printf("%s-%s\n", PACKAGE, VERSION);
    printf(gettext("Use -h option to print full list of available options\n"));
    exit(1);
}

void cfg_usage_long(prg)
gchar *prg;
{
printf(gettext("Usage:  %s  [options] [any number of pictures]\n%s-%s\n") ,prg ,
       PACKAGE, VERSION);

printf(gettext("  -v                    - print version number and exit\n"));
printf(gettext("  -h                    - print this help message and exit\n"));
printf(gettext("  -setup                - start setup window\n"));
printf(gettext("  -scenario $file       - fetch parameters from file $file\n"));
printf(gettext("  -mode $type           - set picture rendering mode, mode is\n"
	       "                          one of following:\n"
	       "                          : tile\n"
	       "                          : mirror\n"
	       "                          : center\n"
	       "                          : maximize\n"
	       "                          : smart\n"
	       "                          : centertile\n"
	       "                          : inttile\n"
	       "                          : symtile\n"
	       "                          : intmirror\n"
	       "                          : symmirror\n"));
printf(gettext("  -effect $nr           - effect type for changing pictures [0-%d]\n"), CHBG_EFECTS);
printf(gettext("  -bg $color            - set default background color 1\n"));
printf(gettext("  -bg2 $color           - set default background color 2\n"));
printf(gettext("  -rand_colors          - randomize background colors\n"));
printf(gettext("  -shader $nr           - set background color shading effect\n"
	       "                          type [0-%d]\n"), shader_get_cnt());
printf(gettext("  -interval $start[-$end]\n"
	       "                        - time between picture changes\n"));
printf(gettext("  -max_grow $nr         - maximal growing ratio of picture in\n"
	       "                          smart mode\n"));
printf(gettext("  -max_size $nr         - maximal size of picture in %% of screen size\n"
	       "                          used in smart mode, acceptable value 10-100\n"));
printf(gettext("  -randomize            - randomize picture order\n"));
printf(gettext("  -screensaver          - act as screensaver\n"));
#ifdef HAVE_XSS_SUPPORT
printf(gettext("  -xscreensaver         - act as xscreensaver client\n"));
#endif
printf(gettext("  -speed $nr            - speed ratio for picture changing effects\n"
	       "                          (lower is faster)\n"));
printf(gettext("  -blank                - show blank screen, when can't load picture\n"));
printf(gettext("  -cycle_blank          - when pictures not present cycle with blanking\n"
	       "                          screen (optionally with shading)\n"));
printf(gettext("  -R                    - recurse through directory\n"));
printf(gettext("  -pattern $ptrn        - wildcard pattern for picture selection in\n"
	       "                          recursion through directories\n"));
printf(gettext("  -inwindow             - run in own window not on root\n"));
printf(gettext("  -windowid $id         - run in existing X window with known window ID\n"));
printf(gettext("  -run                  - start runining, after setup window is popped up\n"
	       "                          (have effect only when used with -setup option)\n"));
printf(gettext("  -min_psize $nr        - minimal allowed size of picture when recursing\n"
	       "                          directories\n"));
printf(gettext("  -deffects $list       - list of disabled effects\n"));
printf(gettext("  -dshaders $list       - list of disabled shaders\n"));
printf(gettext("  -notooltips           - disable tooltips in GUI setup\n"));
printf(gettext("  -friter $nr           - number of iterations for fractal based shaders\n"));
printf(gettext("  -rect_size $nr        - size of rectangle used in picture changing\n"
	       "                          effects\n"));
printf(gettext("  -grad_dir $dir        - supply path to GIMP gradients directory\n"));
printf(gettext("  -use_grad             - use GIMP gradients for shading background\n"));
printf(gettext("  -grad $nr             - number of selected gradient (0 = random)\n"));
printf(gettext("  -remember             - remember last picture\n"));
printf(gettext("  -sort                 - sort picture alphabeticaly\n"));
#ifdef HAVE_ENLIGHTENMENT_SUPPORT
printf(gettext("  -enlightenment        - set desktop background via Enlightenment WM\n"
	       "                          when available\n"));
#endif
printf(gettext("  -once                 - run one cycle of background changing and exit\n"));
printf(gettext("  -thumb $type          - show thumbnail view dialog, choose one of\n"
	       "                          all - all thumbnails in one window\n"
	       "                          dir - make thumbnail pages by directories\n"
	       "                          page - show 25 thumbnails per window\n"));
printf(gettext("  -xpos $pos            - horizontal picture position on screen\n"
	       "                          in smart mode (0.0-1.0)\n"));
printf(gettext("  -ypos $pos            - vertical picture position on screen\n"
	       "                          in smart mode (0.0-1.0)\n"));
printf(gettext("  -use_tiles            - use picture tiles for picture background\n"));
printf(gettext("  -add_tile $file       - directory or file which will be used for tiles\n"));
printf(gettext("  -tile $file           - default tile for picture background\n"));
printf(gettext("  -use_banners          - put banner on top of background picture\n"));
printf(gettext("  -add_banner $file     - directory or file which will be used for banners\n"));
printf(gettext("  -banner $file         - use image as banner\n"));
printf(gettext("  -banner_xpos $pos     - horizontal banner position on screen\n"));
printf(gettext("  -banner_ypos $pos     - vertical banner position on screen\n"));
printf(gettext("  -banner_max_grow $nr  - maximal growing ratio of banner\n"));
printf(gettext("  -banner_max_size $nr  - maximal size of banner in %% of screen size\n"));
printf(gettext("  -banner_mode $mode    - banner painting mode (0-%d)\n"), CHBG_BANNER_MODES);
printf(gettext("  -to_file              - generate picture into file instead of to window\n"));
printf(gettext("  -out_file $file       - name of file in which will be image written\n"));
printf(gettext("  -out_file_width $nr   - width of picture when writing to file\n"
	       "                          0 means - take width of root window\n"));
printf(gettext("  -out_file_height $nr  - height of picture when writing to file\n"
	       "                          0 means - take height of root window\n"));
printf(gettext("  -force_nautilus       - force nautilus to use background picture\n"
	       "                          set by chbg\n"));
printf(gettext("  -send_expose          - send expose event when running in foreign window\n"));
printf(gettext("  -print_fname          - print name of file being displayed\n"));
printf(gettext("  -use_cache            - if option -R is used, then try to load list of\n"));
printf(gettext("                          images from a cache file (and create one, if not there\n"));
printf(gettext("  -log_file filename    - write log messages info filename\n"));

exit(1);
}

void cfg_default()
{
    cfg.properties.max_grow = default_max_grow;
    cfg.properties.interval = default_interval;
    cfg.properties.interval_end = 0.0;
    cfg.properties.max_size = default_max_size;
    cfg.properties.type = default_type;
    cfg.properties.efect = default_efect;
    cfg.properties.shade = default_shader;
    cfg.properties.gradnr = 0;
    cfg.properties.use_grad = FALSE;
    cfg.properties.use_tiles = FALSE;
    cfg.properties.use_banners = FALSE;
    cfg.properties.rand_colors = FALSE;
    cfg.properties.xpos = 0.5;
    cfg.properties.ypos = 0.5;
    cfg.properties.tile = NULL;
    cfg.properties.banner = NULL;
    cfg.properties.banner_prop.max_grow = default_max_grow;
    cfg.properties.banner_prop.max_size = default_max_size;
    cfg.properties.banner_prop.xpos = 0.5;
    cfg.properties.banner_prop.ypos = 0.5;
    cfg.properties.banner_prop.mode = 0;

    cfg.speed = default_speed;
    cfg.pics = NULL;
    cfg.pics_orig = NULL;
    cfg.num_pics = 0;
    gdk_color_parse("#000000", &cfg.properties.background);
    gdk_color_parse("#ffffff", &cfg.properties.background2);
    cfg.screensaver = FALSE;
    cfg.xscreensaver = FALSE;
    cfg.blank = FALSE;
    cfg.setup = FALSE;
    cfg.scenario = NULL;
    cfg.recurse = FALSE;
    cfg.inwindow = FALSE;
    cfg.windowid = 0;
    cfg.runit = FALSE;
    cfg.pattern = NULL;
    cfg.cycle_blank = FALSE;
    cfg.d_effects = NULL;
    cfg.d_shaders = NULL;
    cfg.min_size = 0;
    cfg.tooltips = TRUE;
    cfg.fract_iter = default_fract_iter;
    cfg.box_size = default_box_size;
    cfg.current_name = NULL;
    cfg.remember_last = FALSE;
    cfg.sort = FALSE;
    cfg.use_ebg = FALSE;
    cfg.once = FALSE;
    cfg.thumb = THUMB_NONE;
    cfg.tiles = NULL;
    cfg.banners = NULL;
    cfg.tofile = FALSE;
    cfg.outfile = NULL;
    cfg.outfile_width = 0;
    cfg.outfile_height = 0;
    cfg.fake_nautilus = FALSE;
    cfg.send_expose = FALSE;
    cfg.force_xscreensaver = FALSE;
    cfg.print_fname = FALSE;
    cfg.log_file = NULL;
}

void cfg_clear()
{
    GSList *sptr;
    GList *ptr;

    g_free(cfg.properties.tile);
    g_free(cfg.properties.banner);

    for (sptr = cfg.pattern; sptr; sptr = g_slist_remove_link(sptr, sptr))
	g_free(sptr->data);
    g_slist_free(cfg.d_effects);
    g_slist_free(cfg.d_shaders);

    for (ptr = cfg.pics_orig; ptr; ptr = g_list_remove_link(ptr, ptr))
	picentry_free((picentry_t *)ptr->data);

    for (ptr = cfg.pics; ptr; ptr = g_list_remove_link(ptr, ptr))
	picentry_free((picentry_t *)ptr->data);

    for (ptr = cfg.tiles; ptr; ptr = g_list_remove_link(ptr, ptr))
	g_free(ptr->data);

    for (ptr = cfg.banners; ptr; ptr = g_list_remove_link(ptr, ptr))
	banner_free((banner_t *)ptr->data);

    cfg_default();
}

static rendering_type cfg_get_mode(str)
gchar *str;
{
    if (!strcasecmp("tile", str))
	return RENDERT_TILE;
    else if (!strcasecmp("mirror", str))
	return RENDERT_MIRROR;
    else if (!strcasecmp("center", str))
	return RENDERT_CENTER;
    else if (!strcasecmp("maximize", str))
	return RENDERT_MAXIMIZE;
    else if (!strcasecmp("smart", str)) 
	return RENDERT_SMART;
    else if (!strcasecmp("centertile", str))
	return RENDERT_CENTER_TILE;
    else if (!strcasecmp("inttile", str))
	return RENDERT_INT_TILE;
    else if (!strcasecmp("symtile", str))
	return RENDERT_SYM_TILE;
    else if (!strcasecmp("intmirror", str))
	return RENDERT_INT_MIRROR;
    else if (!strcasecmp("symmirror", str))
	return RENDERT_SYM_MIRROR;
    else
    {
	fprintf(stderr, gettext("unknown rendering type : %s\n"), str);
	return default_type;
    }
}

static thumb_type cfg_get_thumb_type(str)
char *str;
{
    if (!strcasecmp("all", str))
	return THUMB_ALL;
    else if (!strcasecmp("dir", str))
	return THUMB_DIR;
    else if (!strcasecmp("page", str))
	return THUMB_PAGE;
    else
    {
	fprintf(stderr, gettext("unknown thumbnailview type: %s\n"), str);
	return THUMB_NONE;
    }
}

static gchar* cfg_get_mode_str(type)
rendering_type type;
{
    switch(type)
    {
	case RENDERT_TILE: return "tile";
	case RENDERT_MIRROR: return "mirror";
	case RENDERT_CENTER: return "center";
	case RENDERT_MAXIMIZE: return "maximize";
	case RENDERT_SMART: return "smart";
	case RENDERT_CENTER_TILE: return "centertile";
	case RENDERT_INT_TILE: return "inttile";
	case RENDERT_SYM_TILE: return "symtile";
	case RENDERT_INT_MIRROR: return "intmirror";
	case RENDERT_SYM_MIRROR: return "symmirror";
	default: return gettext("bug in chbg - contact autor");
    }
}

static char *banner_modes[] = {
    "normal",
    "average",
    "lighten",
    "darken",
    "divide",
    "multiply",
    "multiplybg",
    "multiplyfg",
    "add",
    "subtractbg",
    "subtractfg",
    "grayscale",
    "bumpmap",
    "bumpmapself",
    NULL,
};

int cfg_find_banner_mode(str)
char *str;
{
    int i;
    for (i = 0; banner_modes[i]; i++)
    {
	if (!strcasecmp(str, banner_modes[i]))
	    return i;
    }

    i = _atoi(str);
    if (errno == ERANGE)
	return -1;
    else
	return i;
}

option_t *cfg_find_option(param)
char *param;
{
    int i;

    for (i = 0; i < sizeof(chbg_opts)/sizeof(chbg_opts[0]); i++)
    {
	if (chbg_opts[i].cmd && !strcasecmp(chbg_opts[i].cmd, param))
	    return &(chbg_opts[i]);
    }

    for (i = 0; i < sizeof(foreign_opts)/sizeof(foreign_opts[0]); i++)
    {
	if (foreign_opts[i].cmd &&
	    !strcasecmp(foreign_opts[i].cmd, param))
	    return &(foreign_opts[i]);
    }

    return NULL;
}

gint cfg_cmdln(argc, argv)
guint argc;
gchar **argv;
{
    gint i,j;
    char *cwd = NULL;
    option_t *opt;
    char *params[4];
    char pom[2048];
    int l;

    for(i = 1; i < argc ; i++)
    {
	opt = cfg_find_option(argv[i]);

	if (!opt)
	{
	    if (*(argv[i]) != '-')
	    {			
		char *name,*p;

		l = strlen(argv[i]);

		if (l>2 && argv[i][0] && argv[i][0] == '\"' &&
		    argv[i][l-1] == '\"')
		{
		    strncpy(pom, argv[i]+1, l-2);
		    pom[l-2] = '\0';
		}
		else
		    strcpy(pom, argv[i]);

		if (*pom == '/')
		    name = g_strdup(pom);
		else
		{
		    if (!cwd)
			cwd = getcwd(NULL, PATH_MAX);
		    name = g_strconcat(cwd, "/", pom, NULL);
		}
		p = simplify_path(name);
		g_free(name);
		cfg.pics = g_list_append(cfg.pics, picentry_new(p));
		cfg.num_pics ++;
	    }
	    else
	    {
		fprintf(stderr, gettext("unknown option : %s\n"), argv[i]);
		return -1;
	    }

	    continue;
	}

	if (opt->nparams + i >= argc)
	{
	    fprintf(stderr, gettext("missing parameters for %s option\n"),
		    opt->cmd);
	    return -1;
	}

	for (j = 0; j < opt->nparams; j++)
	{
	    params[j] = argv[i+j+1];
	}

	i += opt->nparams;

	switch (opt->type)
	{
	    case CHBG_OPT_FOREIGN:
		break;
	    case CHBG_OPT_MODE:
		cfg.properties.type = cfg_get_mode(params[0]);
		break;
	    case CHBG_OPT_BG:
		if (!gdk_color_parse(params[0],
				     &cfg.properties.background))
		{
		    fprintf(stderr, gettext("unknown color: %s\n"),
			    params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_BG2:
		if (!gdk_color_parse(params[0],
				     &cfg.properties.background2))
		{
		    fprintf(stderr, gettext("unknown color: %s\n"),
			    params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_MAX_GROW:
		cfg.properties.max_grow = _atof(params[0]);
		if (errno == ERANGE ||
		    (cfg.properties.max_grow <= (gfloat)0.0))
		{
		    fprintf(stderr, gettext("bad grow value: %s, (positive float number required)\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_MIN_PSIZE:
		cfg.min_size = _atoi(params[0]);
		if (errno == ERANGE)
		{
		    fprintf(stderr, gettext("bad value: %s for option -min_psize\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_DEFFECTS:
		cfg.d_effects = append_nr_to_list(cfg.d_effects, params[0]);
		if (!cfg.d_effects)
		{
		    fprintf(stderr, gettext("bad list: %s for option %s\n"),
			    params[0], opt->cmd);
		    return -1;
		}
		break;
	    case CHBG_OPT_DSHADERS:
		cfg.d_shaders = append_nr_to_list(cfg.d_shaders, params[0]);
		if (!cfg.d_shaders)
		{
		    fprintf(stderr, gettext("bad list: %s for option %s\n"),
			    params[0], opt->cmd);
		    return -1;
		}
		break;
	    case CHBG_OPT_INTERVAL:
		if (parse_interval(params[0], &cfg.properties.interval,
				   &cfg.properties.interval_end))
		{
		    fprintf(stderr, gettext("bad float value: %s\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_MAX_SIZE:
		cfg.properties.max_size = _atoi(params[0]);
		if (errno == ERANGE || cfg.properties.max_size < 10 ||
		    cfg.properties.max_size > 100)
		{
		    fprintf(stderr, gettext("bad max_size value: %s, (10 - 100 required)\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_SPEED:
		cfg.speed = _atoi(params[0]);
		if (errno == ERANGE)
		{
		    fprintf(stderr, gettext("bad speed value: %s\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_EFFECT:
		cfg.properties.efect = _atoi(params[0]);
		if (errno == ERANGE || cfg.properties.efect > CHBG_EFECTS)
		{
		    fprintf(stderr, gettext("bad effect value: %s, (0 - %d required)\n"), params[0], CHBG_EFECTS);
		    return -1;
		}
		break;
	    case CHBG_OPT_SHADER:
		cfg.properties.shade = _atoi(params[0]);
		if (errno == ERANGE || cfg.properties.shade > shader_get_cnt())
		{
		    fprintf(stderr, gettext("bad shader value: %s, (0 - %d required)\n"), params[0], shader_get_cnt());
		    return -1;
		}
		break;
	    case CHBG_OPT_WINDOWID:
		cfg.windowid = _atoi(params[0]);
		cfg.infwindow = TRUE;
		if (errno == ERANGE)
		{
		    fprintf(stderr, gettext("bad windowid: %s\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_FRITER:
		cfg.fract_iter = _atoi(params[0]);
		if ((errno == ERANGE) || (cfg.fract_iter < 2))
		{
		    fprintf(stderr, gettext("bad value: %s for option -friter\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_RECT_SIZE:
		cfg.box_size = _atoi(params[0]);
		if ((errno == ERANGE) || (cfg.box_size < 1))
		{
		    fprintf(stderr, gettext("bad value: %s for option -box_size\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_GRAD:
		cfg.properties.gradnr = _atoi(params[0]);
		if (errno == ERANGE)
		{
		    fprintf(stderr, gettext("bad value: %s for option -grad\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_SCENARIO:
		l = strlen(params[0]);

		if (l>2 && params[0][0] && params[0][0] == '\"' &&
		    params[0][l-1] == '\"')
		{
		    strncpy(pom, params[0]+1, l-2);
		    pom[l-2] = '\0';
		}
		else
		    strcpy(pom, params[0]);

		cfg.scenario = simplify_path(params[0]);
		cfg_scenario(cfg.scenario);
		break;
	    case CHBG_OPT_GRAD_DIR:
		cfg.grad_dir = simplify_path(params[0]);
		break;
	    case CHBG_OPT_ADD_TILE:
		cfg.tiles = g_list_append(cfg.tiles,
					  simplify_path(params[0]));
		break;
	    case CHBG_OPT_ADD_BANNER:
		cfg.banners = g_list_append(cfg.banners,
					    banner_new(simplify_path(params[0])));
		break;
	    case CHBG_OPT_TILE:
		cfg.properties.tile = simplify_path(params[0]);
		break;
	    case CHBG_OPT_BANNER:
		cfg.properties.banner = simplify_path(params[0]);
		break;
	    case CHBG_OPT_BANNER_XPOS:
		cfg.properties.banner_prop.xpos = _atof(params[0]);
		if (errno == ERANGE ||
		    cfg.properties.banner_prop.xpos < 0.0 ||
		    cfg.properties.banner_prop.xpos > 1.0)
		{
		    fprintf(stderr, gettext("bad position value: %s, (0.0 - 1.0 required)\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_BANNER_YPOS:
		cfg.properties.banner_prop.ypos = _atof(params[0]);
		if (errno == ERANGE ||
		    cfg.properties.banner_prop.ypos < 0.0 ||
		    cfg.properties.banner_prop.ypos > 1.0)
		{
		    fprintf(stderr, gettext("bad position value: %s, (0.0 - 1.0 required)\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_BANNER_MAX_SIZE:
		cfg.properties.banner_prop.max_size = _atoi(params[0]);
		if (errno == ERANGE ||
		    cfg.properties.banner_prop.max_size < 10 ||
		    cfg.properties.banner_prop.max_size > 100)
		{
		    fprintf(stderr, gettext("bad max_size value: %s, (10 - 100 required)\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_BANNER_MAX_GROW:
		cfg.properties.banner_prop.max_grow = _atof(params[0]);
		if (errno == ERANGE ||
		    cfg.properties.banner_prop.max_grow <= (gfloat)0.0)
		{
		    fprintf(stderr, gettext("bad grow value: %s, (positive float number required)\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_BANNER_MODE:
		cfg.properties.banner_prop.mode =
		    cfg_find_banner_mode(params[0]);
		if (errno == ERANGE ||
		    cfg.properties.banner_prop.mode > CHBG_BANNER_MODES)
		{
		    fprintf(stderr, gettext("bad banner mode value: %s, (0-%d required)\n"), params[0], CHBG_BANNER_MODES );
		    return -1;
		}
		break;
	    case CHBG_OPT_XPOS:
		cfg.properties.xpos = _atof(params[0]);
		if (errno == ERANGE ||
		    cfg.properties.xpos < 0.0 ||
		    cfg.properties.xpos > 1.0)
		{
		    fprintf(stderr, gettext("bad position value: %s, (0.0 - 1.0 required)\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_YPOS:
		cfg.properties.ypos = _atof(params[0]);
		if (errno == ERANGE ||
		    cfg.properties.ypos < 0.0 ||
		    cfg.properties.ypos > 1.0)
		{
		    fprintf(stderr, gettext("bad position value: %s, (0.0 - 1.0 required)\n"), params[0]);
		    return -1;
		}
		break;
	    case CHBG_OPT_USE_GRAD:
		cfg.properties.use_grad = TRUE;
		break;
	    case CHBG_OPT_USE_TILES:
		cfg.properties.use_tiles = TRUE;
		break;
	    case CHBG_OPT_USE_BANNERS:
		cfg.properties.use_banners = TRUE;
		break;
	    case CHBG_OPT_R:
		cfg.recurse = TRUE;
		break;
	    case CHBG_OPT_SCREENSAVER:
		cfg.screensaver = TRUE;
		break;
	    case CHBG_OPT_REMEMBER:
		cfg.remember_last = TRUE;
		break;
	    case CHBG_OPT_SORT:
		cfg.sort = TRUE;
		break;
#ifdef HAVE_XSS_SUPPORT
	    case CHBG_OPT_XSCREENSAVER:
		cfg.xscreensaver = TRUE;
		break;
#endif
	    case CHBG_OPT_BLANK:
		cfg.blank = TRUE;
		break;
	    case CHBG_OPT_SETUP:
		cfg.setup = TRUE;
		break;
	    case CHBG_OPT_RANDOMIZE:
		cfg.randomize = TRUE;
		break;
	    case CHBG_OPT_INWINDOW:
		cfg.inwindow = TRUE;
		break;
	    case CHBG_OPT_RUN:
		cfg.runit = TRUE;
		break;
	    case CHBG_OPT_CYCLE_BLANK:
		cfg.cycle_blank = TRUE;
		break;
	    case CHBG_OPT_RAND_COLORS:
		cfg.properties.rand_colors = TRUE;
		break;
	    case CHBG_OPT_NOTOOLTIPS:
		cfg.tooltips = FALSE;
		break;
	    case CHBG_OPT_H:
	    case CHBG_OPT_LONG_HELP:
		cfg_usage_long(PACKAGE);
		break;
#ifdef HAVE_ENLIGHTENMENT_SUPPORT
	    case CHBG_OPT_ENLIGHTENMENT:
		cfg.use_ebg = TRUE;
		break;
#endif
	    case CHBG_OPT_ONCE:
		cfg.once = TRUE;
		break;
	    case CHBG_OPT_THUMB:
		cfg.thumb = cfg_get_thumb_type(params[0]);
		break;
	    case CHBG_OPT_V:
	    case CHBG_OPT_LONG_VERSION:
#ifdef HAVE_GDKIMLIB1
#define IMGLOADER	"gdk_imlib1"
#endif
#ifdef HAVE_IMLIB1
#define IMGLOADER	"Imlib1"
#endif
#ifdef HAVE_IMLIB2
#define IMGLOADER	"Imlib2"
#endif
#ifdef HAVE_GDKPIXBUF
#define IMGLOADER	"gdk-pixbuf < (0.7.0)"
#endif
#ifdef HAVE_GDKPIXBUF2
#define IMGLOADER	"gdk-pixbuf >= (0.7.0)"
#endif
		fprintf(stderr, "%s-%s\n", PACKAGE, VERSION);
		fprintf(stderr, gettext("using image loader - %s\n"), IMGLOADER);
		exit(1);
		break;
	    case CHBG_OPT_PATTERN:
		cfg.pattern = g_slist_append(cfg.pattern,
					     g_strdup(params[0]));
		break;
	    case CHBG_OPT_PICTURE:
	    case CHBG_OPT_CURRENT_PICTURE:
		break;
	    case CHBG_OPT_OUT_FILE:
		cfg.outfile = g_strdup(params[0]);
		break;
	    case CHBG_OPT_OUT_FILE_WIDTH:
		cfg.outfile_width = _atoi(params[0]);
		if (errno == ERANGE)
		    fprintf(stderr, gettext("bad width value: %s\n"),
			    params[0]);
		break;
	    case CHBG_OPT_OUT_FILE_HEIGHT:
		cfg.outfile_height = _atoi(params[0]);
		if (errno == ERANGE)
		    fprintf(stderr, gettext("bad height value: %s\n"),
			    params[0]);
		break;
	    case CHBG_OPT_TO_FILE:
		cfg.tofile = TRUE;
		break;
	    case CHBG_OPT_FAKE_NAUTILUS:
		cfg.fake_nautilus = TRUE;
		break;
#ifdef HAVE_XSS_SUPPORT
	    case CHBG_OPT_SEND_EXPOSE:
		cfg.send_expose = TRUE;
		break;
	    case CHBG_OPT_FORCE_XSCREENSAVER:
		cfg.force_xscreensaver = TRUE;
		break;
#endif
	    case CHBG_OPT_PRINT_FNAME:
		cfg.print_fname = TRUE;
		break;
	    case CHBG_OPT_LOG_FILE:
		cfg.log_file = g_strdup(params[0]);
		break;
	    case CHBG_OPT_USE_CACHE:
		cfg.use_cache = TRUE;
		break;
	}
    }

    if (cwd)
	free(cwd);
    return 0;
}

option_t *cfg_find_scn_option(param)
char *param;
{
    int i;

    for (i = 0; i < sizeof(chbg_opts)/sizeof(chbg_opts[0]); i++)
    {
	if (chbg_opts[i].scn && !strncasecmp(chbg_opts[i].scn, param,
					     strlen(chbg_opts[i].scn)))
	{
	    return &(chbg_opts[i]);
	}
    }

    return NULL;
}

static int cfg_parse_bool(param)
char *param;
{
    return !strcasecmp(param, "true");
}

static gfloat cfg_parse_position(param)
char *param;
{
    gfloat rv;

    rv = _atof(param);
    if (errno == ERANGE ||
	(rv < (gfloat)0.0) ||
	(rv > (gfloat)1.0))
    {
	fprintf(stderr, gettext(
				"bad position value: %s, (0.0 - 1.0 required)\n"),
		param);
	rv = 0.5;
    }

    return rv;
}

int cfg_scenario(scenario)
gchar *scenario;
{
    FILE *f;
    char line[PATH_MAX];
    picentry_t *pe = NULL;
    banner_t *bn = NULL;
    rendering_type typev;
    gfloat max_growv;
    gfloat intervalv,intervalev;
    guint max_sizev;
    guint efectv;
    option_t *opt;

#define SET_PROP(prop_name, val) \
    if (pe && !pe->prop) \
	pe->prop = prop_dup(&cfg.properties); \
	    if (pe) \
		pe->prop->prop_name = val; \
	    else \
		cfg.properties.prop_name = val;

#define SET_BANNER_PROP(prop_name, val) \
    if (bn && !bn->prop) \
	bn->prop = banner_prop_new(); \
	    if (bn) \
		bn->prop->prop_name = val; \
	    else \
		cfg.properties.banner_prop.prop_name = val;


    if (!(f = fopen(scenario, "rb")))
    {
	perror(scenario);
	return -1;
    }

    while(fgets(line, sizeof(line), f))
    {
	gchar *lns;

	for(lns = line + strlen(line) - 1; lns >= line && isspace(*lns) ; lns--) 
	    *lns = '\0';
	for(lns = line ; *lns && isspace(*lns) ; lns++);

	if (lns[0] == '#') continue;
	if (!*lns) continue;

	opt = cfg_find_scn_option(lns);

	if (!opt)
	{
	    fprintf(stderr, gettext("Unable to parse line:\n%s\n"), line);
	    continue;
	}

	lns += strlen(opt->scn);
	for( ; *lns && isspace(*lns) ; lns++);

	switch (opt->type)
	{
	    case CHBG_OPT_V:
	    case CHBG_OPT_LONG_VERSION:
	    case CHBG_OPT_H:
	    case CHBG_OPT_LONG_HELP:
	    case CHBG_OPT_SETUP:
	    case CHBG_OPT_SCENARIO:
	    case CHBG_OPT_ONCE:
	    case CHBG_OPT_RUN:
	    case CHBG_OPT_THUMB:
	    case CHBG_OPT_WINDOWID:
	    case CHBG_OPT_FORCE_XSCREENSAVER:
	    case CHBG_OPT_FOREIGN:
		break;

	    case CHBG_OPT_MODE:
		typev = cfg_get_mode(lns);
		SET_PROP(type, typev)
		    break;
	    case CHBG_OPT_DSHADERS:
		{
		    GSList *ptr;

		    ptr = append_nr_to_list(cfg.d_shaders, lns);
		    if (ptr)
		    {
			cfg.d_shaders = ptr;
		    }
		    else
		    {
			fprintf(stderr, gettext("bad list: %s\n"), lns);
			return -1;
		    }
		}
		break;
	    case CHBG_OPT_DEFFECTS:
		{
		    GSList *ptr;

		    ptr = append_nr_to_list(cfg.d_effects, lns);
		    if (ptr)
		    {
			cfg.d_effects = ptr;
		    }
		    else
		    {
			fprintf(stderr, gettext("bad list: %s\n"), lns);
			return -1;
		    }
		}
		break;
	    case CHBG_OPT_INTERVAL:
		intervalv = _atof(lns);
		if (parse_interval(lns, &intervalv, &intervalev))
		{
		    fprintf(stderr, gettext("bad interval value: %s\n"), lns);
		    intervalv = default_interval;
		    intervalev = 0.0;
		}
		SET_PROP(interval, intervalv)
		SET_PROP(interval_end, intervalev)
		break;
	    case CHBG_OPT_MAX_SIZE:
		max_sizev = _atoi(lns);
		if (errno == ERANGE ||
		    max_sizev < 10 || max_sizev > 100)
		{
		    fprintf(stderr, gettext("bad max_size value: %s, (10 - 100 required)\n"), lns);
		    max_sizev = default_max_size;
		}
		SET_PROP(max_size, max_sizev)
		    break;
	    case CHBG_OPT_MIN_PSIZE:
		cfg.min_size = _atoi(lns);
		if (errno == ERANGE)
		{
		    fprintf(stderr, gettext("bad min_psize value: %s\n"), lns);
		}
		break;
	    case CHBG_OPT_SPEED:
		cfg.speed = _atoi(lns);
		if (errno == ERANGE)
		{
		    fprintf(stderr, gettext("bad speed value: %s\n"), lns);
		    cfg.speed = default_max_size;
		}
		break;
	    case CHBG_OPT_EFFECT:
		efectv = _atof(lns);
		if (errno == ERANGE)
		{
		    fprintf(stderr, gettext("bad effect value: %s, (0 - %d required)\n"), lns, CHBG_EFECTS);
		    efectv = default_efect;
		}
		SET_PROP(efect, efectv)
		    break;
	    case CHBG_OPT_SHADER:
		efectv = _atof(lns);
		if (errno == ERANGE)
		{
		    fprintf(stderr, gettext("bad shader value: %s, (0 - %d required)\n"), lns, shader_get_cnt());
		    efectv = default_shader;
		}
		SET_PROP(shade, efectv)
		    break;
	    case CHBG_OPT_MAX_GROW:
		max_growv = _atof(lns);
		if (errno == ERANGE || (max_growv <= (gfloat)0.0))
		{
		    fprintf(stderr, gettext("bad grow value: %s, (positive float number required)\n"), lns);
		    max_growv = default_max_grow;
		}
		SET_PROP(max_grow, max_growv)
		    break;
	    case CHBG_OPT_BG:
		{
		    GdkColor *clr;

		    if (pe)
		    {
			if (!pe->prop) 
			    pe->prop = prop_dup(&cfg.properties);
			clr = &pe->prop->background;
		    }
		    else
			clr = &cfg.properties.background;

		    if (!gdk_color_parse(lns, clr))
		    {
			fprintf(stderr, gettext("unknown color: %s\n"), lns);
			gdk_color_parse("#000000", clr);
		    }
		}
		break;
	    case CHBG_OPT_BG2:
		{
		    GdkColor *clr;

		    if (pe)
		    {
			if (!pe->prop) 
			    pe->prop = prop_dup(&cfg.properties);
			clr = &pe->prop->background2;
		    }
		    else
			clr = &cfg.properties.background2;

		    if (!gdk_color_parse(lns, clr))
		    {
			fprintf(stderr, gettext("unknown color: %s\n"), lns);
			gdk_color_parse("#000000", clr);
		    }
		}
		break;
	    case CHBG_OPT_RAND_COLORS:
		SET_PROP(rand_colors, cfg_parse_bool(lns))
		    break;
	    case CHBG_OPT_FRITER:
		cfg.fract_iter = _atoi(lns);
		if ((errno == ERANGE) || (cfg.fract_iter < 2))
		{
		    fprintf(stderr, gettext("bad fractal iterations value: %s\n"), lns);
		}
		break;
	    case CHBG_OPT_RECT_SIZE:
		cfg.box_size = _atoi(lns);
		if ((errno == ERANGE) || (cfg.box_size < 2))
		{
		    fprintf(stderr, gettext("bad box size value: %s\n"), lns);
		}
		break;
	    case CHBG_OPT_RANDOMIZE:
		if (!strcasecmp(lns, "true"))
		    cfg.randomize = TRUE;
		else
		    cfg.randomize = FALSE;
		break;
	    case CHBG_OPT_USE_GRAD:
		SET_PROP(use_grad, cfg_parse_bool(lns))
		    break;
	    case CHBG_OPT_USE_TILES:
		SET_PROP(use_tiles, cfg_parse_bool(lns))
		    break;
	    case CHBG_OPT_GRAD:
		efectv = _atof(lns);
		if (errno == ERANGE)
		{
		    fprintf(stderr, gettext("bad gradient value: %s\n"), lns);
		    efectv = 0;
		}
		SET_PROP(gradnr, efectv)
		    break;
	    case CHBG_OPT_XPOS:
		SET_PROP(xpos, cfg_parse_position(lns))
		    break;
	    case CHBG_OPT_YPOS:
		SET_PROP(ypos, cfg_parse_position(lns))
		    break;
	    case CHBG_OPT_TILE:
		SET_PROP(tile, g_strdup(lns))
		    break;
	    case CHBG_OPT_GRAD_DIR:
		cfg.grad_dir = g_strdup(lns);
		break;
	    case CHBG_OPT_ADD_TILE:
		cfg.tiles = g_list_append(cfg.tiles, simplify_path(lns));
		break;
	    case CHBG_OPT_R:
		cfg.recurse = cfg_parse_bool(lns);
		break;
	    case CHBG_OPT_BLANK:
		cfg.blank = cfg_parse_bool(lns);
		break;
	    case CHBG_OPT_NOTOOLTIPS:
		cfg.tooltips = cfg_parse_bool(lns);
		break;
	    case CHBG_OPT_INWINDOW:
		cfg.inwindow = cfg_parse_bool(lns);
		break;
	    case CHBG_OPT_CYCLE_BLANK:
		cfg.cycle_blank = cfg_parse_bool(lns);
		break;
	    case CHBG_OPT_SCREENSAVER:
		cfg.screensaver = cfg_parse_bool(lns);
		break;
#ifdef HAVE_ENLIGHTENMENT_SUPPORT
	    case CHBG_OPT_ENLIGHTENMENT:
		cfg.use_ebg = cfg_parse_bool(lns);
		break;
#endif
#ifdef HAVE_XSS_SUPPORT
	    case CHBG_OPT_XSCREENSAVER:
		cfg.xscreensaver = cfg_parse_bool(lns);
		break;
#endif
	    case CHBG_OPT_PICTURE:
		pe = picentry_new(simplify_path(lns));
		cfg.pics = g_list_append(cfg.pics, pe);
		cfg.num_pics ++;
		break;
	    case CHBG_OPT_PATTERN:
		cfg.pattern = g_slist_append(cfg.pattern, g_strdup(lns));
		break;
	    case CHBG_OPT_CURRENT_PICTURE:
		cfg.current_name = g_strdup(lns);
		break;
	    case CHBG_OPT_REMEMBER:
		cfg.remember_last = cfg_parse_bool(lns);
		break;
	    case CHBG_OPT_SORT:
		cfg.sort = cfg_parse_bool(lns);
		break;
	    case CHBG_OPT_USE_BANNERS:
		SET_PROP(use_banners, cfg_parse_bool(lns))
		    break;
	    case CHBG_OPT_BANNER:
		SET_PROP(banner, g_strdup(lns))
		    break;
	    case CHBG_OPT_ADD_BANNER:
		bn = banner_new(simplify_path(lns));
		cfg.banners = g_list_append(cfg.banners, bn);
		break;
	    case CHBG_OPT_BANNER_XPOS:
		SET_BANNER_PROP(xpos, cfg_parse_position(lns))
		    break;
	    case CHBG_OPT_BANNER_YPOS:
		SET_BANNER_PROP(ypos, cfg_parse_position(lns))
		    break;
	    case CHBG_OPT_BANNER_MAX_GROW:
		max_growv = _atof(lns);
		if (errno == ERANGE || (max_growv <= (gfloat)0.0))
		{
		    fprintf(stderr, gettext("bad grow value: %s, (positive float number required)\n"), lns);
		    max_growv = default_max_grow;
		}
		SET_BANNER_PROP(max_grow, max_growv)
		    break;
	    case CHBG_OPT_BANNER_MAX_SIZE:
		max_sizev = _atoi(lns);
		if (errno == ERANGE ||
		    max_sizev < 10 || max_sizev > 100)
		{
		    fprintf(stderr, gettext("bad max_size value: %s, (10 - 100 required)\n"), lns);
		    max_sizev = default_max_size;
		}
		SET_BANNER_PROP(max_size, max_sizev)
		    break;
	    case CHBG_OPT_BANNER_MODE:
		max_sizev = cfg_find_banner_mode(lns);
		if (errno == ERANGE ||
		    max_sizev < 0 || max_sizev > CHBG_BANNER_MODES )
		{
		    fprintf(stderr, gettext("bad banner mode value: %s, (0-%d required)\n"), lns, CHBG_BANNER_MODES );
		    max_sizev = 0;
		}
		SET_BANNER_PROP(mode, max_sizev)
		    break;
	    case CHBG_OPT_OUT_FILE:
		cfg.outfile = g_strdup(lns);
		break;
	    case CHBG_OPT_OUT_FILE_WIDTH:
		cfg.outfile_width = _atoi(lns);
		if (errno == ERANGE)
		    fprintf(stderr, gettext("bad width value: %s\n"), lns);
		break;
	    case CHBG_OPT_OUT_FILE_HEIGHT:
		cfg.outfile_height = _atoi(lns);
		if (errno == ERANGE)
		    fprintf(stderr, gettext("bad height value: %s\n"), lns);
		break;
	    case CHBG_OPT_TO_FILE:
		cfg.tofile = cfg_parse_bool(lns);
		break;
	    case CHBG_OPT_FAKE_NAUTILUS:
		cfg.fake_nautilus = cfg_parse_bool(lns);
		break;
#ifdef HAVE_XSS_SUPPORT
	    case CHBG_OPT_SEND_EXPOSE:
		cfg.send_expose = cfg_parse_bool(lns);
		break;
#endif
	    case CHBG_OPT_PRINT_FNAME:
		cfg.print_fname = cfg_parse_bool(lns);
		break;
	    case CHBG_OPT_LOG_FILE:
		cfg.log_file = g_strdup(lns);
		break;
	    case CHBG_OPT_USE_CACHE:
		cfg.use_cache = cfg_parse_bool(lns);
		break;
	}
    }
    fclose(f);

    return 0;
}

#define HEX(x) (((x) > 9) ? 'a' + (x) - 10 : '0' + (x))

guchar *cfg_get_color_str(color, buf)
GdkColor *color;
char *buf;
{
    guchar r,g,b;
    static guchar str[8];

    r = color->red / 256;
    g = color->green / 256;
    b = color->blue / 256;

    str[0]= '#';
    str[1]= HEX(r >> 4);
    str[2]= HEX(r & 0x0f);
    str[3]= HEX(g >> 4);
    str[4]= HEX(g & 0x0f);
    str[5]= HEX(b >> 4);
    str[6]= HEX(b & 0x0f);
    str[7]= '\0';

    if (buf)
    {
	strcpy(buf, str);
	return buf;
    }
    else
	return str;
}

int cfg_save_scenario(scenario)
gchar *scenario;
{
    FILE *f;
    GSList *ptr;
    GList *lptr;
    picentry_t *pe;

    if (!(f = fopen(scenario, "wb")))
    {
	perror(scenario);
	return -1;
    }

    fprintf(f, gettext("# this is automaticaly generated chbg scenario file\n\n"));
    if (cfg.current_name)
	fprintf(f, "CurrentPicture: %s\n", cfg.current_name);

    fprintf(f, "OutputToFile: %s\n", cfg.tofile ? "true" : "false");
    if (cfg.outfile)
	fprintf(f, "OutputFile: %s\n", cfg.outfile);
    fprintf(f, "OutputFileWidth: %d\n", cfg.outfile_width);
    fprintf(f, "OutputFileHeight: %d\n", cfg.outfile_height);
    fprintf(f, "Blank: %s\n", cfg.blank ? "true" : "false");
    fprintf(f, "Speed: %d\n", cfg.speed);
    fprintf(f, "Tooltips: %s\n", cfg.tooltips ? "true" : "false");
    fprintf(f, "Recurse: %s\n", cfg.recurse ? "true" : "false");
    fprintf(f, "InWindow: %s\n", cfg.inwindow ? "true" : "false");
    fprintf(f, "Randomize: %s\n", cfg.randomize ? "true" : "false");
    fprintf(f, "CycleBlank: %s\n", cfg.cycle_blank ? "true" : "false");
    fprintf(f, "MinPictureSize: %d\n", cfg.min_size);
    fprintf(f, "FractalIterations: %d\n", cfg.fract_iter);
    fprintf(f, "RectSize: %d\n", cfg.box_size);
    fprintf(f, "Screensaver: %s\n", cfg.screensaver ? "true" : "false");
#ifdef HAVE_XSS_SUPPORT
    fprintf(f, "XScreensaver: %s\n", cfg.xscreensaver ? "true" : "false");
#endif
#ifdef HAVE_ENLIGHTENMENT_SUPPORT
    fprintf(f, "Enlightenment: %s\n", cfg.use_ebg ? "true" : "false");
#endif
    fprintf(f, "RememberLast: %s\n", cfg.remember_last ? "true" : "false");
    fprintf(f, "Sort: %s\n", cfg.sort ? "true" : "false");
    fprintf(f, "ForceNautilus: %s\n", cfg.fake_nautilus ? "true" : "false");
#ifdef HAVE_XSS_SUPPORT
    fprintf(f, "SendExpose: %s\n", cfg.send_expose ? "true" : "false");
#endif
    fprintf(f, "PrintFileName: %s\n", cfg.print_fname ? "true" : "false");
    if (cfg.log_file)
	fprintf(f, "LogFile: %s\n", cfg.log_file);
    fprintf(f, "UseCache: %s\n", cfg.use_cache ? "true" : "false");

    if (cfg.grad_dir)
    {
	fprintf(f, "GradientsDir: %s\n", cfg.grad_dir);
    }

    fprintf(f, "\n");

    for (lptr = cfg.tiles; lptr; lptr = lptr->next)
    {
	fprintf(f, "AddTile: %s\n", (char *)lptr->data);
    }

    fprintf(f, "\n");

    fprintf(f, "BannerXPosition: %.3f\n", cfg.properties.banner_prop.xpos);
    fprintf(f, "BannerYPosition: %.3f\n", cfg.properties.banner_prop.ypos);
    fprintf(f, "BannerMaxGrow: %.1f\n", cfg.properties.banner_prop.max_grow);
    fprintf(f, "BannerMaxSize: %d\n", cfg.properties.banner_prop.max_size);
    fprintf(f, "BannerMode: %s\n\n", banner_modes[cfg.properties.banner_prop.mode]);

    for (lptr = cfg.banners; lptr; lptr = lptr->next)
    {
	banner_t *bn = lptr->data;

	fprintf(f, "AddBanner: %s\n", bn->name);

	if (bn->prop)
	{
	    fprintf(f, "BannerXPosition: %.3f\n", bn->prop->xpos);
	    fprintf(f, "BannerYPosition: %.3f\n", bn->prop->ypos);
	    fprintf(f, "BannerMaxGrow: %.1f\n", bn->prop->max_grow);
	    fprintf(f, "BannerMaxSize: %d\n\n", bn->prop->max_size);
	    fprintf(f, "BannerMode: %s\n\n", banner_modes[bn->prop->mode]);
	}
    }

    fprintf(f, "\n");

    ptr = cfg.d_effects;
    if (ptr)
    {
	fprintf(f, "DisabledEffects: %d", (int)ptr->data);
	ptr = ptr->next;
	while(ptr)
	{
	    fprintf(f, ",%d", (int)ptr->data);
	    ptr = ptr->next;
	}
	fprintf(f, "\n");
    }

    ptr = cfg.d_shaders;
    if (ptr)
    {
	fprintf(f, "DisabledShaders: %d", (int)ptr->data);
	ptr = ptr->next;
	while(ptr)
	{
	    fprintf(f, ",%d", (int)ptr->data);
	    ptr = ptr->next;
	}
	fprintf(f, "\n");
    }

    ptr = cfg.pattern;
    while(ptr)
    {
	fprintf(f, "Pattern: %s\n", (char *)ptr->data);
	ptr = ptr->next;
    }

    fprintf(f, "UseGradients: %s\n", cfg.properties.use_grad ? "true" : "false");
    fprintf(f, "UseTiles: %s\n", cfg.properties.use_tiles ? "true" : "false");
    fprintf(f, "UseBanners: %s\n", cfg.properties.use_banners ? "true" : "false");
    fprintf(f, "Gradient: %d\n", cfg.properties.gradnr);


    fprintf(f, "BackgroundColor: %s\n", cfg_get_color_str(&cfg.properties.background, NULL));
    fprintf(f, "BackgroundColor2: %s\n", cfg_get_color_str(&cfg.properties.background2, NULL));

    fprintf(f, "RandomizeColors: %s\n", cfg.properties.rand_colors ? "true" : "false");
    fprintf(f, "RenderingMode: %s\n", cfg_get_mode_str(cfg.properties.type));
    fprintf(f, "MaxGrow: %.1f\n", cfg.properties.max_grow);
    fprintf(f, "MaxSize: %d\n", cfg.properties.max_size);
    fprintf(f, "XPosition: %.3f\n", cfg.properties.xpos);
    fprintf(f, "YPosition: %.3f\n", cfg.properties.ypos);
    if (cfg.properties.tile)
	fprintf(f, "Tile: %s\n", cfg.properties.tile);
    if (cfg.properties.banner)
	fprintf(f, "Banner: %s\n", cfg.properties.banner);

    if (cfg.properties.interval_end == 0.0)
	fprintf(f, "Interval: %.2f\n", cfg.properties.interval);
    else
	fprintf(f, "Interval: %.2f-%.2f\n",
		cfg.properties.interval, cfg.properties.interval_end);

    /* Next line : One "\n" in Efect-line too much. Didnt load efect & shader */
    /* parameters correctly...						  */

    fprintf(f, "Efect: %d\n", cfg.properties.efect);
    fprintf(f, "Shader: %d\n\n", cfg.properties.shade);

    // Use the original list if present
    lptr = (cfg.pics_orig ? cfg.pics_orig : cfg.pics);
    while (lptr)
    {
	char pom1[16],pom2[16];

	pe = (picentry_t *) lptr->data;

	fprintf(f, "Picture: %s\n", pe->name);
	if (pe->prop)
	{
	    if (pe->prop->type != cfg.properties.type)
		fprintf(f, "RenderingMode: %s\n",
			cfg_get_mode_str(pe->prop->type));

	    if (pe->prop->max_grow != cfg.properties.max_grow)
		fprintf(f, "MaxGrow: %.1f\n",
			pe->prop->max_grow);

	    if (pe->prop->max_size != cfg.properties.max_size)
		fprintf(f, "MaxSize: %d\n", pe->prop->max_size);

	    cfg_get_color_str(&pe->prop->background, pom1);
	    cfg_get_color_str(&cfg.properties.background, pom2);
	    if (strcmp(pom1, pom2))
		fprintf(f, "BackgroundColor: %s\n", pom1);

	    cfg_get_color_str(&pe->prop->background2, pom1);
	    cfg_get_color_str(&cfg.properties.background2, pom2);
	    if (strcmp(pom1, pom2))
		fprintf(f, "BackgroundColor2: %s\n", pom1);

	    if (!pe->prop->rand_colors !=
		!cfg.properties.rand_colors)
		fprintf(f, "RandomizeColors: %s\n",
			pe->prop->rand_colors ? "true" : "false");

	    if ((pe->prop->interval_end !=
		 cfg.properties.interval_end) ||
		(pe->prop->interval != cfg.properties.interval))
	    {
		if (pe->prop->interval_end == 0.0)
		    fprintf(f, "Interval: %.2f\n",
			    pe->prop->interval);
		else
		    fprintf(f, "Interval: %.2f-%.2f\n",
			    pe->prop->interval,
			    pe->prop->interval_end);
	    }

	    if (pe->prop->efect != cfg.properties.efect)
		fprintf(f, "Efect: %d\n", pe->prop->efect);

	    if (pe->prop->shade != cfg.properties.shade)
		fprintf(f, "Shader: %d\n", pe->prop->shade);

	    if (!pe->prop->use_grad != !cfg.properties.use_grad)
		fprintf(f, "UseGradients: %s\n",
			pe->prop->use_grad ? "true" : "false");

	    if (!pe->prop->use_tiles != !cfg.properties.use_tiles)
		fprintf(f, "UseTiles: %s\n",
			pe->prop->use_tiles ? "true" : "false");

	    if (pe->prop->gradnr != cfg.properties.gradnr)
		fprintf(f, "Gradient: %d\n", pe->prop->gradnr);

	    if (pe->prop->xpos != cfg.properties.xpos)
		fprintf(f, "XPosition: %.3f\n", pe->prop->xpos);

	    if (pe->prop->ypos != cfg.properties.ypos)
		fprintf(f, "YPosition: %.3f\n", pe->prop->ypos);

	    if (pe->prop->tile &&
		(!cfg.properties.tile ||
		 strcmp(pe->prop->tile, cfg.properties.tile)))
		fprintf(f, "Tile: %s\n", pe->prop->tile);

	    if (pe->prop->banner &&
		(!cfg.properties.banner ||
		 strcmp(pe->prop->banner, cfg.properties.banner)))
		fprintf(f, "Banner: %s\n", pe->prop->banner);
	}
	lptr = lptr->next;
    }

    fclose(f);

    return 0;
}

enum {
    CHBG_MENUACCEL ,
    CHBG_PROP ,
    CHBG_LAST
};

struct rc_terms {
    char *name;
    int id;
};

static struct rc_terms chbgproprc[] = {
    {"MenuAccel:", CHBG_MENUACCEL},
    {"Property:", CHBG_PROP}
};

void cfg_load_prop()
{
    char pom[2048];
    FILE *f;
    char *line;
    int found = FALSE;
    int i;

    sprintf(pom, "%s/.chbg_prop", g_get_home_dir());

    f = fopen(pom, "rb");

    if (!f) return;

    while((line = fgets(pom, sizeof(pom), f)))
    {
	if (*line == '#') continue;
	if (!strcspn(line, " \t\r\n")) continue;

	found = FALSE;
	for (i = 0 ; i < CHBG_LAST ; i++)
	{
	    if (!strncasecmp(chbgproprc[i].name, line, strlen(chbgproprc[i].name)))
	    {
		found = TRUE;
		break;
	    }
	}
	if (!found)
	    fprintf(stderr, gettext("Unable to parse: %s\n"), line);
	else
	{
	    line += strlen(chbgproprc[i].name);
	    line += strspn(line, "\t ");
	    *(line + strcspn(line,"\n\r")) = '\0';
	    switch(chbgproprc[i].id)
	    {
		case CHBG_MENUACCEL:
		    {
			gaccel *ga;

			if ((ga = gaccel_parse_str((guchar *)line)))
			{
			    gaccel_add(ga);
			}
			else
			    fprintf(stderr, gettext("Unable to parse: \"%s\"\n"), line); 
		    }
		    break;
		case CHBG_PROP:
		    {
			gprop *gp;

			if ((gp = gprop_parse(line)))
			{
			    gprop_add(gp);
			}
			else
			    fprintf(stderr, gettext("Unable to parse: \"%s\"\n"), line); 
		    }
		    break;
		default:
		    break;
	    }
	}
    }
    fclose(f);
}

void cfg_save_prop()
{
    char pom[2048];
    FILE *f;

    sprintf(pom, "%s/.chbg_prop", g_get_home_dir());

    f = fopen(pom, "w+b");

    if (!f)
    {
	fprintf(stderr, gettext("Unable to create ~/.chbg_prop file"));
	return;
    }

    fprintf(f, gettext("# This file was generated by chbg\n# You may edit it if you're careful!\n\n"));

    gaccel_save_keys(f);
    gprop_save(f);

    fclose(f);
}

void cfg_add_default_prop()
{
    int i,j;
    char *cmd;
    char pom[256];
    struct {
	char *label;
	char *cmd;
	char *prg;
	int have;
    } default_extop[] = {
	{"Gimp", "gimp %f", "gimp", FALSE},
	{"Electric Eyes", "ee %f", "ee", FALSE},
	{"Eye of Gnome", "eog %f", "eog", FALSE},
	{"GQview", "gqview %f", "gqview", FALSE},
	{"Xpaint", "xpaint %f", "xpaint", FALSE},
	{"Xv", "xv %f", "xv", FALSE},
	{NULL, NULL, NULL, FALSE}
    };

    if (gprop_get_bool("have_default_editors", &i))
	return;

    for (i = 0; default_extop[i].label; i++)
    {
	if (check_program(default_extop[i].prg))
	{
	    default_extop[i].have = TRUE;
	}
    }

    for (i = 0, j = 0; i < CHBG_EXTERNOPENER; i++)
    {
	sprintf(pom, "option-extop%02d-cmd", i);

	if (!gprop_get_str(pom, &cmd))
	{
	    while (default_extop[j].label &&
		   !default_extop[j].have) j++;

	    if (default_extop[j].label)
	    {
		gprop_set_str(pom, default_extop[j].cmd);
		sprintf(pom, "option-extop%02d-label", i);
		gprop_set_str(pom, default_extop[j].label);
		j++;
	    }
	    else
		break;
	}
    }

    gprop_set_bool("have_default_editors", TRUE);
}

