/* 
 * Testbed for GLSL fragment noise() replacement.
 *
 * Shaders are loaded from two external files:
 * "GLSLnoisetest.vert" and "GLSLnoisetest.frag".
 * The program itself draws a spinning sphere
 * with a noise-generated fragment color.
 *
 * This program uses GLFW for convenience, to handle the OS-specific
 * window management stuff. Some Windows-specific stuff is still there,
 * though, so this code is not portable to other platforms.
 * The non-portable parts of the code is the MessageBox() function
 * call, which is easily replaced by a fprintf(stderr) call,
 * and probably most of the stupid loading of function pointers that
 * Windows requires of you to access anything above OpenGL 1.1.
 *
 * Author: Stefan Gustavson (stegu@itn.liu.se) 2004, 2005
 * Port to Glut: Gabriel Zachmann (zach@tu-clausthal.de)(za@, 2007
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#if defined(__linux__)

#include <gl.h>
#include <glu.h>
#include <glut.h>

#elif defined(__APPLE__)

#include <OpenGL/gl.h>
#include <OpenGL/glu.h>
#include <GLUT/glut.h>

#endif

#include "shader_setup.h"


/* Some more global variables for convenience. */
double t0 = 0.0;
int frames = 0;
char titlestring[200];

GLuint permTextureID;
GLuint gradTextureID;
GLuint sphereList;
GLboolean updateTime = GL_TRUE;
GLboolean animateObject = GL_TRUE;

GLuint programObj;

GLint location_permTexture = -1; 
GLint location_gradTexture = -1; 
GLint location_time = -1;

const char *vertexShaderStrings[1];
const char *fragmentShaderStrings[1];
GLint vertexCompiled;
GLint fragmentCompiled;
GLint shadersLinked;
char str[4096]; // For error messages from the GLSL compiler and linker

int perm[256]= {151,160,137,91,90,15,
  131,13,201,95,96,53,194,233,7,225,140,36,103,30,69,142,8,99,37,240,21,10,23,
  190, 6,148,247,120,234,75,0,26,197,62,94,252,219,203,117,35,11,32,57,177,33,
  88,237,149,56,87,174,20,125,136,171,168, 68,175,74,165,71,134,139,48,27,166,
  77,146,158,231,83,111,229,122,60,211,133,230,220,105,92,41,55,46,245,40,244,
  102,143,54, 65,25,63,161, 1,216,80,73,209,76,132,187,208, 89,18,169,200,196,
  135,130,116,188,159,86,164,100,109,198,173,186, 3,64,52,217,226,250,124,123,
  5,202,38,147,118,126,255,82,85,212,207,206,59,227,47,16,58,17,182,189,28,42,
  223,183,170,213,119,248,152, 2,44,154,163, 70,221,153,101,155,167, 43,172,9,
  129,22,39,253, 19,98,108,110,79,113,224,232,178,185, 112,104,218,246,97,228,
  251,34,242,193,238,210,144,12,191,179,162,241, 81,51,145,235,249,14,239,107,
  49,192,214, 31,181,199,106,157,184, 84,204,176,115,121,50,45,127, 4,150,254,
  138,236,205,93,222,114,67,29,24,72,243,141,128,195,78,66,215,61,156,180};

/* These are Ken Perlin's proposed gradients for 3D noise. I kept them for
   better consistency with the reference implementation, but there is really
   no need to pad this to 16 gradients for this particular implementation.
   If only the "proper" first 12 gradients are used, they can be extracted
   from the grad4[][] array: grad3[i][j] == grad4[i*2][j], 0<=i<=11, j=0,1,2
*/
int grad3[16][3] = {{0,1,1},{0,1,-1},{0,-1,1},{0,-1,-1},
                   {1,0,1},{1,0,-1},{-1,0,1},{-1,0,-1},
                   {1,1,0},{1,-1,0},{-1,1,0},{-1,-1,0}, // 12 cube edges
                   {1,0,-1},{-1,0,-1},{0,-1,1},{0,1,1}}; // 4 more to make 16

/* These are my own proposed gradients for 4D noise. They are the coordinates
   of the midpoints of each of the 32 edges of a tesseract, just like the 3D
   noise gradients are the midpoints of the 12 edges of a cube.
*/
int grad4[32][4]= {{0,1,1,1}, {0,1,1,-1}, {0,1,-1,1}, {0,1,-1,-1}, // 32 tesseract edges
                   {0,-1,1,1}, {0,-1,1,-1}, {0,-1,-1,1}, {0,-1,-1,-1},
                   {1,0,1,1}, {1,0,1,-1}, {1,0,-1,1}, {1,0,-1,-1},
                   {-1,0,1,1}, {-1,0,1,-1}, {-1,0,-1,1}, {-1,0,-1,-1},
                   {1,1,0,1}, {1,1,0,-1}, {1,-1,0,1}, {1,-1,0,-1},
                   {-1,1,0,1}, {-1,1,0,-1}, {-1,-1,0,1}, {-1,-1,0,-1},
                   {1,1,1,0}, {1,1,-1,0}, {1,-1,1,0}, {1,-1,-1,0},
                   {-1,1,1,0}, {-1,1,-1,0}, {-1,-1,1,0}, {-1,-1,-1,0}};

/*
 * printError() - Signal an error. MessageBox() is a Windows-specific function,
 * rewrite this to print the error message to the console to make it portable!
 */
void printError(const char *errtype, const char *errmsg)
{
  fprintf(stderr, "%s: %s\n", errtype, errmsg);
}



int getUniLoc( const char * uni_name )
{
	GLint uni_id = glGetUniformLocation( programObj, uni_name );
	if ( uni_id == -1 )
		fprintf(stderr,"uniform 'time' does not exist!\n");
	return uni_id;
}



// rendering

void changeSize(int w, int h)
{
	// Prevent a divide by zero, when window is too short
	if ( h == 0 )
		h = 1;

	float ratio = 1.0* w / h;

	// Reset the coordinate system before modifying
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	
	// Set the viewport to be the entire window
    glViewport(0, 0, w, h);

	// Set the correct perspective.
	gluPerspective(45,ratio,1,100);
	glMatrixMode(GL_MODELVIEW);


}


void drawScene(float t);

void renderScene(void)
{
	static float a = 0.0;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glLoadIdentity();

    // Look from 0,-4,0 towards 0,0,0 with Z as "up" in the image
    gluLookAt( 0.0f, -4.0f, 0.0f,  // Eye position
               0.0f, 0.0f, 0.0f,   // View point
               0.0f, 0.0f, 1.0f ); // Up vector

#if 0
	// trackball
	GLfloat m[4][4];
	build_rotmatrix(m, curquat);
	glMultMatrixf(&m[0][0]);
#endif

	glUniform1f( location_time, a );		// this should be the time
	glUniform1i( location_permTexture, 0 ); // Texture unit 0
	glUniform1i( location_gradTexture, 1 ); // Texture unit 1

	glRotatef(a,0,1,0);

	// glutSolidTeapot(1);
	drawScene(a);

	a+=0.1;

	glutSwapBuffers();
}


void processNormalKeys(unsigned char key, int x, int y)
{
	if (key == 27) 
		exit(0);
}


/*
 * drawTexturedSphere(r, segs) - Draw a sphere centered on the local
 * origin, with radius "r" and approximated by "segs" polygon segments,
 * with texture coordinates in a latitude-longitude mapping.
 */
void drawTexturedSphere(float r, int segs)
{
	int i, j;
	float x, y, z, z1, z2, R, R1, R2;

	// Top cap
	glBegin(GL_TRIANGLE_FAN);
	glNormal3f(0,0,1);
	glTexCoord2f(0.5f,1.0f); // This is an ugly (u,v)-mapping singularity
	glVertex3f(0,0,r);
	z = cos(M_PI/segs);
	R = sin(M_PI/segs);
	for(i = 0; i <= 2*segs; i++)
	{
		x = R*cos(i*2.0*M_PI/(2*segs));
		y = R*sin(i*2.0*M_PI/(2*segs));
		glNormal3f(x, y, z);
		glTexCoord2f((float)i/(2*segs), 1.0f-1.0f/segs);
		glVertex3f(r*x, r*y, r*z);
	}
	glEnd();  

	// Height segments
	for(j = 1; j < segs-1; j++)
	{
		z1 = cos(j*M_PI/segs);
		R1 = sin(j*M_PI/segs);
		z2 = cos((j+1)*M_PI/segs);
		R2 = sin((j+1)*M_PI/segs);
		glBegin(GL_TRIANGLE_STRIP);
		for(i = 0; i <= 2*segs; i++)
		{
			x = R1*cos(i*2.0*M_PI/(2*segs));
			y = R1*sin(i*2.0*M_PI/(2*segs));
			glNormal3f(x, y, z1);
			glTexCoord2f((float)i/(2*segs), 1.0f-(float)j/segs);
			glVertex3f(r*x, r*y, r*z1);
			x = R2*cos(i*2.0*M_PI/(2*segs));
			y = R2*sin(i*2.0*M_PI/(2*segs));
			glNormal3f(x, y, z2);
			glTexCoord2f((float)i/(2*segs), 1.0f-(float)(j+1)/segs);
			glVertex3f(r*x, r*y, r*z2);
		}
		glEnd();
	}

	// Bottom cap
	glBegin(GL_TRIANGLE_FAN);
	glNormal3f(0,0,-1);
	glTexCoord2f(0.5f, 1.0f); // This is an ugly (u,v)-mapping singularity
	glVertex3f(0,0,-r);
	z = -cos(M_PI/segs);
	R = sin(M_PI/segs);
	for(i = 2*segs; i >= 0; i--)
	{
		x = R*cos(i*2.0*M_PI/(2*segs));
		y = R*sin(i*2.0*M_PI/(2*segs));
		glNormal3f(x, y, z);
		glTexCoord2f(1.0f-(float)i/(2*segs), 1.0f/segs);
		glVertex3f(r*x, r*y, r*z);
	}
	glEnd();
}


/*
 * initSphereList(GLuint *listID, GLdouble scale) - create a display list
 * to render the sphere more efficently than calling lots of trigonometric
 * functions for each frame.
 * (A vertex array would be even faster, but I'm a bit lazy here.)
 */
void initSphereList(GLuint *listID, GLdouble scale)
{
	*listID = glGenLists(1);

	glNewList(*listID, GL_COMPILE);
	drawTexturedSphere(scale, 20);
	glEndList();
}


/*
 * initPermTexture(GLuint *texID) - create and load a 2D texture for
 * a combined index permutation and gradient lookup table.
 * This texture is used for 2D and 3D noise, both classic and simplex.
 */
void initPermTexture(GLuint *texID)
{
	char *pixels;
	int i,j;

	glGenTextures(1, texID); // Generate a unique texture ID
	glBindTexture(GL_TEXTURE_2D, *texID); // Bind the texture to texture unit 0

	pixels = (char*)malloc( 256*256*4 );
	for(i = 0; i<256; i++)
		for(j = 0; j<256; j++)
		{
			int offset = (i*256+j)*4;
			char value = perm[(j+perm[i]) & 0xFF];
			pixels[offset] = grad3[value & 0x0F][0] * 64 + 64;   // Gradient x
			pixels[offset+1] = grad3[value & 0x0F][1] * 64 + 64; // Gradient y
			pixels[offset+2] = grad3[value & 0x0F][2] * 64 + 64; // Gradient z
			pixels[offset+3] = value;                     // Permuted index
		}

	// we need GL_NEAREST lookup.
	glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, 256, 256, 0, GL_RGBA, GL_UNSIGNED_BYTE, pixels );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
}

/*
 * initGradTexture(GLuint *texID) - create and load a 2D texture
 * for a 4D gradient lookup table. This is used for 4D noise only.
 */
void initGradTexture(GLuint *texID)
{
	char *pixels;
	int i,j;

	glActiveTexture(GL_TEXTURE1); // Activate a different texture unit (unit 1)

	glGenTextures(1, texID); // Generate a unique texture ID
	glBindTexture(GL_TEXTURE_2D, *texID); // Bind the texture to texture unit 2

	pixels = (char*)malloc( 256*256*4 );
	for(i = 0; i<256; i++)
		for(j = 0; j<256; j++)
		{
			int offset = (i*256+j)*4;
			char value = perm[(j+perm[i]) & 0xFF];
			pixels[offset] = grad4[value & 0x1F][0] * 64 + 64;   // Gradient x
			pixels[offset+1] = grad4[value & 0x1F][1] * 64 + 64; // Gradient y
			pixels[offset+2] = grad4[value & 0x1F][2] * 64 + 64; // Gradient z
			pixels[offset+3] = grad4[value & 0x1F][3] * 64 + 64; // Gradient w
		}

	// we need GL_NEAREST lookup.
	glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, 256, 256, 0, GL_RGBA, GL_UNSIGNED_BYTE, pixels );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );

	glActiveTexture(GL_TEXTURE0); // Switch active texture unit back to 0 again
}


/*
 * drawScene(float t) - the actual drawing commands to render our scene.
 */
void drawScene(float t)
{
    glRotatef(30.0f, 1.0f, 0.0f, 0.0f); // Rotate the view somewhat
    glPushMatrix(); // Transforms to animate the light:
      glRotatef(30.0f*t, 0.0f, 0.0f, 1.0f);
      glTranslatef(5.0f, 0.0f, 0.0f); // Orbit around Z, 5 units from origin
      float lightpos0[4]={0.0f, 0.0f, 0.0f, 1.0f}; // Origin, in hom. coords
      glLightfv(GL_LIGHT0, GL_POSITION, lightpos0); // Set light position
    glPopMatrix(); // Revert to initial transform
    glPushMatrix(); // Transforms to animate the object:
      glRotatef(45.0f*t, 0.0f, 0.0f, 1.0f); // Spin around Z
      glColor3f(1.0f, 1.0f, 1.0f); // White base color
      // Enable lighting and the LIGHT0 we placed before
      glEnable(GL_LIGHTING);
      glEnable(GL_LIGHT0);
      // We have now enabled lighting, so this object is lit.
      glCallList(sphereList); // Draw a sphere using the display list
    glPopMatrix(); // Revert to initial transform
    // Disable lighting again, to prepare for next frame.
    glDisable(GL_LIGHTING);
    glDisable(GL_LIGHT0);
}



/*
 * main(argc, argv) - the standard C entry point for the program
 */
int main(int argc, char *argv[])
{
	glutInit(&argc, argv);
	glutInitDisplayMode( GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGB );
	glutInitWindowPosition(100,100);
	glutInitWindowSize(400,400);
	glutCreateWindow("Simplex Noise Demo");

	glutDisplayFunc(renderScene);
	glutIdleFunc(renderScene);
	glutReshapeFunc(changeSize);
	glutKeyboardFunc(processNormalKeys);

	glEnable(GL_DEPTH_TEST);
	glClearColor(1.0,1.0,1.0,1.0);
	glEnable(GL_CULL_FACE);

    glEnable(GL_TEXTURE_1D); // Enable 1D texturing
    glEnable(GL_TEXTURE_2D); // Enable 2D texturing

	checkShaderExt();

	programObj = setShaders( "noise.vert", "noise.frag" );

	// Locate the uniform shader variables so we can set them later:
	// a texture ID ("permTexture") and a float ("time").
	location_permTexture = getUniLoc( "permTexture" );
	location_gradTexture = getUniLoc( "gradTexture" );
	location_time = getUniLoc( "time" );

    // Create and load the textures (generated, not read from a file)
    initPermTexture(&permTextureID);
    initGradTexture(&gradTextureID);
    
    // Compile a display list for the teapot, to render it more quickly
    initSphereList(&sphereList, 1.0);
    
	glutMainLoop();

    return 0;
}

