#!/usr/bin/python -t -3 -O

from math import sqrt, fsum
import sys
import random
import copy
import functools


class Vector(object):

	# Create a Vector
	# Vector(3) -> vector of dim 3 with all zeros
	# Vector(3, 0.5) -> vector of dim 3 with all 0.5
	# Vector(3, [list] ) -> vector of dim 3 with coordinates init'ed from list
	def __init__(self, dimension, coords=None):
		if coords is None:
			self._coords = [0] * dimension
		elif isinstance( coords, (int,float) ):
			self._coords = [coords] * dimension
		else:
			self._coords = list(coords)
			if len(self._coords) != dimension:
				print >> sys.stderr, 'Vector ctor: dimension mismatch!'
				print >> sys.stderr, "	  dimension=%d, but you gave %d coordinates!" \
										% ( dimension, len(self._coords) )

	@classmethod
	def random( cls, dimension ):								# Vector.random(1000) gives random vector
		return cls( dimension, [ random.random() for i in xrange(dimension) ] )

	@classmethod
	def randomGauss( cls, dimension, center, sigma2 ):			# Vector.randomGauss(1000) gives random vector
		return cls( dimension, [ random.gauss(center[i],sigma2) # with Gaussian distribution around center
									for i in xrange(dimension) ] )

	def copy(self):												# make a deep copy
		return copy.deepcopy( self )

	def __getitem__(self, index):								# implements x[i] for rhs
		return self._coords[index]

	def __setitem__(self, index, value):						# dito for lhs
		self._coords[index] = value

	def __len__(self):											# number of dimensions, NOT vector magnitude
		return len(self._coords)

	def dist(self, other):										# x.dist(y) = || x - y ||
		return abs( Vector(len(self), self - other) )

	def length(self):											# x.length() = || x ||
		return sqrt( fsum(x*x for x in self._coords) )			# using "list comprehension" here;
																# use fsum() for better precision; but can be
																# slower by a factor 2 compared to sum()

	def __abs__(self):											# abs(x) := || x ||
		return self.length()

	def normalize(self):										# make self a unit vector
		mag = self.length()
		if mag > 0:
			self *= (1.0/mag)

	def __str__(self):											# for 'print x'
		return '( ' + ','.join( [str(c) for c in self._coords] ) + ' )'

	def __mul__(self, operand):									# 'vec1 * vec2' (dot product) or  'vec * num' (scaling)
		dim = len(self)
		if isinstance(operand, (int,float)):					# scaling
			return Vector( dim, [ x*operand for x in self._coords] )
		elif isinstance(operand, Vector):						# dot product
			return sum( [ x*y for x,y in zip(self._coords,operand._coords) ] )
		else:
			print >> sys.stderr, 'Vector *: unknown type of operand'
			return None

	def __rmul__(self, operand):								# num * vec
		return self * operand

	def __div__( self, operand):								# vec / scalar
		return self * (1.0/operand)

	def __add__(self, other):									# vec1 + vec2
		return Vector( len(self), [x+y for x,y in zip(self._coords,other._coords) ] )

	def __sub__(self, other):									# vec1 - vec2
		return Vector( len(self), [x-y for x,y in zip(self._coords,other._coords) ] )

	def min( self, other):										# vec1.min( vec2 )
		return Vector( len(self), [ min(x,y) for x,y in zip(self._coords,other._coords) ] )

	def max( self, other):										# vec1.min( vec2 )
		return Vector( len(self), [ max(x,y) for x,y in zip(self._coords,other._coords) ] )



########################## Unit tests #############################


Epsilon = 0.0000001


def checkEqual( z1, z2 ):
	if abs(z1-z2) < Epsilon:
		print "OK"
	else:
		print "Error!"


if __name__ == "__main__":
	print "Unit tests of class Vector"

	x = Vector( 4, (1, 2, 3, 4) )
	y = Vector( 4, (2, 3, 4, 5) )
	z = y - x
	print "1.: ", z
	checkEqual( abs(z), 2.0 )

	a = Vector( 4, 1.0 )
	print "2.:" , a
	checkEqual( z, a )

	a = z * x
	print "3.: ", a
	checkEqual( a, 10.0 )

	a = abs(z)
	b = x.dist(y)
	print "4.: ", a, b
	checkEqual( a, b )

	r = Vector.random(100)
	r2 = r * 2
	s = r2 - r
	print "5.: ", 
	checkEqual( r, s )

	
	# check randomGauss
	points = []
	dim = 3
	cluster_center = Vector.random(dim)
	n = 100
	for j in xrange(n):
		points.append( Vector.randomGauss( dim, cluster_center, 0.01 ) )
	mini = functools.reduce( lambda a, b: a.min(b) , points )		# minimum of all points (coord-wise)
	maxi = functools.reduce( lambda a, b: a.max(b) , points )		# minimum of all points (coord-wise)
	summ = functools.reduce( lambda a, b: a + b , points )
	print "Min: ", mini
	print "Max: ", maxi
	print "Avg: ", (summ / n)

	diag = maxi - mini
	scal = Vector( len(diag), [ 1.0/x for x in diag] )
	for p in points:
		for i in xrange( len(p) ):
			p[i] = ( p[i] - mini[i] ) * scal[i]

	mini = functools.reduce( lambda a, b: a.min(b) , points )
	maxi = functools.reduce( lambda a, b: a.max(b) , points )
	
	checkEqual( mini, Vector(dim,0.0) )
	checkEqual( maxi, Vector(dim,1.0) )
