
/*****************************************************************************\
 *                              Water Waves
\*****************************************************************************/

/*! @file Waterwaves.cpp
 *
 *  @brief
 *	Waterwaves effect class implementation
 *    
 *  @author Michael Nolden (michael120@web.de)
 *  
 */

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

// Qt headers
#include <qtimer.h>
#include <qvgroupbox.h>
#include <qlabel.h>
#include <qslider.h>
#include <qgl.h>
#include <qimage.h>
#include <qcheckbox.h>

// Local Headers
#include "Waterwaves.h"

// this function computes a points distance from (mx,my) and returns it in the variable dist
void Waterwaves::distance(GLfloat _x, GLfloat _y, GLfloat mx, GLfloat my)
{
	dist = sqrt((_x-mx)*(_x-mx)+(_y-my)*(_y-my));	//computes the distance of point (_x,_y) 
													//from (mx,my)
}

// this function computes texture coordinates and returns them in the variables tx, ty and tz
void Waterwaves::texcoords3f(GLfloat x, GLfloat y, GLfloat z)
{
	tx = ((x-3)/6)+1;	//computes the texture coordinate in x direction [0,1]
	ty = ((y-3)/6)+1;	//computes the texture coordinate in y direction [0,1]
	tz = ((z-3)/6)+1;	//computes the texture coordinate in z direction [0,1]
}

// **** Construction/desctruction ****
/**
 * @param parent Pointer to the parent window.
 *  
 * @see GLEffect::GLEffect(), effectInfo
 **/
Waterwaves::Waterwaves(GLFramework* parent):GLEffect(parent)
{
	// the following lines set parameter required by the framework
	effectInfo->effectName = "Water Waves";
	effectInfo->fileDialogNames[0] = "Load Textures";
	effectInfo->needNewTextures = true;
	effectInfo->requiredImageLists = 1;
	effectInfo->shortDescription = "Water Waves";
	effectInfo->version = "1.0";
	effectInfo->author = "Michael Nolden (michael120@web.de)";
	effectInfo->needRawTextures = false;
		
	// texImage1 to texImage3 are set to 0 in order to prevent any previous references from 
	//	interfering with the current effect
	texImage1 = 0;
	texImage2 = 0;
	texImage3 = 0;
	
	/* This has to be included as the last line in your constructor. 
	Don't forget it! */
	createControlPanel(true);
	hideControls();
}

/**
 * 
 *
 * @see GLEffect::~GLEffect()
 **/
Waterwaves::~Waterwaves()
{
}

// Initialization
/**
 * 
 *
 * @see GLEffect::init()
 **/
void Waterwaves::initialize()
{
	// sets the viewport
	fovAngle = 45;
	nearClipPlane = 1;
	farClipPlane = 30;
	viewDistance = 10;
	currentDepthCoord = 0;

	// the following lines set essential Open Gl parameters 
    glShadeModel(GL_SMOOTH);
	glBlendFunc(GL_ONE_MINUS_SRC_ALPHA, GL_SRC_ALPHA);
	glEnable(GL_BLEND);
	glClearColor(0.0, 0.0, 0.0, 1.0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glGenTextures(1, &texture);
	glBindTexture(GL_TEXTURE_2D, texture);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
	glEnable(GL_DEPTH_TEST);

	// the following lines set the initial values of required variables
	speed = 0.1;
	wavelength = 11.0;
	amplitude = 0.4;
	damping = 0.0;
	r = 0.0;
	number_of_wavecenters = 0;
	alpha = polygonalphavalue;
	redC = 1.0;
	greenC = 1.0;
	blueC = 1.0;
	alpha = 0.0;
	res = 100;
	range= 50;
	distortmode = false;
	wavemode = true;
	index = 0;
	max = 5;

	wavedata[0] = 0.0;
	wavedata[1] = 0.0;
	wavedata[2] = 0.0;
	wavedata[3] = 0.0;
	wavedata[4] = 0.0;
	wavedata[5] = 0.0;
	wavedata[6] = 0.0;
	wavedata[7] = 0.0;
	wavedata[8] = 0.0;
	wavedata[9] = 0.0;

	texImage1 = parentFrame->fetchImage(0,0);
}

// this function is used for the slider that sets the alpha value of the textures that
// determine if and how transparent the texture is
void Waterwaves::setpolygonalphavalue(int value)
{
	polygonalphavalue = (float)value / 10;
	update();
}

// this function sets the speed with which the radius is increased, which is equal to the 
// animationspeed
void Waterwaves::setanimationspeed(int value)
{
	speed = (value * 0.1) + 0.1;
}

// this function is used for the slider that sets the dampingfactor
void Waterwaves::setdamping(int factor)
{
	damping = factor * 0.1;
	update();
}

// this function is used for the slider that set the wavelength
void Waterwaves::setwavelength(int wl)
{
	wavelength = (11.0 - (float)wl);
	update();
}

// this function is used for the check box that enables or disables the three dimensional wavemode
void Waterwaves::setwavemode(bool enable)
{
	// set wavemode relative to enable and distortmode relative to !enable
	wavemode = enable;
	distortmode = !enable;
	waveBox->setChecked(enable);
	distortBox->setChecked(!enable);
	
	update();
}

// this function is used for the check box that enables or disables the two dimensional wavemode
void Waterwaves::setdistortmode(bool enable)
{
	// works similar to setwavemode(bool enable);
	wavemode = !enable;
	distortmode = enable;
	waveBox->setChecked(!enable);
	distortBox->setChecked(enable);
	
	update();
}

// this function is used for the slider that sets the amplitude
void Waterwaves::setamplitude(int amp)
{
	amplitude = ((float)amp/25);
	update();
}

// this function is used for the slider that set the red component of the color the textures are
// blended with
void Waterwaves::setred(int r)
{
	redC = ((float)r/10); 
	update();
}

// this function is used for the slider that set the green component of the color the textures are
// blended with
void Waterwaves::setgreen(int g)
{
	greenC = ((float)g/10); 
	update();
}

// this function is used for the slider that set the blue component of the color the textures are
// blended with
void Waterwaves::setblue(int b)
{
	blueC = ((float)b/10); 
	update();
}

//this function is used for the slider that sets the number of wavecenters
//void Waterwaves::setnumberofcenters(int num)
//{
//	number_of_wavecenters = num + 1;
//	update();
//}

// this function is used for the slider that sets the grid resolution
void Waterwaves::setresolution(int num)
{
	// sets grid resolution to res * res
	if(num == 0){res = 100; range = 50;}
	if(num == 1){res = 140; range = 70;}
	if(num == 2){res = 170; range = 85;}
	if(num == 3){res = 200; range = 100;}
    if(num == 4){res = 220; range = 110;}
	if(num == 5){res = 240; range = 120;}
	if(num == 6){res = 260; range = 130;}
	if(num == 7){res = 280; range = 140;}
	if(num == 8){res = 300; range = 150;}
	if(num == 9){res = 316; range = 158;}
}

/**
 * 
 * 
 * @see GLEffect::createControlPanel()
 **/
void Waterwaves::createControls(QWidget* parentWindow)
{
	controlPanel = new QVGroupBox("Water Waves",parentWindow,"stars");
	
	// creates a timer that is used to animate the effect and connects it to animate()
	effectTimer = new QTimer(this, "none");
	connect(effectTimer, SIGNAL(timeout()), this, SLOT(animate()) );

	// creates the slider that is used to set the grid resolution and connects it to
	// setresolution(int)
	resolution_label = new QLabel("Resolution (min. 100000 - max. 100000): ", controlPanel);
	resolution_slider = new QSlider(0,9,0,0, QSlider::Horizontal, controlPanel, "resolution_slider");
	resolution_slider->setTickmarks(QSlider::Above);
	resolution_slider->setTickInterval(1);
	connect(resolution_slider, SIGNAL(valueChanged(int)), this, SLOT(setresolution(int)) );

	// creates the slider that is used to set the animationspeed and connects it to
	// setanimationspeed(int)
	animationspeed_label = new QLabel("Animationspeed:", controlPanel);
	animationspeed_slider = new QSlider(0,9,0,0, QSlider::Horizontal, controlPanel, "animationspeed_slider");
	animationspeed_slider->setTickmarks(QSlider::Above);
	animationspeed_slider->setTickInterval(1);
	connect(animationspeed_slider, SIGNAL(valueChanged(int)), this, SLOT(setanimationspeed(int)) );

	// creates the slider that is used to set the wavelength and connects it to
	// setwavelength(int)
	wl_label = new QLabel("Wavelength:", controlPanel);
	wl_slider = new QSlider(0,10,0,0, QSlider::Horizontal, controlPanel, "wl_slider");
	wl_slider->setTickmarks(QSlider::Above);
	wl_slider->setTickInterval(1);
	connect(wl_slider, SIGNAL(valueChanged(int)), this, SLOT(setwavelength(int)) );
	
	// creates the slider that is used to set the amplitude and connects it to
	// setamplitude(int)
	amp_label = new QLabel("Amplitude:", controlPanel);
	amp_slider = new QSlider(0,10,0,10, QSlider::Horizontal, controlPanel, "amp_slider");
	amp_slider->setTickmarks(QSlider::Above);
	amp_slider->setTickInterval(1);
	connect(amp_slider, SIGNAL(valueChanged(int)), this, SLOT(setamplitude(int)) );
	
	// creates the slider that is used to set the damping factor and connects it to
	// setdamping(int)
	damp_label = new QLabel("Damping:", controlPanel);
	damp_slider = new QSlider(0,10,0,0, QSlider::Horizontal, controlPanel, "damp_slider");
	damp_slider->setTickmarks(QSlider::Above);
	damp_slider->setTickInterval(1);
	connect(damp_slider, SIGNAL(valueChanged(int)), this, SLOT(setdamping(int)) );

	// creates the slider that is used to set the red component of the blending color and connects
	// it to setred(int)
	red_label = new QLabel("RED:", controlPanel);
	red_slider = new QSlider(0,10,0,10, QSlider::Horizontal, controlPanel, "red_slider");
	red_slider->setTickmarks(QSlider::Above);
	red_slider->setTickInterval(1);
	connect(red_slider, SIGNAL(valueChanged(int)), this, SLOT(setred(int)) );

	// creates the slider that is used to set the green component of the blending color and connects
	// it to setgreen(int)
	green_label = new QLabel("GREEN:", controlPanel);
	green_slider = new QSlider(0,10,0,10, QSlider::Horizontal, controlPanel, "green_slider");
	green_slider->setTickmarks(QSlider::Above);
	green_slider->setTickInterval(1);
	connect(green_slider, SIGNAL(valueChanged(int)), this, SLOT(setgreen(int)) );

	// creates the slider that is used to set the blue component of the blending color and connects
	// it to setblue(int)
	blue_label = new QLabel("BLUE:", controlPanel);
	blue_slider = new QSlider(0,10,0,10, QSlider::Horizontal, controlPanel, "blue_slider");
	blue_slider->setTickmarks(QSlider::Above);
	blue_slider->setTickInterval(1);
	connect(blue_slider, SIGNAL(valueChanged(int)), this, SLOT(setblue(int)) );

	// creates the slider that is used to set the alpha value of the texture and connects
	// it to setpolygonalphavalue(int)
	alpha_label = new QLabel("ALPHA:", controlPanel);
	alpha_slider = new QSlider(0,10,0,0, QSlider::Horizontal, controlPanel, "blue_slider");
	alpha_slider->setTickmarks(QSlider::Above);
	alpha_slider->setTickInterval(1);
	connect(alpha_slider, SIGNAL(valueChanged(int)), this, SLOT(setpolygonalphavalue(int)) );

	// creates the checkbox that determines if the three dimensional wavemode is switched on or
	// off and connects it to setwavemode(bool)
	waveBox = new QCheckBox("Waterwaves mode",controlPanel);
	waveBox->setChecked(true);
	connect(waveBox, SIGNAL(toggled(bool)), this, SLOT(setwavemode(bool)) );

	// creates the checkbox that determines if the two dimensional wavemode is switched on or
	// off and connects it to setdistortmode(bool)
	distortBox = new QCheckBox("Distortion mode",controlPanel);
	distortBox->setChecked(false);
	connect(distortBox, SIGNAL(toggled(bool)), this, SLOT(setdistortmode(bool)) );
}

// **** Animation and Rendering Control functions ****
/**
 * 
 * Starts effect animation. 
 *
 **/
void Waterwaves::play()
{
	effectTimer->start(10,false);	
}

/**
 * 
 * Stops animation. 
 *
 **/
void Waterwaves::stop()
{
	effectTimer->stop();	
}

/**
 * 
 * Resets effect parameters. 
 *
 **/
void Waterwaves::reset()
{
	effectTimer->stop();

	texImage1 = parentFrame->fetchImage(0,0);
	
	wl_slider->setValue(0);
	amp_slider->setValue(10);
	damp_slider->setValue(0);
	red_slider->setValue(10);
	green_slider->setValue(10);
	blue_slider->setValue(10);
	alpha_slider->setValue(0);
	resolution_slider->setValue(0);
	animationspeed_slider->setValue(0);
	waveBox->setChecked(true);
	distortBox->setChecked(false);

	index = 0;
	number_of_wavecenters = 0;

	r = 0.0; // resets the radius
	update();
}

/**
 * 
 * Pauses animation. 
 *
 **/
void Waterwaves::pause()
{
	effectTimer->stop();
}

/**
 * 
 * Renders the effect. 
 *
 **/
void Waterwaves::render()
{
	alpha = polygonalphavalue; 
	glColor4f(redC, greenC, blueC, alpha); // sets color that is used to draw the polygons

	// creates texture
	if(texImage1!=0){
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, texImage1->width(), texImage1->height(), 0, GL_RGBA, GL_UNSIGNED_BYTE, texImage1->bits());
	}

	glBindTexture(GL_TEXTURE_2D, texture);

	// assigns the color values that are used during blending to the array alphav
	alphav[0] = 1.0-redC;
	alphav[1] = 1.0-greenC;
	alphav[2] = 1.0-blueC;
	alphav[3] = 1.0-alpha;
	
	glEnable(GL_TEXTURE_2D); // enables texturing
	
	glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, alphav); //sets environment color
	
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);	// clears screen and depth buffer
	
	var = 3.0/range; // is used to compute coordinates from array indices

	resetGrid(); // resets the grid

	// computes up to five distortions
	if( number_of_wavecenters > 0) 
		computesinusvalues(wavedata [0], wavedata [1]);
	if( number_of_wavecenters > 1) 
		computesinusvalues(wavedata [2], wavedata [3]);
	if( number_of_wavecenters > 2) 
		computesinusvalues(wavedata [4], wavedata [5]);
	if( number_of_wavecenters > 3) 
		computesinusvalues(wavedata [6], wavedata [7]);
	if( number_of_wavecenters > 4) 
		computesinusvalues(wavedata [8], wavedata [9]);
	
	// renders the effect if the two dimensional wavemode is enabled
	if(wavemode == false && distortmode == true){
	
		// sets texture modes
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

		draw2d(); // draws the rendered effect
	}
	
	// renders the effect if the three dimensional wavemode is enabled
	if(wavemode == true && distortmode == false){
	
		// sets texture modes
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

		draw3d(); // draws the rendered effect
	}

	glDisable(GL_TEXTURE_2D); // disables texturing

}

// animates the effect

void Waterwaves::animate()
{
	r = r + speed; // increases the radius
	if(r > 6.2) // resets the radius to 0 if it gets greater than 6.2 = 360 degree
		 r = 0.0;
	texImage1 = parentFrame->fetchImage(0,0);
	update();
}

// resets the grid
void Waterwaves::resetGrid()
{
	for(j = -158; j < 159; j++)
	{
		for(i = -158; i < 159; i++)
		{
			data [i+158] [j+158] = 0; // the array is filled with zeros
		}
	}
}

// computes sinus values for every grid point
void Waterwaves::computesinusvalues(GLfloat mx, GLfloat my)
{
	for(j = -range; j < (range+1); j++)
	{
		for(i = -range; i < (range+1); i++)
		{
			distance( (i*var), (j*var), mx, my); // computes the points distance from the wavecenter
			
			if(dist <=(8.5*(1.5-damping))) // sets the damping factor
				dampingfactor = 1-((dist/4.25)*damping);
			else
				if(damping == 0.0)
					dampingfactor = 1.0;
				else
					dampingfactor = 0.0;
			
				// computes the sinus function values and saves them in data
			data [i+range] [j+range] = data [i+range] [j+range] + (dampingfactor * amplitude * sin((wavelength * dist - r)));
		}
	}
}

// draws the distortion if distortion in x- and y-direction is desired
void Waterwaves::draw2d()
{
	for(j = -range; j < range; j++)
	{
		glBegin(GL_TRIANGLE_STRIP);		
		for(i = -range; i < (range+1); i++)
		{
						
			// computes the texture coordinates and draws the coresponding textures  
			texcoords3f(i*var + data [i+range] [j+range], j*var + data [i+range] [j+range], 0.0);
			glTexCoord2f(tx, ty);			
			glVertex3f( i*var, j*var, 0.0 );
			
			// computes the texture coordinates and draws the coresponding textures
			texcoords3f(i*var + data [i+range] [j+range+1], (j+1)*var + data [i+range] [j+range+1], 0.0);
			glTexCoord2f(tx, ty);
			glVertex3f( i*var, (j+1)*var, 0.0 );
		}
		glEnd();
	}
}

// draws the distortion if distortion in z-direction is desired
void Waterwaves::draw3d()
{
	for(j = -range; j < range; j++)
	{
		glBegin(GL_TRIANGLE_STRIP);		
		for(i = -range; i < (range+1); i++)
		{
						
			// computes the texture coordinates and draws the coresponding textures
			texcoords3f(i*var, j*var, data [i+range] [j+range]);
			glTexCoord2f(tx, ty);			
			glVertex3f( i*var, j*var, data [i+range] [j+range]);
			
			// computes the texture coordinates and draws the coresponding textures
			texcoords3f(i*var, (j+1)*var, data [i+range] [j+range+1]);
			glTexCoord2f(tx, ty);
			glVertex3f( i*var, (j+1)*var, data [i+range] [j+range+1]);
		}
		glEnd();
	}
}

void Waterwaves::mousePressEvent(QMouseEvent* mEv)
{
	mouseX = mEv->x(); 
	mouseY = mEv->y();
	mouseButtonPressed = mEv->button();
	
	// transform to 3d mouse coordinates
	get3DMouseCoords(currentDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);

	// optionally provide normalized 2d mouse coordinates
	// normalize coordinates to [-1...1]
	normMouseX = mouseX / (GLdouble(glView->width())/2);
	normMouseX -= 1.0;

	normMouseY = mouseY / (GLdouble(glView->height())/2);
	normMouseY -= 1.0;

	// stores the mouse click coordinates in wavedata[]
	if(index < max)
	{
	wavedata[2*index] = 10 * mouse3D_X;
	wavedata[2*index+1] = -(10 * mouse3D_Y);
	index = index +1;
	number_of_wavecenters = index;
	}
	update();
}
