
/*****************************************************************************\
 *                              ShadowCaster
\*****************************************************************************/

/** 
 * @defgroup ShadowCasterClasses Shadow Caster effect classes
 *
 */

/*! @file ShadowCaster.h
 *
 *  @brief
 *	Shadowcast effect class declarations
 *   
 *  @author Moritz Voss
 *     
 */

/*
    Copyright  2001 Moritz Voss All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


/*! @class ShadowCaster
 *
 *  @ingroup ShadowCasterClasses
 *
 *  @brief Casts all sorts of wacky shadows.
 *
 *  OpenGL 1.2 based Shadow Simulator (-unfinished-) by <b>Moritz Voss</b> 2k+2
 *  This program demonstrates, though not without errors and artifacts, how one
 *  can utilize the SGIX_shadow extension available on consumer grade hardware
 *  and professional boards alike. It also provides a brief introduction to
 *  fractal terrain synthesis and OpenGL lighting models. The implementation of
 *  this software is flawed. Apart from some depth-buffer artifact problems I
 *  was unable to resolve, the lighting calculations result in some blending
 *  colours which I didn't intend to result from them. Apart from that, it's a
 *  fun little piece of not-too-shabby code =^-.-^="
 *  
 *  
 **/

#ifndef ShadowCaster_H
#define ShadowCaster_H

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

//forward references
class QTimer;
class QComboBox;
class QCheckBox;
class QSlider;
class QLabel;

#include <stdlib.h>

// local headers
#include "../main/GLEffect.h"
#include "ShadowUI.h"


const GLuint MapXRes = 128;
const GLuint MapYRes = 128;

/** \internal
  *
  */

struct tVertex
{
	GLfloat x, y, z; //spatial
	GLfloat nx, ny, nz; //vertex normal
	GLfloat tx, ty; //2D texture coordinates
};

class ShadowCaster: public GLEffect
{
	Q_OBJECT

	public:

	ShadowCaster(GLFramework* pr);
	virtual ~ShadowCaster();

	//! Mouse move events handler.
	virtual void mouseMoveEvent(QMouseEvent*);
	//! Mouse press events handler.
	virtual void mousePressEvent(QMouseEvent*);
	//! Mouse release events handler
	virtual void mouseReleaseEvent(QMouseEvent*);

	//! Mouse wheel events handler
	virtual void wheelEvent(QWheelEvent*);

	//! Keyboard events handler.
	virtual void keyPressEvent(QKeyEvent*);
	
	//re-implemented control functions
	virtual void stop();
	virtual void play();
	virtual void reset();
	virtual void pause();

	protected:
	virtual void initialize();
	virtual void render();


	void initExtensions(void);
	void createControls(QWidget* parentWindow);
	void makeTexture();

	//heightfield stuff

	void genNormals();

	void performFIR();
	void prepareZero();

	void performFF();
	void prepareFF();

	void performMPD(int minX, int minY, int maxX, int maxY, float dHeight, float r, int recursions);
	void prepareMPD();


	protected slots:

	private:

	// control panel
    shadowControls* controls;
	// helper functions
	void initGrid();
	void drawGrid();
	void drawVisualAids();

	void initLightStates();
	void generateShadowMap();
	void generateTexCoords(bool enable);

	void crossV3(GLfloat v[3], GLfloat w[3], GLfloat out[3]);
	void normalizeV3(GLfloat v[3]);

	//member vars

	// used to animate effect
	QTimer* effectTimer;

	
	// current texture & shadow map
	GLuint texture;
	GLuint shadow;
	GLuint luminance;

	// current random seed
	unsigned long seed;

	//shadow map refresh flag
	bool shadowsDirty;

	// light colours
	GLfloat lightCol[4];
	GLfloat lightPos[4];
	GLfloat lightDst[4];

	//light matrices
	GLfloat lightProjection[16];
	GLfloat lightModelView[16];

	//temporary viewport
	GLint tempViewPort[4];

	//light source sphere
	GLUquadric* quadric;

	//auxiliary depths (for Lighting)
	GLfloat startDepthCoord, endDepthCoord;

	// vertex /color /tex coord arrays
	tVertex grid[MapXRes][MapYRes];

	//shadow map data
	GLfloat shadowMap[256*256];
	
	private slots:

	void animate();	
	void updateLighting();
	void updateLightColor();
	void updateHeightField();
	void reRandomize() {srand(seed=rand());updateHeightField();}; //simple little inline bitch
};

#endif


