
/*****************************************************************************\
 *                              ShadowCaster
\*****************************************************************************/

/*! @file ShadowCaster.cpp
 *
 *  @brief
 *	Shadowcaster effect class implementation
 *    
 *  @author Moritz Voss
 *  
 */


// system library headers
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <time.h>

#ifndef _WIN32
#	include <unistd.h>
#	include <sys/times.h>
#endif

// Qt headers
#include <qimage.h>
#include <qtimer.h>

#include <qvgroupbox.h>
#include <qlabel.h>
#include <qslider.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qcombobox.h>

//local headers
#include "ShadowCaster.h"
#include "ShadowUI.h"

#ifdef _WIN32
#	include "glstuff.h"
#endif

const double PI = 3.14159;

ShadowCaster::ShadowCaster(GLFramework* pr)
	:	GLEffect(pr)
{

	//provide some basic info about the effect
	effectInfo->effectName = "Shadow Caster";
	effectInfo->version = "1.0";
	effectInfo->versionMajor = 1;
	effectInfo->shortDescription = "Casts all sorts of wacky shadows.";
	effectInfo->author  ="Moritz Thygrrr Voss";

	effectInfo->requiredImageLists = 1;
	effectInfo->needNewTextures = true;
	effectInfo->needStreaming = false;
	effectInfo->needRawTextures = false;

	effectInfo->fileDialogNames[0]= "Load Textures";
	
	createControlPanel(true);

	texImage1 = 0;
	texImage2 = 0;
	texImage3 = 0;

	hideControls();	
}

ShadowCaster::~ShadowCaster()
{
	gluDeleteQuadric(quadric);
}


void ShadowCaster::createControls(QWidget* parentWindow)
{

	controlPanel = new QVGroupBox("Shadow Controls",parentWindow,"provides control over the effect");
	controls = new shadowControls(controlPanel);
		

	//timer to enable animation in this effect
	effectTimer = new QTimer(this, "none");
	

	// bind interface functions
	connect(controls->colorRed, SIGNAL(sliderMoved(int)), 
		this, SLOT(updateLightColor()) );

	connect(controls->colorGreen, SIGNAL(sliderMoved(int)), 
		this, SLOT(updateLightColor()) );

	connect(controls->colorBlue, SIGNAL(sliderMoved(int)), 
		this, SLOT(updateLightColor()) );

	connect(controls->colorAlpha, SIGNAL(sliderMoved(int)), 
		this, SLOT(updateLightColor()) );

	connect(controls->randomButton, SIGNAL(clicked()),
		this, SLOT(reRandomize()) );

	connect(controls->amplitudeSlider, SIGNAL(sliderMoved(int)),
		this, SLOT(updateHeightField()) );

	connect(controls->frequencySlider, SIGNAL(sliderMoved(int)),
		this, SLOT(updateHeightField()) );

	connect(controls->falloffSlider, SIGNAL(sliderMoved(int)),
		this, SLOT(updateHeightField()) );

	connect(controls->filterSlider, SIGNAL(sliderMoved(int)),
		this, SLOT(updateHeightField()) );

	connect(effectTimer, SIGNAL(timeout()), 
		this, SLOT(animate()) );

	connect(controls->terrainMethod, SIGNAL(activated(int)), 
		this, SLOT(updateHeightField()) );

	connect(controls->aidDepth, SIGNAL(toggled(bool)), 
		this, SLOT(updateLightColor()) );

	connect(controls->lightSource, SIGNAL(toggled(bool)), 
		this, SLOT(updateLightColor()) );

	connect(controls->lightVectors, SIGNAL(toggled(bool)), 
		this, SLOT(updateLightColor()) );

	connect(controls->showNormals, SIGNAL(toggled(bool)), 
		this, SLOT(updateLightColor()) );

	connect(controls->lightView, SIGNAL(toggled(bool)), 
		this, SLOT(updateLightColor()) );

	connect(controls->lightMethod, SIGNAL(activated(int)), 
		this, SLOT(updateLightColor()) );

	connect(controls->glossMethod, SIGNAL(activated(int)), 
		this, SLOT(updateLightColor()) );

	connect(controls->lightDirection, SIGNAL(activated(int)), 
		this, SLOT(updateLightColor()) );

	//nur mal ne ganz kleine, dumme Frage... wer braucht denn jetzt noch MFC oder VCL?
}

// events
void ShadowCaster::keyPressEvent(QKeyEvent* kEv)
{
//		parentFrame->addToStatus("KP() called.");
};

void ShadowCaster::mouseMoveEvent(QMouseEvent* mEv)
{
	mouseX = mEv->x(); 
	mouseY = mEv->y();
	mouseButtonPressed = mEv->state();

	// transform to 3d mouse coordinates
	if (mouseButtonPressed & LeftButton) get3DMouseCoords(currentDepthCoord = startDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);
	if (mouseButtonPressed & RightButton) get3DMouseCoords(currentDepthCoord = endDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);

	// optionally provide normalized 2d mouse coordinates

	// normalize coordinates to [-1...1]
	normMouseX = mouseX / (GLdouble(glView->width())/2);
	normMouseX -= 1.0;

	normMouseY = mouseY / (GLdouble(glView->height())/2);
	normMouseY -= 1.0;

	updateLighting();
	update();
};

void ShadowCaster::mousePressEvent(QMouseEvent* mEv)
{
	mouseX = mEv->x(); 
	mouseY = mEv->y();
	mouseButtonPressed = mEv->state();

	// transform to 3d mouse coordinates
	if (mouseButtonPressed & LeftButton) get3DMouseCoords(currentDepthCoord = startDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);
	if (mouseButtonPressed & RightButton) get3DMouseCoords(currentDepthCoord = endDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);

	// optionally provide normalized 2d mouse coordinates

	// normalize coordinates to [-1...1]
	normMouseX = mouseX / (GLdouble(glView->width())/2);
	normMouseX -= 1.0;

	normMouseY = mouseY / (GLdouble(glView->height())/2);
	normMouseY -= 1.0;

	updateLighting();
	update();
};

void ShadowCaster::mouseReleaseEvent(QMouseEvent* mEv)
{
	mouseX = mEv->x(); 
	mouseY = mEv->y();
	mouseButtonPressed = mEv->state();

	// transform to 3d mouse coordinates
	if (mouseButtonPressed & LeftButton) get3DMouseCoords(currentDepthCoord = startDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);
	if (mouseButtonPressed & RightButton) get3DMouseCoords(currentDepthCoord = endDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);

	// optionally provide normalized 2d mouse coordinates

	// normalize coordinates to [-1...1]
	normMouseX = mouseX / (GLdouble(glView->width())/2);
	normMouseX -= 1.0;

	normMouseY = mouseY / (GLdouble(glView->height())/2);
	normMouseY -= 1.0;

	updateLighting();
	update();
};

void ShadowCaster::wheelEvent(QWheelEvent* wEv)
{
	//mouseX = wEv->x(); 
	//mouseY = wEv->y();

//	mouseButtonPressed = mEv->state();  <---- taken out because wheel events have no proper state!

	if (mouseButtonPressed & LeftButton)
	{
		currentDepthCoord = startDepthCoord -= (GLfloat) wEv->delta() / 1200000.0f; //I hate big numbers
	}

	if (mouseButtonPressed & RightButton)
	{
		currentDepthCoord = endDepthCoord -= (GLfloat) wEv->delta() / 1200000.0f;
	}

	if (mouseButtonPressed & LeftButton) get3DMouseCoords(currentDepthCoord = startDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);
	if (mouseButtonPressed & RightButton) get3DMouseCoords(currentDepthCoord = endDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);

	// transform to 3d mouse coordinates

	// optionally provide normalized 2d mouse coordinates

	// normalize coordinates to [-1...1]
	normMouseX = mouseX / (GLdouble(glView->width())/2);
	normMouseX -= 1.0;

	normMouseY = mouseY / (GLdouble(glView->height())/2);
	normMouseY -= 1.0;

	updateLighting();
	update();
};



// slots

void ShadowCaster::updateHeightField(void) //when user moves the sliders...
{
	if (controls->terrainMethod->currentItem() == 0) prepareZero();
	if (controls->terrainMethod->currentItem() == 1) performFF();
	if (controls->terrainMethod->currentItem() == 2) prepareMPD();

	update();
};

void ShadowCaster::updateLighting(void) //updates light positioning vectors
{
	char string[256];

	if (mouseButtonPressed & LeftButton)
	{
		lightPos[0]= mouse3D_X;
		lightPos[1]= -mouse3D_Y;
		lightPos[2]= -mouse3D_Z;

		sprintf( string, "%10g", mouse3D_X );
		controls->positionX->setText(string);
		sprintf( string, "%10g", mouse3D_Y );
		controls->positionY->setText(string);
		sprintf( string, "%10g", mouse3D_Z );
		controls->positionZ->setText(string);
	};

	if (mouseButtonPressed & RightButton)
	{
		lightDst[0]= mouse3D_X;
		lightDst[1]= -mouse3D_Y;
		lightDst[2]= -mouse3D_Z;

		sprintf( string, "%10g", mouse3D_X );
		controls->directionX->setText(string);
		sprintf( string, "%10g", mouse3D_Y );
		controls->directionY->setText(string);
		sprintf( string, "%10g", mouse3D_Z );
		controls->directionZ->setText(string);
	};
	shadowsDirty = true;

};

//main update function for all light changes
void ShadowCaster::updateLightColor(void) //updates light color vector
{
	lightCol[0]	= GLfloat(255-controls->colorRed->value())/255.0f;
	lightCol[1]	= GLfloat(255-controls->colorGreen->value())/255.0f;
	lightCol[2]	= GLfloat(255-controls->colorBlue->value())/255.0f;
	lightCol[3]	= GLfloat(255-controls->colorAlpha->value())/255.0f;

///	qDebug("%f - %f - %f - %f", lightCol[0],  lightCol[1],  lightCol[2],  lightCol[3]);

	if(controls->lightMethod->currentItem()) controls->lightView->setEnabled(true); else {controls->lightView->setEnabled(false); controls->lightView->setChecked(false);}

	shadowsDirty = true;
	update();
};

// control functions

void ShadowCaster::play()
{
	// start animation
	effectTimer->start(25,false);

}

void ShadowCaster::stop()
{
	// stop and go back to the first frame
	effectTimer->stop();
	texImage1 = parentFrame->fetchImage(0,0);
	update();
}

void ShadowCaster::reset()
{
	// stop and reset effect
	effectTimer->stop();
	texImage1 = parentFrame->fetchImage(0,0);
	currentDepthCoord = startDepthCoord = 1.0f-4.0f/29.0f;
	endDepthCoord = 1.0f-5.5f/29.0f;

	fovAngle  = 60;

	viewDistance  = 5;
	nearClipPlane = 1;
	farClipPlane  = 30;

	mouseX = mouseY = 0;
	lightCol[0] = lightCol[1] = lightCol[2] = lightCol[3] = 1;
	lightPos[0] = lightPos[1] = 0;
	lightPos[2] = -2;
	lightPos[3] = 1;
	lightPos[2] = 1;
	lightDst[0] = lightDst[1] = lightDst[2] = 1;
	lightDst[3] = 1;


	glClearColor(0.0,0.0,0.0,1);
	glShadeModel(GL_SMOOTH);

	controls->positionX->setText("0");
	controls->positionY->setText("0");
	controls->positionZ->setText("0");

	controls->directionX->setText("-1");
	controls->directionY->setText("-1");
	controls->directionZ->setText("-1");

	glLoadIdentity();

	// set texturing parameters
	glEnable(GL_DEPTH_TEST);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	
	glGenTextures(1, &texture);  //obtain texture object
	glGenTextures(1, &shadow);   //obtain shadow map object
	glGenTextures(1, &luminance);//obtain visualization map object

	glActiveTextureARB(GL_TEXTURE0_ARB); //multi texturing

	glBindTexture(GL_TEXTURE_2D, texture); //set up normal texture
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);	
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);



	glActiveTextureARB(GL_TEXTURE1_ARB); //multi texturing

	glBindTexture(GL_TEXTURE_2D, shadow); //set up shadow texture
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);	
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

	glActiveTextureARB(GL_TEXTURE1_ARB); //multi texturing

	glBindTexture(GL_TEXTURE_2D, luminance); //set up shadow texture
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);	
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);


	glLightModelf(GL_LIGHT_MODEL_TWO_SIDE, 0.0); //make sure light model is NOT two-sided

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST); 
	
	texImage1 = parentFrame->fetchNextImage(0);

	makeTexture();

	initGrid();
	prepareZero();

	controls->lightMethod->setCurrentItem(0);
	controls->glossMethod->setCurrentItem(0);
	controls->lightDirection->setCurrentItem(0);

	controls->lightSource->setChecked(true);
	controls->lightVectors->setChecked(true);
	controls->showNormals->setChecked(false);
	controls->lightView->setChecked(false);
	controls->lightView->setEnabled(false);
	controls->aidDepth->setChecked(true);

	controls->colorRed->setValue(0);
	controls->colorGreen->setValue(0);
	controls->colorBlue->setValue(0);
	controls->colorAlpha->setValue(0);

	controls->amplitudeSlider->setValue(100);
	controls->frequencySlider->setValue(100);
	controls->falloffSlider->setValue(100);
	controls->filterSlider->setValue(100);

	updateLighting();
	updateLightColor();
}

void ShadowCaster::pause()
{
  //pause animation
  effectTimer->stop();
}


void ShadowCaster::animate()
{
  // go to next image 
  texImage1 = parentFrame->fetchNextImage(0); 
  makeTexture();
  update();
}


//grid initialization, also adapts to current aspect ratios!

void ShadowCaster::initGrid()
{
	GLfloat x_aspect = (GLfloat)texImage1->width() / (GLfloat)texImage1->height();
	GLfloat y_aspect = (GLfloat)texImage1->height() / (GLfloat)texImage1->width();
	
	if (x_aspect == 0) x_aspect = 1; //fail safe, don't want to know what kind of FUCK
	if (y_aspect == 0) y_aspect = 1; //the users spam into this program :->

	for (int i=0; i < MapXRes; i++)
		for (int j=0; j < MapYRes; j++)
		{
			grid[i][j].x = y_aspect > x_aspect ? (((GLfloat)i - (GLfloat)MapXRes/2.0f))/(GLfloat)MapXRes : (((GLfloat)i - (GLfloat)MapXRes/2.0f) * x_aspect)/(GLfloat)MapXRes;
			grid[i][j].y = y_aspect <= x_aspect  ? (((GLfloat)j - (GLfloat)MapYRes/2.0f))/(GLfloat)MapYRes : (((GLfloat)j - (GLfloat)MapYRes/2.0f) * y_aspect)/(GLfloat)MapYRes;

			grid[i][j].x *= 2.0;
			grid[i][j].y *= 2.0;

	//		grid[i][j].z = 0.0; //<--- z is business of SLOT updateHeightField();

			grid[i][j].tx = (GLfloat)i / ((GLfloat)MapXRes-1.0f);
			grid[i][j].ty = (GLfloat)j / ((GLfloat)MapYRes-1.0f);
		}
		makeTexture();
}


// heightfield ops

void ShadowCaster::genNormals()
{
		float vec1[3];
		vec1[0] = 0.075f;
		vec1[1] = 0;
		float vec2[3];
		vec2[0] = 0;
		vec2[1] = 0.075f;
		float norm1[3];
		float norm2[3];
		float norm3[3];
		float norm4[3];
		
		
		// Walk through and create normals averaging the surrounding normals together to give smooth appearance
		for(unsigned int curRow=0;curRow<MapYRes;++curRow)				//Row
		{	
			for(unsigned int curColumn=0;curColumn<MapXRes;++curColumn)	//Column
			{	
				
				if(curRow != MapYRes-1 && curColumn != MapXRes-1)	// If not in Bottem Right Corner
				{
					vec1[2] = grid[curRow][curColumn+1].z-grid[curRow][curColumn].z;
					vec2[2] = grid[curRow+1][curColumn].z-grid[curRow][curColumn].z;
					
					crossV3(vec1,vec2,norm1);
					norm1[0] *= -1.0;	// Corrects for invert sign on norms
				}
				else
				{
					norm1[0] = 0;
					norm1[1] = 0;
					norm1[2] = 1;
				}
				
				if(curRow != MapYRes-1 && curColumn != 0)	// If not in Bottem Left Corner
				{
					vec1[2] = grid[curRow][curColumn-1].z-grid[curRow][curColumn].z;
					vec2[2] = grid[curRow+1][curColumn].z-grid[curRow][curColumn].z;
					
					crossV3(vec1,vec2,norm2);
				}
				else
				{
					norm2[0] = 0;
					norm2[1] = 0;
					norm2[2] = 1;
				}
				if(curRow != 0 && curColumn != 0)	// If not in Upper Left Corner
				{
					vec1[2] = grid[curRow][curColumn-1].z-grid[curRow][curColumn].z;
					vec2[2] = grid[curRow-1][curColumn].z-grid[curRow][curColumn].z;
					
					crossV3(vec1,vec2,norm3);
					norm3[1] *= -1.0;	// Corrects for invert sign on norms
				}
				else
				{
					norm3[0] = 0;
					norm3[1] = 0;
					norm3[2] = 1;
				}
				
				if(curRow != 0 && curColumn != MapXRes-1)	// If not in Upper Right Corner
				{
					vec1[2] = grid[curRow][curColumn+1].z-grid[curRow][curColumn].z;
					vec2[2] = grid[curRow-1][curColumn].z-grid[curRow][curColumn].z;
					crossV3(vec1,vec2,norm4);
					norm4[0] *= -1.0;	// Corrects for invert sign on norms
					norm4[1] *= -1.0;	// Corrects for invert sign on norms
				}
				else
				{
					norm4[0] = 0;
					norm4[1] = 0;
					norm4[2] = 1;
				}
				
				// Now we add the 4 normals around the point we want and normalize
				norm1[0] += norm2[0] + norm3[0] + norm4[0];
				norm1[1] += norm2[1] + norm3[1] + norm4[1];
				norm1[2] += norm2[2] + norm3[2] + norm4[2];
				normalizeV3(norm1);
				
				grid[curRow][curColumn].nx = -norm1[1];
				grid[curRow][curColumn].ny = -norm1[0];
				grid[curRow][curColumn].nz = norm1[2];
			}
		}

	};

void ShadowCaster::performFIR(void)
{
		float k = ((float) (200 - controls->filterSlider->value())) / 200.0f;
		int r;
		int c;
		
		// Covers left to right
		for(r=0; r<MapYRes; ++r)
		{
			for(c=1; c<MapXRes; ++c)
			{
				grid[r][c].z = k*grid[r][(c-1)].z+
					(1-k)*grid[r][c].z;
			}
		}
		
		// Covers right to left
		for(r=0; r<MapYRes; ++r)
		{
			for(c=MapXRes-2; c !=0 ; --c)
			{
				grid[r][c].z = k*grid[r][(c+1)].z+
					(1-k)*grid[r][c].z;
			}
		}
		
		// Covers top to bottem
		for(c=0; c<MapXRes; ++c)
		{
			for(r=1; r<MapYRes; ++r)
			{
				grid[r][c].z = k*grid[(r-1)][c].z+
					(1-k)*grid[r][c].z;
			}
		}
		
		// Covers bottem to top
		for(c=0; c<MapXRes; ++c)
		{
			for(r=MapYRes-2; r!=0; --r)
			{
				grid[r][c].z = k*grid[(r+1)][c].z+
					(1-k)*grid[r][c].z;
			}
		}
};

void ShadowCaster::normalizeV3(GLfloat v[3])
	{
		GLfloat len = (float)sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2]); 
		if(len > 0){
			v[0] /= len;
			v[1] /= len;
			v[2] /= len;
		}
	}


void ShadowCaster::crossV3(GLfloat v[3], GLfloat w[3], GLfloat out[3])
	{
		out[0] = v[1]*w[2]-w[1]*v[2];
		out[1] = w[0]*v[2]-v[0]*w[2];
		out[2] = v[0]*w[1]-w[0]*v[1];
	}


void ShadowCaster::prepareZero(void) //zero terrain, doesn't need perform (trivial)
{
	for (int i=0; i < MapXRes; i++)
		for (int j=0; j < MapYRes; j++)
		{
			grid[i][j].z = 0.0;
			grid[i][j].nx = 0;
			grid[i][j].ny = 0;
			grid[i][j].nz = 1;
		}
};


void ShadowCaster::performFF(void) //fault formation
{
	srand(seed);
	static int ff_old_seed;
	static int ff_old_fir;
	static int ff_old_magn;
	static int ff_old_iter;
	
	if (controls->frequencySlider->value() != ff_old_iter || controls->amplitudeSlider->value() != ff_old_magn || controls->filterSlider->value() != ff_old_fir)
	{
		float dHeighti;// The current depth height
		
		srand(seed);
		
		prepareZero();
		
		for( int i=0; i < (200-controls->frequencySlider->value())/2; ++i)	// This is used to generate the dHeighti variables
		{
			// Create 2 random points that are located inside the height and width of the hrightmap
			int point1[2] = {(int)((float)rand()/RAND_MAX*MapXRes), (int)((float)rand()/RAND_MAX*MapYRes)};
			int point2[2] = {(int)((float)rand()/RAND_MAX*MapXRes), (int)((float)rand()/RAND_MAX*MapYRes)};
			float vec1[3] = {point2[0]-(float)point1[0], point2[1]-(float)point1[1], 0};
			
			dHeighti = (((float) (200-controls->amplitudeSlider->value()) / (float)2000));	// These are the current depth heights
			if (dHeighti > 0) 
			{ 
				
				// Now go through height map and adjust all the heights
				for( int j=0;j < MapYRes; ++j)
				{
					for(  int k=0; k < MapXRes; ++k)
					{
						float vec2[3] = {k-(float)point1[0], j-(float)point1[1], 0};					
						float vec3[3];
						crossV3(vec1, vec2, vec3);					
						
						if(vec3[2] < 0)
							grid[j][k].z += dHeighti; 	// Add to height if on one side of line but not other		
						else
							grid[j][k].z -= dHeighti; 	// Sub from height if on one side of line but not other		
					}
				}
			};
		}
		performFIR();
		genNormals();
	};
};


void ShadowCaster::performMPD(int minX, int minY, int maxX, int maxY, float dHeight, float r, int recursions)
{
	int midX = (maxX-minX)/2+minX;		// X midpoint
	int midY = (maxY-minY)/2+minY;		// Y midpoint
	float randNum = (float)rand()/RAND_MAX*2.0f -1.0f;	// A random number between -1 and 1
	float centerHeight = randNum * dHeight;
	float currentHeight;
	float distFromCenter;
	float distFromEdge;
	
	// Go through the height map and adjust all the heights
	for( int i = minY; i< maxY; ++i )	// y
	{
		for( int j = minX; j<maxX; ++j)	// x
		{
			distFromCenter = (float)sqrt((i-midY)*(i-midY) + (j-midX)*(j-midX));	
			if(i<=midY && j<=midX)			// Upper Left*
			{
				if( (i-minY)<(j-minX) )
				{
					distFromEdge = (float)sqrt( (i-minY)*(i-minY) + ((midX-j)*(i-minY)/(midY-i+1))*((midX-j)*(i-minY)/(midY-i+1)) );
				}
				else
				{
					distFromEdge = (float)sqrt( (j-minX)*(j-minX) + ((midY-i)*(j-minX)/(midX-j+1))*((midY-i)*(j-minX)/(midX-j+1)) );
				}
				
			}
			else if(i<=midY && j>=midX)		// Upper Right
			{
				if( (i-minY)<(maxX-j) )
				{
					distFromEdge = (float)sqrt( (i-minY)*(i-minY) + ((j-midX)*(i-minY)/(midY-i+1))*((j-midX)*(i-minY)/(midY-i+1)) );
				}
				else
				{
					distFromEdge = (float)sqrt( (maxX-j)*(maxX-j) + ((midY-i)*(maxX-j)/(j-midX+1))*((midY-i)*(maxX-j)/(j-midX+1)) );
				}
			}
			else if(i>=midY && j>=midX)		// Lower Right*
			{
				if( (maxY-i)<(maxX-j) )
				{
					distFromEdge = (float)sqrt( (maxY-i)*(maxY-i) + ((j-midX)*(maxY-i)/(i-midY+1))*((j-midX)*(maxY-i)/(i-midY+1)) );
				}
				else
				{
					distFromEdge = (float)sqrt( (maxX-j)*(maxX-j) + ((i-midY)*(maxX-j)/(j-midX+1))*((i-midY)*(maxX-j)/(j-midX+1)) );
				}
			}
			else							// Lower Left
			{
				if( (maxY-i)<(j-minX) )
				{
					distFromEdge = (float)sqrt( (maxY-i)*(maxY-i) + ((midX-j)*(maxY-i)/(i-midY+1))*((midX-j)*(maxY-i)/(i-midY+1)) );
				}
				else
				{
					distFromEdge = (float)sqrt( (j-minX)*(j-minX) + ((i-midY)*(j-minX)/(midX-j+1))*((i-midY)*(j-minX)/(midX-j+1)) );
				}
			}
			currentHeight = distFromEdge/(distFromCenter + distFromEdge)*centerHeight;
			
			grid[i][j].z += currentHeight;
		}
	}
	
	--recursions;
	dHeight*= (float)pow(2, -r);
	
	// keep going deeper into recursion if the number of resursive iterations has not been met
	if(recursions>0)
	{
		performMPD(minX, minY, midX, midY, dHeight, r, recursions);
		performMPD(minX, midY, midX, maxY, dHeight, r, recursions);
		performMPD(midX, minY, maxX, midY, dHeight, r, recursions);
		performMPD(midX, midY, maxX, maxY, dHeight, r, recursions);
	}
}


void ShadowCaster::prepareMPD(void)
{

	static int mpd_old_seed;
	static int mpd_old_fir;
	static int mpd_old_magn;
	static int mpd_old_recs;

	if (controls->frequencySlider->value() != mpd_old_recs || controls->amplitudeSlider->value() != mpd_old_magn || controls->filterSlider->value() != mpd_old_fir)
	{

		srand(seed);

		prepareZero();

		float dHeight = (float) (200-controls->amplitudeSlider->value()) / 200.0f;
		int recursions =  (200-controls->frequencySlider->value())/30;
		float r = (float) (200-controls->falloffSlider->value())/200.0f;

		int minX;
		int minY;
		int maxX;
		int maxY;

		minX = 0;
		minY = 0;
		maxX = MapXRes-1;
		maxY = MapYRes-1;

		performMPD(minX, minY, maxX, maxY, dHeight, r, recursions);

		performFIR();
		genNormals();
	}

};

// initialization functions

void ShadowCaster::makeTexture(void)
{
	if(texImage1!=0)
	{
		glActiveTextureARB(GL_TEXTURE0_ARB);	
		glBindTexture(GL_TEXTURE_2D, texture);
		glTexImage2D(GL_TEXTURE_2D,0,4,texImage1->width(), texImage1->height(),
				0, GL_RGBA, GL_UNSIGNED_BYTE, texImage1->bits());
	}
	
};


void ShadowCaster::initialize()
{
	//parentFrame->glView()->getOpenGLCaps();

	quadric = gluNewQuadric();


	initExtensions();

	// initialize effect parameters and controls
	currentDepthCoord = startDepthCoord = 1.0f-4.0f/29.0f;
	endDepthCoord = 1.0f-5.5f/29.0f;

	fovAngle  = 60;

	mouseX = mouseY = 0;

	viewDistance  = 5;
	nearClipPlane = 1;
	farClipPlane  = 30;

	lightCol[0] = lightCol[1] = lightCol[2] = lightCol[3] = 1;
	lightPos[0] = lightPos[1] = 0;
	lightPos[2] = -2;
	lightPos[3] = 1;
	lightPos[2] = 1;
	lightDst[0] = lightDst[1] = lightDst[2] = 1;
	lightDst[3] = 1;

	glClearColor(0.0,0.0,0.0,1);
	glShadeModel(GL_SMOOTH);

	controls->positionX->setText("0");
	controls->positionY->setText("0");
	controls->positionZ->setText("0");

	controls->directionX->setText("-1");
	controls->directionY->setText("-1");
	controls->directionZ->setText("-1");

	glLoadIdentity();

	// set texturing parameters
	glEnable(GL_DEPTH_TEST);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	
	glGenTextures(1, &texture);  //obtain texture object
	glGenTextures(1, &shadow);   //obtain shadow map object
	glGenTextures(1, &luminance);//obtain visualization map object

	glActiveTextureARB(GL_TEXTURE0_ARB); //multi texturing

	glBindTexture(GL_TEXTURE_2D, texture); //set up normal texture
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);	
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);



	glActiveTextureARB(GL_TEXTURE1_ARB); //multi texturing

	glBindTexture(GL_TEXTURE_2D, shadow); //set up shadow texture
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);	
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	glActiveTextureARB(GL_TEXTURE1_ARB); //multi texturing

	glBindTexture(GL_TEXTURE_2D, luminance); //set up shadow texture
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);	
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);


	glLightModelf(GL_LIGHT_MODEL_TWO_SIDE, 0.0); //make sure light model is NOT two-sided

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST); 
	
	texImage1 = parentFrame->fetchNextImage(0);

	makeTexture();

	initGrid();
	prepareZero();


	updateLighting();

}	

void ShadowCaster::drawGrid() //single rendering pass
{

	glActiveTextureARB(GL_TEXTURE0_ARB); //multi texturing
	glBindTexture(GL_TEXTURE_2D, texture); //set up normal texture
	glEnable(GL_TEXTURE_2D);

	glBegin(GL_TRIANGLES);

	glColor4f(1,1,1,1);

	for (int i = 0; i < MapXRes-2; i++)
		for (int j = 0; j < MapXRes-2; j++)
		{
				glMultiTexCoord2fvARB(GL_TEXTURE0_ARB, &(grid[i][j].tx));
				glNormal3f(grid[i][j].nx,		grid[i][j].ny,		grid[i][j].nz);
				glVertex3f(grid[i][j].x,		grid[i][j].y,		grid[i][j].z);
				
				glMultiTexCoord2fvARB(GL_TEXTURE0_ARB, &(grid[i+1][j].tx));
				glNormal3f(grid[i+1][j].nx,		grid[i+1][j].ny,	grid[i+1][j].nz);
				glVertex3f(grid[i+1][j].x,		grid[i+1][j].y,		grid[i+1][j].z);

				glMultiTexCoord2fvARB(GL_TEXTURE0_ARB, &(grid[i][j+1].tx));
				glNormal3f(grid[i][j+1].nx,		grid[i][j+1].ny,	grid[i][j+1].nz);
				glVertex3f(grid[i][j+1].x,		grid[i][j+1].y,		grid[i][j+1].z);

				
				glMultiTexCoord2fvARB(GL_TEXTURE0_ARB, &(grid[i+1][j].tx));
				glNormal3f(grid[i+1][j].nx,		grid[i+1][j].ny,	grid[i+1][j].nz);
				glVertex3f(grid[i+1][j].x,		grid[i+1][j].y,		grid[i+1][j].z);

				glMultiTexCoord2fvARB(GL_TEXTURE0_ARB, &(grid[i][j+1].tx));
				glNormal3f(grid[i][j+1].nx,		grid[i][j+1].ny,	grid[i][j+1].nz);
				glVertex3f(grid[i][j+1].x,		grid[i][j+1].y,		grid[i][j+1].z);

				glMultiTexCoord2fvARB(GL_TEXTURE0_ARB, &(grid[i+1][j+1].tx));
				glNormal3f(grid[i+1][j+1].nx,	grid[i+1][j+1].ny,	grid[i+1][j+1].nz);
				glVertex3f(grid[i+1][j+1].x,	grid[i+1][j+1].y,	grid[i+1][j+1].z);

		}
	glEnd();
};


//initExtensions - initialize extensions
void ShadowCaster::initExtensions(void)
{
#ifdef _WIN32
  /* Retrieve some ARB_multitexture routines. */
  glMultiTexCoord2iARB =
    (PFNGLMULTITEXCOORD2IARBPROC)
    GET_PROC_ADDRESS("glMultiTexCoord2iARB");
  glMultiTexCoord2fvARB =
    (PFNGLMULTITEXCOORD2FVARBPROC)
    GET_PROC_ADDRESS("glMultiTexCoord2fvARB");
  glMultiTexCoord3fARB =
    (PFNGLMULTITEXCOORD3FARBPROC)
    GET_PROC_ADDRESS("glMultiTexCoord3fARB");
  glMultiTexCoord3fvARB =
    (PFNGLMULTITEXCOORD3FVARBPROC)
    GET_PROC_ADDRESS("glMultiTexCoord3fvARB");
  glActiveTextureARB =
    (PFNGLACTIVETEXTUREARBPROC)
    GET_PROC_ADDRESS("glActiveTextureARB");
# endif
}


//displays various visual aids if active
void ShadowCaster::drawVisualAids()
{

		glActiveTextureARB(GL_TEXTURE0_ARB);  //these bitches need to go
		glDisable(GL_TEXTURE_2D);
		glActiveTextureARB(GL_TEXTURE1_ARB);
		glDisable(GL_TEXTURE_2D);


	if (controls->showNormals->isChecked()) // do normals
	{
		for (int i = 8; i < MapXRes; i+=8)
			for (int j = 8; j < MapXRes; j+=8)
			{
				//qDebug("%f  %f  %f",grid[i][j].x+grid[i][j].nx,		grid[i][j].y+grid[i][j].ny,		grid[i][j].z+grid[i][j].nz);
				glBegin(GL_LINES);
					glVertex3f(grid[i][j].x,grid[i][j].y,grid[i][j].z);	
					glVertex3f(grid[i][j].x+grid[i][j].nx,grid[i][j].y+grid[i][j].ny,grid[i][j].z+grid[i][j].nz);	
				glEnd();
			};
	}

	if (controls->aidDepth->isChecked()) glEnable(GL_DEPTH_TEST); else glDisable(GL_DEPTH_TEST);

	glColor4fv(lightCol);
	
	glDisable(GL_LIGHTING);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); //standard transparency equation

	if (controls->lightSource->isChecked())
	{
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
			glTranslatef(lightPos[0],lightPos[1],lightPos[2]);
			gluQuadricNormals(quadric, GLU_SMOOTH);
			gluSphere(quadric, 0.14, 20, 20);
		glPopMatrix();
	};

	if (controls->lightVectors->isChecked()) //lite vector
	{
		glBegin(GL_LINES);
			glVertex3fv(lightPos); glVertex3fv(lightDst);
		glEnd();
	};
	glDisable(GL_BLEND);

	if (controls->lightView->isChecked()) //show light view in 'window'
	{
		glDisable(GL_DEPTH_TEST);
		glDisable(GL_LIGHTING);


		glColor4f(1,1,1,1);

		glActiveTextureARB(GL_TEXTURE0_ARB);
		glDisable(GL_TEXTURE_2D);

		glActiveTextureARB(GL_TEXTURE1_ARB);
		glEnable(GL_TEXTURE_2D);
		
		glBindTexture(GL_TEXTURE_2D, luminance); //use the visualization proxy, arbitrary precision

		glMatrixMode(GL_PROJECTION);
		glPushMatrix();
		glLoadIdentity();	
		gluOrtho2D(0, 500, 0, 500);
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		glTranslatef(-2.5, 2.0, 0);
		glBegin(GL_QUADS);
			glMultiTexCoord2iARB(GL_TEXTURE1_ARB, 0,0);
			glVertex3i(0,0,0);

			glMultiTexCoord2iARB(GL_TEXTURE1_ARB, 1,0);
			glVertex3i(100,0,0);

			glMultiTexCoord2iARB(GL_TEXTURE1_ARB, 1,1);
			glVertex3i(100,103,0);

			glMultiTexCoord2iARB(GL_TEXTURE1_ARB, 0,1);
			glVertex3i(0,103,0);
		glEnd();

		glPopMatrix();
		glMatrixMode(GL_PROJECTION);
		glPopMatrix();
		glMatrixMode(GL_MODELVIEW);
	};
};

// initializes the light & specularity states
void ShadowCaster::initLightStates()
{
	GLfloat materialProps[4] = {1.0f,1.0f,1.0f,1.0f}; //proxy for normal material
	GLfloat materialGloss[4] = {0.6f,0.6f,0.6f,0.6f}; //proxy for full reflectivity
	GLfloat materialDull[4]  = {0.0f,0.0f,0.0f,0.0f}; //proxy for zero reflectivity

	GLfloat materialZero[4] = {0,0,0,0}; //proxy for 0 ambience

	GLfloat lightDir[4] = {lightDst[0]-lightPos[0], lightDst[1]-lightPos[1], 
						   lightDst[2]-lightPos[2], 0};

	glEnable(GL_COLOR_MATERIAL);
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,  materialProps);

	
	switch (controls->glossMethod->currentItem())
	{
		case 0 : //no gloss
			glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, materialDull);
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
			glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SINGLE_COLOR);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,  materialProps);
			glMateriali(GL_FRONT_AND_BACK,  GL_SHININESS, 0);
			break;
		case 1 :  //image gloss
			glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, materialGloss);
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
			glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SINGLE_COLOR);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,  materialProps);
			glMateriali(GL_FRONT_AND_BACK,  GL_SHININESS, 5);
			break;
		case 2 :  //photo gloss
			glDisable(GL_BLEND);
			glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, materialGloss);
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_BLEND);
			glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SEPARATE_SPECULAR_COLOR);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,  materialGloss);
			glMateriali(GL_FRONT_AND_BACK,  GL_SHININESS, 5);
			break;
	}
	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT,  materialZero);


//spotlight
	glLightfv(GL_LIGHT1, GL_POSITION,		lightPos);
	glLightfv(GL_LIGHT1, GL_SPOT_DIRECTION, lightDir);
	glLightfv(GL_LIGHT1, GL_DIFFUSE,		lightCol);
	glLightfv(GL_LIGHT1, GL_SPECULAR,		lightCol);
	glLighti(GL_LIGHT1,  GL_SPOT_CUTOFF,	90);
	glLighti(GL_LIGHT1,  GL_SPOT_EXPONENT,	1);


//directional light
	lightDir[0] *= -1.0f;  //*merf* I don't know why! Maybe ask Karinne? Yip!
	lightDir[1] *= -1.0f;  //*merf* I don't know why! Maybe ask Karinne? Yip!
	lightDir[2] *= -1.0f;  //*merf* I don't know why! Maybe ask Karinne? Yip!
	glLightfv(GL_LIGHT0, GL_DIFFUSE,		lightCol);
	glLightfv(GL_LIGHT0, GL_SPECULAR,		lightCol);
	glLightfv(GL_LIGHT0, GL_POSITION,		lightDir); //I'm a nifty muddafugga


	glEnable(GL_LIGHTING);


	if (controls->lightDirection->currentItem())
	{
		glDisable(GL_LIGHT1);
		glEnable(GL_LIGHT0);
	}
	else //spotlight
	{
		glDisable(GL_LIGHT0);
		glEnable(GL_LIGHT1);
	};

};

//generates texcoords for shadow mapping
void ShadowCaster::generateTexCoords(bool enable)
{
	/* Scale and bias [-1,1]^3 clip space into [0,1]^3 texture space. */
	GLfloat Smatrix[16] = {
    0.5, 0,   0,   0,
    0,   0.5, 0,   0,
    0,   0,   0.5, 0,
    0.5, 0.5, 0.5, 1.0
	};

	GLfloat resultMatrix[16];
	if (enable)
	{
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
			glLoadMatrixf(Smatrix);
			glMultMatrixf(lightProjection);
			glMultMatrixf(lightModelView);
			glGetFloatv(GL_MODELVIEW_MATRIX, resultMatrix);
		glPopMatrix();

		glActiveTextureARB(GL_TEXTURE1_ARB);
		glBindTexture(GL_TEXTURE_2D, shadow);
		glEnable(GL_TEXTURE_2D); 

		GLfloat p[4];

  /* S and Q are always needed */
		p[0] = resultMatrix[0];
		p[1] = resultMatrix[4];
		p[2] = resultMatrix[8];
		p[3] = resultMatrix[12];
		glTexGenfv(GL_S, GL_EYE_PLANE, p);

		p[0] = resultMatrix[3];
		p[1] = resultMatrix[7];
		p[2] = resultMatrix[11];
		p[3] = resultMatrix[15];
		glTexGenfv(GL_Q, GL_EYE_PLANE, p);

	    p[0] = resultMatrix[1];
	    p[1] = resultMatrix[5];
	    p[2] = resultMatrix[9];
	    p[3] = resultMatrix[13];
	    glTexGenfv(GL_T, GL_EYE_PLANE, p);

	    p[0] = resultMatrix[2];
	    p[1] = resultMatrix[6];
	    p[2] = resultMatrix[10];
	    p[3] = resultMatrix[14];
	    glTexGenfv(GL_R, GL_EYE_PLANE, p);

		glEnable(GL_TEXTURE_GEN_S);
		glEnable(GL_TEXTURE_GEN_Q);
		glEnable(GL_TEXTURE_GEN_T);
		glEnable(GL_TEXTURE_GEN_R);
	} else
	{
		glActiveTextureARB(GL_TEXTURE1_ARB);
//		glBindTexture(GL_TEXTURE_2D, shadow);
		glDisable(GL_TEXTURE_2D);

		glDisable(GL_TEXTURE_GEN_S);
		glDisable(GL_TEXTURE_GEN_Q);
		glDisable(GL_TEXTURE_GEN_T);
		glDisable(GL_TEXTURE_GEN_R);
	};

};

void ShadowCaster::generateShadowMap()
{
	static bool virginCondition = 1;
	if (shadowsDirty) //save processing time
	{	
		glColorMask(0, 0, 0, 0); //only want depth values!
		glColor4f(1,1,1,1);
		glGetIntegerv(GL_VIEWPORT, tempViewPort);
		glViewport(0, 0, 256, 256);

		glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
		glDisable(GL_LIGHTING);
			glMatrixMode(GL_PROJECTION);
				glPushMatrix();
				glLoadIdentity();
				//set up square aspect ratios for texture
				gluPerspective(140, 1.0, 0.85, 5);
		

				
			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
				glLoadIdentity();

				glScalef(4,4,4);
				//ever wondered what this command was good for? *g*
				gluLookAt(lightPos[0], lightPos[1], lightPos[2], lightDst[0],  lightDst[1],  lightDst[2], 0, 0, 1);

				glGetFloatv(GL_PROJECTION_MATRIX, lightProjection);  //store for texture coord generation
				glGetFloatv(GL_MODELVIEW_MATRIX, lightModelView);  //store for texture coord generation

				drawGrid();

				glActiveTextureARB(GL_TEXTURE1_ARB); //work with the shadow channels
				if (controls->lightView->isChecked())
				{
					glReadPixels(0, 0, 256, 256, GL_DEPTH_COMPONENT, GL_FLOAT, shadowMap); //load depth image
					glBindTexture(GL_TEXTURE_2D, luminance);
					glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB8, 256, 256, 0, GL_LUMINANCE, GL_FLOAT, shadowMap); //load helper image
				};

				glBindTexture(GL_TEXTURE_2D, shadow);
				if (virginCondition) {glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT, 0, 0, 256, 256, 0); virginCondition = 0;}//load light view
					else glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, 256, 256); //load light view

	//		glMatrixMode(GL_MODELVIEW);
			glPopMatrix();

			glMatrixMode(GL_PROJECTION);
			glPopMatrix();
		glViewport(tempViewPort[0], tempViewPort[1], tempViewPort[2], tempViewPort[3]);
		glColorMask(1, 1, 1, 1);
	}


};


void ShadowCaster::render()
{

	initGrid();

	glEnable(GL_DEPTH_TEST);

	glClearColor(0,0,0,0);

	glEnable(GL_TEXTURE_2D);

	glActiveTextureARB(GL_TEXTURE0_ARB);
	glEnable(GL_TEXTURE_2D);

	glActiveTextureARB(GL_TEXTURE1_ARB);
	glDisable(GL_TEXTURE_2D);
	//do the map thing
	if (controls->lightMethod->currentItem())
	{
		generateShadowMap();
		generateTexCoords(true);
	} else generateTexCoords(false);

	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

	//self-explanatory
	initLightStates();

	//render final scene
	drawGrid();

	generateTexCoords(false);
	//process visual aids
	drawVisualAids();

	//done!
	glFlush();

}




