/*****************************************************************************\
 *                              GLView
\*****************************************************************************/

/*! @file GLView.h
 *
 *  
 *
 *  @brief
 *	GL rendering view class declarations
 *   
 *  @author Ibraguim Kouliev
 *     
 */

/*
    Copyright  2001 Ibragium Kouliev All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @class GLView
 *
 *  @ingroup appclasses
 *
 *  @brief GL rendering view class.
 *
 *  This class encapsulates the GL rendering viewport and provides basic 
 *  rendering functionality. The viewport calls in the user-defined 
 *  implementation of render() from the selected effect to display it. 
 *  Also contained in the class are event hadlers for keyboard and mouse
 *  (see descriptions for details), as well as functions that automatically
 *  set and handle all necessary view/projection tranformations as well 
 *  as resizing and updates.
 *
 *  @note  See notes on individual functions for details.
 *  
 **/

#ifndef GLVIEW_H
#define GLVIEW_H

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000


//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

//Qt headers
#include <qgl.h>

// GL headers
#include <GL/gl.h>
#include <GL/glu.h>

//---------------------------------------------------------------------------
//  Forward references
//---------------------------------------------------------------------------

// DO NOT REMOVE !
class GLFramework; 
class GLEffect;

class QPixmap;
class QCursor;

//**************************************************************************
// GLView
//**************************************************************************

class GLView : public QGLWidget
{
	Q_OBJECT

//---------------------------------------------------------------------------
//  Public Instance Methods
//---------------------------------------------------------------------------

public:
//---------------------------------------------------------------------------

//! @name Construction/Desctruction
// @{

	//! Primary class contructor
		GLView(QWidget* parent, GLFramework* parentGui, 
				QGLFormat* glFormat, const char* name=0);
	
	//! Destructor
		virtual ~GLView();
//---------------------------------------------------------------------------
// @}

//! @name Public API functions
// @{
		//! Sets the camera angles.
		void setCameraAngles(GLdouble azimuth, 
					GLdouble elev, GLdouble twist);

		//! Sets FOV (Field Of View) angle.
		void setFieldOfView(GLdouble fov);

		/** @brief Provided for convenience so effects 
		 could change projection 
		parameters if required. */

		void setProjectionParams(GLdouble fov, GLdouble nearClip, 
					 GLdouble farClip,GLdouble distance);

		//! Sets the current effect. 
		void setCurrentEffect(GLEffect* effect);
		
		//! Called by the current effect to update the window.
		void updateWindow()
		{updateGL();}

// ------------------------------------------------------------------------------
// @}

//! @name Layout-related functions
// @{

	/** @brief Returns the size policy for this widget. 
	See Qt documentation on how this works... */
	QSizePolicy sizePolicy() const;

	//! Returns the size hint.
	QSize sizeHint() const;

// --------------------------------------------------------------------------
// @}

// --------------------------------------------------------------------------

public slots:
	
	//! Post-construction final init.
	virtual void polish();

	//! Toggles between pespective and orthographic projection.
	void changeProjectionMode(bool orthoMode);

	//! Toggles between normal and camera control views
	void switchToTrackball(bool on);

	//! Resets the view orientation 
	void resetView();

	//! Resets the view extents
	void resetFOV();

	//! Saves the current rendered frame to a file
	void saveCurrentRenderedImage();

signals:
// --------------------------------------------------------------------------

//---------------------------------------------------------------------------
//  Private/Protected Instance Methods
//---------------------------------------------------------------------------

	protected: 

//! @name Internal GL-related functions
// @{
		//! Initializes the GL window.
		virtual void initializeGL();
		
		//! Draws the GL window.
		virtual void paintGL();
		
		// Used to initiate redraw....
		//virtual void updateGL();

		//! Called when the GL window is resized
		virtual void resizeGL(int width, int height);
//---------------------------------------------------------------------------
// @}

//! @name Event handlers
// @{
		//! Handles mouse move events.	
		virtual void mouseMoveEvent(QMouseEvent*);
		
		//! Handles mouse press events
		virtual void mousePressEvent(QMouseEvent*);

		//! Handles mouse release events
		virtual void mouseReleaseEvent(QMouseEvent*);
		
		//! Handles keyboard events
		virtual void keyPressEvent(QKeyEvent*);

		//! Handles mouse wheel events
		virtual void wheelEvent(QWheelEvent*);
		
//---------------------------------------------------------------------------
// @}

		
//---------------------------------------------------------------------------
//  Private/Protected Instance Variables
//---------------------------------------------------------------------------

	protected:
		// **** Data members ****
			
		// Pointer to the parent widget
		GLFramework* parentFrame;

		// Pointer to current effect
		GLEffect* currentEffect;

		// ---- Projection parameters ----

		// Distance from the origin to the chosen viewpoint 
		GLdouble viewDistance;

		// FOV (Field Of View) angle for pespective projection
		GLdouble fovAngle;
		
		// distance to near and far clipping planes
		GLdouble nearClipPlane, farClipPlane;

		/* extents of the orthographic view volume, used to zoom in
		 * /out in orthographic mode */
		GLdouble orthoVolumeExtents; 

		// projection mode flag
		bool perspectiveOn;

		// ---- Components of the virtual trackball control ----

		// virtual trackball mode flag
		bool trackballOn;
		bool hadTracking; //for setting/resetting mouse tracking

		// orbiting/rotation/fov state cursors and their icons
		QPixmap* orbitIcon;
		QCursor* orbitCursor;
	
		QPixmap* orbitHorizontalIcon;
		QCursor* orbitHorizontalCursor;
		
		QPixmap* orbitVerticalIcon;
		QCursor* orbitVerticalCursor;

		QPixmap* rotationIcon;
		QCursor* rotationCursor;

		QPixmap* fovIcon;
		QCursor* fovCursor;

		// single-axis rotation locks
		bool xLock,yLock,zLock; 
		int lockThreshold; // lock threshold distance
		
		bool mouseMoving; // is the mouse moving ?

		float fovAngleDelta; // FOV angle increment value
		float orthoDelta; // ortho volume increment value

		/* for initial setup of modelview matrices */
		bool first_run; 
		
		/* the following stuff is used to calculate the
		 axis and angle of rotation for the scene while in trackball
		 mode */	
	
		// direction vectors and resulting rotation axis
		double trackVector1[3],trackVector2[3];
		double rotationAxis[3];
		double rotationAngle; // scene rotation angle
	
		int mouse_x, mouse_y; //current mouse coordinates
		int prev_mouse_x, prev_mouse_y; // previous mouse coordinates

		double norm_x, norm_y; //normalized mouse coordinates
		double vec_length; // temporary var used in calculations

		float angleScale; // scale value for angles
		float ballRadius; // trackball sphere radius

		/* these are used to track object orientation */
		GLdouble currentModelviewMatrix[16]; // current modelview matrix
		
		/* current modelview matrix for the tripod symbol */
		GLdouble currentTripodMatrix[16]; 
		GLdouble tempMatrix[16];

	private:

//---------------------------------------------------------------------------
//  Class methods and variables
//---------------------------------------------------------------------------
public:
	//! Displays OpenGL renderer capabilities. 
	static void getOpenGLCaps();

};

#endif




