/*****************************************************************************\
 *                              GLEffect
\*****************************************************************************/

/*! @file GLView.cpp
 *
 *  
 *
 *  @brief
 *	GL rendering view implementation
 *    
 *  @author Ibraguim Kouliev
 *  
 */

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

// System library headers
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <math.h>
#include <string.h>

// Qt headers
#include <qgl.h>

#include <qsize.h>
#include <qvgroupbox.h>
#include <qpushbutton.h>
#include <qcursor.h>
#include <qdialog.h>
#include <qfiledialog.h>
#include <qlayout.h>

#include <qpixmap.h>
#include <qcursor.h>

#include <qimage.h>

// local headers
#include "GLView.h"
#include "GLEffect.h"
#include "GLFramework.h"
#include "icons.h"

// Implementation of GLView

/**
 *
 *  @param parent Pointer to the parent widget. 
 *  
 *  @param parentGui Pointer to the parent GUI. Must point to the parent
 *  instance of GLFramework, as opposed to the 1st parameter, which can also 
 *  point to another widget.
 *
 *  @param glFormat Pointer to a QGLFormat context descriptor object.
 *  @param name Symbolic name for this widget.
 *
 *  Constructs an object and obtains a pointer to the parent GLFramework
 *  widget. Internal parameters are initialized 
 *  here as well.
 *
 *  @todo Some additional init may be added here later.
 * 
 **/

GLView::GLView(QWidget* parent, GLFramework* parentGui, 
			   QGLFormat* glFormat, const char* name)
		:	QGLWidget(*glFormat, parent, name)
{
	parentFrame = parentGui;

	// initialize the projection parameters to acceptable default values
	viewDistance = 5.0f;
	fovAngle = 40.0f;
	nearClipPlane = -1.0f;
	farClipPlane = 1.0f;

	orthoVolumeExtents = 2.5f;
	orthoDelta = 0.1f;
	fovAngleDelta = 1.0f;

	// set the current effect pointer to NULL
	currentEffect = 0;

	perspectiveOn = true;
		
	// init the trackball components
	trackballOn = false;
	hadTracking = false;
	mouseMoving = false;

	first_run = false;

	xLock = yLock = zLock = false;
	lockThreshold = 10; // in pixels

	mouse_x = mouse_y = 0;
	prev_mouse_x = prev_mouse_y = 0;

	trackVector1[0] = trackVector1[1] = trackVector1[2] = 0.0f;
	trackVector2[0] = trackVector2[1] = trackVector2[2] = 0.0f;
	rotationAxis[0] = rotationAxis[1] = rotationAxis[2] = 0.0f;
	rotationAngle = 0.0f;

	memset(currentModelviewMatrix,0x00,16*sizeof(GLdouble));
	memset(currentTripodMatrix, 0x00, 16*sizeof(GLdouble));
	memset(tempMatrix,0x00,16*sizeof(GLdouble));

	angleScale = 1.0f;
	ballRadius = sqrt(2.0f);

	// create custom cursors for various trackball modes
	orbitIcon = new QPixmap(const_cast<const char**>
				(TRACKBALL_ORBIT_CURSOR));
	orbitCursor = new QCursor(*orbitIcon);
	
	orbitHorizontalIcon = new QPixmap(const_cast<const char**>
					  (TRACKBALL_ORBIT_HORIZONTAL));
	orbitHorizontalCursor = new QCursor(*orbitHorizontalIcon);

	orbitVerticalIcon = new QPixmap(const_cast<const char**>
					(TRACKBALL_ORBIT_VERTICAL));
	orbitVerticalCursor = new QCursor(*orbitVerticalIcon);

	rotationIcon = new QPixmap(const_cast<const char**>
					(TRACKBALL_ROTATION_CURSOR));
	rotationCursor = new QCursor(*rotationIcon);

	fovIcon = new QPixmap(const_cast<const char**>
			      (TRACKBALL_FOV_CURSOR));
	fovCursor = new QCursor(*fovIcon);
	
	// set to default cross-hair cursor
	setCursor(crossCursor);
}

/**
 * Destroys the object. Empty for now, but may be modified later to
 * include removal of dynamically allocated memory.
 *
 **/
GLView::~GLView()
{
	// clean up...
	delete orbitIcon;
	delete orbitCursor;
	delete rotationIcon;
	delete rotationCursor;

	delete fovIcon;
	delete fovCursor;

	delete orbitHorizontalIcon;
	delete orbitHorizontalCursor;

	delete orbitVerticalIcon;
	delete orbitVerticalCursor;

}

/** This function initializes internal states and parameters.  It also sets the
 * size of GL viewport and establishes the  perspective projection according to
 * fovAngle, nearClipPlane, farClipPlane, and aspect ratio of the viewport,
 * whose values are obtained from the currently selected effect.  The
 * projection parameters are kept until a new effect is set. 
 * 
 * Most basic openGL functionality is initially \b de-activated here in  order
 * to reset the GL rendering context when calling a new effect.  This is done
 * to avoid such situations as one effect enabling GL_ALPHA_TEST and
 * accidentally leaving it enabled, and the next selected effect looking
 * incorrectly as a result because it didn't want GL_ALPHA_TEST. 
 *
 * @attention The above mechanism protects you from unexpected side effects,
 * but you \b will have to explicitly enable any openGL modes your effect
 * requires.
 *
 *  @see setProjectionParams()
 **/
void GLView::initializeGL()
{	
	// default clear color to black
	glClearColor(0.0, 0.0, 0.0, 1.0);

	// size of viewport
	glViewport(0,0, 
		   (GLsizei)(this->width()),
		   (GLsizei)(this->height()) );

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	if(perspectiveOn)
	{
		// define perspective projection
		gluPerspective(fovAngle,
				(GLdouble)(this->width())/
				(GLdouble)(this->height()),
				nearClipPlane, farClipPlane);
	}
	else
	{
		// define orthographic projection
		glOrtho(-orthoVolumeExtents,orthoVolumeExtents,
			-orthoVolumeExtents,orthoVolumeExtents,
			-1, farClipPlane);

	}

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	if(perspectiveOn) 
	   gluLookAt(0,0,viewDistance, 0,0,0, 0,1,0);

	// disable basic openGL modes

    // Framebuffer parameters
	glDisable(GL_DEPTH_TEST); 
	glDisable(GL_ALPHA_TEST);
	glDisable(GL_STENCIL_TEST);
	glDisable(GL_SCISSOR_TEST);
	glDisable(GL_LOGIC_OP);
	
	// Geometry parameters
	glDisable(GL_NORMALIZE);
	glDisable(GL_AUTO_NORMAL);
	glDisable(GL_CULL_FACE);

	glDisable(GL_POINT_SMOOTH);
	glDisable(GL_POLYGON_SMOOTH);
	glDisable(GL_POLYGON_STIPPLE);
	
	glDisable(GL_LINE_SMOOTH);
	glDisable(GL_LINE_STIPPLE);
	
	glDisable(GL_COLOR_MATERIAL);

	// Evaluator parameters
	glDisable(GL_MAP1_COLOR_4);
	glDisable(GL_MAP1_INDEX);
	glDisable(GL_MAP1_NORMAL);
	
	glDisable(GL_MAP1_TEXTURE_COORD_1);
	glDisable(GL_MAP1_TEXTURE_COORD_2);
	glDisable(GL_MAP1_TEXTURE_COORD_3);
	glDisable(GL_MAP1_TEXTURE_COORD_4);

	glDisable(GL_MAP1_VERTEX_3);
	glDisable(GL_MAP1_VERTEX_4);

	glDisable(GL_MAP2_COLOR_4);
	glDisable(GL_MAP2_INDEX);
	glDisable(GL_MAP2_NORMAL);
	
	glDisable(GL_MAP2_TEXTURE_COORD_1);
	glDisable(GL_MAP2_TEXTURE_COORD_2);
	glDisable(GL_MAP2_TEXTURE_COORD_3);
	glDisable(GL_MAP2_TEXTURE_COORD_4);

	glDisable(GL_MAP2_VERTEX_3);
	glDisable(GL_MAP2_VERTEX_4);

	// Blending parameters
	glDisable(GL_BLEND);
	
	// Lighting & fog parameters
	glDisable(GL_LIGHTING); 
	glDisable(GL_FOG);

	// Texturing parameters
	glDisable(GL_TEXTURE_1D);
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_TEXTURE_GEN_Q);
	glDisable(GL_TEXTURE_GEN_R);
	glDisable(GL_TEXTURE_GEN_S);
	glDisable(GL_TEXTURE_GEN_T);

	// Rendering parameters
	glDisable(GL_DITHER);
}

/**
 * @param width New width of viewport, in pixels.
 * @param height New height of viewport, in pixels.
 *
 * This function is used internally by GLView to re-establish the 
 * viewport & projection matrix. It is automatically invoked when 
 * the application window is resized and sets the size of 
 * GL viewport and perspective projection mode dependent on 
 * current values of fovAngle, nearClipPlane, farClipPlane and aspect
 * ratio of the viewport, as well as initial viewpoint position. 
 *   
 * @note This function must never be explicitly called.
 * 
 * @see initializeGL(), setProjectionParams()
 **/

// Used when resizing the window 
void GLView::resizeGL(int width, int height)
{
	//set the viewport size
	glViewport(0,0,(GLsizei)width,(GLsizei)height);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	if(perspectiveOn)
	{
		// define perspective projection
		gluPerspective(fovAngle,
				(GLdouble)width/(GLdouble)height,
				nearClipPlane, farClipPlane);
	}
	else
	{
		// define orthographic projection
		glOrtho(-orthoVolumeExtents, orthoVolumeExtents,
			-orthoVolumeExtents, orthoVolumeExtents,
			-5, farClipPlane);
	}

	glMatrixMode(GL_MODELVIEW);
}

/** Positions the camera and draws the window. Displaying the currently
 * selected effect is accomplished by invoking its render() function.  The
 * rendering context is double-buffered by default. 
 * This method also renders the axis tripod and axes visual aids when in
 * virtual trackball mode. 
 *  
 **/
void GLView::paintGL()
{
	// remove previous transformations 
	glLoadIdentity();

	/* This is used only during the very first rendering of
	the scene to set up the correct camera viewpoint. Later on the required
	trasformation is already present in the modelview matrix, hence this
	if() clause */

	if(first_run)
	{
	  if(perspectiveOn) 
	   gluLookAt(0,0,viewDistance, 0,0,0, 0,1,0);
	}
	else glLoadMatrixd(currentModelviewMatrix);

	// rotate the scene only if the mouse is really moving
	if(mouseMoving)
	glRotated(rotationAngle, 
	  	  rotationAxis[0], rotationAxis[1], rotationAxis[2]);

	// call in user-defined rendering routine
	//Only if an effect is present 
	if(currentEffect!=0)
		currentEffect->render();
	
	// draw the axis tripod as visual aid 

	/* Note: all of additional push/pop operations and projection
	changes here are done so that the axis tripod may be drawn
	independently of the current FOV,aspect ratio, view distance and / or
	projection type...*/ 
	
 	glMatrixMode(GL_PROJECTION);
	 
	//set up orthographic projection for the tripod
	 glPushMatrix();
	 glLoadIdentity();
	 glOrtho(-1, -1, 1, 1, nearClipPlane, farClipPlane); 

	 glMatrixMode(GL_MODELVIEW);
	 glPushMatrix();

	 glDisable(GL_TEXTURE_2D);

	 /*This sets up temporary x- and y- axes visual aids
	 while the trackball mode is on */

	 if(trackballOn)
	 {
	  glPushMatrix();
	   glLoadIdentity();
	   glBegin(GL_LINES);
	   
	    // x - axis 
		glColor3f(1,0,0); 
		glVertex3f(-1.0f,0.0f,0.0f);
		glVertex3f(1.0f,0.0f,0.0f);
		
		// y - axis
		glColor3f(0,1,0);
		glVertex3f(0.0f,-1.0f,0.0f);
		glVertex3f(0.0f,1.0f,0.0f);

	   glEnd();
	  glPopMatrix();
	 }
	
	 // see explanation above...
	 if(first_run)
	 {
		glLoadIdentity();
		glTranslated(0.8f, -0.8f, 0.0f);
	 }
	 else glLoadMatrixd(currentTripodMatrix);

	 // rotate the tripod same as the scene
	 if(mouseMoving)
	 glRotated(rotationAngle,
		   rotationAxis[0], rotationAxis[1],rotationAxis[2]);

	 // save tripod's modelview matrix for future use
	 glGetDoublev(GL_MODELVIEW_MATRIX, tempMatrix);
 
	 // draw the tripod
	 glBegin(GL_LINES);

	 // y - axis 
	 glColor3f(0,1,0); 
	 
	 glVertex3f(0,0,0);
	 glVertex3f(0.0f, 0.1f, 0.0f); 
	 
	 // arrow
	 glVertex3f(-0.02f, 0.07f, 0.0f);
	 glVertex3f( 0.0f,  0.1f,  0.0f);
	 glVertex3f( 0.02f, 0.07f, 0.0f);
	 glVertex3f( 0.0f,  0.1f,  0.0f);
	
	 // x - axis
	 glColor3f(1,0,0);
	 
	 glVertex3f(0,0,0);
	 glVertex3f(0.1f,  0.0f,  0.0f);

	 //arrow
	 glVertex3f(0.07f,-0.02f, 0.0f);
	 glVertex3f(0.1f,  0.0f,  0.0f);
	 glVertex3f(0.07f, 0.02f, 0.0f);
	 glVertex3f(0.1f,  0.0f,  0.0f); 
	
	 // z-axis
	 glColor3f(0,0,1);

	 glVertex3f(0,0,0);
	 glVertex3f( 0.0f,  0.0f, 0.1f);
	
	 //arrow
	 glVertex3f(-0.02f, 0.0f, 0.07f);
	 glVertex3f( 0.0f,  0.0f, 0.1f);
	 glVertex3f( 0.02f, 0.0f, 0.07f);
	 glVertex3f( 0.0f,  0.0f, 0.1f);

	glEnd(); 
	glEnable(GL_TEXTURE_2D);
	
	//restore the original projection and modelview matrices

	glPopMatrix();
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);

	swapBuffers();
}

/** This is called to perform any initialization that's necessary before the
 * widget is displayed, but is too early to be called from the class
 * constructor. Does nothing for now but may be modified later if required. 
 *    
 * @note For internal use only.
 * 
 **/
void GLView::polish()
{
	QGLWidget::polish(); //call base-class implementation
	
}


/**
 * @param effect The effect object to be set as current effect.
 *
 * Sets the current effect and initializes it.  This also resets and 
 * reinitializes the GL viewport (see initializeGL() for explanation 
 * why this is done). 
 *
 *  @throw Can issue an error message and quit the program if the pointer 
 *  contained in \b effect parameter was bad.
 *   
 * @see initializeGL(), setProjectionParams()
 * 
 *
 **/
void GLView::setCurrentEffect(GLEffect* effect)
{
	first_run = true;

	CHECK_PTR(effect); 
	currentEffect = effect;

	// reinitialize the GL rendering context
	initializeGL();

	// initialize effect parameters
	currentEffect->init();
	this->setFocus();

	resizeGL(width(),height());
	updateGL();
	
}

// **** Event handlers ****

/** @param mEv A mouse event pointer supplied by the window. 
 *
 * Passes mouse press events to the corresponding event handler of the current
 * effect. This event handler also sets up the first direction vector and any
 * necessary rotation locks while in virtual trackball mode. Furthermore this
 * changes the mouse cursor to appropriate images when entering various
 * rotation/zoom modes.
 * 
 * 
 *@see mouseMoveEvent() **/
void GLView::mousePressEvent(QMouseEvent* mEv)
{
	/* if the view is in virtual trackball mode, pick up the 
	mouse coordinates and generate the 1st mouse position vector */
	
	if((trackballOn) & (mEv->button()==LeftButton) )
	{		
	 // ignore mouse event if mouse coordinates are out of window bounds
	 if( ( (mEv->x()>this->width()) | (mEv->x()<0) ) |
	  ( (mEv->y()>this->height()) | (mEv->y()<0) )  )
	  {
	   return;
	  }
	  else
	  {			
		//reset vectors
		trackVector1[0] = trackVector1[1] = trackVector1[2] = 0.0f; 
		trackVector2[0] = trackVector2[1] = trackVector2[2] = 0.0f;
	
		rotationAxis[0] = rotationAxis[1] = rotationAxis[2] = 0.0f; 
		rotationAngle = 0.0f;

		//get and normalize mouse coordinates to [-1...1]
		mouse_x = mEv->x(); mouse_y = mEv->y();
			
		norm_x = ((double)mouse_x/this->width())*2.0f - 1.0f;

		/* Y-Coordinate has to be negated because the window
		coordinates system has the positive Y-axis 'looking down'*/

		norm_y = - ( ((double)mouse_y/this->height())*2.0f - 1.0f);
			
		/* If mouse x or y coords lie within the direction
		lock threshold, constrain the rotation to the
		corresponding axis till the mouse button is released.
		The lock release occurs within mouseReleaseEvent() The
		corresponding mouse coordinate will be ignored,
		resulting in a purely 2-dimensional 2nd mouse position
		vector. The 1st position vector is not calculated at
		all, instead being set to a fixed reference point, to
		increase the angle precision. */

		//special case for zLock (view tilt)
		if( ( abs( norm_x*(width()/2)) < lockThreshold) & 
		  ( abs( norm_y*(height()/2)) < lockThreshold) )
		{				
			zLock = true;
			setCursor(*rotationCursor);

			rotationAxis[0]=0.0;
			rotationAxis[1]=0.0;
			rotationAxis[2]=1.0;

			// take (0,1,0) as fixed reference point
			trackVector1[0]=0.0;
			trackVector1[1]=1.0;
			trackVector1[2]=0.0;

		}
		else // lock to x (horizontal) axis
		if(abs( norm_y*(height()/2) )<lockThreshold)
		{				
			xLock = true;
			setCursor(*orbitHorizontalCursor);
				
			rotationAxis[0]=1.0;
			rotationAxis[1]=0.0;
			rotationAxis[2]=0.0;

			// take (0,0,1) as fixed reference point
			trackVector1[0]=0.0;
			trackVector1[1]=0.0;
			trackVector1[2]=1.0;
		}
		else // lock to y (vertical axis)
		if(abs( norm_x*(width()/2) )<lockThreshold)
		{
			yLock = true;
			setCursor(*orbitVerticalCursor);
			
			rotationAxis[0]=0.0;
			rotationAxis[1]=1.0;
			rotationAxis[2]=0.0;

			// take (0,0,1) as fixed reference point
			trackVector1[0]=0.0;
			trackVector1[1]=0.0;
			trackVector1[2]=1.0;
		}
		else // no locks active, normal rotation mode 
		{
			trackVector1[0] = norm_x;
			trackVector1[1] = norm_y;
			
			trackVector1[2] = sqrt(ballRadius*ballRadius
						-(norm_x*norm_x + norm_y*norm_y));
		}
	 }
	 		
	}

	else if((trackballOn) & (mEv->button()==RightButton))
	{
	  // switch to zoom mode 
	  setCursor(*fovCursor);
	  prev_mouse_y = mEv->y();
	}
	// otherwise pass event to the currently selected effect
	else currentEffect->mousePressEvent(mEv);
}

/**
 * @param mEv A mouse event pointer supplied by the window
 *
 * Passes mouse release events to the corresponding event handler of
 * the current effect. This handler also resets any rotation locks 
 * and rotation angle & axis parameters, and returns to normal orbiting
 * cursor.
 *
 **/
void GLView::mouseReleaseEvent(QMouseEvent* mEv)
{
	if((trackballOn) & (mEv->button()==LeftButton) )
	{
		/* release rotation locks and switch back to default 
		orbiting cursor */
		
		xLock = yLock = zLock = false;
		mouseMoving = false;
		setCursor(*orbitCursor);

		rotationAngle = 0.0f;
		rotationAxis[0]=rotationAxis[1]=rotationAxis[2]=0.0f;

		//store the current modelview matrix
		glGetDoublev(GL_MODELVIEW_MATRIX, currentModelviewMatrix);
		memcpy(currentTripodMatrix,tempMatrix,16*sizeof(GLdouble));

		if(first_run)
		 first_run = false;
	}
	else 
	if((trackballOn) & (mEv->button()==RightButton))
	{      
		setCursor(*orbitCursor);

	} else currentEffect->mouseReleaseEvent(mEv);
}

/**
 * @param mEv A mouse event pointer supplied by the window.
 *
 * Passes mouse move events to the corresponding event handler of the 
 * current effect. 
 * 
 * This event handler also calculates the rotation axis and angle 
 * when in virtual trackball mode, or changes the zoom extents when in 
 * zoom mode. 
 *
 **/
void GLView::mouseMoveEvent(QMouseEvent* mEv)
{
  /* if the view is in virtual trackball mode, pick up the 
	mouse coordinates, generate the 2nd mouse position vector, calculate
	the rotation axis and angle, and rotate the scene */

	if((trackballOn) & (mEv->state()==LeftButton) )
	{		
		/* ignore mouse event if mouse coordinates are out of window
		 * bounds */

		if( ( (mEv->x()>this->width()) | (mEv->x()<0) ) |
		 ( (mEv->y()>this->height()) | (mEv->y()<0) )  )
		{
			return;
		}
		else
		{
			mouseMoving = true;

			// first, we generate the 2nd mouse position vector...

			//get and normalize mouse coordinates to [-1...1]
			mouse_x = mEv->x(); mouse_y = mEv->y();
			
			norm_x = ((double)mouse_x/this->width())*2.0f - 1.0f;
			norm_y = 
			    - (((double)mouse_y/this->height())*2.0f - 1.0f);

			if(xLock) 
			{norm_x = 0.0;}

			else if(yLock) 
			{ norm_y = 0.0;}

			trackVector2[0] = norm_x;
			
			/* ensure that in zLock mode, 2nd position vector
			has sufficient magnitude, and constrain rotation to
			xy-plane */

			trackVector2[1] = (zLock == true)?
						sqrt(ballRadius*ballRadius-
						(norm_x*norm_x)):norm_y;

			trackVector2[2] = (zLock == true)?0.0:
						sqrt(ballRadius*ballRadius
						-(norm_x*norm_x + norm_y*norm_y));
			
			//reverse the rotation axis if necessary
			
			if(xLock)
			{
				rotationAxis[0]=(norm_y>0.0)?-1.0:1.0; 
			}

			if(yLock)
			{
				rotationAxis[1]=(norm_x<0.0)?-1.0:1.0; 
			}

			if(zLock)
			{
				rotationAxis[2]=(norm_x<0.0)?-1.0:1.0; 
			}

			angleScale = 2.0f;

			/* the rotation axis is only constructed if none of
			the rotation locks is active */

			if( (xLock==false) &
			    (yLock==false) &
			    (zLock==false))

			{
			 /* the rotation axis is a normalized cross product
			 between the calculated mouse position vectors */

			 rotationAxis[0] = trackVector1[1]*trackVector2[2]-
			  		   trackVector1[2]*trackVector2[1];

			 rotationAxis[1] = trackVector1[2]*trackVector2[0]-
					   trackVector1[0]*trackVector2[2];

			 rotationAxis[2] = trackVector1[0]*trackVector2[1]-
					   trackVector1[1]*trackVector2[0];

			 vec_length = sqrt(rotationAxis[0]*rotationAxis[0] 
					   + rotationAxis[1]*rotationAxis[1]
					   + rotationAxis[2]*rotationAxis[2]);

			 // normalize the resulting vector 
			 
			 rotationAxis[0] /= vec_length;
			 rotationAxis[1] /= vec_length;
			 rotationAxis[2] /= vec_length;
			
			}

			//normalize 1st direction vector
			vec_length = sqrt(trackVector1[0]*trackVector1[0]+ 
					          trackVector1[1]*trackVector1[1]+
				 	          trackVector1[2]*trackVector1[2]);

			trackVector1[0] /= vec_length;
			trackVector1[1] /= vec_length;	
			trackVector1[2] /= vec_length;	

			//normalize 2nd direction vector
			vec_length = sqrt(trackVector2[0]*trackVector2[0]+ 
					          trackVector2[1]*trackVector2[1]+
					          trackVector2[2]*trackVector2[2]);

			trackVector2[0] /= vec_length;
			trackVector2[1] /= vec_length;	
			trackVector2[2] /= vec_length;	
			
			/* now we find out the rotation angle by taking an
			 * inverse cosine of the dot product between the mouse
			 * position vectors
			*/

			rotationAngle = 
				angleScale*(180.0/3.14159)*acos(
					   (trackVector1[0]*trackVector2[0]+
					    trackVector1[1]*trackVector2[1]+
					    trackVector1[2]*trackVector2[2]));
		}
		updateGL(); //redraw the scene....
	}
	else

	// when in zoom mode
	if((trackballOn) & (mEv->state()==RightButton))
	{
		mouse_y = mEv->y();
				
		if(mouse_y - prev_mouse_y > 0)
		{
			// zoom out view
			fovAngle += fovAngleDelta;
			orthoVolumeExtents += orthoDelta;
		}

		if(mouse_y - prev_mouse_y < 0)
		{
			// zoom in view
			fovAngle -= fovAngleDelta;
			orthoVolumeExtents -= orthoDelta;
		}

		prev_mouse_y = mouse_y;

		resizeGL(width(), height());
		updateGL();
	}

	// otherwise pass event to the currently selected effect
	else currentEffect->mouseMoveEvent(mEv);
}

/**
 * @param kEv A keyboard event pointer supplied by the window.
 *
 * Passes keyboard press events to the corresponding event handler of the
 * current event.
 **/
void GLView::keyPressEvent(QKeyEvent* kEv)
{
	// pass keyboard event on to GLEffect 
	currentEffect->keyPressEvent(kEv);
}

/** @param wEv A wheel event pointer supplied by the window.
 *
 * Passes mouse wheel events to the corresponding event handler of the current
 * effect. The GLView object is set up to automatically assume focus when a
 * wheel event arrives. 
 *
 * @note The default implementation of this handler in GLEffect ignores wheel
 * events, because this is required by Qt framework to properly pass such
 * events to the parent window.
 *
 * @see GLEffect::wheelEvent() **/

void GLView::wheelEvent(QWheelEvent* wEv)
{
	//pass wheel event on to GLEffect

	/* Just a temporary hack for working around the 
	wheel event crashes. Seems that setEnabled(false) doesn't really
	deactivate the wheel events.....strange! */
	
	if(currentEffect==NULL)
		wEv->ignore();
	else currentEffect->wheelEvent(wEv);
}

/**
 * @param azimuth Azimuth angle of the viewpoint.
 * @param elevation Elevation angle of the viewpoint.
 * @param twist Twist (Rotation on Z axis) angle of the viewpoint.
 *
 * Sets the camera azimuth, elevation and twist angles and redraws the window.
 *
 * @note This function will be changed as the virtual trackball is rewritten.
 * @see setFieldOfView()
 **/

void GLView::setCameraAngles(GLdouble azimuth, GLdouble elevation, 
							   GLdouble twist)
{
	//azimuthAngle = azimuth;
	//elevationAngle = elevation;
	//twistAngle = twist;
		
	updateGL();

}

/**
 * @param fov New FOV angle to be set.
 *
 * Sets the Field of View angle and adjusts the projection matrix accordingly.
 * The window is updated after projection matrix has been adjusted.
 * 
 * @see setProjectionParams()
 **/

void GLView::setFieldOfView(GLdouble fov)
{
	fovAngle = fov;
	glMatrixMode(GL_PROJECTION);

	glLoadIdentity(); // discard previous projection trasform
	gluPerspective(fovAngle, 
			(GLdouble)(this->width())/(GLdouble)(this->height()),
			nearClipPlane, farClipPlane);

	glMatrixMode(GL_MODELVIEW);
	updateGL();
}

/**
 * @param fov New FOV angle.
 * @param nearClip New distance to the near clipping plane.
 * @param farClip New distance to the far clipping plane.
 * @param distance New distance between the camera and the world origin. 
 * 
 * Sets the perspective projection parameters. Used internally by 
 * GLEffect::init() to pass projection parameters of the current effect
 * to GLView.
 *
 * @see resizeGL()
 *
 **/
void GLView::setProjectionParams(GLdouble fov, GLdouble nearClip, 
				 GLdouble farClip, GLdouble distance)
{
	fovAngle = fov;
	nearClipPlane = nearClip;
	farClipPlane = farClip;
	viewDistance = distance;

}

/** @param orthoMode If true, the view switches to orthographic
 * projection,otherwise the view assumes pespective projection.
 *
 * Toggles the view between perspective and orthographic projection modes.
 * 
 * @see resizeGL()
 *
 **/
void GLView::changeProjectionMode(bool orthoMode)
{
	/* toggle the view between perspective and orthographic
	projection mode */

	if(orthoMode) perspectiveOn = false;
	else perspectiveOn = true;

	// update the projection matrix and redraw the scene
	resizeGL(width(), height());
	updateGL();
}

/** @param activate If true, the view switches to virtual trackball
 * mode,otherwise it will return to normal mode.
 *
 * Toggles the view between virtual trackball mode and normal viewing mode.
 * This function also remembers the previous state of mouse tracking and then
 * disables it when in virtual trackball mode. After the trackball is
 * deactivated, the previous mouse tracking state will be restored.
 * 
 * @see paintGL()
 *
 **/
void GLView::switchToTrackball(bool activate)
{
	/* toggle the view between virtual trackball mode and
	normal (user) mode */

	if(activate) 
	{
		trackballOn = true;
		setCursor(*orbitCursor);

		hadTracking = hasMouseTracking();
		if(hadTracking) setMouseTracking(false);
	}
	else
	{
		trackballOn = false;
		setCursor(crossCursor);

		setMouseTracking(hadTracking);
	}

	updateGL();
	
}

/**
 * Resets the camera orientation (but not the zoom extents/FOV)
 *
 * @see resetFOV()
 *
 **/
void GLView::resetView()
{
	rotationAngle = 0.0;

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	
	// separate modelview matrix for axis tripod 
	glPushMatrix();
	 glTranslated(0.8f,-0.8f,0.0f);
	 glGetDoublev(GL_MODELVIEW_MATRIX, currentTripodMatrix);
	glPopMatrix();
	
	if(perspectiveOn) 
	   gluLookAt(0,0,viewDistance, 0,0,0, 0,1,0);

	rotationAngle = 0.0f;
	rotationAxis[0]=rotationAxis[1]=rotationAxis[2]=0.0f;

	glGetDoublev(GL_MODELVIEW_MATRIX, currentModelviewMatrix);
	updateGL();
}

/**
 * Resets the FOV (or zoom extents when in orhtographic mode)
 *
 * @see resetView()
 *
 **/
void GLView::resetFOV()
{
	fovAngle = 50;
	orthoVolumeExtents = 2.5f;

	resizeGL(width(), height());
	updateGL();
}

/**
 * This function ask the user to specify a file name on disk
 * and uses it to save the current rendering to a file. 
 * If no name is specified or the dialog is cancelled,
 * the image will not be saved. 
 *
 * @note Only PNG images are supported at this moment
 * (being the most versatile of all formats directly supported 
 * by Qt without use of external imaging libraries)
 *
 */
void GLView::saveCurrentRenderedImage()
{
	// for file dialog exit code...
	int returnCode;

	//render current scene to a pixmap
	const QPixmap& frameToSave = this->renderPixmap(width(), 
					          height(), 
						  true);

	//construct a simple file save dialog
	QFileDialog saveDialog(parentFrame,"save",true);
	saveDialog.setCaption("Save Current Rendered Image As...");
	saveDialog.setFilter("Portable Network Graphics (*.png)" );
	
	returnCode = saveDialog.exec();
	QString fileToSave(saveDialog.selectedFile());

	// check whether a file name was really specified
	if((returnCode == QDialog::Rejected) | (fileToSave.isEmpty()))
		return;

	// if no extension was supplied, automatically append ".png"
	if(fileToSave.find(".png",0,false)==-1)
	{
		fileToSave.append(".png");
	}

	// save the image under selected file name
	if(frameToSave.save(fileToSave, "PNG")==false)
	qDebug("There was an error while to trying to save "+fileToSave);
	else
	qDebug("Sucessfully saved "+fileToSave);
			
}

/**
 * This function reports back general openGL capabilities. The output is sent 
 * to the console. 
 *
 * @remarks In the future I will probably modify this to send the report to a
 * file or a suitable dialog window.
 * 
 **/

void GLView::getOpenGLCaps()
{
	// console version for now

	// get general information about openGL implementation

	qDebug("\n\nVendor: %s", glGetString(GL_VENDOR));
	qDebug("OpenGL Version: %s", glGetString(GL_VERSION));
	qDebug("Renderer: %s", glGetString(GL_RENDERER));
	
	QString extString((const char*)glGetString(GL_EXTENSIONS) );

	// parse extension string and insert newlines
	unsigned int itr=0;
	unsigned int insPos = 0;
	
	while(itr<extString.length()-5)
	{
		insPos = extString.find(" ");
		
		extString.replace(insPos,1, "");
		extString.insert(insPos,"\n");
		itr = insPos; 
	}
	
	// report GL extensions
	qDebug("\nOpenGL extensions:\n\n%s", (const char*)extString);

	// report GLU extensions
	qDebug("GLU version: %s",gluGetString(GLU_VERSION));
	qDebug("GLU extensions: %s",gluGetString(GLU_EXTENSIONS));

	// report maximum implementation-dependent values

	/* NOTICE: this scans for capabilities than can be reported by OpenGL
	 * 1.1 for now. Some additional OpenGL 1.2 stuff will be added later */

	GLint temp=0; // used for querying integer values
	
	qDebug("\nOpen GL capabilities:\n");
	
	// general renderer capabilities
	glGetIntegerv(GL_MAX_LIGHTS, &temp);
	qDebug("\nMaximum number of lights: %d ", temp);
	
	glGetIntegerv(GL_MAX_MODELVIEW_STACK_DEPTH, &temp);
	qDebug("Maximum modelview matrix stack depth: %d", temp);
	
	glGetIntegerv(GL_MAX_PROJECTION_STACK_DEPTH, &temp);
	qDebug("Maximum projection matrix stack depth: %d", temp);

	glGetIntegerv(GL_MAX_TEXTURE_STACK_DEPTH, &temp);
	qDebug("Maximum texture matrix stack depth: %d", temp);

	glGetIntegerv(GL_MAX_CLIP_PLANES, &temp);
	qDebug("\nMaximum number of clipping planes: %d", temp);
	
	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &temp);
	qDebug("Maximum height/width of textures: %d", temp);

	//****  Framebuffer color resolution characteristics ****
	glGetIntegerv(GL_SUBPIXEL_BITS, &temp);
	qDebug("\nNumber of bits of subpixel precision: %d", temp);

	// Color buffer resolution
	glGetIntegerv(GL_RED_BITS, &temp);
	qDebug("\nColor buffer red channel depth: %d bits", temp);
	
	glGetIntegerv(GL_GREEN_BITS, &temp);
	qDebug("Color buffer green channel depth: %d bits", temp);
	
	glGetIntegerv(GL_BLUE_BITS, &temp);
	qDebug("Color buffer blue channel depth: %d bits", temp);

	glGetIntegerv(GL_ALPHA_BITS, &temp);
	qDebug("Color buffer alpha channel depth: %d bits", temp);

	// Depth buffer depth
	glGetIntegerv(GL_DEPTH_BITS, &temp);
	qDebug("\nZ-buffer depth: %d bits", temp);
	
	// Stencil buffer depth
	glGetIntegerv(GL_STENCIL_BITS, &temp);
	qDebug("Stencil buffer depth: %d bits", temp);

	// Accumulation buffer resolution
	glGetIntegerv(GL_ACCUM_RED_BITS, &temp);
	qDebug("\nAccumulation buffer red channel depth: %d bits", temp);

	glGetIntegerv(GL_ACCUM_GREEN_BITS, &temp);
	qDebug("Accumulation buffer green channel depth: %d bits", temp);
	
	glGetIntegerv(GL_ACCUM_BLUE_BITS, &temp);
	qDebug("Accumulation buffer blue channel depth: %d bits", temp);

	glGetIntegerv(GL_ACCUM_ALPHA_BITS, &temp);
	qDebug("Accumulation buffer alpha channel depth: %d bits", temp);

	
	// Other implementation-dependent values
	glGetIntegerv(GL_MAX_PIXEL_MAP_TABLE , &temp);
	qDebug("\nMaximum size of a glPixelMap translation table: %d", temp);

	glGetIntegerv(GL_MAX_NAME_STACK_DEPTH , &temp);
	qDebug("Maximum selection-name stack depth: %d", temp);

	glGetIntegerv(GL_MAX_LIST_NESTING, &temp);
	qDebug("Maximum display-list call nesting: %d", temp);

	glGetIntegerv(GL_MAX_EVAL_ORDER, &temp);
	qDebug("Maximum evaluator polynomial order: %d", temp);

	glGetIntegerv(GL_MAX_VIEWPORT_DIMS, &temp);
	qDebug("Maximum viewport dimensions: %d", temp);

	glGetIntegerv(GL_MAX_ATTRIB_STACK_DEPTH, &temp);
	qDebug("Maximum depth of the attribute stack: %d", temp);

	GLboolean temp2=false;

	glGetBooleanv(GL_AUX_BUFFERS, &temp2);
	qDebug("\nNumber of auxiliary buffers: %d", temp2);

	glGetBooleanv(GL_RGBA_MODE, &temp2);
	qDebug("GL_RGBA_MODE: %s", (temp2)?"Yes":"No");

	glGetBooleanv(GL_DOUBLEBUFFER , &temp2);
	qDebug("Double-buffered: %s", (temp2)?"Yes":"No");

	glGetBooleanv(GL_STEREO , &temp2);
	qDebug("Capable of stereoscopic rendering: %s", (temp2)?"Yes":"No");
	
	GLfloat temp3[2];
	
	glGetFloatv(GL_POINT_SIZE_RANGE, temp3);
	qDebug("\nRange (low to high) of antialiased point sizes: %.3f - %.3f",
			temp3[0], temp3[1]);

	glGetFloatv(GL_LINE_WIDTH_RANGE , temp3);
	qDebug("Range (low to high) of antialiased line widths: %.3f - %.3f",
			temp3[0], temp3[1]);

}

// **** layout-related functions ****

QSizePolicy GLView::sizePolicy() const
{
 return QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding); 
}


QSize GLView::sizeHint() const
{return QSize(600,600);}



