/*****************************************************************************\
 *                              GLFramework
\*****************************************************************************/

/*!
 * @defgroup appclasses GL Effects Framework application classes
 *
 * @brief The following classes encapsulate the framework application:
 * 
 * GLFramework is the primary application class which provides the primary
 * application window, associated controls, file loading mechanism and does all
 * top-level event handling. 
 * 
 * GLView is the openGL-based rendering window which displays the images
 * generated by the selected effect. This class also contains the functions
 * that provide the virtual trackball mode.  
 * 
 * GLEffect is the abstract base class for all user-defined effects.
 */

/*! @file GLFramework.h
 *
 *  @version 1.0 first public release
 *
 *  @brief
 *	Primary application class declarations
 *   
 *  @author Ibraguim Kouliev
 *     
 */

/*
    Copyright  2001 Ibragium Kouliev All Rights Reserved.
 
    This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License along
    with GL Effects Framework; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @class GLFramework
 *
 *  @ingroup appclasses
 *
 *  @brief Primary application class.
 *  
 * This is the primary application GUI class.  All primary gui controls, as well
 * as the openGL viewport are instantiated and placed into a common layout
 * here. This class provides functions for loading image sequences, controlling 
 * the view via a virtual trackball control, effect selection and controls.
 * Also takes care of all necessary initialization/event handling at the top level 
 * of the application.
 *  
 * @todo
 * There are a few small things left here and there. In the near future I will link in 
 * the FreeImage library so that the program can support additional file formats like TGA,
 * JPG, etc. 
 *  
 *  
 **/

#ifndef GLFRAMEWORK_H
#define GLFRAMEWORK_H

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

/* This will instruct the program to show the about box when quitting.
 * The value of USE_ABOUT_BOX is the number of milliseconds to keep
 * the about box on screen, after which the program is closed.
 */

#define USE_ABOUT_BOX 5000

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

// system library headers
#ifdef _WIN32
#include <windows.h>
#endif

#include <qwidget.h>

#ifdef USE_ABOUT_BOX
#include "aboutbox.h"
#endif

//---------------------------------------------------------------------------
//  Forward references
//---------------------------------------------------------------------------

class GLEffect; // do NOT remove!
class GLView; // do NOT remove!
class GLTrackball; //do NOT remove!

// controls classes
class QPushButton;
class QToolButton;
class QComboBox;
class QMultiLineEdit;
class QLabel;
class QCheckBox;
class QProgressDialog;

// geometry management classes
class QGrid;
class QGridLayout;
class QHGroupBox;
class QVGroupBox;
class QHBox;
class QVBox;

class QImage;
class QPixmap;
class QStringList;
class QTimer;

class QGLFormat;

//**************************************************************************
// GLFramework
//**************************************************************************
class GLFramework:public QWidget 
{
	Q_OBJECT

//---------------------------------------------------------------------------
//  Public Instance Methods
//---------------------------------------------------------------------------
public:

// --------------------------------------------------------------------------
//! @name    Creation, desctruction
// @{
	
	//! Primary class contructor 
	GLFramework(QWidget* parent=0, const char* name = 0);
	
	//! Destructor
	~GLFramework();

// --------------------------------------------------------------------------
// @} 

//! @name Public API functions
// @{

public:

	//! Adds a line to the status display.
	void addToStatus(const QString& line);

	//! Clears the contents of the status display.
	void clearStatusDisplay(void);
	
	//! Enables/disables memory tracker. 
	void enableMemTracker(bool en);

	//! Returns a pointer to the current effect object.
	GLEffect* currentGLEffect() const
	{return currentEffect;}

	//! Returns a pointer to the GL rendering window.
	GLView* glView() const
	{return glWindow;}

	//! Returns the number of loaded images in the specified list 
	unsigned int getNumImages(unsigned int listNum);

// ------------------------------------------------------------------------------
// @}

//! @name The following functions feed image data to an effect: 
// @{

	QImage* fetchPreviousImage(unsigned int listNum=0); //!<goes to previous frame in sequence
	QImage* fetchCurrentImage(unsigned int listNum=0); //!<goes to current frame in sequence
	QImage* fetchNextImage(unsigned int listNum=0); //!<goes to next frame in sequence
	QImage* fetchImage(unsigned int listNum, unsigned int imgNum); /**< @brief 
	goes to the specified frame */
	
// --------------------------------------------------------------------------
// @}


//! @name    Layout-related Functions
// @{
	
	/** @brief Returns the size policy for this widget. 
	 * See Qt documentation on how this works...*/
	QSizePolicy sizePolicy() const;

	//! Returns the size hint.
	QSize sizeHint() const;

	//! Places controls of an effect into main GUI.
	void placeEffectControlPanel(QWidget* panel);

// @}
// --------------------------------------------------------------------------

protected:

	/** @brief Used to automatically hide  user-defined controls
	 * @brief when the main window is minimized / out of view
	*/
	virtual void hideEvent(QHideEvent*);

	/** @brief Used to automatically show user-defined controls
	 * @brief when the main window is made visible
	 *
	 */
	virtual void showEvent(QShowEvent*);


// --------------------------------------------------------------------------
//! @name	Signals:
// @{ 

signals:

// @}

//! @name	Slots:
// @{

public slots:  // ...which control the following:
	
private slots:
	
	void updateMemoryTracker(); 
	//!< Updates the memory tracker indicators of the status display

// @}

// --------------------------------------------------------------------------

// --------------------------------------------------------------------------
//! @name Event handlers
// @{	

//! @remarks The onPlay(), onPause(), onStop(), onReset() handlers 

	void onPlay(); //!< Handler for the "Play" button
	void onPause();//!< Handler for the "Pause" button
	void onStop(); //!< Handler for the "Stop" button
	void onReset();//!< Handler for the "Reset" button

	void onLoadFiles();//!< Handler for the "Load Images..." button
	void onSave(); //!< Handler for the "Save Image..." button
	
	void onQuit(); //!< Handler for the "Quit" button

	void onSelectEffect(int effectNum); //!< Selects an effect

	void onViewMode(bool on); //!< Toggles between pespective/ortho view mode
	
	void onTrackball(bool on); //!< Switches to virtual trackball mode
	void onResetView(); //!< Resets the camera 
	void onResetFOV(); //!< Resets FOV / zoom extents 


// @}
// --------------------------------------------------------------------------

//---------------------------------------------------------------------------
//  Private/Protected Instance Methods
//---------------------------------------------------------------------------

protected:
	
	//! Performs any necessary post-construction initialization.
	virtual void polish();

private:
	// initialization/clean up functions
	
	//! Initializes the member controls, slot/signal connections etc. 
	void initFramework();

	//! Called when the window is destroyed
	void cleanUp();

//---------------------------------------------------------------------------
//! @name Helper Functions
// @{

	/** @brief Gets list(s) of open file names */
	void getFileNames(unsigned int listCount);

//---------------------------------------------------------------------------
// @}
	

//---------------------------------------------------------------------------
//  Private/Protected Instance Variables
//---------------------------------------------------------------------------

private:

// --------------------------------------------------------------------------
//@! name Member controls
//@{

	// **** member controls ****
	
	// primary control panel and its components
	QGrid* mainControls;

	// Control buttons
	QPushButton* resetButton;
	QPushButton* loadImagesButton;
	QPushButton* saveButton;

	QPushButton* quitButton;

	QPushButton* playButton;
	QPushButton* stopButton;
	QPushButton* pauseButton;
	
	//Effect selector box
	QHGroupBox* selectorBox;
	QComboBox* effectSelector;

	// status window
	QMultiLineEdit* statusDisplay;
	
	// memory status indicators
	QLabel* memUsageGauge;
	QLabel* bytesTakenGauge;
	
	// Amount of free bytes gauge and... 
	QLabel* bytesFreeGauge;
	
	//... a dynamic increase/decrease indicator and its icons
	QLabel* bytesFreeUpDown;
	QHBox*	bytesFreeBox;
	QPixmap* arrowUp;
	QPixmap* arrowDown;

	QLabel* bytesTotalGauge;
	QLabel* texelDataGauge;
	
	// periodical memory status update timer
	QTimer* memTicker; 
	
	// memory tracker state 
	bool memTrackerEnabled;

	// Layout helper objects
	
	// contains the status display and indicators
	QVGroupBox* statusBox;
	
	// main Grid layout object for the entire frame
	QGridLayout* mainGrid;

	//sub-grid in the side panel, required for proper positioning of controls
	QGridLayout* panelGrid;
	
	// This one contains and constraints the openGL rendering window
	QVBox* glFrame;

	// The GL rendering window
	GLView* glWindow;

	// The virtual trackball control 
	
	// Containers for v.trackball controls
	QHGroupBox* trackBox;
	QVBox* trackButtonsBox;
	
	QPushButton* cameraResetButton;
	QPushButton* fovResetButton;
	QPushButton* cameraModeButton; // toggle trackball mode
	QPushButton* projectionButton; // toggle persp./ortho mode


//@}
// --------------------------------------------------------------------------

// --------------------------------------------------------------------------

//@! name Variables
//@{

	// **** Member variables ****
 
	// GL rendering context format descriptor
	QGLFormat* glContextFormat;

	// Internal effect objects

	// the current effect class instance 
	GLEffect* currentEffect;
	
	// an array containing the effect objects
	GLEffect** effectPool;
	
	// used to check whether the program has been just started
	bool startUp;

	QImage** imageLists; // texture data objects
	QStringList* fileNameLists; // file names

	QImage* dummyImage; // empty placeholder for fallback...

	unsigned int prevNumImageLists; //previous number of image lists, 
	//required for deallocation
	
	int currentImage[3]; // frame counters
	unsigned int totalTexelBytes; // current total size of texel data 

	// **** Win32-Specific ****
	
	// info struct to be used with GlobalMemoryStatus()
	#ifdef _WIN32
		MEMORYSTATUS currentMemStatus;
		
	#endif

//@}
	
//---------------------------------------------------------------------------
//  Class methods and variables
//---------------------------------------------------------------------------


};

#endif



		

