/*****************************************************************************\
 *                              GLFramework
\*****************************************************************************/

/*! @file GLFramework.cpp
 * 
 *  
 *
 *  @brief
 *	Primary application class implementation
 *   
 *
 *  @author Ibraguim Kouliev
 *  
 * @attention Some bug traps in the following section are temporary and we 
 * might decide to remove them later. So try to check you data anyway, 
 * before you use it.
 *  
 *  
 */

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

// Standard system library headers
#include <stdlib.h>

// Qt headers
#include <qwidget.h>
#include <qapplication.h>
#include <qdialog.h>
#include <qpopupmenu.h>
#include <qgl.h>


#include <qlayout.h>
#include <qvbox.h>
#include <qhbox.h>
#include <qvgroupbox.h>
#include <qhgroupbox.h>
#include <qgrid.h>

#include <qcombobox.h>
#include <qmultilineedit.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qprogressdialog.h>
#include <qprogressbar.h>

#include <qpixmap.h>
#include <qimage.h>

#include <qfiledialog.h>
#include <qstringlist.h>

// local headers
#include "icons.h"

#include "GLFramework.h"
#include "GLView.h"
#include "GLEffect.h"

// an example of a typical effect
#include "SimpleEffect.h"

// student-submitted classes
#include "../accum_buffer/AccumEffect.h"
#include "../water_waves/Waterwaves.h"
#include "../shadowcaster/ShadowCaster.h"
#include "../b_spline_warp/CBSplineWrapperEffect.h"
#include "../contour_warp/ContourWarpEffect.h"


/** 
 * @param parent A pointer to the parent widget (In this case 0, because it 
 * is top-level widget)	
 * 
 * @param name A name for this widget, used \e only internally by Qt. Can be 
 * safely set to 0. Both parameters are passed on to the QWidget constructor. 
 *	
 * This the primary class constructor. Most of GUI initialization is actually
 * done in initFramework(), while the constructor instantiates the effect 
 * objects and does other required initialization. 
 * See implementation for details.
 * 
 * @throw 
 *  Contains pointer checks. Will issue an error and quit if there isn't 
 *  enough memory available.
 *
 * @todo
 *	New effects will be gradually added here as they are ready  
 *
 * @see
 *   initFramework()
 **/
GLFramework::GLFramework(QWidget* parent, const char* name):
			QWidget(parent, name)
{

	prevNumImageLists = 0;
	
	// initialize all window components and slot/signal connections
	initFramework();

	// set initial run state
	startUp=true;

	qDebug("Framework initialized....");


	// create effect pool and effects
	currentEffect = 0;
	effectPool= new GLEffect* [20]; // max 20 effects for now

	/* ATTENTION: this is one of major areas, which needs to be modified.
	Follow the pattern and add your effect(s) here. */

	effectPool[0] = new AccumEffect(this);
	CHECK_PTR(effectPool[0]);

	effectSelector->insertItem("Accumulation Buffer Effects");

	effectPool[1] = new Waterwaves(this);
	CHECK_PTR(effectPool[1]);

	effectSelector->insertItem("Water wave effects");

	effectPool[2] = new ShadowCaster(this);
	CHECK_PTR(effectPool[2]);

	effectSelector->insertItem("Lights & Shadows");
	
	effectPool[3] = new CBSplineWrapperEffect(this);
	CHECK_PTR(effectPool[3]);

	effectSelector->insertItem("B-Spline Warp");
	
	effectPool[4] = new ContourWarpEffect(this);
	CHECK_PTR(effectPool[4]);

	effectSelector->insertItem("Contour Warping");

	effectPool[5] = new SimpleEffect(this);
	CHECK_PTR(effectPool[5]);

	effectSelector->insertItem("Simple Effect (from Tutorial)");

	// init image & filename lists pointers
	imageLists=0;
	fileNameLists=0;
	
	// init frame counters
	currentImage[0]=currentImage[1]=currentImage[2]=0;

	totalTexelBytes=0;

	// activate memory tracker
	enableMemTracker(true);
	
	//create an empty placerholder QImage object for fallback situations
	dummyImage = new QImage();

}

/**  
 * Destroys the widget. Most cleanup is done via cleanUp(). 
 *
 * @warning
 *   May cause a crash if you prematurely deallocate imageLists, fileNameLists
 *   or any member controls (buttons, labels etc.), because those vars are 
 *   eventually freed by cleanUp(). 
 *
 * @see cleanUp()
 *
 **/

GLFramework::~GLFramework() 
{
	// clean up everything
	cleanUp();

}

/**   
 *
 * This function performs any post-construction initialization, just before
 * the window is displayed. Some additional code may be included here later,
 * for now it's empty. 
 *
 **/
void GLFramework::polish()
{
	
}

/**   
 *
 * This carries out all essential GUI initialization by creating the 
 * components of the main window and establishing any necessary 
 * slot/signal connections. 
 *
 *
 * @see GLFramework::GLFramework()
 *
 **/
void GLFramework::initFramework()
{
	// primary window initialization... 
	
	// create main GUI components and add them to the layout grid
	
	// primary button group - grid with 2 columns
	mainControls =	new QGrid(2, this,"main_controls");
	mainControls -> setSpacing(5);
	mainControls -> setMargin(5);

	mainControls -> setFrameStyle(QFrame::Box | QFrame::Sunken);

	// create icons for buttons
	QPixmap play_ic(const_cast<const char**>(PLAY_ICON) );
	QPixmap open_ic(const_cast<const char**>(OPEN_ICON) );
	QPixmap quit_ic(const_cast<const char**>(QUIT_ICON) );
	QPixmap stop_ic(const_cast<const char**>(STOP_ICON) );
	QPixmap pause_ic(const_cast<const char**>(PAUSE_ICON) );
	QPixmap reset_ic(const_cast<const char**>(RESET_ICON) );
	QPixmap save_ic(const_cast<const char**>(SAVE_ICON) );

	// ....and buttons
	loadImagesButton = new QPushButton(open_ic,"Load Image(s)..",
					   mainControls, "load_images");

	playButton 	 = new QPushButton(play_ic, "Play",mainControls, "play");
	saveButton   = new QPushButton(save_ic, "Save Image...",mainControls,"save");
	stopButton	 = new QPushButton(stop_ic, "Stop",mainControls, "stop");
	resetButton	 = new QPushButton(reset_ic,"Reset",mainControls, "reset");
	pauseButton	 = new QPushButton(pause_ic,"Pause", mainControls, "pause");
	quitButton	 = new QPushButton(quit_ic, "Quit",mainControls, "quit");


	// create the effect selector combo box 
	selectorBox = new QHGroupBox("Select effect type:",this, 0);
	effectSelector  = new QComboBox(false, selectorBox, "selector");
	selectorBox -> setFrameStyle(QFrame::Box | QFrame::Sunken);

	// establish signal/slot connection 

	connect(quitButton, SIGNAL(clicked()), this, SLOT(onQuit())	);
	connect(playButton, SIGNAL(clicked()), this, SLOT(onPlay())	);
	connect(resetButton,SIGNAL(clicked()), this, SLOT(onReset()));
	connect(stopButton, SIGNAL(clicked()), this, SLOT(onStop())	);
	connect(pauseButton,SIGNAL(clicked()), this, SLOT(onPause()));
	connect(saveButton, SIGNAL(clicked()), this, SLOT(onSave()) );

	connect(loadImagesButton, SIGNAL(clicked()), 
			this, SLOT(onLoadFiles()));


	connect(effectSelector, SIGNAL(activated(int)), 
			this, SLOT(onSelectEffect(int)) );

	// create box containing the memory status indicators and display
	statusBox = new QVGroupBox("Status", this,"status_box");
	statusBox->setMargin(10);

	// initialize indicators
	bytesTakenGauge = new QLabel("Bytes taken:",statusBox);
	
	/* put an QHBox around bytesFreeGauge and bytesFreeUpDown 
	- this might look freaky... :) Need it to keep them both on a 
	single line */

	bytesFreeBox = new QHBox(statusBox);
	bytesFreeBox->setSpacing(10);

	bytesFreeGauge = new QLabel("Free bytes:",bytesFreeBox);
	bytesFreeUpDown= new QLabel("", bytesFreeBox);

	//create icons for up/down indicator
	arrowUp = new QPixmap(const_cast<const char**>(ARROW_UP));
	arrowDown = new QPixmap(const_cast<const char**>(ARROW_DOWN));
	
	// the rest...
	bytesTotalGauge = new QLabel("Total system memory:",statusBox);
	memUsageGauge = new QLabel("Memory load:",statusBox);
	texelDataGauge = new QLabel("Total texel data size:", statusBox);

	// create memory update timer
	memTicker = new QTimer(this);

	connect(memTicker, SIGNAL(timeout()), 
			this, SLOT(updateMemoryTracker()));

	// create status window and set its appearance
	statusDisplay = new QMultiLineEdit(statusBox, "status");
	statusDisplay -> setFrameStyle(QFrame::Panel | QFrame::Sunken);
	
	// this is meant only for showing information, so disable editing
	// capability
	statusDisplay->setReadOnly(TRUE);
	//statusDisplay->setFixedVisibleLines(10);

	// create the virtual trackball control and its child controls

    trackBox = new QHGroupBox("Camera and view mode control",this);

	QPixmap trackball_ic(const_cast<const char**>(TRACKBALL_ICON));

	cameraModeButton = new QPushButton(trackball_ic,"",trackBox);
	cameraModeButton->setFixedSize(40,40);
	cameraModeButton->setToggleButton(true);

	trackButtonsBox = new QVBox(trackBox);
	trackButtonsBox->setSpacing(5);

	cameraResetButton = new QPushButton("Reset camera", trackButtonsBox);
	fovResetButton = new QPushButton("Reset FOV", trackButtonsBox);
	projectionButton = new QPushButton("Perspective view", trackButtonsBox);

	projectionButton->setToggleButton(true);
	
	// hook up the virtual trackball mode switch
	connect(cameraModeButton, SIGNAL(toggled(bool)),
			this, SLOT(onTrackball(bool)) );

	// hook up the perspective/ortho switch 
	connect(projectionButton, SIGNAL(toggled(bool)),
			this, SLOT(onViewMode(bool)) );

	// hook up the reset view and reset FOV buttons
	connect(cameraResetButton, SIGNAL(clicked()),
			this, SLOT(onResetView()) );

	connect(fovResetButton, SIGNAL(clicked()),
			this, SLOT(onResetFOV()) );

	// now create the left panel layout grid

	/*notice: this layout has to be created as a top-level layout,
	hence the 0 instead of 'this' as the 1st param to the constructor,
	otherwise it would conflict with the main layout grid */

	panelGrid = new QGridLayout(0, 5, 1, 5, -1, "left_panel");

	// add general controls to the panel grid
	
	panelGrid->addWidget(mainControls,0,0, Qt::AlignTop);
	panelGrid->addWidget(selectorBox,1,0,Qt::AlignLeft | Qt::AlignTop);
	panelGrid->addWidget(statusBox,4,0, Qt::AlignBottom);
	
	// add trackball
	panelGrid->addWidget(trackBox,3,0,Qt::AlignVCenter | Qt::AlignHCenter);

	// set panel grid behaviour... not fine-tuned for now
	panelGrid->setRowStretch(0,0);
	panelGrid->setRowStretch(2,0);
	panelGrid->setRowStretch(1,0);

	/* necessary to avoid drastic layout distortion during 
	 child GUI hide/show operations */

	panelGrid->setRowStretch(4,1);
	panelGrid->setRowStretch(3,1);
	
	// this box contains (and constraints) the GL rendering window
	glFrame = new QVBox(this,"context_box");
	glFrame->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	
	//set the openGL rendering context format
    glContextFormat = new QGLFormat();

	glContextFormat->setAlpha(true);
	glContextFormat->setRgba(true);
	glContextFormat->setDepth(true);
	glContextFormat->setDirectRendering(true); 
	
	// create the openGL rendering window
	glWindow = new GLView(glFrame, this, glContextFormat); 

	// init main window layout grid
	mainGrid = new QGridLayout(this,1,2,10,-1,"grid");
		
	// add GUI components to the layout
	mainGrid->addWidget(glFrame,0,0);

	// add left panel sub-grid to main layout - row #1, column #2
	mainGrid->addLayout(panelGrid,0,1);

	// set main layout grid behaviour
	mainGrid->setColStretch(1,0);
	mainGrid->setColStretch(0,1);


	/* Ok, this is IMPORTANT: 
	
	  Here's how this  QGridLayout beast operates:

	First you set the # of rows/columns, in this case 1 row and 2 columns,
	then add in some widgets. If left in this state, each column will by
	default grow/shrink depending on the size policy of a given widget(s).
	If however the column stretch is set to anything other that 0, that
	particular column will grow to provide its widget with the space that
	the widget requires to expand to its full size (to the best of my
	knowledge for now...), whereas setting the column stretch to 0 will
	keep the affected column from getting too wide.  I.e. here to achieve
	the max. size of the GL viewport it's column stretch is set to 1, and
	the 'controls' column's stretch is set to 0 so that the final resulting
	configuration is with full-sized GL view side by side with the controls
	column, the controls column being stretched to maximum width possible
	under given horizontal size of the window.  The size policy of the GL
	viewport in both directions is set to "Expanding".
	
	Here are some important types of size policies and their explanations:
	
	  Fixed - the sizeHint() is the only acceptable alternative, so widget
	  can never grow or shrink 
	  (eg. the vertical direction of a pushbutton). 
	
	  Minimum - the sizeHint() is minimal, and sufficient. The widget can 
	  be expanded, but there is no advantage to it being larger 
	  (eg. the horizontal direction of a pushbutton). 
	
	  Maximum - the sizeHint() is a maximum. The widget can be shrunk 
	  any amount without detriment if other widgets need the space 
	  (eg. a separator line). 
	
	  Preferred - the sizeHint() is best, but the widget can be shrunk 
	  below that and still be useful. The widget can be expanded, but 
	  there is no advantage to it being larger than sizeHint() 
	  (the default QWidget policy). 
	
	  Expanding - the sizeHint() is a sensible size, but the widget can 
	  be shrunk below that and still be useful. The widget can make use 
	  of extra space, so it should get as much space as possible 
	  (eg. the horizontal direction of a slider). 
		
	*/

	// --- Final init steps ---

	/* Tell the GL widget to atomatically assume focus upon 
	any wheel events. */

	glWindow->setFocusPolicy(QWidget::WheelFocus);

	/* Because upon startup no effect has been selected yet, 
	deactive the main controls (except quit) initially so that they don't
	acidentally acess pointers that have not yet been initialized.	
	*/
	
	glWindow->setEnabled(false);
	
	playButton->setEnabled(false);
	pauseButton->setEnabled(false);
	stopButton->setEnabled(false);
	loadImagesButton->setEnabled(false);
	resetButton->setEnabled(false);
	saveButton->setEnabled(false);

	cameraModeButton->setEnabled(false);
	cameraResetButton->setEnabled(false);
	fovResetButton->setEnabled(false);
	projectionButton->setEnabled(false);
	
	quitButton->setEnabled(true); 
 
}

/**   
 * This function performs all necessary clean-up upon exiting the program
 *
 * @warning
 *   May cause a crash if you prematurely deallocate imageLists, 
 *   fileNameLists or any member controls (buttons, labels etc.). 
 *   Be careful with those pointers.
 *   
 * @todo Some additional cleanup may be added here later
 *
 **/
void GLFramework::cleanUp()
{
	/*still not sure whether I have to manually delete 
	the child widgets upon destruction... the Qt documentation claims that
	the child widgets are destroyed automatically with the parent
	object....*/

	delete mainControls;
	delete statusBox;
	delete statusDisplay;
	delete glWindow;

    	delete[] imageLists;
	delete[] fileNameLists;
	delete[] effectPool;

	// delete temporary icon objects
	delete arrowUp;
	delete arrowDown;

}

// slots

/**  
 * Updates memory tracker indicators with current values. The green/red arrow
 * indicator is simply for additional visual reference and shows whether the 
 * amount of free memory is currently increasing or decreasing. 
 *
 * @attention This function currently shows the \b global memory status, i.e.
 * it doesn't reflect the amount of memory consumed by the program 
 * (except the \b texel \b data size indicator). It should not be considered 
 * precise, and should be used rather as a debugging aid for
 * checking for possible memory leaks etc.
 *
 * @warning
 *   Available only on Win32 platforms for now. UNIX part will be added later.
 *
 * @pre See above
 *   
 * @todo UNIX part of implementation
 *
 **/

void GLFramework::updateMemoryTracker()
{
	// temporary variables
	unsigned long int	prevBytesFree;
	unsigned long int	prevBytesTaken;
	unsigned int		prevLoad;
		
	/* section only defined for Win32 targets, 
	uses Win32-specific GlobalMemoryStatus() function & struct */

#ifdef _WIN32
	
	// remember previous values for comparison
	prevBytesFree = currentMemStatus.dwAvailPhys;
	prevBytesTaken = currentMemStatus.dwTotalPhys - prevBytesFree;
	prevLoad = currentMemStatus.dwMemoryLoad;

	GlobalMemoryStatus(&currentMemStatus);
	
	// display current memory load
	QString memLoad = "Memory load: "+
		QString::number(currentMemStatus.dwMemoryLoad);
	
	memLoad += "%    ";
	memLoad += (int(currentMemStatus.dwMemoryLoad-prevLoad<0))?"":"+";
	memLoad += QString::number
		(int(currentMemStatus.dwMemoryLoad-prevLoad))+" %";

	memUsageGauge->setText(memLoad);

	// display current amount of free (unallocated) memory
	QString bytesFree = "Free memory: "+
		QString::number(currentMemStatus.dwAvailPhys/1024);
	
	bytesFree += " KBytes    ";
	bytesFreeGauge->setText(bytesFree);


	bytesFreeUpDown->setPixmap(
		(long int(currentMemStatus.dwAvailPhys-prevBytesFree)<0)?
		 (*arrowDown):
		 (long int(currentMemStatus.dwAvailPhys-prevBytesFree)>0)?
		 (*arrowUp):0);
	
	// display current amount of used memory
	QString bytesTaken = "Used memory: "+
		QString::number
		((currentMemStatus.dwTotalPhys-currentMemStatus.dwAvailPhys)/1024);
	
	bytesTaken += " KBytes";
	bytesTakenGauge->setText(bytesTaken);


	bytesTotalGauge->setText("Total physical memory: "+
		QString::number(currentMemStatus.dwTotalPhys/1024)+" KBytes");

	texelDataGauge->setText("Total size of texel data: "+
		QString::number(totalTexelBytes/1024) + " KBytes");

#endif

// Later on here will be the section for UNIX targets....

}

// event handlers for main control buttons

void GLFramework::onPlay()
{
	currentEffect->play(); 
}

void GLFramework::onStop()
{
	currentEffect->stop(); 
}

void GLFramework::onPause()
{
	currentEffect->pause();
}

void GLFramework::onReset()
{
	currentEffect->reset();
}

void GLFramework::onTrackball(bool on)
{
	glWindow->switchToTrackball(on);
}

void GLFramework::onViewMode(bool on)
{
	if(on) projectionButton->setText("Orthographic view");
	else projectionButton->setText("Perspective view");

	glWindow->changeProjectionMode(on);
}

void GLFramework::onResetView()
{
	glWindow->resetView();
}

void GLFramework::onResetFOV()
{
	glWindow->resetFOV();
}

void GLFramework::onSave()
{
	glWindow->saveCurrentRenderedImage();
}

/** 
 * Called when "Load Images..." button is clicked or a new effect requiring 
 * a different set of textures is selected. This calls the file loading 
 * dialog, up to 3 times, depending on requirements of the current effect, 
 * and loads the specified files into memory. The loaded files are 
 * automatically converted into an OpenGL-compatible image format and can 
 * be subsequently acessed through the fetch*Image() family of functions.
 * The scene is then re-rendered with new textures applied.
 *  
 * @throw Contains pointer checks. Will issue an error and quit if there 
 * isn't enough memory available.
 *  
 * @pre
 *   The information about the number of required texture sets is picked up
 *   from the effectInfo struct of the current effect, so make sure you 
 *   supply correct values in requiredImageLists and needNewTextures members
 *   of this struct. If you don't need any textures, set the 
 *   requiredImageLists to 0 and needNewTextures to \b false. Optionally, 
 *   you can provide more descriptive names for the file dialogs by setting
 *   fileDialogNames[0],[1], and [2] to desired text, 
 *   for example \b effectInfo->fileDialogNames[0] = "Load texture images" .
 *
 *
 * @warning
 * There is currently only a simple protection available against uninitialized
 * image data arrays. If you want to load the images, and cancel file loading
 * in one or more file dialogs, or do not specify any images, the program will
 * automatically fall back and provide one empty QImage object for each list 
 * that was not loaded. Thus. subsequent access by fetch*Image() functions 
 * will return these objects instead of dangling pointers, thereby avoiding 
 * possible crash situations. Your objects will then simply be rendered 
 * with no texture(s).\n <b>Be advised</b> that this protection is 
 * probably not <b>100 %</b> bulletproof, so be careful and do not overly
 * rely on this protection mechanism....
 *  
 * 
 * @todo
 *   Streaming mode
 * 
 * @see
 *   fetchNextImage(),fetchPreviousImage(),fetchCurrentImage(),
 *    fetchImage(),onSelectEffect()
 *
 **/

void GLFramework::onLoadFiles()
{
    // no idea how to introduce a wildcard for numbered files...for now
	
    // if no image lists are required, don't bother processing further
	if(currentEffect->numImageLists()==0) return;

	/*WARNING: There may or may not exist a possible memory leak within 
	 this function. I'm not yet able to tell whether it will 
	 really show up anywhere.... */

	//explicitly de-allocate previous file names & image lists

	if(imageLists != 0)
	{
		for(unsigned int i=0;i<prevNumImageLists; i++)
		{
			for(unsigned int j=0;j<fileNameLists[i].count();j++)
			{
			  imageLists[i][j].reset();
			}
		}
		delete[] imageLists;
		imageLists = 0;
	}

	if(fileNameLists != 0)
	{
		for(unsigned int i=0;i<prevNumImageLists;i++)
		{
			fileNameLists[i].clear();
		}
			
		delete[] fileNameLists;
		fileNameLists = 0;
	} 

	// create temporary buffer for image loading
	QImage* buffer = new QImage();

	totalTexelBytes = 0; //reset texel data size counter
	
	// get the number of required image lists from the effect
	unsigned int numImgLists = currentEffect->numImageLists();

	//save this as previous number of image lists for future use as well
	prevNumImageLists = numImgLists;

	qDebug("Number of image lists: %d", numImgLists);

	// temporary bug trap 
	if(numImgLists>3)
		qFatal("More than 3 different image lists specified!");

	// create file name lists & image lists
	fileNameLists = new QStringList[numImgLists];
	CHECK_PTR(fileNameLists);

	imageLists = new QImage* [numImgLists];
	CHECK_PTR(imageLists);

	// load image files 
	unsigned int listNumber;

	for(listNumber=0; listNumber<numImgLists; listNumber++)
	{
		/* Get the list of open file names. Don't know yet whether 
		GCC will complain about these multiple "" constants... */

		fileNameLists[listNumber] =

			QFileDialog::getOpenFileNames(
			"All graphic files (*.png *.bmp *.jpg);;"
			"Portable Network Graphics(*.png);;"
			"Windows Bitmap (*.bmp);;"
			"JPEG images (*.jpg)",
			QString::null,this, "files", 
			currentEffect->dialogName(listNumber) );

		fileNameLists[listNumber].sort();

		//create at least one empty QImage object for protection....
		if(fileNameLists[listNumber].count() == 0)
		{
			imageLists[listNumber]=new QImage[1];
			CHECK_PTR(imageLists[listNumber]);

			qWarning("Warning: no images were specified or file loading\n"
				 "was cancelled for image list # %d \n" 
				 "Falling back to single default QImage object...",
				 listNumber+1);
		}
		else
		{
		
			// Create the corresponding image list
			imageLists[listNumber]=
				new QImage[fileNameLists[listNumber].count()];
			
			CHECK_PTR(imageLists[listNumber]);
			qDebug("\nList %d contains: ",listNumber);
		}
		
		unsigned int itr=0;
		unsigned int totalBytes=0;

		//create a progress dialog for possible lengthy 
		//loading operations.
		
		QProgressDialog progressDialog
			("Loading images...",
			"Cancel",
			fileNameLists[listNumber].count(),
			this, "progress", TRUE);
		
		// This is to ensure that the loading dialog is always shown.
		progressDialog.setMinimumDuration(0);
			
		while(itr<fileNameLists[listNumber].count())
		{
			//show current progress and continue loading
			progressDialog.setProgress(itr);
			progressDialog.setGeometry
				(this->width()/2,this->height()/2,200,100);
			qApp->processEvents();
		
			
			// load next image file
			if( (buffer->load((const QString&)fileNameLists[listNumber][itr]))==false)
			{
				// check for errors
				qFatal("Could not load file %s !",
					(const char*)fileNameLists[listNumber][itr]);
			}
		
			/* if no original textures are required,
			 generate texture and convert to 
			 openGL-compatible format */

			if(currentEffect->needRawTextures() == false)
			{
				imageLists[listNumber][itr] =
					QGLWidget::convertToGLFormat(*buffer);
			}
			else
			{
				/*Do not convert to openGL-compatible format 
				  and simply copy loaded image into the 
				  image list */

				imageLists[listNumber][itr] = buffer->copy();
			}
			
			totalBytes += imageLists[listNumber][itr].numBytes();
	
			// add this texture's size to total texel 
			// data size counter
			totalTexelBytes += imageLists[listNumber][itr].numBytes();

			itr++;
		
		}

		//Set progress to 100% when finished.
		progressDialog.setProgress(fileNameLists[listNumber].count());

		//Show total size of loaded pixel data for this list
		qDebug("Total size of pixel data: %d Bytes (%d KBytes)", 
				totalBytes, (int)(totalBytes/1024) ); 
	}

	delete buffer;

	// tell the current effect that new texture data has been loaded.
	currentEffect->reloadTextures();

}

/**   
 * @param listCount  specifies the number of required  file name lists
 *
 * @return None. Generates file name lists from the file loading dialogs
 * 
 * @note This function is \b not used in the actual program yet. It will like
 * be utilized for the streaming mode (still working on it). It shows the file
 * loading dialog(s), just like onLoadFiles(), but only creates the file name
 * lists, without actually loading any images into memory. 
 *
 * @throw  Contains pointer checks. Will issue an error and quit if there 
 * isn't enough memory available.
 *
 * @todo
 *   Streaming mode....
 **/

void GLFramework::getFileNames(unsigned int listCount)
{
		// create file name lists & image lists
	fileNameLists = new QStringList[listCount];
	CHECK_PTR(fileNameLists);

	// load image files 
	unsigned int listNumber;

	for(listNumber=0; listNumber<listCount; listNumber++)
	{
		/* Get the list of open file names. Don't know yet whether 
		GCC will complain about these multiple "" constants... */

		fileNameLists[listNumber] =

			QFileDialog::getOpenFileNames(
			"All graphic files (*.png *.bmp *.jpg);;"
			"Portable Network Graphics(*.png);;"
			"Windows Bitmap (*.bmp);;"
			"JPEG images (*.jpg)",
			QString::null,this, "files", 
			currentEffect->dialogName(listNumber) );

		fileNameLists[listNumber].sort();

		qDebug("\nList %d contains: ",listNumber);
			
	}
}

/**  
 * @param listNum  Specifies the number of the image list to be acessed.
 * Allowed parameter range is <b> 0 - 2 </b>. 
 *  @param imgNum Specifies the image to be retrieved. Allowed parameter
 *  range is <b> 0 - (number of images in the list) -1</b>. 	 
 *
 * @return  A pointer to the retrieved image. Will return an empty QImage
 * placeholder if called and the current  effect does not require any 
 * textures, i.e. its numImageLists is \b 0.  
 * 
 * Fetches the image specified by imgNum from the list specified by listNum. 
 *
 * @throw 
 *  Contains checks for possible out-of-bounds errors. Will issue a suitable
 *  error and quit the apllication. 
 *
 * @warning See warning for onLoadFiles(). 
 * @pre  Properly initialized and loaded image lists
 *   
 * @todo
 *   Streaming mode....
 *
 * @bug The description in \b warning might be considered a bug by some....
 *
 * @remarks The reason for returning an empty QImage object if the 
 * numImageLists of an effect is 0 is to provide at least a rudimentary 
 * (for now) protection mechanism. Consider a case where you specify an effect
 * that doesn't need any textures and yet accidentally forget and try to 
 * access the image lists in your effect implementation. What will happen?\n
 * Thus, with at least an empty QImage object to operate on, you will easily
 * avoid crashes. Same applies for all other fetch*Image() functions. 
 * See \b warning section of onLoadFiles() for additional precautions....

  @see fetchNextImage(), fetchPreviousImage(), fetchCurrentImage()
 *
 **/
 
QImage* GLFramework::fetchImage(unsigned int listNum, unsigned int imgNum)
{
	// error traps
	if(currentEffect->numImageLists()==0) return dummyImage;

	// list number out of bounds
	if(listNum>2)
		qFatal("Image list number out of allowed range (0-2) !");
	
	// image number out of bounds
	if(imgNum > fileNameLists[listNum].count()-1)
		qFatal("Image number out of maximum sequence range (0-%d)!",
				fileNameLists[listNum].count()-1);
	
	// update currentImage counter
	currentImage[listNum] = imgNum;

	/* Protection against uninitialized image lists. If no images were
	specified during loading or the loading dialog(s) were cancelled, 
	the program will automatically fall back and  use a single 
	default (empty) QImage object as data */
	
	if(fileNameLists[listNum].count() == 0) return &imageLists[listNum][0];

	return &imageLists[listNum][imgNum];
}

/**  
 * @param listNum  Specifies the number of the image list to be acessed. 
 * Allowed parameter range is <b> 0 - 2 </b>. Default value of \b 0 is 
 * suitable for effects using only one set of textures (i.e. one image list). 
 * 
 * @return  A pointer to the retrieved image. Will return \b 0 if called and
 * the current effect does not require any textures, i.e. its numImageLists 
 * is \b 0.  
 * 
 * Retrieves the previous image (relative to the current one) from the 
 * sequence of the specified image list. The default behaviour of this 
 * function is as follows: if with the next call to it the \b currentImage 
 * counter 'underflows' i.e. is less than \b 0 that counter is set to the 
 * number of the \b last image in the sequence - the function loops 
 * \b backwards through the image sequence during animation.
 *
 * @throw 
 *  Contains checks for possible out-of-bounds errors. Will issue a suitable
 *  error and quit the apllication. 
 *
 * @warning See fetchImage()

 * @pre Properly initialized and loaded image lists.
 *   
 * @todo
 *   Streaming mode....
 *
 * @bug See fetchImage(), fetchNextImage(), fetchCurrentImage()
 *
 **/

QImage* GLFramework::fetchPreviousImage(unsigned int listNum)
{
	if(currentEffect->numImageLists()==0) return dummyImage;

	// error trap
	// list number out of bounds
	if(listNum>2)
		qFatal("Image list number out of allowed range (0-2) !");


	currentImage[listNum]--;
	
	if(currentImage[listNum] < 0) 
		currentImage[listNum] = fileNameLists[listNum].count()-1;

	/* Protection against uninitialized image lists. If no images were
	specified during loading or the loading dialog(s) were cancelled, 
	the program will automatically fall back and  use a single 
	default (empty) QImage object as data */
	
	if(fileNameLists[listNum].count() == 0) return &imageLists[listNum][0];
	
	return &imageLists[listNum][currentImage[listNum]];

}

/**  
 * @param listNum  Specifies the number of the image list to be acessed. 
 * Allowed parameter range is <b> 0 - 2 </b>.Default value of \b 0 is suitable
 * for effects using only one set of textures (i.e. one image list). 
 * 
 * @return  A pointer to the retrieved image. Will return \b 0 if called and
 * the current effect does not require any textures, i.e. its numImageLists
 * is \b 0.  
 * 
 * Retrieves the current image from the sequence of the specified image list. 
 *
 * @throw 
 *  Contains checks for possible out-of-bounds errors. Will issue a suitable
 *  error and quit the apllication. 
 *
 * @warning See fetchImage()
 *
 * @pre  Properly initialized and loaded image lists
 *   
 * @todo
 *   Streaming mode....
 *
 * @bug See fetchImage(), fetchNextImage(), fetchPreviousImage()
 *
 **/

// this is pretty damn self-explanatory....
QImage* GLFramework::fetchCurrentImage(unsigned int listNum)
{
	if(currentEffect->numImageLists()==0) return dummyImage;

	// list number out of bounds
	if(listNum>2)
		qFatal("Image list number out of allowed range (0-2) !");

	/* Protection against uninitialized image lists. If no images were
	specified during loading or the loading dialog(s) were cancelled, 
	the program will automatically fall back and  use a single 
	default (empty) QImage object as data */
	
	if(fileNameLists[listNum].count() == 0) return &imageLists[listNum][0];

	return &imageLists[listNum][currentImage[listNum]];
}

/**  
 * @param listNum Specifies the number of the image list to be acessed. Allowed
 * parameter range is <b> 0 - 2 </b>. Default value of \b 0 is suitable for 
 * effects using only one set of textures (i.e. one image list). 
 * 
 * @return  A pointer to the retrieved image. Will return \b 0 if called and 
 * the current 
 effect does not require any textures, i.e. its numImageLists is \b 0.  
 * 
 * Retrieves the next image from the sequence of the specified image list.  
 * The default behaviour of this function is as follows: if with the next call 
 * to it the \b currentImage counter 'overflows' i.e. is greater than the 
 * number of corresponding texture objects, that counter is set back 
 * to \b 0. - the function loops through the image sequence during animation.
 *
 * @throw 
 * Contains checks for possible out-of-bounds errors. Will issue a suitable
 * error and quit the apllication. 
 *
 * @warning See fetchImage()
 *
 * @pre Properly initialized and loaded image lists
 *   
 * @todo
 *   Streaming mode....
 *
 * @bug See fetchImage(), fetchPreviousImage(), fetchCurrentImage()
 *
 **/

QImage* GLFramework::fetchNextImage(unsigned int listNum)
{
	// a simple protection in case an effect didn't require any textures
	if(currentEffect->numImageLists()==0) return dummyImage;

	// list number out of bounds
	if(listNum>2)
		qFatal("Image list number out of allowed range (0-2) !");

	currentImage[listNum]++;
	
	if(currentImage[listNum] > 
		static_cast<int>(fileNameLists[listNum].count()-1)) 
		currentImage[listNum] = 0;
	
	/* Protection against uninitialized image lists. If no images were
	specified during loading or the loading dialog(s) were cancelled, 
	the program will automatically fall back and  use a single 
	default (empty) QImage object as data */
	
	if(fileNameLists[listNum].count() == 0) return &imageLists[listNum][0];

	return &imageLists[listNum][currentImage[listNum]];

}

/** 
 * @param listNum   Specifies the image list to be acessed.
 *
 * @return
 *   Number of images in the specified list. 
 *
 * @throw Contains checks for possible out-of-bounds errors. Will issue a
 * suitable error and quit the apllication. 
 * 
 * @warning
 * The returned number is the real full number of images in the list. 
 * If you want to use it in say, a loop for working with the image list,
 * use <b>this number - 1</b>, or else you will get an "off by 1" error 
 * and likely a crash, too.
 *
 * @pre
 *   Properly initialized and loaded image lists
 * 
 **/

unsigned int GLFramework::getNumImages(unsigned int listNum)
{
	if(listNum>2)
		qFatal("Image list number out of allowed range (0-2) !");
	return fileNameLists[listNum].count();

}

/**
 * Called when the "Quit" button is clicked. Presents the user with a 
 * "yes/no" dialog box asking whether or not he/she really wants to exit
 * the application. Will stop the animation of the currently active effect
 * before showing the dialog box.
 *    
 * @todo
 *   Fix that damn dialog box...
 *
 * @bug
 *   For some reason some modal dialog widgets in Qt such as dialog boxes 
 *   don't properly recieve move() and setGeometry() commands sometimes. As 
 *   the result, depending on the current (operating system????) conditions,
 *   the aforementioned "yes/no" box might or might not sometimes appear at 
 *   the upper-left corner of the main window, or at coordinates of another
 *   recently displayed dialog window... I have yet to find a fix for this.
 *
 * @see
 *   onStop() (perhaps...)
 *
 **/

void GLFramework::onQuit()
{
	//stop animation

	// (only if there's really an effect in progress)
	if(currentEffect!=0)
	this->onStop();

	QPixmap icon(const_cast<const char**>(QUIT_ICON));
	
	// causes a message box asking the user whether he/she really wishes
	// to quit
	QMessageBox* quit_box = new QMessageBox("Wrapping up..", 
			"Do you really wish to quit?",
			QMessageBox::Critical,
	    		QMessageBox::Yes | QMessageBox::Default, 
			QMessageBox::No,QMessageBox::NoButton, this, 
			"quit_box", true);
	
	quit_box->setIcon(icon);
	
	// center msg box in window - this part is still not working
	// correctly ....	
	quit_box->setFixedSize(200,100);
	
	// send the quit signal if the answer is positive
	if(quit_box->exec()==QMessageBox::Yes)
	{
      #ifdef USE_ABOUT_BOX
		QTimer* boxTimer = new QTimer(this);
		QFrame* aboutFrame = 
		new QFrame(0, 0, WStyle_Customize | WStyle_NoBorderEx);

		QPixmap aboutPixmap(const_cast<const char**>(ABOUT_BOX_IMAGE));

		aboutFrame->setFrameStyle(QFrame::Box | QFrame::Plain);
		aboutFrame->setBackgroundPixmap(aboutPixmap);

		connect(boxTimer, SIGNAL(timeout()), this, SLOT(close()) );
	
		this->hide();
		aboutFrame->show();
		aboutFrame->setGeometry(this->width()/3, this->height()/3, 500,300);
		boxTimer->start(USE_ABOUT_BOX, true);
     #else

		this->close();
	 #endif
	}
	
}

/**
 * @param effectNumber  The number of effect to be selected.
 * 
 * Uses the number returned by the effect selector combo box to select into 
 * the effect pool array. This functions stops and resets the previously 
 * active effect and hides its child control panel, then sets the current 
 * effect to the selected one and passes a pointer to it to the GL rendering 
 * window. Also checks if the selected effect is already current, to avoid 
 * unnecessary hiding/showing etc. 
 *  
 * @warning
 * This function automatically calls onLoadFiles() when a new effect is 
 * selected, provided the following conditions are met:
 * the selected effect really needs a new set of textures
 * \b OR has a different required number of image lists
 * \b AND its required number of image lists is not 0. This kind of 
 * behaviour may not be liked by some people.... 
 * Talk to me and we'll work something out....
 *
 * @pre
 *  All effects available in the selector box must be properly instantiated. 
 *  
 * @see onLoadFiles(), glView() 
 *  
 **/

void GLFramework::onSelectEffect(int effectNumber)
{
	/* the following if is only needed upon initial selection, 
	where no effect has been enabled yet */
	
	if(startUp==true)
	{	
		// activate the main controls
		//trackball->setEnabled(true);
		
		playButton->setEnabled(true);
		pauseButton->setEnabled(true);
		stopButton->setEnabled(true);
		resetButton->setEnabled(true);
		
		loadImagesButton->setEnabled(true);
		saveButton->setEnabled(true);

		cameraModeButton->setEnabled(true);
		cameraResetButton->setEnabled(true);
		fovResetButton->setEnabled(true);
		projectionButton->setEnabled(true);
		
		// set the current effect to the selected one and show it
		currentEffect = effectPool[effectNumber];

		/* load textures. This is called regardless of needNewTextures 
		 of the selected effect, to provide some initial
		texture data */

		onLoadFiles();
		
		// set the current effect in GL window
		glWindow->setEnabled(true);
		glWindow->setCurrentEffect(currentEffect);
		
		currentEffect->show();
		currentEffect->showControls();

		this->addToStatus(currentEffect->describeEffect());
		this->setCaption(currentEffect->name());

		startUp=false;
	}

	if(currentEffect != effectPool[effectNumber])
	{
		// stop, reset and hide the previously active effect
		this->onStop();
		this->onReset();

		currentEffect->hideControls();
		currentEffect->hide();

		// switch mouse to normal mode
		cameraModeButton->setOn(false);
		

		// destroy and re-create the GL rendering context...

		/* This is done for the following reason:
		Even though the openGL states and modes are reset to their
		defaults every time a different effect is selected, the
		internal data structures , tags, flags etc. of the current
		openGL rendering context remain as they were used by the
		previous effect - regretfully, there is no global reset
		function in openGL. As a result, the next selected effect may
		end up having incorrect, distored or missing textures, lighting
		and so on - the initialization within GLView::initializeGL()
		cannot completely reset the rendering context.  The only
		effective workaround for this problem that I have come up with
		for now is to destroy the rendering context alltogether, after
		the current effect is completely done with it, and create a new
		one instead.

		The GL viewport and controls are disabled temporarily while
		the GLView object is deleted, to avoid accidental access to
		[temporarily] dangling pointers... */	
		
		glWindow->setEnabled(false);
	
		playButton->setEnabled(false);
		pauseButton->setEnabled(false);
		stopButton->setEnabled(false);
		loadImagesButton->setEnabled(false);
		resetButton->setEnabled(false);
		saveButton->setEnabled(false);

		delete glWindow;
		
		// create the openGL rendering window
		glWindow = new GLView(glFrame, this, glContextFormat); 
		CHECK_PTR(glWindow); 
	
				
		// set the current effect to the selected one 
		currentEffect = effectPool[effectNumber];

		// check whether new textures need to be loaded

		// new textures will be loaded if:
		// the selected effect really needs a new set of textures
		bool needTextures = currentEffect->needNewTextures();

		// AND its required number of image lists is not 0
		bool needImgLists  = (currentEffect->numImageLists()!=0);
			
		if( needTextures & needImgLists )
		{			
		  onLoadFiles();
		}

		// set the current effect in GL window and activate it 
		glWindow->setCurrentEffect(currentEffect);

		/* After the current effect is selected into the GL renderer,
		all event handlers, pointers etc. are once again safe to 
		access, so re-enable the viewport and all controls. */

		glWindow->setEnabled(true);
		glWindow->setFocusPolicy(QWidget::WheelFocus);
		glWindow->show();
	
		playButton->setEnabled(true);
		pauseButton->setEnabled(true);
		stopButton->setEnabled(true);
		loadImagesButton->setEnabled(true);
		saveButton->setEnabled(true);
		resetButton->setEnabled(true); 

		currentEffect->show();
		currentEffect->showControls();
		this->addToStatus(currentEffect->describeEffect());
		this->setCaption(currentEffect->name());

		glWindow->resetView();
		
	}
	else 
	{		
		// the selected effect is already current, so do nothing
	}
}


/** Clears the status display
 */ 
void GLFramework::clearStatusDisplay(void)
{statusDisplay->setText("");}

/**
 * @param line  The line of text to display
 * 
 * Takes the contents of \e line and adds it as the last line of the status
 * display. The display will automatically scroll if full or the line is 
 * longer than the width of the display.
 *
 * @warning The display capacity is set to \b 500 lines of text. When it 
 * overflows, it automatically deletes the first line 
 * (that is, the oldest one). Keep this in mind if you output large 
 * messages such as series of mouse coords etc.
 *    
 * @todo Provide a smarter status display...
 * 
 * @see
 *  clearStatusDisplay()
 *
 **/
void GLFramework::addToStatus(const QString& line)
{
	// add single line to line edit box 
	statusDisplay->insertLine(line, (statusDisplay->numLines())+1);
	
	// scroll down to the last added line
	statusDisplay->setCursorPosition(statusDisplay->numLines(), 0, false);
	
	// prevent overflow... this one is not clean yet 
	if((statusDisplay->numLines())>500) statusDisplay->removeLine(0);
}

/**
 * @param enable  Set to \b false to disable the tracker, to \b true to 
 * enable it.
 * 
 * Enables/disables the memory tracker display & tracker update timer
 *
 * @warning
 * The update interval is set to 100 ms. I found this to be an optimal 
 * value. We can change this later if needed. However, I would not recommend
 * it since using a smaller interval wont't make the tracker much preciser,
 * and will probably increase the processor load since more timer events 
 * will need to be processed. (The triangle indicator will probably
 * also flash too fast and only annoy the users...)
 *
 * @todo
 *   UNIX-specific section in updateMemoryTracker()
 *
 * @see updateMemoryTracker()
 * 
 **/

void GLFramework::enableMemTracker(bool enable)
{
	bytesFreeGauge->setEnabled(enable);
	bytesTakenGauge->setEnabled(enable);
	bytesTotalGauge->setEnabled(enable);
	memUsageGauge->setEnabled(enable);

	if(enable)
		memTicker->start(100,false);
	else memTicker->stop();

}

// layout-related functions
// still not sure whether this works correctly....
QSize GLFramework::sizeHint() const
{
	return QSize(1024,768);

}

QSizePolicy GLFramework::sizePolicy() const
{
	return QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
}


/**
 * @param panel  Pointer to an effect's control panel
 *
 *  Places the control panel of an effect into the 
 *  2nd column of the primary GUI and centers it vertically.
 *  Notice: To be precise, this function will take the control box and
 *  place it into the 2nd row of the 1st (and only) column of the panelGrid 
 *  sub-layout.
 *
 * @warning
 *  The control panel can be anything that derives from a QFrame, including 
 *  the 'geometry management' widget types, but not from QLayout!
 *
 * @pre The control panel should be properly initialized and instantiated 
 *  before passing it to this function
 *  
 *
 * @see GLEffect::createControls(), GLEffect::createControlPanel(),
 *  GLEffect::showControls(), GLEffect::hideControls() 
 *
 **/

void GLFramework::placeEffectControlPanel(QWidget* panel)
{
	panelGrid->addWidget(panel,2,0,QGridLayout::AlignVCenter);
}


void GLFramework::hideEvent(QHideEvent* hEv)
{
	
	if(currentEffect!=0)
	{
		currentEffect->hideControls();
		currentEffect->hide();
	}
	QWidget::hideEvent(hEv);
}

void GLFramework::showEvent(QShowEvent* sEv)
{
	
	if(currentEffect!=0)
	{
		currentEffect->showControls();
		currentEffect->show();
	}
	QWidget::showEvent(sEv);
}




