/*****************************************************************************\
 *                              GLEffect
\*****************************************************************************/

/*! @file GLEffect.h
 *
 *  
 *
 *  @brief
 *	Abstract effect base class declarations
 *   
 *  @author Ibraguim Kouliev
 *  
 *  
 */

/*
    Copyright  2001 Ibragium Kouliev All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @class GLEffect
 *
 * @ingroup appclasses
 *
 * @brief Abstract base class for various user-defined effects.
 *
 * This abstract base class provides basic functionality common for all
 * effects, i.e. instantiation/hiding/showing of child GUI controls, 
 * mouse/keyboard event handlers, a basic information structure, and some 
 * other useful functions. (See list of API functions for details). It also 
 * provides common predefined member variables than can be used by an effect 
 * to keep pointers to texture data, access current 2d/3d mouse coordinates,
 * pressed keys, perspective projection parameters etc. 
 *
 * @todo
 *
 **/

#ifndef GLEFFECT_H
#define GLEFFECT_H

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

//Qt headers
#include <qobject.h>
#include <qgl.h> 

// GL headers
#include <GL/gl.h>
#include <GL/glu.h>

/* Some essential headers - included here one time so that they 
 * don't have to be included
 * into each and every derived class.... */

#include <qimage.h>

#include "GLFramework.h"
#include "GLView.h"

//---------------------------------------------------------------------------
//  Forward references
//---------------------------------------------------------------------------

class QFrame; 
class QDialog;
class QGridLayout;
class QImage;

//**************************************************************************
// GLEffect
//**************************************************************************

class GLEffect : public QObject
{
	Q_OBJECT
//---------------------------------------------------------------------------
//  Public Instance Methods
//---------------------------------------------------------------------------

public:
// --------------------------------------------------------------------------
//! @name    Construction/desctruction
// @{
	
	//! Primary class contructor 
	GLEffect(GLFramework* parent_gui);

	//! Destructor
	virtual ~GLEffect();

// --------------------------------------------------------------------------
// @} 

// --------------------------------------------------------------------------
//! @name  Public API functions 
// @{
	//! Wrapper for calling initialize() from outside. 
	void init(); 

	//! A function to instantiate and place the settings panel 
	void createControlPanel(bool needSeparateWindow=false);

	//! User-defined implementation of the settings panel
	
	/** 
	@param parentWindow Pointer to the parent widget.
		
	This is where you create your own conrol panel for your effect.
	The absolute minimum here is to provide an empty definition of this 
	function, i.e \n void createControls(QWidget* parent) {}, if you don't
	need any control widgets. This will put up an empty frame with 
	"No controls available" title in the right panel of the main window.\n
	Otherwise, here's how you build such a panel:\n 
	The \b controlPanel member variable will already contain a pointer to 
	the correct parent widget(depending on what you chose 
	in createControlPanel() ) at the time this function is called. So, 
	construct some kind of container first, a \b QVBox for instance, with 
	controlPanel as the parent widget. Then construct your control 
	widgets with your \b QVBox container as the parent, and don't forget to 
	define suitable slots for these widgets in your class. 
	Finally, establish all necessary signal/slot connections, and the 
	panel is ready to use.
		
	\b Attention: When you have implemented the panel, its instantiation
	is taken care of automatically in GLEffect, so make sure you don't 
	call this function from anywhere - the resulting behaviour of the 
	program would be upredictable!
	**/
	virtual void createControls(QWidget* parentWindow)=0;
		

	// Child GUI visibility control functions
		
	//! Hides the settings panel of an effect.
	void hideControls(); 
		
	//! Shows the settings panel of an effect.
	void showControls();
	
	//! Called when an effect is hidden
	virtual void hide();
	//! ... or shown
	virtual void show();
		
	// **** Helper functions ****

	/** @brief This provides a digested textual description 
	of the effect. Does NOT provide individual parameters.
	*/
	QString describeEffect() const;

	//! Returns the name of this effect.
	const char* name() const
	{return effectInfo->effectName;}

	//! Returns the number of image lists required by this effect.
	unsigned int numImageLists() const
	{return effectInfo->requiredImageLists;}

	//! Tells whether an effect requires a new set of textures to be 
	//loaded.
	bool needNewTextures() const
	{return effectInfo->needNewTextures;}

	//! Returns a name for specified selection dialogue.
	const char* dialogName(unsigned int number) const
	{return effectInfo->fileDialogNames[number];}

	//! Returns whether an effect needs streaming mode (Not used yet)
	bool needsStreaming() const
	{return effectInfo->needStreaming;}

	/** @brief Returns whether an effect needs original loaded images 
        not converted into openGL-compatible format. */

	/** This function is used by GLFramework to ask an effect whether it
	wants to handle originally loaded image data. If you set the 
	\b newRawTextures member of the EffectInfo struct to \b true, the loaded
	images will no longer be automatically converted to the most common 
	openGL-compatible format (32-bit RGBA) and you will be responsible 
	for handling the data yourself in an appropriate way. You may wish to
	do so if the automatic conversion doesn't meet your requirements 
	for some reason. */

	bool needRawTextures() const
	{return effectInfo->needRawTextures;}

	/** @brief Used internally by GLFramework to signal an effect that 
	new texture data has been loaded. */
	void reloadTextures();


// --------------------------------------------------------------------------
//@}

// --------------------------------------------------------------------------
/** @name Rendering, animation and effect control functions 
 *  @attention Derived effect classes should override and implement 
 *   these as necessary. Default implementations do nothing nor are they
 *   supposed to doanything!
 **/
	
// @{		  
	//! stops animation of an effect...
	virtual void stop();
	
	//! runs animation..
	virtual void play();
		
	//! Resets the effect's parameters.
	virtual void reset();

	//! Pauses animation...
	virtual void pause();
		
	//! Effect-specific implementation of the rendering routine.
	/** This is the 'heart' of the effect rendering system. 
	 Derived classes will need to put implementation of their 
	 rendering algorithm into this function. */
		
	virtual void render()=0;
// --------------------------------------------------------------------------
// @} 

// --------------------------------------------------------------------------

/** @name Event handlers
 * @attention These can (but don't need to) be reimplemented in the derived 
 * classes. They retrieve
 * all necessary information from events and make it available through 
 * predefined variables, and also automatically call update(). 
 * You can modify / reimplement them for specific
 * purposes if you need. See further documentation for details. */  
// @{ 
	//! Mouse move events handler.
	virtual void mouseMoveEvent(QMouseEvent*);
	//! Mouse press events handler.
	virtual void mousePressEvent(QMouseEvent*);
	//! Mouse release events handler
	virtual void mouseReleaseEvent(QMouseEvent*);

	//! Mouse wheel events handler
	virtual void wheelEvent(QWheelEvent*);

	//! Keyboard events handler.
	virtual void keyPressEvent(QKeyEvent*);

// --------------------------------------------------------------------------
// @} 

// --------------------------------------------------------------------------
//! @name Signals/Slots
// @{ 

signals:

public slots:
// --------------------------------------------------------------------------
// @} 


//---------------------------------------------------------------------------
//  Private/Protected Instance Methods
//---------------------------------------------------------------------------

protected: 

// **** Initalization functions ****
		
	/*
	WARNING: Derived classes may want to override this
	as necessary, but will need to call the base
	implementation as the first line of their implementation
	to properly initialize the GL window, unless they
	completely override all initialization parameters.
	... some decisions will be taken here yet...*/
		
	//! Initializes parameters / GL modes of an effect.
	virtual void initialize();

	// **** GL-drawing functions ****
		
	//! Used to initiate redraw....
	virtual void update();

//---------------------------------------------------------------------------
//! @name Helper functions
// @{ 

	//! Wrapper for setMouseTracking().
	/** Activates mouse tracking in GLView to track the mouse
	position regardless of mouse button states. This is useful
	in situations where you want your effect to react to mouse movements
	even if the button(s) aren't clicked. */

	void forceMouseTracking(bool enable)
	{glView->setMouseTracking(enable);}
	
	/** @brief Returns 3-D mouse coordinates by "un-projecting"
	2-D window mouse coordinates. */

	void get3DMouseCoords(GLdouble DepthCoord, GLdouble& worldX, 
				GLdouble& worldY, GLdouble& worldZ);

		
	//! Optional texture generation...
	void generateTextures(); 
	//! ... and deletion
	void deleteTextures();

//---------------------------------------------------------------------------
// @}

//---------------------------------------------------------------------------
//  Private/Protected Instance Variables
//---------------------------------------------------------------------------

protected:

//---------------------------------------------------------------------------
//! @name Data members
// @{ 
		
	/** @brief Provides general information about an effect.
	\b IMPORTANT: Don't forget to initialize this struct with values 
	appropriate for your effect, as these control major aspects of the 
	image loading mechanism. The fileDialogNames[] is optional, but 
	initializing these with some descriptive names like 'Textures', 
	'Alpha masks' etc. may be a good idea. ;) 
	*/

	struct EffectInformation
	{
		const char* effectName; //!< Effect's name
		const char* shortDescription; //!< Short description
		const char* author; //!< Author's name
			
		const char* version; //! <Version number
		unsigned int versionMajor; //!< Major version number

		unsigned int requiredImageLists; //!< Number of required image sets
		bool needNewTextures; //!< Really need new textures to be loaded?
		const char* fileDialogNames[3]; //!< Custom names for file loading dialogs
		bool needStreaming; //!< Require streaming? (not in use yet)
			
		bool needRawTextures; //!< Convert loaded images into OpenGL format ?

	};
		
	//! Pointer to the parent widget
	GLFramework* parentFrame;

	//! Pointer to the GL rendering display
	GLView* glView;

	//! Effect information structure 
	EffectInformation* effectInfo;

	//! A common 'container' for possible user-supplied control elements
	QFrame* controlPanel;

	//! A (possible) top-level dialog for a separate control panel window
	QDialog* controlWindow;

	//!... and its layout grid
	QGridLayout* controlWindowGrid;
		
	//! Hidden/visible status
	bool hidden;

	//! Current mouse coordinates (relative to the GL window)
	GLint mouseX, mouseY;

	//! Mouse coordinates normalized to [-1...1]
	GLdouble normMouseX, normMouseY;

	//! 3-D mouse coordinates
	GLdouble mouse3D_X, mouse3D_Y, mouse3D_Z;

	/** @brief Current mouse button. ButtonState is defined in the global 
	 * Qt namespace. see Qt documentation for possible values of 
	 * ButtonState */
	ButtonState mouseButtonPressed;
		
	/* @brief Current keyboard key. Key is also defined in global Qt 
	 * namespace.  See Qt documentation for possible values of Key */
	Key keyPressed;

	//!  Pointers to current texture data frame(s)
	QImage* texImage1; 
	QImage* texImage2; 
	QImage* texImage3;

	//!  Pointers to the optional texture objects
	GLuint* textures1; 
	GLuint* textures2; 
	GLuint* textures3;
   
	//! Current frame position
	GLuint currentFrame;
	
//---------------------------------------------------------------------------
// @}
		
//---------------------------------------------------------------------------
//! @name GL parameters/states
// @{				

	GLint viewportCoords[4]; //! Viewport coordinates
	GLdouble modelViewMatrix[16]; //! Current modelview matrix 
	GLdouble projectionMatrix[16]; //! Current projection matrix

	GLdouble currentDepthCoord; //! Current depth coordinate
	
	//! Distance from the origin to the chosen viewpoint 
	GLdouble viewDistance;
		
	// Projection parameters
		
	//! FOV (Field Of View) angle
	GLdouble fovAngle;
		
	//! Distance to near and far clipping planes
	GLdouble nearClipPlane, farClipPlane;

//---------------------------------------------------------------------------
// @}

	private:

	//disabled default constructor and assignment operator
	GLEffect();
	GLEffect& operator = (const GLEffect& source);

//---------------------------------------------------------------------------
//  Class methods and variables
//---------------------------------------------------------------------------

};

#endif


