/*****************************************************************************\
 *                              GLEffect
\*****************************************************************************/

/*! @file GLEffect.cpp
 *
 *  
 *
 *  @brief
 *	Abstract base effect class implementation
 *    
 *  @author Ibraguim Kouliev
 *  
 *  
 */

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

// Qt headers
#include <qobject.h>

#include <qsize.h>
#include <qvgroupbox.h>
#include <qpushbutton.h>
#include <qcursor.h>
#include <qdialog.h>
#include <qlayout.h>

// System library headers
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>

// Local headers
#include "GLEffect.h"

// **** Construction/desctruction ****

/**
 * @param parentGui Pointer to the parent window.
 *  
 * Constructs an effect object and initializes internal state variables. The
 * parentGui pointer is passed on to QObject constructor. Also obtains a
 * pointer to the GL rendering window and places it into the \b glView
 * variable.
 *
 * @warning Almost all of general variables/states are set to 0 initially, so
 * do not assume anything as default values for mouse coords, image pointers
 * etc. Also, please keep in mind that the default projection parameters are as
 * follows: fov angle - 60 degrees, view distance from the origin - 5 units (in
 * world coordinates), distance to near clipping plane - 0, to far - 30 units.
 * While this seems to be acceptable in usual cases, it is advisable to
 * nevertheless set those parameters to your own acceptable values, as some
 * effects might not look correctly with the defaults.
 *
 * @pre The parent window object must be fully constructed and available prior
 * to creating any effect instances. Furthermore, the GL rendering window
 * should also be fully constructed and avaiable, as the constructor will try
 * to obtain a pointer to it. 
 *
 * @remarks Even though the effects are not derived from QWidget, they still
 * need to be constructed with a pointer to the parent window because they need
 * to be able to access various functions contained in GLFramework and GLView.
 *
 * @todo Additional init may be added later
 *
 * @see GLFramework::glView, effectInfo..
 **/

GLEffect::GLEffect( GLFramework* parentGui) : QObject(parentGui)
{
	// save pointer to the parent window for future use
	parentFrame = parentGui;

	// create the effect description structure
	effectInfo = new EffectInformation;

	// init effectInfo with some default data
	effectInfo->author = "";
	effectInfo->fileDialogNames[0] = "";
	effectInfo->fileDialogNames[1] = "";
	effectInfo->fileDialogNames[2] = "";
	effectInfo->effectName = "";
	effectInfo->needNewTextures = false;
	effectInfo->needStreaming = false;
	effectInfo->requiredImageLists = 0;
	
	//object has initially no control panel and is not visible
	hidden=true;
	controlPanel=0;

	// init texture pointers to null
	texImage1 = 0;
	texImage2 = 0;
	texImage3 = 0;

	// init the projection parameters to acceptable default values
	viewDistance = 5;
	fovAngle = 60;
	nearClipPlane = 0;
	farClipPlane = 30;

	mouseX = mouseY = 0;
	mouse3D_X = mouse3D_Y = mouse3D_Z = 0;
	normMouseX = normMouseY = 0;

	// obtain a pointer to the GL rendering window
	glView = parentFrame->glView();

}

/**
 * 
 * Destroys an effect object. The settings panel and local effectInfo struct
 * are deleted here.
 *
 * @throw see Warning...
 *
 * @warning The local effectInfo struct and settings panel must still exist at
 * the time the object is destroyed, otherwise pointer exceptions may occur.
 * 
 *  
 **/

GLEffect::~GLEffect()
{
	delete effectInfo;
	delete controlPanel;
}

/**
 *
 * This function initializes the parameters of an effect and its associated GL
 * modes/states. Default implementation does \b nothing, so override this as
 * necessary. Be sure to set up and initialize any necessary variables / data
 * here, and to enable/disable any required openGL states.
 *
 * @throw None, but check for possible GL errors when
 * enabling/disabling/initalizing various GL states....
 *
 * @warning Please remember that this function is called every time the effect
 * is selected. This is done to properly re-initialize all necessary parameters
 * and GL states in order to avoid conflicts with any residual data/GL modes
 * that may have been left by the previously active effect.  Therefore, write
 * your initialization code accordingly, so that its behaviour is consistent.
 *   
 * @see GLView::initializeGL(), init()
 *   
 *
 **/
void GLEffect::initialize()
{
	
}

/**
 *
 * This is really just a wrapper for calling initialize() from outside.
 * Although having such a simple function might seem silly(or stupid....),
 * there are two important reasons for having it around: First, I tried to keep
 * initialize() protected, as to not expose any unnecessary implementation
 * details. Second, what it also does is set the perspective projection
 * parameters in GLView that correspond to the selected effect. This guarantees
 * that the selected effect will look correctly. (If you had to manually call
 * the setProjectionParams() in initialize() and for some reason forgot to do
 * it, your effect would probably look incorrectly).
 * 
 * 
 * @see GLView::setProjectionParams()
 *
 **/
void GLEffect::init()
{
	initialize();
	glView->setProjectionParams(fovAngle, nearClipPlane, 
					farClipPlane, viewDistance);

}

/**
 * This is the base implementation - it will simply redraw the GL window. If
 * you need something more sophisticated before redraw, re-implement this in
 * your class and don't forget to include glView->update(); as the last line of
 * your implementation.  @note Try to keep all necessary transforms etc. within
 * your implementation of render(). Do not override this function unless it is
 * really necessary. 
 *
 * @pre Assumes all params are set up correctly..
 *
 * @remarks This function, among several others, interfaces to the GL window to
 * provide rendering and animation control. It is actually used as a wrapper
 * for glView->update() to keep the class interface of GLEffect consistent.
 *   
 * @see GLView::update()
 *
 **/
void GLEffect::update()
{
	glView->update();
  	
}

/** 
 * This function is used internally by image loading mechanism within
 * GLFramework to notify the current effect when the user loads a new set of
 * images. The texture pointers are set to first image frames from each
 * of 3 possible image lists (depending on how many image lists there are). The
 * scene is then immediately updated to reflect the changes. 
 * 
 * @see GLFramework::onLoadFiles()
 */

void GLEffect::reloadTextures()
{
  //nothing to do if there're no textures...
  if(this->effectInfo->requiredImageLists==0) return;

  //otherwise fetch first image frames from every image list
  
  switch(this->effectInfo->requiredImageLists)
  {
	case 1:
	  texImage1 = parentFrame->fetchImage(0,0);
	break;

	case 2:
	 texImage1 = parentFrame->fetchImage(0,0);
	 texImage2 = parentFrame->fetchImage(1,0);
	break;

	case 3:
	 texImage1 = parentFrame->fetchImage(0,0);
	 texImage2 = parentFrame->fetchImage(1,0);
	 texImage3 = parentFrame->fetchImage(2,0);
	break;

  }
  // re-render the scene
  update();

  /* On the side note here I should mention that the call to update()
  here might be redundant in some situations, depending on designs of your
  rendering routines, but it is safer to just update the scene anyway after
  loading a new set of images. */

}

// **** Event handlers ****

/** @param mEv  Pointer to the incoming QMouseEvent. The event is supplied by
 * the GL window.
 * 
 * @note A general note on all of these event handlers: if you want to override
 * their default implementations, you have two options:\n 1. You override them
 * completely and will then need to explicitly call get3DMouseCoords() and
 * update() functions as necessary...\n 2. You override them with your specific
 * code and add GLEffect::mouseMoveEvent (or others, respectively) as the last
 * line of your implementation.  Normalized and 3-d mouse coordinates will be
 * then generated automatically.  Be aware that update() will be called
 * automatically,too, so don't include a call to update() yourself.
 *
 * This event handler processes mouse click events generated by the GL window.
 * The resulting information is stored as X and Y 2-D mouse coordinates
 * (relative to the GL window) in mouseX and mouseY, respectively. The button
 * states are stored in the mouseButtonPressed variable. See the Qt
 * documentation for details on possible values of ButtonState and how to
 * interpret them. The coordinates are also automatically normalized to
 * <b>[-1...1]</b> range and are stored in normMouseX and normMouseY. As a
 * convenience, 2-D mouse coordinates are also always automatically
 * "unprojected" back into 3-D world coordinate space and are given back in
 * mouse3D* variables. The scene is updated immediately after an event has been
 * processed.
 *
 * @throw None, but see description of get3DMouseCoords() for possible
 * "unprojection" errors/difficulties...
 *   
 * @pre See get3DMouseCoords() for details...
 *   
 * @see get3DMouseCoords(), mouseMoveEvent(), GLView::mouseClickEvent()
 *   
 **/

void GLEffect::mousePressEvent(QMouseEvent* mEv)
{
	mouseX = mEv->x(); 
	mouseY = mEv->y();
	mouseButtonPressed = mEv->button();
	
	// transform to 3d mouse coordinates
	get3DMouseCoords(currentDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);

	// optionally provide normalized 2d mouse coordinates
	// normalize coordinates to [-1...1]
	normMouseX = mouseX / (GLdouble(glView->width())/2);
	normMouseX -= 1.0;

	normMouseY = mouseY / (GLdouble(glView->height())/2);
	normMouseY -= 1.0;

	update();
}

/** @param mEv Pointer to the incoming QMouseEvent. The event is supplied by
 * the GL window.
 *
 * Reacts to the mouse being released after a click and/or drag. The default
 * implementation does nothing, as all the relevant information is normally
 * gathered by the preceding mousePressEvent() or mouseMoveEvent().
 * 
 **/
void GLEffect::mouseReleaseEvent(QMouseEvent* mEv)
{
	//default implementation does nothing
	//mEv->ignore();
}


/** @param mEv Pointer to the incoming QMouseEvent. The event is supplied by
 * the GL window.
 * 
 * Description, warnings etc. almost identical to mousePressEvent(), except
 * this one processes mouse movement events, i.e. what is normally known as
 * "dragging" the mouse.
 * 
 * @see mousePressEvent, GLView::mouseMoveEvent()
 *   
 **/
void GLEffect::mouseMoveEvent(QMouseEvent* mEv)
{
	mouseX = mEv->x(); 
	mouseY = mEv->y();
	mouseButtonPressed = mEv->state();

	// transform to 3d mouse coordinates
	get3DMouseCoords(currentDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);

	// optionally provide normalized 2d mouse coordinates

	// normalize coordinates to [-1...1]
	normMouseX = mouseX / (GLdouble(glView->width())/2);
	normMouseX -= 1.0;

	normMouseY = mouseY / (GLdouble(glView->height())/2);
	normMouseY -= 1.0;

	update();
}

/**
 * @param kEv A pointer to the incoming QKeyEvent. The event is supplied by the
 * GL window
 *
 * This event handler processes incoming keyboard events. Nothing complicated
 * here, the pressed key is stored in the keyPressed variable. Look into
 * documentation of the global Qt namespace for possible key values.  Many of
 * them have convenient symbolic names.  The scene is updated immediately after
 * an event has been processed.
 *  
 * @see GLView::keyPressEvent()
 *   
 **/
void GLEffect::keyPressEvent(QKeyEvent* kEv)
{
	keyPressed = (Key)(kEv->key());
	update();
}

/** @param wEv  A pointer to the incoming QWheelEvent. The event is supplied by
 * the GL window.
 *
 * This handler processes incoming mouse wheel events. The default
 * implementation does nothing and ignores the event (as required by Qt event
 * processing hierarchy). The GLView window is set up to automatically assume
 * focus upon any incoming wheel events to facilitate your event handler
 * programming. 
 *
 * @see GLView::wheelEvent()
 *
 **/
void GLEffect::wheelEvent(QWheelEvent* wEv)
{
	//default implementation does nothing
	wEv->ignore();
}


// **** Control functions ****

void GLEffect::play()
{

}

void GLEffect::stop()
{

}

void GLEffect::reset()
{

}

void GLEffect::pause()
{

}


// **** Helper functions ****

/**
 * @param DepthCoord Current value of the depth coordinate.  @param worldX,
 * worldY, worldZ The variables in which the resulting 3-D world space
 * coordinates will be returned.
 *
 *
 * This is a helper function that is used internally to "unproject" the 2-D
 * mouse coordinates coming from the window back into 3-D world coordinate
 * system, in order to facilitate such things as picking points on a plane etc.
 * The results are returned in worldX, worldY, worldZ. They depend on the
 * current modelview & projection matrices and viewport bounds, as well as
 * supplied value of DepthCoord. 
 *
 * @throw Will send warning messages to the console if the given coordinates
 * cannot be unprojected for some reason. 
 *
 * @warning Quoting the OpenGL Programming Guide, "There are inherent
 * difficulties in trying to reverse the transformation process.".  In certain
 * cases, the unprojection can become impossible, because of a non-invertible
 * matrix etc,etc. The two pretty common culprits I have experienced were
 * incorrect distances to near/far clipping planes and coordinates that
 * originated outside the GL window. The latter has been fixed in event
 * handlers. There still are some points to check if you see warnings that the
 * coordinates could not be unprojected - current transformations, camera
 * position, fov angle, and those clipping plane distances being likely
 * candidates. 
 *   
 *
 * @pre You'll need to set the value of currentDepthCoord to something
 * sensible. If you set it to 0.0, this function will obtain the 3-D
 * coordinates lying on the near clipping plane. Likewise, setting it to 1.0
 * will obtain the 3-D coordinates on the far clipping plane. Will either of
 * those, it's quite simple to "shoot" a ray through the obtained point and
 * then pick up any point on that ray.
 *   
 * 
 * @see mousePressEvent(), mouseMoveEvent()
 *   
 **/

// Transforms current 2d mouse coordinates back into 3d world coordinates
void GLEffect::get3DMouseCoords(GLdouble DepthCoord, GLdouble& worldX, 												GLdouble& worldY, GLdouble& worldZ)
{
	//retrieve current viewport dimensions, modelview
	//and projection matrices
	glGetIntegerv(GL_VIEWPORT, viewportCoords);
	glGetDoublev(GL_MODELVIEW_MATRIX, modelViewMatrix);
	glGetDoublev(GL_PROJECTION_MATRIX, projectionMatrix);


	// transform 2d mouse coordinates back to 3d world coordinates
	if(gluUnProject((GLdouble)mouseX, (GLdouble)mouseY, DepthCoord, 
					 modelViewMatrix, projectionMatrix, 
					 viewportCoords, 
					 &worldX, &worldY, &worldZ)==GL_FALSE)
	{
		//error trap 
		qDebug("Could not unproject these coordinates (%d, %d)!",
				mouseX, mouseY);
	}

	/* NOTE: This implementation doesn't use the gluUnProject4() supplied
	 * with OpenGL 1.2 (for reasons of compatibility with standard OpenGL
	 * implementation in win32). Therefore, it won't be able to handle
	 * non-standard depth ranges. Should you require such behaviour, you'll
	 * need to change this implementation. */
}


/* The following functions are currently stubs. These might be
implemented some time in the future. */

void GLEffect::generateTextures()
{

}

void GLEffect::deleteTextures()
{

}

// Returns a short textual description of the effect
QString GLEffect::describeEffect() const
{
	//build up the description string
	QString text;
	
	text += "\n";
	text += effectInfo->effectName;
	text += "\n";
	text += effectInfo->author;
	text += "\nVersion ";
	text += effectInfo->version;
	text += "\n";
	text += effectInfo->shortDescription;

	return text;
}

// **** layout-related functions ****

/**
 * @param needSeparateWindow Controls where the control panel will be placed.
 *
 * This instantiates the control panel and places it into a separate top-level
 * dialog window if needSeparateWindow is set to TRUE. Otherwise ,the control
 * panel will be placed into the side panel of the primary window (default
 * option). The details of implementation are a bit tricky - you're advised to
 * look at the internal comments if you wish to fully understand how this
 * function works. It calls the user-defined implementation of
 * createControls(), passing it the pointer to the correct parent... 
 * 
 * @throw May generate warning messages (these originate from Qt's inner
 * mechanisms), if incorrect child-to-parent widget relationships are detected.
 * This could happen in createControls() if you acidentally construct some
 * widget with a wrong/unacceptable parent...
 *
 * @warning  You're responsible for calling createControlPanel() from the
 * constructor of your derived class - it won't happen automatically.  (The
 * reason for this is that a control panel is part of an individual effect
 * class, and as such it cannot be instantiated in the constructor of the base
 * class). If you don't want any controls, you'll still have to provide at
 * least an empty implementation of createControls() and call
 * createControlPanel() for the effect selection mechanism to be able to
 * properly show/hide your effect. A small frame with "No controls available"
 * title will be generated automatically for you.
 *   
 * @pre The base class part of an effect object must be initialized...
 *  
 * @see showControls(), hideControls(), show(), hide(), createControlPanel()
 *
 **/

void GLEffect::createControlPanel(bool needSeparateWindow)
{
	// needed for proper init of the control panel object
	QWidget* tempParent;

	if(needSeparateWindow)
	{
		// create a top-level modeless dialog if required
		controlWindow=new QDialog(0,0,false, Qt::WStyle_Customize | 
			Qt::WStyle_NormalBorder | WStyle_Title | Qt::WStyle_Tool
			 | Qt::WStyle_StaysOnTop | Qt::WStyle_Minimize);

		/* NOTICE: This function cannot be reimplemented, because we
		 * tried to provide a standard way to instantiate control
		 * panels for effects, and its operation relies on this code.
		 * 
		 */
		
		controlWindowGrid = new QGridLayout(controlWindow,1,1,10);
		controlWindowGrid->setSpacing(10);

		tempParent=(QWidget*)controlWindow;
	
	}
	else
	{
		// controls will be placed into the main GUI
		controlWindow=0; 
		tempParent = (QWidget*)parentFrame;
		
	}

	// instantiate a used-defined control panel... 
	createControls(tempParent);
	
	
	if(controlPanel==0)
	{
		/* This takes care of a rare case where an effect
		doesn't need any visual controls at all.  NOTICE: the
		createControls() function can still be imlemented in order to
		initialize any necessary non-visual controls, for example
		timers.. 
		*/

		//provides an empty frame
		controlPanel = new QGroupBox("No controls available", 
						tempParent);
	}

	if(needSeparateWindow)
	{
		// ...and place it into top-level dialog window
		controlWindowGrid->addWidget(controlPanel,0,0);
				
	}
	else
	{
		// ...or into the main GUI
		parentFrame->placeEffectControlPanel(controlPanel);

	}

}


/**
 * This function is called when an effect is hidden. It is doesn't currently do
 * much, except reset the parameters of the effect being hidden to their
 * defaults. Some additional commands might be added here later.
*/

void GLEffect::hide()
{
	hidden=true; //set visibility status 

	// reset the effect parameters
	reset();

}

/** This function is called when an effect is shown. May be later used for a
 * variety of different purposes. */

void GLEffect::show()
{
	hidden=false;
}

// these twins hide / show the control panel

/** This function hides the control panel of the current effect. You should
 * call it once at the end of your derived class' constructor so that the
 * control panel of your effect is initially hidden from view (otherwise the
 * main GUI will look weird...)
 *
 *
 *  @remarks This may seem strange, but this call to hideControls() at the end
 *  of your constructor is really necessary to ensure proper operation of the
 *  program. If I later come up with a better, more intuitive design, I'll
 *  change this implementation.
 *
 * @warning May cause a crash if the control panel has been deallocated or
 * didn't exist at the moment this function was called, so be careful with
 * those controlPanel and controlWindow pointers. 
 * 
 * @see showControls(), hide(), show() 
 */

void GLEffect::hideControls()
{
	// if controls were in a separate window
	if(controlWindow!=0)
	{
		controlWindow->hide();
	}

	// otherwise...
	else controlPanel->hide();
}

/**
 * This shows an effect's control panel. Both showControls() and hideControls()
 * are used internally by GLFramework::onSelectEffect() member of the
 * GLFramework class to control the visibility of individual effects upon
 * selection. You should never call showControls() explicitly.  \n By default,
 * this function detects if the panel had been instantiated as a separate
 * top-level window. It then aligns it to the right border of the main window
 * (if the main window is normal) or in the right side panel of the main window
 * (if the main window is maximized). 
 *
 * @warning Same as in hideControls()
 *
 * @todo If this 'docking' behaviour proves to be irritating, I'll introduce an
 * additional parameter with which to turn it on and off.
 *
 */
void GLEffect::showControls()
{
	// in case of separate top-level dialog window
	if(controlWindow!=0)
	{
		controlWindow->show();

		int place_x;
		int place_y;

		// this is placement code - still experimental
		if(parentFrame->isMaximized())
		{
			place_x = parentFrame->x()+parentFrame->width()
								*(float)3/4;
			
			place_y = parentFrame->y()+parentFrame->height()/3;
		}
		else
		{
			place_x = parentFrame->x() + 
				  parentFrame->frameGeometry().width()+5;
			place_y = parentFrame->y()+25;
		}
		

		// displace and resize dialog window
		// this section is not optimal yet...
		controlWindow->setGeometry(place_x,place_y,
					   controlWindow->childrenRect().width(),
					   controlWindow->childrenRect().height() ); 

	}
	else controlPanel->show(); // in case of normal control panel
}

/** @param 
 *
 * @return Detaillierte Beschreibung ...
 *
 * @throw 
 *
 * @warning @pre
 *   
 * @todo
 * 
 * @bug
 * 
 * @see
 *   
 ***/










