// some vector utility functions

#ifndef VECTOR_H
#define VECTOR_H

/**
 * @file vector.h
 *
 * @ingroup contourWarpClasses
 *
 */

/** \internal 
 * The data type declared here is a 3-dimensional vector along with its
 * associated linear operations. This one is used internally by the Contour
 * Warping effect classes, but one could certainly use this for own purposes.
 */
typedef struct VECTOR3D{
	float x;
	float y;
	float z;

#if (defined (__cplusplus) | !defined (__STDC__))
	VECTOR3D () {};
	VECTOR3D ( float _x, float _y, float _z);

	VECTOR3D& operator += (const VECTOR3D&);
	VECTOR3D& operator -= (const VECTOR3D&);
	VECTOR3D& operator *= (float);
	VECTOR3D& operator /= (float);

	VECTOR3D operator + () const;
	VECTOR3D operator - () const;

	VECTOR3D operator + ( const VECTOR3D&) const;
	VECTOR3D operator - ( const VECTOR3D&) const;
	VECTOR3D operator * ( float) const;
	VECTOR3D operator / ( float) const;

	friend VECTOR3D operator * ( float, const struct VECTOR3D& );

	int operator == (const VECTOR3D&) const;
	int operator != (const VECTOR3D&) const;
#endif

} VECTOR3D;


#if (defined (__cplusplus) | !defined (__STDC__))
	inline VECTOR3D::VECTOR3D ( float _x, float _y, float _z)
	{
		x = _x; y = _y; z = _z;
	}

	inline VECTOR3D& VECTOR3D::operator += (const VECTOR3D& v)
	{
		x += v.x; y += v.y; z += v.z;
		return *this;
	}

	inline VECTOR3D& VECTOR3D::operator -= (const VECTOR3D& v)
	{
		x -= v.x; y -= v.y; z -= v.z;
		return *this;
	}

	inline VECTOR3D& VECTOR3D::operator *= (float f)
	{
		x *= f; y *= f; z *= f;
		return *this;
	}

	inline VECTOR3D& VECTOR3D::operator /= (float f)
	{
		float fInv = 1.0f / f;
		x *= fInv;
		y *= fInv;
		z *= fInv;
		return *this;
	}
	
	inline VECTOR3D VECTOR3D::operator + () const
	{
		return *this;
	}

	inline VECTOR3D VECTOR3D::operator - () const
	{
		return VECTOR3D (-x, -y, -z);
	}

	inline VECTOR3D VECTOR3D::operator + (const VECTOR3D& v) const
	{
		return VECTOR3D (x + v.x, y + v.y, z + v.z);
	}

	inline VECTOR3D VECTOR3D::operator - (const VECTOR3D& v) const
	{
		return VECTOR3D (x - v.x, y - v.y, z - v.z);
	}

	inline VECTOR3D VECTOR3D::operator * (float f) const
	{
		return VECTOR3D (x * f, y * f, z * f);	
	}

	inline VECTOR3D VECTOR3D::operator / (float f) const
	{
		float fInv = 1.0f / f;
		return VECTOR3D (x * fInv, y * fInv, z * fInv);
	}

	inline VECTOR3D operator * (float f, const struct VECTOR3D& v )
	{
		return VECTOR3D (f * v.x, f * v.y, f * v.z);
	}

	inline int VECTOR3D::operator == (const VECTOR3D& v) const
	{
		return x == v.x && y == v.y && z == v.z;
	}

	inline int VECTOR3D::operator != (const VECTOR3D& v) const
	{
		return x != v.x || y != v.y || z != v.z;
	}
#endif

float vecSqrMagnitude (const VECTOR3D& v);
float vecMagnitude (const VECTOR3D& v);
VECTOR3D vecNormalize (const VECTOR3D& v);
float vecDotProduct (const VECTOR3D& v1, const VECTOR3D& v2);
VECTOR3D vecCrossProduct (const VECTOR3D& v1, const VECTOR3D& v2);

#endif

