/*
 * Copyleft 2001 Lukas Degener, Ingmar Kanitscheider. All rights granted ;-)
 *----------------------
 * file: src/geomObj.h
 *----------------------
 * This file is part of the CountourWarp Effect based on the GLEffect Framework.
 * In this file you will find declarations for various data structures and methods, that
 * are frequently used within the actual WarpAlghorithm implementations.
 */

/*! @file geomObj.h
 *
 * @ingroup contourWarpClasses
 *
 * @brief Declaration of Edge class. Declaration of geometry helper functions
 * @author Lukas Degener, Ingmar Kanitscheider
 */

#ifndef GEOMOBJ_H
#define GEOMOBJ_H

// We need some vector utility functions and operator overloading
// collected in Vector.h/.cpp (type VECTOR3D)
#include "vector.h"

//--------------------------------------------
// Data structures:
//--------------------------------------------

// Types of edges
enum EDGE {line, ray, edge};

/** @class Edge
 *
 * @ingroup contourWarpClasses
 * 
 * @brief Edge type which covers lines, rays and edges
 */
class Edge {
public:
	VECTOR3D point1; //!< first point
	VECTOR3D point2; //!< second point
	EDGE	 type; //!< type of the edge
				   /*!< can be <code>line</code>, <code>ray</code> or
	                    <code>edge</code>*/
	//! default constructor
	Edge();
	//! copy constructor
    Edge(const Edge &original);
	//! constructor with VECTOR3D pair and edge type
	Edge(VECTOR3D p1, VECTOR3D p2,EDGE type=edge);
	//! assignment operator
	Edge & operator=(const Edge &orig);
	//! comparement operator
	int operator==(Edge &other);
};


/**
 * A struct that holds information on an intersection between two
 * Edges. (similar to the Intersection class in my java demo)
 *
 * @ingroup contourWarpClasses
 * 
 */
struct Intersection {
	/**
	 * First involved edge
	 */
	Edge edge1;

	/**
	 * Second involved edge
	 */
	Edge edge2;

	/**
	 * edge1.point1 + lambda1(edge1.point2 - edge1.point1) = point
	 */
	float lamda1;

	/**
	 * edge2.point1 + lambda2(edge2.point2 - edge2.point1) = point
	 */
	float lamda2;

	/**
	 * the point where edge1 and edge2 intersect
	 */
	VECTOR3D point;
};

// forward declaration
class PolygonShape;

//--------------------------------------------
// Method declarations:
//--------------------------------------------

	/**
 	 * returns the point on edge with minimal distance to x
	 * For a finit edge (type edge), this will be one of the
	 * end points, if the orthogonal projection of x isn't lying between
	 * the two end points.
	 * For a ray, this will be the start point (point1) of the edge, if the
	 * orthogonal projection is lying 'behind' the ray
    */
	VECTOR3D projection(Edge edge, VECTOR3D x);

	/**
	 *	returns the intersection of e1 and e2, or null if there is no intersection
	 */
	Intersection intersect(Edge e1, Edge e2);

	/** Checks if there is an intersection between the edge and the plane in which the
	*   parallelogram lies.
	*   If the edges of the parallelogram aren't infinite the function projects the point
	*   on the bounds of the parallelogram.
	*   @note It is assumed that the first point of the two edges is identical!
	*   @param sectEdge The intersecting edge
	*   @param firstParEdge First edge of the parallelogramm
	*	@param secParEdge Second edge of the parallelogramm
	*	@param pIntersection pointer to a vector to be filled with the intersection point.
	*   @returns 1 if there is an intersection, 0 otherwise.
	*/
	int IntersectEdgeParallelogram (Edge sectEdge, Edge firstParEdge, Edge secParEdge,
											VECTOR3D *pIntersection);


	/**
	* applies an afine transformation to point
	* matrix should be a 4x4 matrix, stored line after line.
	*/
	VECTOR3D atrans(VECTOR3D point, const float* matrix);

#endif

