/*
 * Copyleft 2001 Lukas Degener, Ingmar Kanitscheider. All rights granted ;-)
 *----------------------
 * file: src/WarpAlgorithm.h
 *----------------------
 * This file is part of the CountourWarp Effect based on the GLEffect
 * Framework.  In this file you will find the declaration of the WarpAlgorithm*
 * classes Implementation is in WarpAlgorithm.cpp
 *
 * Comments are doxygen compatible
 */

/*! @file WarpAlgorithm.h
 *
 *	@brief
 *	Declaration of WarpAlgorithm* classes and CVertexGrid class
 *	@author Lukas Degener, Ingmar Kanitscheider
 */

#ifndef WARPALGORITHM_H
#define WARPALGORITHM_H


//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

//  Qt headers
#include <qpixmap.h>

// Local headers
#include "vector.h"
#include "PolygonShape.h"
#include "geomObj.h"


/*****************************************************************************\
 *                              Vertex Grid Helper Class
\*****************************************************************************/

/*! @class CVertexGrid
 *
 *  @ingroup contourWarpClasses
 *
 * @brief Covers the vertex grid underlying the image
 *
 * The main functionality provided by this class is to encapsulate a grid of
 * vertices which can be put at an arbitrary position and can have arbitrary
 * orientation. But in our project, the vertex grid lies in the z=0 plane.
 * 
 * Moreover, the resolution of the field can be set dynamically.
 * @see CVertexGrid::SetGridData(VECTOR3D, VECTOR3D, VECTOR3D, int, int)
 *
 * The data holded by this structure is modified with the warping
 * transformation and rendered in the ContourWarpEffect class.
 */ 

class CVertexGrid {
	
private:
	/*! 
	 * @brief The vertex positions
	 */
	VECTOR3D *m_pVertices;

	/*!
	 * @brief Backup vertex array
	 * 
	 * In order to accelerate the resetting of the vertices
	 * this vertex array stores the initial positions of the vertices.
	 */
	VECTOR3D *m_pInitVertices;

	/*!
	 * @brief The texture coordinate array
	 * 
	 * This information is only used for rendering.
	 */
	float	 *m_pTexCoords;
private:
	/*!
	 * @brief Initializes the vertex positions
	 */
	void InitializeVertices();
public:
//! @name grid rectangle
//! @{
	/*!
	 * @brief The top left vertex of the grid
	 */
	VECTOR3D  m_vTopLeft;
	/*!
	 * @brief The top right vertex of the grid
	 */
	VECTOR3D  m_vTopRight;
	/*!
	 * @brief The bottom left vertex of the grid
	 */
	VECTOR3D  m_vBottomLeft;
//! @}

//! @name grid resolution
//!	@{
	/*!
	 * @brief number of columns
	 */ 
	int		  m_numCols;
	/*!
	 * @brief number of rows
	 */
	int		  m_numRows;
//!	@}

public:
	/*!
	 * @brief Constructor.
	 * 
	 * Initializes the pointers to NULL and sets the grid resolution to 0.
	 */
	CVertexGrid();
	
	/*!
	 * @brief Overloaded Constructor.
	 * 
	 * Creates the field by calling
	 * SetGridData(VECTOR3D,VECTOR3D,VECTOR3D,int,int)
	 */
	CVertexGrid(VECTOR3D vTopLeft, VECTOR3D vTopRight, VECTOR3D vBottomLeft,
			int numCols, int numRows);

	/*!
	 * @brief Destructor.
	 * 
	 * Deallocates all memory needed.
	 */
	~CVertexGrid ();

	/*!
	 * @brief initializes the vertex grid
	 *
	 * This method initializes a rectangular vertex grid in 3D-space.
	 *
	 * @param vTopLeft The top left vertex of the grid
	 * @param vTopRight The top right vertex of the grid
	 * @param vBottomLeft The bottom left vertex of the grid
	 * 
	 * @param numCols number of columns in grid
	 * @param numRows number of rows in grid
	 *
	 * @returns <code>false</code> if there isn't enough memory it fails,
	 * <code>true</code> otherwise.
	 */
	bool SetGridData (VECTOR3D vTopLeft, 
			  VECTOR3D vTopRight, 
			  VECTOR3D vBottomLeft,
			  int numCols, int numRows);
	
	/*!
	 * @brief Sets vertices to their initial positions
	 *
	 * This is achieved by copying the backup array to the vertex array
	 */
	void ResetVertices();
	
//! @name data access methods
//! @{
	/*!
	 * @param index The index of the vertex in the vertex grid
	 * @returns a reference to the coordinates
	 */
	inline VECTOR3D& Vector(int index) {return m_pVertices[index];}
	/*!
	 * @param index The index of the vertex in the vertex grid
	 * @returns a pointer to the texture coordinates
	 */
	inline float *TexCoord(int index) {return &m_pTexCoords[2*index];}
	/*!
	 * @param x Column of the vertex in the vertex grid
	 * @param y Row of the vertex in the vertex grid
	 * @returns a pointer to the coordinates
	 */
	inline VECTOR3D& Vector(int x, int y) 
		{return m_pVertices[y*m_numCols+x];}
	/*!
	 * @param x Column of the vertex in the vertex grid
	 * @param y Row of the vertex in the vertex grid
	 * @returns a pointer to the texture coordinates
	 */
	inline float *TexCoord(int x, int y)
		{return &m_pTexCoords[2*(y*m_numCols+x)];}
	/*!
	 * @param x Column of the vertex in the vertex grid
	 * @param y Row of the vertex in the vertex grid
	 * @param index specifies whether the s or t texcoord is returned.
	 * @returns the texture coordinate
	 */
	inline float& TexCoord(int x, int y, int index)
		{return m_pTexCoords[2*(y*m_numCols+x)+index];}
// @}
};


//---------------------------------------------------------------------------
//  Defines
//---------------------------------------------------------------------------

/*! 
 * Parameter type constants
 */
#define WA_TYPE_BOOL	1
#define WA_TYPE_INT		2
#define WA_TYPE_FLOAT	3

//---------------------------------------------------------------------------
//  Types
//---------------------------------------------------------------------------

/*! @struct paramInfo
 *
 * @brief A struct that holds information about a single parameter
 */
struct paramInfo {
   char name[255];      //!< The name of the parameter
   char descr[255];     //!< A description of what the parameter does
   char minDescr[255];  //!< A (short) description of what small values mean
   char maxDescr[255];  //!< A (short) description of what big values mean
    
   QPixmap minIcon;   //!< Icon for min values
   QPixmap maxIcon;   //!< Icon for max values

   int type;            //!< The data type of the param.
			/*!< The possible values are <code>WA_TYPE_BOOL</code>,
			 * <code>WA_TYPE_INT</code> and
			 * <code>WA_TYPE_FLOAT</code>
			 */
   float max;           //!< The largest valid value, cast to float
   float min;           //!< the smallest valid value,  cast to float
   float dflt;          //!< The default value, cast to float

};


/*! @class WarpAlgorithm
 *
 *  @ingroup contourWarpClasses
 *	
 * @brief The abstract parent class of algorithms
 *
 * This is an abstract parent class for different algorithm variants that could
 * have been implemented during the work on this project. This class is used
 * for calling the methods from the main effect class.
 *
 * The main information in this class is:
 * - constant edges: 
 *		It is possible to specify additional edges which don't change
 *		position in the warping process. E.g. this is used in order to
 *		force the image in a constant frame when the "frame as
 *		edges"-option is enabled.  \b Note: The data structures for the
 *		constant edges are provided by derived classes.  
 * - vertex grid pointer:
 *		Each instance of a WarpAlgorithm derivate receives a pointer to
 *		an CVertexGrid instance at startup via the setVertexGrid()
 *		method that holds the vertices being warped.
 * .
 *
 * Additional functionality:
 * - parameters: 
 *		WarpAlgorithm derivates can declare parameters which influence
 *		the warp result.  The number and type of the parameters is
 *		arbitrary.
 * - precalculations:
 *		WarpAlgorithm derivates can do some precalculation with the
 *		original contour.  This is useful if the original contour
 *		stays fixed for a longer time than the transformed contour and
 *		the algorithm can do some precalculations only depending of the
 *		original contour.
 * .
 */
class WarpAlgorithm
{

protected:
	/*!
	 * @brief Constant edge flag
	 *
	 * A flag for enabling and disabling the integration of the
	 * constant edges in the calculation.
	 */
	bool  bUseConstantEdges;

	/*!
	 * @brief Vertex grid pointer
	 *
	 * This attribute is set via the setVertexGrid() method
	 */
	CVertexGrid *m_vertexGrid;

public:
	
	/*!
	 * @brief Algorithm precalculation
	 * 
	 * Allows the algorithm to do some precalculations only depending
	 * of the original contour.
	 * @param origShape The original contour
	 */
	virtual void doPrecalculation (PolygonShape *origShape) = 0;

	/*!
	 * @brief Precalculation memory deallocation
	 *
	 * Most kinds of precalculations as e.g. caching allocates some memory
	 * so this method allows the WarpAlgorithm derivate to free it up.
	 */
	virtual void undoPrecalculation() = 0;
	
	/*!
	 * @brief Vertex grid warp
	 *
	 * The main vertex grid warping method.
	 * @param origShape The original contour
	 * @param trafoShape The warped contour
	 * @param vPlaneNormal The normal vector of the plane in which the image lies.
	 */
	virtual void transformGrid (PolygonShape *origShape,
				    PolygonShape *trafoShape,
				    VECTOR3D vPlaneNormal) = 0;
	
//! @name parameter setter and getter
//! @{

	/*!
	 * @brief Sets a parameter
	 * @param ParamID parameter ID
	 * @param pValue pointer to the parameter (which type is unknown)
	 */
	virtual void setParam (int ParamID, void *pValue) = 0;

	/*!
	 * @brief Gets a parameter
	 * @param ParamID parameter ID
	 * @returns a pointer to a parameter (of unknown type)
	 */
	virtual void *getParam (int ParamID) = 0; 

	/*!
	 * @brief Gets parameter info
	 * @param ParamID parameter ID
	 * @param descr reference to a paramInfo structure to be filled.
	 * @note This information is used in AlgorithmControl for labeling.
	 */
	virtual void getParamInfo(int ParamID,paramInfo &descr) = 0;

	/*!
	 * @brief Gets number of parameters
	 */
	virtual int getParamCount() = 0; 
// @}

	/*! 
	 * @brief precalculation state
	 * @returns whether the algorithm does optimizations with
	 * precalculation
     */
	virtual bool doesPrecalculation() = 0;

//! @name constant edges
//! @{
	/*! 
	 * @brief Sets all constant edges
	 * @param pEdges pointer to an array of edges
	 * @param numEdges number of edges in the array
	 */
	virtual void setConstantEdges (Edge *pEdges, int numEdges) = 0;

	/*! 
	 * @brief Gets the number of constant edges
	 * @returns The number of constant edges
	 */
	virtual int  getConstantEdgeCount() = 0;

	/*! 
	 * @brief Gets all constant edges
	 * @param pEdges pointer to an array to be filled with the edges
	 * @param maxEdges maximal number of edges that have place in the array
	 */
	virtual void getConstantEdges(Edge *pEdges, int maxEdges) = 0;

	/*!
	 * @brief Sets whether constant edges are used in the warp algorithm
	 * @param value true, if constant edges are used in the warp algorithm.
	 */
	virtual void useConstantEdges(bool value){bUseConstantEdges = value;}
	/*!
	 * @brief Gets whether constant edges are used in the warp algorithm
	 * @returns whether constant edges are used in the warp algorithm.
	 */
	virtual bool getUseConstantEdges(){return bUseConstantEdges;}
//! @}

	/*!
	 * @brief Sets the vertex grid
	 * @param vertexGrid a valid pointer to a CVertexGrid instance
	 */
	virtual void setVertexGrid(CVertexGrid* vertexGrid)
	{m_vertexGrid = vertexGrid;}
	
	/*!
	 * @brief Gets the vertex grid
	 * @returns The vertex grid pointer
	 */
	virtual CVertexGrid *getVertexGrid(){return m_vertexGrid;}
};

/*! @class WarpAlgorithm1
 *
 *  @ingroup contourWarpClasses
*
* @brief A warp algorithm derived from WarpAlgorithm
*
* This is a warping algorithm we've implemented. It compiles the final position
* of a warped point from the affine transformations of _all_ edges, wedged
* depeding on the inverse of the distance to the point.  For math details see
* the specification
* 
* The parameters in this algorithm only influence the wedge function.  If
* changed, the weighted mean between the transformations of the edges is
* calculated differently.
*
* The sort of precalculation performed by this algorithm is wedge caching If
* the original contour stays fixed for a longer time than the transformed one
* the wedge per point and per edge can already be calculated.
*
* @note In differential mode which is supported in ContourWarpEffect the
* original contour passed to this algorithm changes between subsequent
* transform-calls, so ContourWarpEffect turns wedge caching off in this mode.
*/

/**
 * Number of parameters and parameter type macro
 */
#define WA1_PARAMCOUNT	 3
#define WA1_PARAMTYPE(x) WA_TYPE_FLOAT

class WarpAlgorithm1 : public WarpAlgorithm
{
private:

	/*!
	 * @brief Parameter array
	 * @note All parameters supported by this algorithm are of type float.
	 * @see setParam(int,void*)
	 */
	float Params[3];
	
	/*!
	 * @brief Array of constant edges
	 */
	Edge *constEdges;

	/*!
	 * @brief Number of array elements in <code>constEdges</code>
	 */
	int	  constEdgeCount;

	/*!
	 * @brief The wedge cache 
	 */
	float *wedgeCache;

	/*!
	 * @brief Flag whether wedge caching is enabled
	 */
	bool bWedgeCached;

// ---------------------------------------------------------------
// Private Methods
// ---------------------------------------------------------------
	
	/*!
	 * @brief An initialization method called by both constructors
	 */
	void initialize ();

	/*!
	 * @brief Transforms a point only with a single edge pair
	 *
	 * This method method "warps" a point with the affine transformation
	 * between an original edge and a warped one.
	 *
	 * For further details, see specification 2.3
	 *
	 * @param point pointer to the point being warped
	 * @param origEdge the original edge
	 * @param trafoEdge the warped edge
	 * @param vPlaneNormal the normal of the plane the image (and the two edges)
	 *          lies in.
	 */
	void trafoWithEdgePair (VECTOR3D *point, const Edge & origEdge,
				const Edge & trafoEdge, VECTOR3D vPlaneNormal);

	/*!
	 * @brief Returns the inverse distance weight of the edge for this point.
	 * 
	 * This helper method determines how important is the transformation of
	 * the given edge for the point at this position.  The final
	 * transformation of the pointed is calculated with a weighted mean of
	 * the warped edges (contoures).  The wedge result is also influenced
	 * by the parameters of the algorithm.	
	 *
	 * For further details, see specification 2.4
	 *
	 * @param vPoint the point being warped
	 * @param edge a warped edge
	 */
	float wedge (VECTOR3D vPoint, Edge edge);

	
	/*!
	 * @brief Transforms a single point
	 * 
	 * This method transform the point at index <code>index</code> in the
	 * vertex grid
	 * 
	 */
	void transform (int index, PolygonShape *origShape,
			PolygonShape *trafoShape, VECTOR3D vPlaneNormal);	

public:
	
	/*!
	 * @brief Wedge caching
	 *
	 * The sort of Precalculation done in WarpAlgorithm1 is wedge caching
	 * The wedge depends only from the vertex positions relative to the
	 * original shape.
	 * 
	 * @param origShape The original contour (shape)
     */
	virtual void doPrecalculation (PolygonShape *origShape);

	/*!
	 * @brief Precalculation memory deallocation
	 *
	 * Frees up the memory used by wedge caching.
	 */
	virtual void undoPrecalculation ();

	/*!
	 * @brief Vertex grid warp
	 *
	 * The main vertex grid warping method.
	 * @param origShape The original contour
	 * @param trafoShape The warped contour
	 * @param vPlaneNormal The normal vector of the plane in which the
	 * image lies.
	 */
	virtual void transformGrid (PolygonShape *origShape,
					PolygonShape *trafoShape, 
					VECTOR3D vPlaneNormal);
		
//! @name parameter setter and getter
//! @{
	
	/*!
	 * @brief Gets a parameter
	 * @param ParamID parameter ID
	 * @param descr reference to a paramInfo structure to be filled.  
     *
	 * @see setParam(int, void*)
	 */
	void getParamInfo(int ParamID,paramInfo &descr);

	/*!
	 * @brief Sets a parameter
	 *
	 * The meaning of the parameters are:
	 * - <code>Param[0]</code>: paramter a in the specification.
	 * - <code>Param[1]</code>: paramter b in the specification.
	 * - <code>Param[2]</code>: paramter p in the specification.
	 * For further details, see specification 2.4
	 *
	 * @param ParamID parameter ID
	 * @param pValue pointer to the parameter (which type is unknown)
	 *  
	 * @returns a pointer to a parameter (which type is always float here)
	 */
	void setParam (int ParamID, void *pValue);

	/*!
	 * @brief Gets parameter info
	 * @param ParamID parameter ID
	 *
	 * @see setParam(int, void*)
	 */
	void *getParam (int ParamID);

	/*!
	 * @brief Gets number of parameters (here always 3)
	 * @see setParam(int, void*)
	 */
	int getParamCount(){return WA1_PARAMCOUNT;}

// @}
	/*! 
	 * @brief precalculation state
	 * @returns whether the algorithm does optimizations with wedge caching
     */
	bool doesPrecalculation(){return bWedgeCached;}
	
//! @name constant edges
//! @{
	/*! 
	 * @brief Sets all constant edges
	 * @param pEdges pointer to an array of edges
	 * @param numEdges number of edges in the array
	 */
	void setConstantEdges (Edge *pEdges, int numEdges);

	/*! 
	 * @brief Gets the number of constant edges
	 * @returns The number of constant edges
	 */
	int  getConstantEdgeCount(){return constEdgeCount;}

	/*! 
	 * @brief Gets all constant edges
	 * @param pEdges pointer to an array to be filled with the edges
	 * @param maxEdges maximal number of edges that have place in the array
	 */
	void getConstantEdges(Edge *pEdges, int maxEdges);

	
	/*!
	 * @brief Sets whether constant edges are used in the warp algorithm
	 * @param value true, if constant edges are used in the warp algorithm.
	 * @note This method is reimplemented here in order to disable wedge
	 * caching.
	 */
	void useConstantEdges (bool value);
//! @}

	/*!
	 * @brief Sets the vertex grid
	 * @param vertexGrid a valid pointer to a CVertexGrid instance
	 * @note This method is reimplemented here in order to disable wedge
	 * caching.
	 */
	void setVertexGrid (CVertexGrid *vertexGrid);

	/*!
	 * @brief Constructor.
	 * 
	 * Initializes parameters and flags, sets pointer to NULL.
	 */
	WarpAlgorithm1();

	/*!
	 * @brief Constructor.
	 * 
	 * Initializes parameters and flags, sets pointer to NULL.
	 * Calls the setVertexGrid method.
	 */
	WarpAlgorithm1(CVertexGrid *vertexGrid);

	/*!
	 * @brief Destructor.
	 * 
	 * Deallocates all used memory
	 */
	~WarpAlgorithm1();
};

#endif

