/*
 * Copyleft 2001 Lukas Degener, Ingmar Kanitscheider. All rights granted ;-)
 *----------------------
 * file: src/WarpAlgorithm.cpp
 *----------------------
 * This file is part of the CountourWarp Effect based on the GLEffect Framework.
 * In this file you will find the implementation of the WarpAlgorithm* classes
 * Declaration is in WarpAlgorithm.h
 */


// include some standard libs
#include <string.h>
#include <math.h>
// We need some vector utility functions and operator overloading
// collected in Vector.h/.cpp (type VECTOR3D)
#include "vector.h"
// We need the edge class
#include "geomObj.h"
// We need the PolygonShape class holding the edges the points are being warped with
#include "Icon.h"
#include "PolygonShape.h"

// include the declaration of WarpAlgoritm* classes
#include "WarpAlgorithm.h"


//---------------------------------------------------------------------------
//  Some usefull macros
//---------------------------------------------------------------------------

#define SAFE_DELETE(x) {if (x) {delete(x);(x)=NULL;}}
#define SAFE_DELETE_ARRAY(x) {if (x) {delete[](x);(x)=NULL;}}

// one of this usefull macros to access dynamic arrays of higher dimensions...
// Caution: This array doesn't make a difference between cols and rows!
#define WEDGECACHE(i,k) wedgeCache[(k)*m_vertexGrid->m_numRows*m_vertexGrid->m_numCols+\
								   (i)]

/*****************************************************************************\
 *                              CVertexGrid class
\*****************************************************************************/

CVertexGrid::CVertexGrid ()
{
	// initialize pointers to NULL indicating they are uninitialized.
	m_pVertices = NULL;
	m_pInitVertices = NULL;
	m_pTexCoords = NULL;
	m_numCols = 0;
	m_numRows = 0;
}

CVertexGrid::CVertexGrid (VECTOR3D vTopLeft, VECTOR3D vTopRight,
						  VECTOR3D vBottomLeft,
						  int numCols, int numRows)
{
	// initialize pointers to NULL because otherwise,
	// SAFE_DELETE in SetGridData may fail.
	m_pVertices = NULL;
	m_pInitVertices = NULL;
	m_pTexCoords = NULL;
	SetGridData(vTopLeft, vTopRight, vBottomLeft, numCols, numRows);
}

CVertexGrid::~CVertexGrid ()
{
	// clean up
	SAFE_DELETE(m_pVertices);
	SAFE_DELETE(m_pInitVertices);
	SAFE_DELETE(m_pTexCoords);
}

bool CVertexGrid::SetGridData (VECTOR3D vTopLeft, VECTOR3D vTopRight,
						  VECTOR3D vBottomLeft,
						  int numCols, int numRows)
{
	SAFE_DELETE (m_pVertices);
	SAFE_DELETE (m_pInitVertices);
	SAFE_DELETE (m_pTexCoords);
	
	// allocate memory
	m_pVertices = new VECTOR3D[numCols*numRows];
	m_pInitVertices = new VECTOR3D[numCols*numRows];
	m_pTexCoords = new float[2*numCols*numRows];
	if (!m_pVertices || !m_pInitVertices || ! m_pTexCoords)
		return false;
	// copy number of rows and cols 
	// We need them already in the InitializeVertices() method
	m_numRows = numRows;
	m_numCols = numCols;

	// initialize member data
	m_vTopLeft = vTopLeft;
	m_vTopRight = vTopRight;
	m_vBottomLeft = vBottomLeft;

	// Distribute vertices equally over the field
	InitializeVertices();

	return true;
}

void CVertexGrid::InitializeVertices()
{
	for (int i=0; i < m_numCols; i++)
		for (int j=0; j < m_numRows; j++)
		{
			// Interpolate vertices in the rectangle.
			// There are e.g. numCols-1 intervalls between the TopLeft and TopRight
			// corner.
			Vector(i,j) = m_vTopLeft +
							i / (float) (m_numCols-1) * (m_vTopRight - m_vTopLeft) +
							j / (float) (m_numRows-1) * (m_vBottomLeft - m_vTopLeft);
			// set s-texcoord
			TexCoord(i, j, 0) = i / (float) (m_numCols-1);
			// set t-texcoord
			// tex coord origin in OpenGL is BottomLeft -> mirror t-texcoord
			TexCoord(i, j, 1) = 1 - j / (float) (m_numRows-1);
		}

	// make a backup so we can use ResetVertices in the future
	// instead of resetting each point
	if (m_pInitVertices)
		memcpy (m_pInitVertices, m_pVertices, m_numCols*m_numRows*sizeof(VECTOR3D));
}

void CVertexGrid::ResetVertices()
{
	if (m_pInitVertices)
		memcpy(m_pVertices, m_pInitVertices, m_numCols*m_numRows*sizeof(VECTOR3D));
}

//---------------------
// class WarpAlgorithm1
//---------------------
//****************************************************
//* For method documentation look in WarpAlgorithm.h *
//****************************************************


/* Implementation note:
* As described in the specification the algorithm is divorced in two parts:
* In the first part, the transformation is calculated for every edge pair.
* In the second part, the final position of the point being transformed
* is calculated by wedging between the edge-pair-transformations
*/

void WarpAlgorithm1::initialize()
{
	// set a=0
	Params[0] = 0;
	// set b=1
	Params[1] = 1;
	// set p=0
	Params[2] = 0;

	// set frame as edges by default
	bUseConstantEdges = true;

	// further initializations
	constEdgeCount = 0;
	constEdges = NULL;

	bWedgeCached = false;
	wedgeCache = NULL;
}

WarpAlgorithm1::WarpAlgorithm1()
{
	// call the constructor-common initialization method
	initialize();
}

WarpAlgorithm1::WarpAlgorithm1(CVertexGrid *vertexGrid)
{
	// call the constructor-common initialization method
	initialize();

	setVertexGrid (vertexGrid);
}

WarpAlgorithm1::~WarpAlgorithm1()
{
	// use the well-known SAFE_DELETE macro
	SAFE_DELETE(constEdges);
	SAFE_DELETE(wedgeCache);
}

void WarpAlgorithm1::getParamInfo(int ParamID,paramInfo &descr){
   switch(ParamID){
      // parameter a
	  case 0:
		strcpy (descr.name,"Grip");
		strcpy (descr.descr,"Gibt an, wie stark das Bild\n an den Skelletkanten klebt.");
		strcpy (descr.minDescr, "mehr grip");
		strcpy (descr.maxDescr, "gleichmigere\nVerzerrung");
        descr.minIcon=I_PARAM_MAXGRIP;
        descr.maxIcon=I_PARAM_MINGRIP;
		descr.type=WA_TYPE_FLOAT;
		descr.max=1.0;
		descr.min=0.0;
		descr.dflt=0.0;
      return;
	  // parameter b
      case 1:
		strcpy (descr.name,"Einflu der Distanz");
		strcpy (descr.descr,"Legt fest, wie stark der Einflu einer\nKante von ihrer Entfernung\nabhngt");
		strcpy (descr.minDescr, "gar nicht");
		strcpy (descr.maxDescr, "sehr stark");
        descr.minIcon=I_PARAM_MINDISTANCE;
        descr.maxIcon=I_PARAM_MAXDISTANCE;
		descr.type=WA_TYPE_FLOAT;
		descr.max=5.0;
		descr.min=0.0;
		descr.dflt=1.0;
      return;
	  // parameter p
      case 2:
		strcpy (descr.name,"Einflu der Lnge");
		strcpy (descr.descr,"Gibt an, ob kurze oder lange\nKanten greren Einflu haben");
		strcpy (descr.minDescr, "kurze Kanten");
		strcpy (descr.maxDescr, "lange Kanten");
        descr.minIcon=I_PARAM_MINLENGTH;
        descr.maxIcon=I_PARAM_MAXLENGTH;
		descr.type=WA_TYPE_FLOAT;
		descr.max=5.0;
		descr.min=-5.0;
		descr.dflt=0.0;
	  return;
   }
}


void WarpAlgorithm1::doPrecalculation (PolygonShape *origShape)
{
	// if this method is called, previous caching is surely obsolete
	SAFE_DELETE (wedgeCache);
	bWedgeCached = false;

	// get number of edges in origShape
	int ShapeCount = origShape->getEdgeCount();
	// perhaps we must add some constant edges
	int EdgeCount = ShapeCount + (bUseConstantEdges ? constEdgeCount : 0);
	// allocate memory for each point and edge
	wedgeCache = new float[m_vertexGrid->m_numRows*m_vertexGrid->m_numCols*EdgeCount];
	// This could be a lot of memory, so if it's too much, exit now...
	if (!wedgeCache)
		return;

	for (int k=0; k<EdgeCount; k++)
	{
		// store the current edge temporarily
		Edge edge;
		if (bUseConstantEdges)
		{	
			// write the wedges for the constant edges at the end of the array
			if (k >= ShapeCount)
				edge = constEdges[k-ShapeCount];
			else
				edge = origShape->getEdge(k);
		} else
			edge = origShape->getEdge(k);

		// calculate the wedge of this edge for all points in the grid
		for (int i=0; i<m_vertexGrid->m_numCols*m_vertexGrid->m_numRows; i++)
			WEDGECACHE(i,k) = wedge(m_vertexGrid->Vector(i), edge);
	}

	// now we have again a valid wedge cache
	bWedgeCached = true;
}

void WarpAlgorithm1::undoPrecalculation()
{
	// deallocate memory
	SAFE_DELETE (wedgeCache);
	// turn wedge caching off
	bWedgeCached = false;
}

void WarpAlgorithm1::transformGrid (PolygonShape *origShape,
									PolygonShape *trafoShape, VECTOR3D vPlaneNormal)
{
	// make sure all pointers are valid
	if (!m_vertexGrid || !origShape || !trafoShape)
		return;

	// calculate the transformation for each point
	for (int i=0; i<m_vertexGrid->m_numCols*m_vertexGrid->m_numRows; i++)
		transform (i, origShape, trafoShape, vPlaneNormal);
}


void WarpAlgorithm1::transform(int index, PolygonShape *origShape,
		PolygonShape *trafoShape, VECTOR3D vPlaneNormal)
{
	// get the vector
	VECTOR3D & point = m_vertexGrid->Vector(index);

	// ask for the number of edges
	int origEdgeCount = origShape->getEdgeCount();
	int trafoEdgeCount = trafoShape->getEdgeCount();

	// perhaps add number of constant edges
	if (bUseConstantEdges && constEdges)
	{
		origEdgeCount += constEdgeCount;
		trafoEdgeCount += constEdgeCount;
	}

	// if there are no edges, exit now...
	if (origEdgeCount == 0 || trafoEdgeCount == 0)
		return;

	// For different number of edges the algorithm isn't implemented yet.
	if (origEdgeCount != trafoEdgeCount)
		return;
	
	// initialize two arrays of edges
	Edge *origEdges = new Edge [origEdgeCount];
	Edge *trafoEdges = new Edge [trafoEdgeCount];
	// return if there is no memory
	if (!origEdges || !trafoEdges)
		return;

	// The Visual C++ compiler doesn't terminate the valuability of a variable declared in
	// the header of a for-loop at the end
	// of the loop, so I must declare it before, if I wish to use the same variable
	// in several for-loops.
	int i;
	
	// first, copy edges from PolygonShape to edge array
	for (i=0; (i < origShape->getEdgeCount() && i < trafoShape->getEdgeCount()); i++)
	{
		origEdges[i] = origShape->getEdge(i);
		trafoEdges[i] = trafoShape->getEdge(i);
	}

	// then, copy the constant edges
	if (bUseConstantEdges && constEdges)
	{
		memcpy (origEdges+i, constEdges, constEdgeCount*sizeof(Edge));
		memcpy (trafoEdges+i, constEdges, constEdgeCount*sizeof(Edge));
	}

	// DANGER!!!!
	// The following code must be changed if there are different numbers of edges
	// in the two shapes
	// For now, I'm taking the minimum of the numbers of edges (incl. constant edges)
	int minEdgeCount = ((origEdgeCount < trafoEdgeCount) ? origEdgeCount : trafoEdgeCount);
	
	// I need an array to storing the function value for the transformation function
	// with a single edge pair. (see specification 2.3)
	VECTOR3D *functValue = new VECTOR3D [minEdgeCount];

	for (i=0; i<minEdgeCount; i++)
	{
		// initialize functValue array with the vector of the point being transformed
		functValue[i] = point;
		// transform with the i-th edge pair.
		trafoWithEdgePair (&functValue[i], origEdges[i], trafoEdges[i], vPlaneNormal);
	}

	VECTOR3D sum1 = VECTOR3D (0.0f, 0.0f, 0.0f);
	float sum2 = 0.0f;
	float w;
	for (i=0; i<minEdgeCount; i++)
	{
		// if we have a valid wedge cache, use it
		if (bWedgeCached)
			w = WEDGECACHE(index, i);
		else
			// calculate wedge
			w = wedge(point, origEdges[i]);
		
		if (w>1e35)
		{
			// There was a division through zero in the wedge calculate
			// This means the point lies directly on the edge (if a = 0);
			// force sum1/sum2 to be zero
			sum1 = VECTOR3D (0.0f,0.0f,0.0f);
			sum2 = 1.0f;
			break;
		}
		
		// (functValue[i] - point) is difference function
		// build sums to calculate wedged medium
		// for further details see specification 2.4
		sum1 = sum1 + w*(functValue[i]-point);
		sum2 = sum2 + w;
	}

	// add the difference to the position of the point
	point += sum1/sum2;

	// clean up
	delete functValue;
	delete origEdges;
	delete trafoEdges;
}

void WarpAlgorithm1::trafoWithEdgePair(VECTOR3D *pPoint, const Edge &origEdge, 
									   const Edge &trafoEdge, VECTOR3D vPlaneNormal)
{
	// build point for easier handling
	// in the following code, I use the same names as in the specification
	VECTOR3D x = *pPoint;
	
	// PQ = Q - P
	VECTOR3D PQ = origEdge.point2 - origEdge.point1;
	// for easier handling
	VECTOR3D P = origEdge.point1;
	// turn 90 degrees by taking the cross product with the normalized plane normal vector
	// DANGER!!: If the angle between vPlaneNormal and PQ isn't 90 degrees, the 
	//           magnitude of PQ90 is less than PQ.
	VECTOR3D PQ90 = vecCrossProduct(PQ,vecNormalize(vPlaneNormal));

	// u = <x - P, Q - P> / ||Q-P||^2
	float u = vecDotProduct(x - P, PQ)/vecSqrMagnitude(PQ);
	// v = <x - P, rho90(Q-P)> / ||Q-P||
	// if PQ90 is really less than PQ, the following variant is safer:
	float v = vecDotProduct(x - P, PQ90)/vecMagnitude(PQ90);

	// P'Q' = Q' - P'
	VECTOR3D P1Q1 = trafoEdge.point2 - trafoEdge.point1;
	// for easier handling
	VECTOR3D P1 = trafoEdge.point1;
	// turn 90 degrees by taking the cross product with the normalized plane normal vector
	VECTOR3D P1Q190 = vecCrossProduct(P1Q1,vecNormalize(vPlaneNormal));

	// x = P' + u(Q'-P') + v*rho90(Q'-P')/||Q'-P'||
	*pPoint = P1 + u*P1Q1 + v*P1Q190/vecMagnitude(P1Q1);
	
}

float WarpAlgorithm1::wedge(VECTOR3D vPoint, Edge edge)
{
	// if you want to find the variables from the specification (equation (11) in 2.4):
	// a is Params[0]
	// b is Params[1]
	// p is Params[2]
	float length = vecMagnitude (edge.point2-edge.point1);
	float distance = vecMagnitude(vPoint-projection(edge,vPoint));

	return pow (pow(length,Params[2])/(Params[0]+distance),Params[1]);
}

void WarpAlgorithm1::setParam(int ParamID, void *pValue)
{
	switch (ParamID)
	{
		case 0:
		case 1:
		case 2:
			Params[ParamID] = *(float *)pValue;
			break;
	}
}

void *WarpAlgorithm1::getParam(int ParamID)
{
	switch (ParamID)
	{
		case 0:
		case 1:
		case 2:
			return (void *)&Params[ParamID];
			break;
		default:
			return NULL;
	}
}

void WarpAlgorithm1::getConstantEdges(Edge *pEdges, int maxEdges)
{
	// get as many edges as the minimum of maxEdges and constEdgeCount
	if (pEdges)
		memcpy (pEdges, constEdges,
			((maxEdges < constEdgeCount) ? maxEdges : constEdgeCount)*sizeof (Edge));
}

void WarpAlgorithm1::setConstantEdges(Edge *pEdges, int numEdges)
{
	SAFE_DELETE(constEdges);
	constEdges = new Edge[numEdges];
	constEdgeCount = numEdges;
	memcpy (constEdges, pEdges, numEdges*sizeof (Edge));

	// the wedge cache is now invalid
	undoPrecalculation();
}

void WarpAlgorithm1::useConstantEdges(bool value)
{
	WarpAlgorithm::useConstantEdges(value);

	// the wedge cache is now invalid
	undoPrecalculation();
}

void WarpAlgorithm1::setVertexGrid(CVertexGrid *vertexGrid)
{
	WarpAlgorithm::setVertexGrid(vertexGrid);

	// the wedge cache is now invalid
	undoPrecalculation();
}

