/*
 * Copyleft 2001 Lukas Degener, Ingmar Kanitscheider. All rights granted ;-)
 *----------------------
 * file: src/PolygonShape.h
 *----------------------
 * This file is part of the CountourWarp Effect based on the GLEffect Framework.
 * In this file you will find the declaration of the PolygonShapeEditor class
 * Implementation is in PolygonShapeEditor.cpp
 */

/*! @file PolygonShapeEditor.h
 *
 *  @brief
 *  Declaration of PolygonshapeEditor class
 *  @author Ingmar Kanitscheider, Lukas Degener
 */


#ifndef POLYGONSHAPEEDITOR_H
#define POLYGONSHAPEEDITOR_H

#include<qwidget.h>
#include<qpopupmenu.h>

#include "vector.h"
#include "PolygonShape.h"


//states of the editor (mState)
#define STATE_READY                 0
#define STATE_DRAWING_STRIP         1
#define STATE_TRANSLATING           2
#define STATE_SCALING               3
#define STATE_ROTATING              4
#define STATE_SELECTING_CIRCLE      5
#define STATE_SELECTING_BOX         6

//the states of the Interpreter (mIState);
#define IS_READY                    0
#define IS_TRANSFORM                1
#define IS_REGION                   2
#define IS_DRAW                     3
#define IS_CANCEL                   4

//the editmodes of the editor (mEditMode)
#define MODE_DRAW                   0
#define MODE_TRANSLATE              1
#define MODE_SCALE                  2
#define MODE_ROTATE                 3


//the region modes for selecting a region (mRegionMode)
#define REGION_BOX                  4
#define REGION_CIRCLE               5

#define ACTION_DELETE               7

/*! @class PolygonShapeEditor
 *
 *  @ingroup contourWarpClasses
 *
 * @brief a class to edit a PolygonShape by interpreting user interactions
 *
 * This class is supposed to do th actual creating/editing of the contour lines
 * represented by an instance of PolygonShape.
 *
 * This is my new Implenetation of the old PolygonShapeEditor. This new version
 * makes full use of the enhanced Features of the new PolygonShape class.
 * As a result, here are some highlights of the new variant:
 *
 * -  An aditional layer of Abstraction was added to seperate the event handlers from the
 *    actual edit actions. You can think of this as of dividing the member function into
 *    two categories: One implements the interface towards the user, the other one does
 *    the actual communication with the PolygonShape. The methods in the second category
 *    are moreless an abstraction of the user actions.
 *
 * -  Consequently editing is now done in to steps: First the user actions are
 *    interpreted. During this phase, all state information is put into account to decide
 *    what the user wants (and what he is allowed!) to do.
 *    The second phase caries out the actual editing of the shape, or more precise,
 *    tells the Shape what to do.
 *
 * -  As a result of the previos point, interpretation of the user interaction is now
 *    much more flexible. Just replace the methods dedicated to interpretion.
 *
 * -  As a result of the PolygonShape's new selection model and it's transformation
 *    feature, a lot of new feature were added to the Editor. It's now possible to
 *    select multible nodes for editing i.e. translate/rotate/scale them.
 *
 * @note Now follows a very detailed description of how the Interpreter figures out what
 * ever the user is intending to do. I doubt that anyone besides me is realy interested
 * in the gory details, but i put it down aswell, just to make sure i know what the editor
 * should act like, and how far we are away from it.
 * So please everybody feel free to skip this part ;-)
 *
 * STATEs and MODEs:
 * There are currently four state variables to reflect the current state of
 * interpretation:
 * -  mState represents the current state of the Editor
 *    This state reflects which action is currently taken by the Editor
 * -  mIState represents the current state of the Interpreter
 *    It reflects what the Interpreter currently thinks about what the user is doing
 * -  mEditMode represents the current Transformation mode.
 *    It tells the Interpreter what kind of Modifications are requested by the user
 * -  mRegionMode represents the current selection mode
 *    When doing region selects, this determines what kind if region the user
 *    intends to select. rectangular or spherical regions are possible
 *
 *    @todo Only rectangular regions are implemented so far. The code for spherical
 *    regions is implemented, but needs testing and adaequat UI representation
 *
 * Interpretation of User Interactions:
 * From the Interpretors point of view, there are five basic things the user can do
 * apart from the usual things like asking stupid questions or looking concerned due to
 * serious damage or criticizing this imaculate artwork of software:
 * He can
 * -  Add/remove Edges/Nodes
 * -  Select things which he thinks are beautifull, like nodes for instance, or me,
 *    or Ingmar.
 * -  Do some transformation to the selected stuff, which was previously choosen by
 *    mEditMode. (Argh! please don't rotate me... i'm gonna be sick! ;-) )
 * -  Change his mind like he always does and cancel his current action
 * -  Change edit mode or region mode.
 *
 * This is how the interpreter decides what to do:
 * Imagine the interpreter as some kind of state machine (DFA)
 * Its initial state is <code>IS_READY</code>
 * -  Left-Clicking on a node without any modifier key will select this node.
 *    If the node was previosly selected, selection remains unchanged. Otherwise,
 *    the previous selection will be discarded,
 *    After this, the Interpreter jumps into state <code>IS_TRANSFORM</code>, unless the
 *    current edit mode is <code>MODE_DRAW_STRIP</code>. In this case, the interpreter
 *    will be pushed into state <code>IS_DRAW</code>, and an edge strip will be started
 *    at this node.
 * -  Left-Clicking on a node while holding down SHIFT will toggle the node between
 *    selected or unselected. The Interpreter remains in state <code>IS_READY</code>
 * -  Left-Clicking into an empty area will clear the selection and push the
 *    Interpretor into state <code>IS_REGION</code>, unless the current edit mode is
 *    <code>MODE_DRAW</code>. In this case, a new edge strip will be started at the
 *    mouse position, and the interpreter is pushed into state <code>IS_DRAW</code>
 * -  Left-Clicking into an empty area while holding down Shift will keep current
 *    Selection and switch into state <code>IS_REGION</code>
 * -  Middle-Clicking into an empty area will set the origin for rotating/scaling to
 *    this position -> <code>IS_READY</code>
 *
 * In State <code>IS_TRANSFORM</code>
 * -  Moving the mouse while left button down will cary out the transformation
 *    specified by <code>mEditMode</code> -> <code>IS_TRANSFORM</code>
 * -  Releasing all mouse buttons will finish the job. -> <code>IS_READY</code>
 * -  Pressing right button (while still holding down left) will cancel the current
 *    Transformation -> <code>IS_CANCEL</code>
 *
 * In State <code>IS_DRAW</code>
 * -  pressing the left mouse button will add a node to a strip. If there is no node
 *    below the pointer, a new one is created. -><code>IS_DRAW</code>
 * - left-clicking while holding down CTRL will finish the strip, adding the last node
 *    at this position. if no node is under the cursor, a node will be added.
 *    -><code>IS_READY</code>
 * -  right-clicking will cancel the current edge, i.e. no last node will be added, but the
 *    interpreter will directly jump to -> <code>IS_CANCEL</code>
 *
 * In State <code>IS_REGION</code>
 * -  Moving the mouse while left Button down will draw a region (either rectangular
 *    or circular, depending on <code>mRegionMode</code>) -> <code>IS_REGION</code>
 * -  Releasing all Mouse buttons will finish the job, i.e. select all nodes in the
 *    region.  -> <code>IS_READY</code>
 * -  Pressing right button (while still holding down left) will cancel selection
 *    No nodes are selected, i.e. the selection keeps as it is. -> <code>IS_CANCEL</code>
 *
 * In State <code>IS_CANCEL</code>
 * -  Releasing all buttons will switch the Interpreter back to -> <code>IS_READY</code>
 */
class PolygonShapeEditor: public QWidget{
   Q_OBJECT

//--------------------------------------------------------------------------------------
// Slots
//--------------------------------------------------------------------------------------

public slots:
   /**
    * @brief set the "size" of nodes
    *
    * Sets the radius of the nodes. A  node will be regarded as "under the mouse"
    * if the distance between this node and the mouse coordinates is equal or less
    * than radius
    * @param r the new radius
    * @see getRadius(), mRadius
    */
   void setRadius(float r);

   /**
    * @brief enables/disables restricted mode
    * @param preserve if true, restricted mode is enabled
    * @see getRestricted(), mRestricted
    */
   void setRestricted(bool preserve=true);

   /**
    * @brief Sets the current edit mode.
    * @param mode the new edit mode. Should be one of the modes mentioned
    * at getEditMode()
    * @see getEditMode(), mEditMode
    */
   void setEditMode(int mode);

   /**
    * @brief Sets the current region mode
    * @param mode the new region mode. Should be one of the modes mentioned
    * at getRegionMode()
    * @see getRegionMode(), mRegionMode
    */
   void setRegionMode(int mode);

   /**
    * @brief cycles up through editmodes.
    *
    * This is usefull for instance to connect WheelEvents to it
    */
   void nextEditMode();

   /**
    * @brief cycles down through editmodes.
    *
    * This is usefull for instance to connect WheelEvents to it
    */
   void previousEditMode();
//--------------------------------------------------------------------------------------
// Signals
//--------------------------------------------------------------------------------------
signals:
   /**
    * @brief emitted when the edit mode changes
    * @param mode the new edit mode
    */
   void editModeChanged(int mode);

   /**
    * @brief emitted when the region mode changes
    * @param mode the new region mode
    */
   void regionModeChanged(int mode);

   /**
    * @brief emitted when the PolygonShape is modified
    *
    * This signal is emitted on every modification done to the nodes/edges of
    * the current PolygonShape. Use this to decide when to redraw the shape.
    */
   void shapeChanged();

//--------------------------------------------------------------------------------------
// Constructor/destructor/initializer
//--------------------------------------------------------------------------------------

//! @name Constructing/deconstructing
//! @{
public:
   /*!
    * @brief creates a new editor
    *
    * This will create an editor instance.
    * @note All parameters will be passed to the QWidget constructor.
    *       Please note that initialy no PolygonShape is set, so editing is only
    *       possible after seting it with setShape(PolygonShape *)
    *
    * @todo Have to check if mShape is null. Right now, no such check is performed,
    *       which could lead to seg faults if the user tries to edit a shape before
    *       pointing the editor to it.
    */
   PolygonShapeEditor(QWidget* parent = 0, const char* name = 0, WFlags fl = 0 );

   /*! @brief deconstuctor
    *
    * Does nothing.
    */
   ~PolygonShapeEditor();
//! @}
//--------------------------------------------------------------------------------------
// Properties
//--------------------------------------------------------------------------------------
//! @name Properties and information retrieval
//! @{

   /*!
    * @brief sets the PolygonShape to edit
    *
    * tells the Editor to edit the PolygonShape pointed at by <code>s</code>
    */
   void setShape(PolygonShape * s);

   /*!
    * @brief returns the current shape
    *
    * @returns a pointer to the PolygonShape currently edited by this editor
    */
   PolygonShape * getShape();



   /**
    * @brief gets the virtual size of the nodes
    *
    * The size of a node specifies the radius of its sensitive area.
    * A node will be regarded as beeing "under the mouse" if the position of the
    * mouse pointer is within a distance from the node's position
    * smaller than or equal to this radius.
    * @returns the virtual radius (aka size) of the nodes
    */
   float getRadius();


   /**
    * @brief checks wether the editor is in restricted mode
    *
    * while being in restirced mode, the editor will only allow isomorph modifications
    * of the edited PolygonShape, i.e. all operation that would modify the shape's
    * topology are be prohibited. In paritcular, no nodes/edges will be added/removed,
    * while by the editor.
    *
    * @note restricted mode is what is typicaly desired during waping. As our warping
    * algorithms rely on on the fact that the original and the transformed shape are
    * isomorph to each other, adding or removing nodes during this phase could lead to
    * unpredictable results.
    * @returns true if the editor is currently in restricted mode
    */
   bool getRestricted();


   /**
    * @brief retrieve the current edit mode
    * @returns the current transformation mode of the editor.
    * the returned value is one of
    * <code>MODE_DRAW, MODE_TRANSLATE, MODE_SCALE, MODE_ROTATE</code>
    */
   int getEditMode();


   /**
    * @brief retrieve the current region mode
    * @returns the current region mode. The returned value is one of
    * the returned value is
    * <code>REGION_BOX</code> or <code>REGION_CIRCLE<code>
    */
   int getRegionMode();

   /**
    * @brief retrieve the editor's popup
    *
    * The editor is equiped with a popup menu that will show when you click the right
    * mouse button while in state <code>IS_READY</code>. In this menu the user
    * may choose the edit mode, the region mode and is given the oporunity to remove
    * all selected nodes from the shape being edited. You can retrieve a pointer to this
    * popup to make it available at different locations within your application.
    *
    * @returns a pointer to the popup menu used in this editor
    */
   QPopupMenu * getPopup();

   /**
    * @brief retrieve the editors origin.
    *
    * @returns the origin assumed for rotation/scalation
    */
   VECTOR3D getCenter();
//--------------------------------------------------------------------------------------
// hints for rendering:
//--------------------------------------------------------------------------------------
   /**
    * @brief retrieve the current edited Edge
    *
    * Actualy their are two slightly different semantics connected to this method:
    *
    * -  during the drawing of a strip, the current Edge is returned
    * -  while doing a region select, this will return an edge
    *    consisting of the start point (where the button was pressed)
    *    and the current mouse position.
    *    Use this to draw a box, or a circle (first point is center), or what ever you
    *    feel like.
    *
    * @returns an Edge composed of the two points.
    */
   Edge getEditEdge();

   /**
    * @brief retrieve the current editor state
    *
    * This might be usefull to decide what to draw (or not to draw)
    * during rendering. You might for example want to draw the selection box during
    * a region select.
    *
    * @returns the current state of the editor. The returned value will be one of
    * <code>STATE_READY, STATE_DRAWING_STRIP, STATE_TRANSLATING, STATE_SCALING,
    * STATE_ROTATING, STATE_SELECTING_CIRCLE, STATE_SELECTING_BOX</code>
    */
   int getEditorState();
//! @}
//--------------------------------------------------------------------------------------
//Event handler
//This handlers are for mouse actions regarding the GL Panel.
//Therefor i have to rename them, to make sure they don't interfere with our own ones
//--------------------------------------------------------------------------------------
//! @name Event handlers
//! @{

   /**
    * @brief Mouse move events handler.
    * @param e the QMouseEvent representing this event
    * @param point the mouse position in 3-D space. Please Note that the
    *        editor does nothing to asure that this is on the image plane.
    *        So projection to this plane (if desired!) and the like should be
    *        done BEFORE passing the position to this method
    */
   void glPanelMoveEvent(QMouseEvent* e, VECTOR3D point);
   /**
    * @brief Mouse press events handler.
    * @param e the QMouseEvent representing this event
    * @param point the mouse position in 3-D space. Please Note that the
    *        editor does nothing to asure that this is on the image plane.
    *        So projection to this plane (if desired!) and the like should be
    *        done BEFORE passing the position to this method
    */
   void glPanelPressEvent(QMouseEvent* e, VECTOR3D point);
   /**
    * @brief Mouse release events handler.
    * @param e the QMouseEvent representing this event
    * @param point the mouse position in 3-D space. Please Note that the
    *        editor does nothing to asure that this is on the image plane.
    *        So projection to this plane (if desired!) and the like should be
    *        done BEFORE passing the position to this method
    */
   void glPanelReleaseEvent(QMouseEvent* e, VECTOR3D point);
//! @}
//--------------------------------------------------------------------------------------
// private edit methods (Abstracted user actions)
//--------------------------------------------------------------------------------------
//! @name abstracted user actions
//! @{
private:
   /**
    * @brief Edit mode dispatcher
    *
    * this method will pass <code>point</code> on to
    * the apropriate transformation method, determined by the value of
    * mEditMode
    * @param point the mouse position. will be passed on to the
    *       apropriate transformation method.
    * @see doTranslate(VECTOR3D), doScale(VECTOR3D), doRotate(VECTOR3D)
    */
   void doTransform(VECTOR3D point);

   /**
    * @brief Edit mode dispatcher
    *
    * this method will call
    * the apropriate transformation method, determined by the value of
    * mEditMode
    * @see finishTranslate(), finishScale(), finishRotate()
    */
   void finishTransform();

   /**
    * @brief Edit mode dispatcher
    *
    * this method will call
    * the apropriate transformation method, determined by the value of
    * mEditMode
    * @see cancelTranslate(), cancelScale(), cancelRotate()
    */
   void cancelTransform();

   /**
    * @brief Region mode dispatcher
    *
    * this method will pass <code>point</code> on to
    * the apropriate region select method, determined by the value of
    * mRegionMode
    * @param point the mouse position. will be passed on to the
    *       apropriate method.
    * @see doBox(VECTOR3D), doCircle(VECTOR3D)
    */
   void doRegion(VECTOR3D point);

   /**
    * @brief Region mode dispatcher
    *
    * this method will call
    * the apropriate region select method, determined by the value of
    * mRegionMode
    * @see finishBox(), finishCircle()
    */
   void finishRegion();

   /**
    * @brief Region mode dispatcher
    *
    * this method will call
    * the apropriate region select method, determined by the value of
    * mRegionMode
    * @see cancelBox(), cancelCircle()
    */
   void cancelRegion();

      /**
       * @brief start/continue an edge strip
       *
       * You can leeeave your hat on...;-) Hehe, nasty you!
       * No, not what you think. This starts or continues drawing
       * a strip of lines.
       * @returns the value returned by PolygonShape::continueStrip(VECTOR3D)
       */
      int doStrip(VECTOR3D point);

      /**
       * @brief finish an edge strip
       *
       * Finish a Strip, taking point as the last point.
       *
       * @returns the value returned by PolygonShape::continueStrip(VECTOR3D)
       */
      int finishStrip(VECTOR3D point);

      /**
       * @brief cancel drawing of current edge
       *
       * Cancels a strip, that is, the current line is not added.
       * @returns the key of the node last added to the strip.
       */
      int cancelStrip();

      /**
       * @brief start/continue translation
       *
       * starts or continues translations of selected nodes
       * @param point the position of the mouse. the translation is calculated so that
       *        the mouse positoin before the start of scalation is mapped
       *        exactly to the current mouse position.
       * @returns a VECTOR3D representing the difference of the new position
       * to the positions before the call of this method
       */
      VECTOR3D doTranslate(VECTOR3D point);

      /**
       * @brief finish translation
       *
       * @returns a VECTOR3D representing the total difference of the new node
       * positions to the positions before the beginnung of the translation
       */
      VECTOR3D finishTranslate();

      /**
       * @brief cancles translation
       *
       * resets selected nodes to their old
       * positions.
       * @returns a VECTOR3D representing the total difference of the
       * positions the node would had before canceling the translation
       * to the positions before the beginnung of the translation
       */
      VECTOR3D cancelTranslate();

      /**
       * @brief start/continue scaling
       *
       * starts or continues scaling of selected nodes
       * @param point the position of the mouse. the factors are calculated so that
       *        the mouse positoin before the start of scalation is mapped
       *        exactly to the current mouse position.
       * @returns a VECTOR3D containing the factors along each axis that where applied
       * by this method call.
       */
      VECTOR3D doScale(VECTOR3D point);

      /**
       * @brief finish scaling
       * @returns a VECTOR3D containing the total factors along each axis that were aplied
       * since the begining of the scalation.
       */
      VECTOR3D finishScale();

      /**
       * @brief cancles scaling
       *
       * resets selected nodes to their old
       * positions.
       * @returns a VECTOR3D containing the factors along each axis that would have been
       * applied since the begining of scalation if the user hadn't canceled.
       */
      VECTOR3D cancelScale();

      /**
       * @brief start/continue rotation
       *
       * starts or continues rotating along the z axis
       * @note theoreticaly there would be no problem to alow another axis
       *       But actualy i din't see the point in that regarding our situation.
       *       Apart that, i wouldn't know how to communicate this to the user.
       * @param point the position of the mouse. the angle is calculated so that
       *        the mouse positoin before the start of scalation is mapped
       *        exactly to the current mouse position.
       * @returns the angle about which the selected points are rotated during this
       * method call.
       */
      float doRotate(VECTOR3D point);

      /**
       * @brief finish rotation
       * @returns the total angle applied since the beginning of the rotation
       */
      float finishRotate();

      /**
       * @brief cancles rotation
       *
       * resets selected nodes to their old
       * positions.
       * @returns the total angle that would have been applied if the user hadn't canceled
       */
      float cancelRotate();

      /**
       * @brief starts/continues box selection
       * @param point if the selection has just started, this will be
       *        the first corner of the box. Otherwise it will be the opposite corner
       */
      void doBox(VECTOR3D point);

      /**
       * @brief finish box selection
       *
       * everything in the marked rectangluar region becomes selected
       */
      void finishBox();

      /**
       * @brief cancels box selection
       *
       * selection is left unchanged
       */
      void cancelBox();

      /**
       * @brief starts/continues circle selection
       * @param point if selection has just started, this will be the center of the
       *        spheric region. Otherwise, the distance between this point an the center
       *        will be taken as the radius of the region.
       */
      void doCircle(VECTOR3D point);

      /**
       * @brief finish circle selection
       *
       * everything in the spheric region becomes selected
       */
      void finishCircle();

      /**
       * @brief cancels circle selection
       *
       * selection is left unchanged
       */
      void cancelCircle();

      /**
       * @brief Select all nodes
       */
      void selectAll();

      /**
       * @brief Unselect all nodes
       */
      void clearSelection();

      /**
       * @brief Select a node
       */
      void select(int key);

      /**
       * @brief Unselect a node
       */
      void unselect(int key);
//! @}
//--------------------------------------------------------------------------------------
// private utility functions
//--------------------------------------------------------------------------------------
//! @name private utility function
//! @{
private:
   /**
    * @brief creates child widgets
    */
   void initGui();

   /**
    * @brief find a node at <code>point</code>
    * @returns the key of a node which is under the mouse
    * or -1 if there is no such node.
    */
   int findNode(VECTOR3D point);

   /**
    * @brief calculates factors for scaling
    *
    * calculates facors that map the components of <code>from</code> to their
    * pendants in <code>to</code>. This method is need in doScale(VECTOR3D) to
    * set up the transformation matrix
    *
    * @returns a VECTOR3D containing the three factors
    */
   VECTOR3D getFactors(VECTOR3D from, VECTOR3D to);

   /**
    * @brief calculate the polar angle
    *
    * @returns the polar angel of v in radians
    */
   float radians(VECTOR3D v);

   /**
    * @brief calculate the angle between two lines
    *
    * calculate the angle between two line going from mCenter to <code>from</code>
    * and from mCenter to <code>to</code>
    * @returns the angle between this lines at mCenter
    */
   float getAngle(VECTOR3D from, VECTOR3D to);

   /**
    * @brief create Identity matrix
    *
    * fills a float[16] array so it represents 4x4 Identity
    * @param matrix should point to memory big enough to store a float[16] array
    */
   void fillMatrix(float* matrix);

   /**
    * @brief creates a translation matrix
    *
    * alters matrix to represent an translation.
    * @param matrix is assumed to represent 4x4 Identity as produced by fillMatrix(float *)
    * @param t the VECTOR3D along which the translation should accure
    */
   void translate(float* matrix,VECTOR3D t);

   /**
    * @brief creates an affine rotation matrix
    *
    * alters matrix to represent a rotation around an axis oriented by the line
    * R*(0,0,1) + c along angle angle rad
    * @param matrix is assumed to represent 4x4 Identity as produced by fillMatrix(float *)
    * @param c the center assumed for the rotation
    * @param rad the rotation angle in radians
    */
   void rotate(float* matrix,VECTOR3D c,float rad);

   /**
    * @brief creates an affine scalation matrix
    *
    * alters a matrix to represent an affine scalation with moved origin
    * @param matrix is assumed to represent 4x4 Identity as produced by fillMatrix(float *)
    * @param c the center assumed for the rotation
    * @param s a VECTOR3D holding the three factors
    */
   void scale(float* matrix,VECTOR3D c,VECTOR3D s);
//! @}
//--------------------------------------------------------------------------------------
// Member Fields
//--------------------------------------------------------------------------------------
   /**
    * @brief The Popup Menu
    * @see getPopup()
    */
   QPopupMenu * mPopup;

   /**
    * @brief the current state of the Editor
    * @see getEditorState
    */
   int mState;

   /**
    * @brief the current state of the Interpreter
    *
    * This holds the state in which the interpretation has currently reached
    * The value of <code>mIStae</code> is always one of the following:
    * <code>IS_READY, IS_TRANSFORM, IS_REGION, IS_DRAW, IS_CANCEL</code>
    */
   int mIState;

   /**
    * @brief the current transformation mode
    * @see getEditMode()
    */
   int mEditMode;

   /**
    * @brief the current region mode
    * @see getRegionMode()
    */
   int mRegionMode;

   /**
    * @brief The center assumed for rotation/scaling
    * @see getCenter(), setCenter(VECTOR3D)
    */
   VECTOR3D mCenter;

   /**
    * @brief The mouse position at the beginning of an user action
    */
   VECTOR3D mFirstPoint;

   /**
    * @brief The current mouse position during an user action
    */
   VECTOR3D mLastPoint;

   /**
    * @brief the "size" of the nodes
    * @see getRadius()
    */
   float mRadius;

   /**
    * @brief pointer to the current shape
    *
    * A pointer to the Shape that is to be edited. Maybe null.
    * @see getShape(), setShape(PolygonShape *)
    */
   PolygonShape *mShape;

   /**
    * @brief true, when in restricted mode
    * @see getRestricted(), setRestricted(bool)
    */
   bool mRestricted;

   /**
    * @brief true if last selected node should be deselected again.
    *
    * OK, i admitt, this is a hack. It solves the following problem:
    *
    * if the user shift-clicks on a node, that is selected and
    * immediatly after this begins a transformation, it is assumed,that he/she
    * did not actualy want to deselect the node. On the other hand, if no translation
    * accurs, he propably wants to deselect the node. So we have to remember if
    * the node should be deselected again on mouse up.
    *
    * Yust keep it like this ;-=)
    */
   bool mDeselectLast;

   /**
    * @brief last selected key
    *
    * the key of the node that was selected before a transformation
    * @see mDeselectLast
    */
   int mLastKey;

};

#endif


