/*
 * Copyleft 2001 Lukas Degener, Ingmar Kanitscheider. All rights granted ;-)
 *----------------------
 * file: src/NewPolygonShapeEditor.cpp
 *----------------------
 * This file is part of the CountourWarp Effect based on the GLEffect Framework.
 * In this file you will find the implementation of the PolygonShapeEditor class
 * declaration is in NewPolygonShapeEditor.h
 */

// to reduce warning messages in Visual C++
#if _MSC_VER >= 1000
#pragma warning(disable:4786)
#endif // _MSC_VER >= 1000

//include this declaration only if NEW_POLYGONSHAPE is defined.
//#ifdef NEW_POLYGONSHAPE_H
//---------------------------------------------------------------------------
//  Some usefull macros
//---------------------------------------------------------------------------

#define SAFE_DELETE(x) {if (x) {delete(x);(x)=NULL;}}
#define SAFE_DELETE_ARRAY(x) {if (x) {delete[](x);(x)=NULL;}}


//#include<iostream.h>
#include<qobject.h>
#include<qwidget.h>
#include<qpopupmenu.h>
#include<qpixmap.h>
#include<qnamespace.h>
#include<qevent.h>
#include<math.h>
// Visual C++ needs also this include
#include<assert.h>
#include"PolygonShapeEditor.h"
#include"geomObj.h"
#include"vector.h"
#include"Icon.h"

// We need it, but we don't know why...
#if _MSC_VER >= 1000
#define M_PI 3.14159265358979323846
#endif

PolygonShapeEditor::PolygonShapeEditor(QWidget* parent, const char* name, WFlags fl): QWidget( parent, name, fl ){
   //set initial values
   mShape=NULL;
   mRadius=.03f;
   mRestricted=false;
   mEditMode=MODE_DRAW;
   mRegionMode=REGION_BOX;
   mState=STATE_READY;
   mIState=IS_READY;
   mFirstPoint=VECTOR3D(0,0,0);
   mLastPoint=VECTOR3D(0,0,0);
   mCenter=VECTOR3D(0,0,0);
   mDeselectLast=false;
   //create the childwidgets
   initGui();
}

void PolygonShapeEditor::initGui(){
   //create popup pallette
   mPopup = new QPopupMenu(this);
   //add menuitems
   mPopup->insertItem(I_ACTION_DRAW,"Polygonzug zeichnen",MODE_DRAW);
   mPopup->insertItem(I_ACTION_TRANSLATE,"Knoten verschieben",MODE_TRANSLATE);
   mPopup->insertItem(I_ACTION_SCALE,"Knoten skalieren",MODE_SCALE);
   mPopup->insertItem(I_ACTION_ROTATE,"Knoten rotieren",MODE_ROTATE);
   mPopup->insertSeparator();
   mPopup->insertItem(I_ACTION_REGIONBOX,"Rechteckige Auswahl",REGION_BOX);
   mPopup->insertItem(I_ACTION_REGIONCIRCLE,"Runde Auswahl",REGION_CIRCLE);
   mPopup->insertSeparator();
   mPopup->insertItem(I_ACTION_DELETE, "Auswahl lschen",ACTION_DELETE);
   connect(mPopup, SIGNAL(activated(int)), this, SLOT(setEditMode(int)));
}

PolygonShapeEditor::~PolygonShapeEditor(){
   //nothing to do.
   //well ok, better delete mPopup. Don't know if this is done by QT
   SAFE_DELETE(mPopup);
}

//--------------------------------------------------------------------------------------
// Properties
//--------------------------------------------------------------------------------------
QPopupMenu * PolygonShapeEditor::getPopup(){
   return mPopup;
}
void PolygonShapeEditor::setShape(PolygonShape *s){
   mShape=s;
}

PolygonShape * PolygonShapeEditor::getShape(){
   return mShape;
}

void PolygonShapeEditor::setRadius(float r){
   mRadius = r;
}

float PolygonShapeEditor::getRadius(){
   return mRadius;
}

void PolygonShapeEditor::setRestricted(bool r){
   mRestricted=r;
   //enable/disable add and remove popupitems
   mPopup->setItemEnabled(MODE_DRAW,!r);
   mPopup->setItemEnabled(ACTION_DELETE,!r);
   //if we are currently in MODE_DRAW and are now entering restricted mode,
   //switch to MODE_TRANSLATE
   if(r&&(mEditMode==MODE_DRAW)) setEditMode(MODE_TRANSLATE);
}

bool PolygonShapeEditor::getRestricted(){
   return mRestricted;
}

void PolygonShapeEditor::setEditMode(int mode){
   switch(mode){
      default:             mEditMode=mode;emit editModeChanged(mode);break;
      case REGION_BOX:     setRegionMode(mode);break;
      case REGION_CIRCLE:  setRegionMode(mode);break;
      case ACTION_DELETE:
         //fetch an iterator over Selection
         map<int,VECTOR3D> selection = mShape->getSelection();
         map<int,VECTOR3D>::iterator it = selection.begin();
         while(it!=selection.end()){
            mShape->removeNode((*it).first);
            it++;
         }
         emit shapeChanged();
      break;
   }
}

void PolygonShapeEditor::nextEditMode(){
   //For some reason Ingmar's mouse driver thinks it 
   //is funky to doubele-generate this event.
   //Giveme a break. ignore every second event.
   //cout<<"Peng!"<<endl;
#ifdef HAVE_INGMAR
   static bool fuck=false;
   if(fuck)
   {
		fuck = false;
		return;
   }
   fuck = true;
#endif
   int newMode=mEditMode+1;
   if(newMode>MODE_ROTATE) newMode=MODE_DRAW;
   if(mRestricted&&(newMode==MODE_DRAW)) newMode=MODE_TRANSLATE;
   setEditMode(newMode);
}

void PolygonShapeEditor::previousEditMode(){

   //For some reason Ingmar's mouse driver thinks it 
   //is funky to doubele-generate this event.
   //Giveme a break. ignore every second event.
#ifdef HAVE_INGMAR
   static bool fuck=false;
   if(fuck)
   {
		fuck = false;
		return;
   }
   fuck = true;
#endif
   int newMode=mEditMode-1;
   if(newMode<MODE_DRAW) newMode=MODE_ROTATE;
   if(mRestricted&&(newMode==MODE_DRAW)) newMode=MODE_ROTATE;
   setEditMode(newMode);
}

int PolygonShapeEditor::getEditMode(){
   return mEditMode;
}

void PolygonShapeEditor::setRegionMode(int mode){
   mRegionMode=mode;
}

int PolygonShapeEditor::getRegionMode(){
   return mRegionMode;
}

Edge PolygonShapeEditor::getEditEdge(){
   return Edge(mFirstPoint,mLastPoint);
}

int PolygonShapeEditor::getEditorState(){
   return mState;
}

VECTOR3D PolygonShapeEditor::getCenter(){
	return mCenter;
}
//--------------------------------------------------------------------------------------
//Event handler
//--------------------------------------------------------------------------------------

void PolygonShapeEditor::glPanelMoveEvent(QMouseEvent* e, VECTOR3D point){
   Qt::ButtonState button = e->state();
   if(mIState==IS_DRAW){
      mLastPoint=point; //update mouseposition for drawing new Edge while
                        //during a strip. This is only needed for rendering.
   }
   if(button&&Qt::LeftButton){
      switch(mIState){
         default:break;
         case IS_TRANSFORM:
            doTransform(point);
         break;
         case IS_REGION:
            doRegion(point);
         break;
      }
   }

}

void PolygonShapeEditor::glPanelPressEvent(QMouseEvent* e, VECTOR3D point){
   int left =(Qt::LeftButton);
   int shiftleft =(Qt::LeftButton|Qt::ShiftButton);
   int controlleft =(Qt::LeftButton|Qt::ControlButton);
   int altleft=(Qt::LeftButton|Qt::AltButton);
   int right =(Qt::RightButton);
   int leftright =(Qt::LeftButton|Qt::RightButton);
   int middle=(Qt::MidButton);
   int button = e->stateAfter();
   int key = findNode(point);
   //let's see what condition my condition is in;-)
   switch(mIState){
      default: break;
      case IS_READY:
         //first see, if Editor is in MODE_DRAW:
         if(mEditMode==MODE_DRAW){
            //don't change to IS_STRIP is mRestriced flag is set!
            if(!mRestricted&&button==left){
               mIState=IS_DRAW;
               //cout<<"->IS_DRAW"<<endl;
               doStrip(point);
            }
            if(button==right){
               //if the mouse is on a key and it's not selected,
               //clear selection and select key
               if(key!=-1){
                  if(!mShape->isSelected(key)){
                     clearSelection();
                     select(key);
                  }
               }
               //pop up mPopup
               mPopup->exec(e->globalPos());
            }
         }

         //what is the buttonstate after the event?
         else if(button==left){

            if(key==-1){ // no node found
               clearSelection();
               mIState=IS_REGION;
               doRegion(point);
               //cout<<"->IS_REGION"<<endl;
            }
            else{
               if(!mShape->isSelected(key)){
                  clearSelection();
                  select(key);
               }
               mIState=IS_TRANSFORM;
               //cout<<"->IS_TRANSFORM"<<endl;
               doTransform(point);
            }
         }
         else if(button==right){
            //if the mouse is on a key and it's not selected,
            //clear selection and select key
            if(key!=-1){
               if(!mShape->isSelected(key)){
                  clearSelection();
                  select(key);
               }
            }
            //pop up mPopup
            mPopup->exec(e->globalPos());
         }
         else if(button==shiftleft){
            if(key==-1){ // no node found
               mIState=IS_REGION;
               //cout<<"->IS_REGION"<<endl;
               doRegion(point);
            }
            else{
               if(mShape->isSelected(key)) mDeselectLast=true;
               select(key);
               mLastKey=key;
               mIState=IS_TRANSFORM;
               //cout<<"->IS_TRANSFORM"<<endl;
               doTransform(point);
            }
         }

         else if(button==altleft){
         if(key==-1){ // no node found
               //TODO:Should remove all nodes.
               //not implemented for now!
            }
            else{
               mShape->removeNode(key);
               //cout<<"->TODO: find another modifier, alt is occupied by xserver."<<endl;
            }
         }
         else if(button==middle){
            mCenter=point;
         }

      break;

      case IS_TRANSFORM:
         if(button==leftright) {//remember: left button should still be down!
            mIState=IS_CANCEL;
            //cout<<"->IS_CANCEL"<<endl;
            cancelTransform();
         }

      break;

      case IS_DRAW:
         if(button==left){
            doStrip(point);
            //cout<<"(doStrip)"<<endl;
         }
         else if(button==controlleft){
            finishStrip(point);
            mIState=IS_READY;
			// enter trafo mod
			setEditMode (MODE_TRANSLATE);
            //cout<<"->IS_READY"<<endl;
         }
         else if(button==right){
            cancelStrip();
			// enter trafo mod
			setEditMode (MODE_TRANSLATE);
            mIState=IS_CANCEL;
            //cout<<"->IS_CANCEL"<<endl;
         }

      break;

      case IS_REGION:
         if(button==leftright){
            cancelRegion();
            mIState=IS_CANCEL;
            //cout<<"->IS_CANCEL"<<endl;
         }

      break;
   }
}

void PolygonShapeEditor::glPanelReleaseEvent(QMouseEvent* e, VECTOR3D point){
   Qt::ButtonState button = e->button();
   if((button==Qt::LeftButton)||(button==Qt::RightButton)){
      switch(mIState){
         default: break;
         case IS_TRANSFORM:
            finishTransform();
            if((mFirstPoint==mLastPoint)&&mDeselectLast){
               unselect(mLastKey);
               mDeselectLast=false;
            }
            mIState=IS_READY;
            //cout<<"->IS_READY"<<endl;
         break;
         case IS_REGION:
            finishRegion();
            mIState=IS_READY;
            //cout<<"->IS_READY"<<endl;
         break;
         case IS_CANCEL:
            mIState=IS_READY;
            //cout<<"->IS_READY"<<endl;
         break;
      }
   }
}
//--------------------------------------------------------------------------------------
// other public functions
//--------------------------------------------------------------------------------------



//--------------------------------------------------------------------------------------
// edit methods (Abstracted user actions)
//--------------------------------------------------------------------------------------

//####
//# Dispatch functions:
//####
void PolygonShapeEditor::doTransform(VECTOR3D point){
   switch(mEditMode){
      default: break;
      case MODE_TRANSLATE:doTranslate(point);break;
      case MODE_ROTATE:doRotate(point);break;
      case MODE_SCALE:doScale(point);break;

   }
   emit shapeChanged();
}

void PolygonShapeEditor::finishTransform(){
   switch(mEditMode){
      default: break;
      case MODE_TRANSLATE:finishTranslate();break;
      case MODE_ROTATE:finishRotate();break;
      case MODE_SCALE:finishScale();break;

   }
   //no emit here! nothing changed since last doTransform()
}

void PolygonShapeEditor::cancelTransform(){
   switch(mEditMode){
      default: break;
      case MODE_TRANSLATE:cancelTranslate();break;
      case MODE_ROTATE:cancelRotate();break;
      case MODE_SCALE:cancelScale();break;

   }
   emit shapeChanged();
}

void PolygonShapeEditor::doRegion(VECTOR3D point){
   switch(mRegionMode){
      default: break;
      case REGION_BOX:doBox(point);break;
      case REGION_CIRCLE:doCircle(point);break;

   }
}

void PolygonShapeEditor::finishRegion(){
   switch(mRegionMode){
      default: break;
      case REGION_BOX:finishBox();break;
      case REGION_CIRCLE:finishCircle();break;
   }
}


void PolygonShapeEditor::cancelRegion(){
   switch(mRegionMode){
      default: break;
      case REGION_BOX:cancelBox();break;
      case REGION_CIRCLE:cancelCircle();break;
   }
}



//####
//# actual work starts here:
//####


int PolygonShapeEditor::doStrip(VECTOR3D point){
   //if we are ready, start a strip
   int result;
   if(mState==STATE_READY){
      //cout<<"->STATE_DRAWING_STRIP"<<endl;
      mLastPoint=point;
      mState=STATE_DRAWING_STRIP;
      int key=findNode(point);
      if(key==-1){
         mFirstPoint=point;
         result=mShape->startStrip(point);
         return result;
      }
      else{
         mFirstPoint=mShape->getNode(key);
         result=mShape->startStrip(key);
         return result;
      }

   }
   //if we are drawing, continue
   else if(mState==STATE_DRAWING_STRIP){
      //cout<<"(continue drawing strip)"<<endl;
      int key=findNode(point);
      if(key==-1){
         mFirstPoint=point;
         result=mShape->continueStrip(point);
         emit shapeChanged();
         return result;
      }
      else{
         mFirstPoint=mShape->getNode(key);
         result=mShape->continueStrip(key);
         emit shapeChanged();
         return result;
      }

   }
   //if we are in some other state, do nothing
   return -1;
}

int PolygonShapeEditor::finishStrip(VECTOR3D point){
   //only do something if we are allready drawing
   if(mState==STATE_DRAWING_STRIP){
      mState=STATE_READY;
      int key=findNode(point);
      if(key==-1){
         return mShape->continueStrip(point);
      }
      else{
         return mShape->continueStrip(key);
      }
      emit shapeChanged();
   }
   //if we are in some other state, do nothing
   return -1;
}

int PolygonShapeEditor::cancelStrip(){
   //only do something if we are allready drawing
   if(mState==STATE_DRAWING_STRIP){
      mState=STATE_READY;
      return mShape->getLastKey();
   }
   //if we are in some other state, do nothing
   return -1;
}

VECTOR3D PolygonShapeEditor::doTranslate(VECTOR3D point){
   //if we are ready, start translating
   if(mState==STATE_READY){
      mState=STATE_TRANSLATING;
      //remember mouse position, so we can
      //calculate the diff later on.
      mLastPoint=point;
      mFirstPoint=point;
      //Now save the state of the shape, so we can undo the change
      //NOTE TO INGMAR:
      //If we are in a differential warp, this will most propably
      //not revert to the state before the translation started.
      //we have to work out something more sophisticated to
      //achieve this.
      mShape->save();

      //as we didn't translate anything yet, the difference is zero
      return VECTOR3D(0,0,0);
   }
   //if we are already translating, continue with it
   if(mState==STATE_TRANSLATING){
      //get difference to last point
      VECTOR3D diff = point-mFirstPoint;
      //remembter mouse position for further translations
      mLastPoint=point;
      //create the transformation matrix
      //cout<<"creating matrix..."<<endl;
      float *t= new float[16];
      //cout<<"filling matrix..."<<endl;
      fillMatrix(t);
      //cout<<"calculating translation"<<endl;
      translate(t,diff);
      //let the shape apply the matrix.
      //cout<<"applying transformation..."<<endl;
      mShape->applyTransformation(t,true);
      //cout<<"done!"<<endl;
      //don't forget do delete t!!
      delete t;

      //return the difference:
      return diff;
   }
   //return zero, if we are in any other state.
   return VECTOR3D(0,0,0);
}

VECTOR3D PolygonShapeEditor::finishTranslate(){
   //not much to do.
   //if we are in STATE_TRANSLATING reset to STATE_READY and return
   //the overall diff.
   if(mState==STATE_TRANSLATING){
      mState=STATE_READY;
      //update Selection
      //cout<<"updating selection..."<<endl;
      mShape->updateSelection();
      //cout<<"done"<<endl;
      return mLastPoint-mFirstPoint;
   }
   //otherwise return zero
   return VECTOR3D(0,0,0);
}

VECTOR3D PolygonShapeEditor::cancelTranslate(){
   //if we are in STATE_TRANSLATING,
   //calculate the current total diff,
   //revert the shape and return the
   //total diff.
   if(mState==STATE_TRANSLATING){
      mState=STATE_READY;
      mShape->revert();
      return mFirstPoint-mLastPoint;
   }
   //otherwise, return zero
   return VECTOR3D(0,0,0);
}

VECTOR3D PolygonShapeEditor::doScale(VECTOR3D point){
   //if we are ready, start scaling
   if(mState==STATE_READY){
      mState=STATE_SCALING;
      //remember mouse position, so we can
      //calculate the diff later on.
      mLastPoint=point;
      mFirstPoint=point;
      //Now save the state of the shape, so we can undo the change
      mShape->save();

      //as we didn't scale anything yet, the factors are all 1
      return VECTOR3D(1,1,1);
   }
   //if we are already scaling, hang on.
   if(mState==STATE_SCALING){
      //calculate factors
      //absolute:
      VECTOR3D f = getFactors(mFirstPoint,point);
      //relative
      VECTOR3D r = getFactors(mLastPoint,point);
      //remembter mouse position for further scalations
      mLastPoint=point;
      //create the transformation matrix
      float *t= new float[16];
      fillMatrix(t);
      scale(t,mCenter,f);
      //let the shape apply the matrix.
      mShape->applyTransformation(t,true);
      //don't forget do delete t!!
      delete t;

      //return the relative factor
      return r;
   }
   //return 1,1,1, if we are in any other state.
   return VECTOR3D(1,1,1);
}

VECTOR3D PolygonShapeEditor::finishScale(){
   //not much to do.
   //if we are in STATE_SCALING reset to STATE_READY and return
   //the overall factor.
   if(mState==STATE_SCALING){
      mState=STATE_READY;
      //update Selection
      mShape->updateSelection();
      return getFactors(mFirstPoint,mLastPoint);
   }
   //otherwise return 1,1,1
   return VECTOR3D(1,1,1);
}

VECTOR3D PolygonShapeEditor::cancelScale(){
   //if we are in STATE_SCALING,
   //calculate the current total factor,
   //revert the shape and return the
   //total factor.
   if(mState==STATE_SCALING){
      mState=STATE_READY;
      mShape->revert();
      return getFactors(mFirstPoint,mLastPoint);
   }
   //otherwise, return 1,1,1
   return VECTOR3D(1,1,1);

}

float PolygonShapeEditor::doRotate(VECTOR3D point){
//if we are ready, start scaling
   if(mState==STATE_READY){
      mState=STATE_ROTATING;
      //remember mouse position, so we can
      //calculate the diff later on.
      mLastPoint=point;
      mFirstPoint=point;

      //Now save the state of the shape, so we can undo the change
      mShape->save();

      //as we didn't rotate anything yet, the angle is zero
      return 0;
   }
   //if we are already scaling, hang on.
   if(mState==STATE_ROTATING){
      //calculate factors
      //absolute:
      float a = getAngle(mFirstPoint,point);
      //relative
      float r = getAngle(mLastPoint,point);
      //remembter mouse position for further scalations
      mLastPoint=point;
      //create the transformation matrix
      float *t= new float[16];
      fillMatrix(t);
      rotate(t,mCenter,a);
      //let the shape apply the matrix.
      mShape->applyTransformation(t,true);
      //don't forget do delete t!!
      delete t;

      //return the relative angle
      return r;
   }
   //return zero, if we are in any other state.
   return 0;
}

float PolygonShapeEditor::finishRotate(){
   //not much to do.
   //if we are in STATE_ROTATING reset to STATE_READY and return
   //the overall factor.
   if(mState==STATE_ROTATING){
      mState=STATE_READY;
      //update Selection
      mShape->updateSelection();
      return getAngle(mFirstPoint,mLastPoint);
   }
   //otherwise return zero
   return 0;
}

float PolygonShapeEditor::cancelRotate(){
   //if we are in STATE_SCALING,
   //calculate the current total factor,
   //revert the shape and return the
   //total factor.
   if(mState==STATE_ROTATING){
      mState=STATE_READY;
      mShape->revert();
      return getAngle(mFirstPoint,mLastPoint);
   }
   //otherwise, return zero
   return 0;
}

void PolygonShapeEditor::doBox(VECTOR3D point){
   //if we are ready, start selecting
   if(mState==STATE_READY){
      mState=STATE_SELECTING_BOX;
      //remember mouse position, so we can
      //calculate the diff later on.
      mLastPoint=point;
      mFirstPoint=point;
      mShape->save();
      return;
   }
   //if we are already selecting, hang on.
   if(mState==STATE_SELECTING_BOX){
      mLastPoint=point;
      return;
   }
}

void PolygonShapeEditor::finishBox(){
   if(mState==STATE_SELECTING_BOX){
      mState=STATE_READY;
      mShape->selectRegion(mFirstPoint,mLastPoint);
   }
}

void PolygonShapeEditor::cancelBox(){
if(mState==STATE_SELECTING_BOX){
      mState=STATE_READY;
      mShape->revert();
   }
}

void PolygonShapeEditor::doCircle(VECTOR3D point){
//if we are ready, start selecting
   if(mState==STATE_READY){
      mState=STATE_SELECTING_CIRCLE;
      //remember mouse position, so we can
      //calculate the diff later on.
      mLastPoint=point;
      mFirstPoint=point;
      mShape->save();
      return;
   }
   //if we are already selecting, hang on.
   if(mState==STATE_SELECTING_CIRCLE){
      mLastPoint=point;
      return;
   }
}

void PolygonShapeEditor::finishCircle(){
   if(mState==STATE_SELECTING_CIRCLE){
      mState=STATE_READY;
      mShape->selectRegion(mFirstPoint,vecSqrMagnitude(mLastPoint-mFirstPoint));
   }
}

void PolygonShapeEditor::cancelCircle(){
   if(mState==STATE_SELECTING_CIRCLE){
      mState=STATE_READY;
      mShape->revert();
   }
}

void PolygonShapeEditor::selectAll(){
   mShape->selectAll();
}

void PolygonShapeEditor::clearSelection(){
   mShape->clearSelection();
}

void PolygonShapeEditor::select(int key){
   mShape->select(key);
}

void PolygonShapeEditor::unselect(int key){
   mShape->unselect(key);
}

//--------------------------------------------------------------------------------------
// private utility functions
//--------------------------------------------------------------------------------------




int PolygonShapeEditor::findNode(VECTOR3D point){
   int key = mShape->findNearestNode(point);
   if (key==-1) return -1;
  if(vecSqrMagnitude(mShape->getNode(key)-point)<=mRadius*mRadius) return key;
   return -1;
}

VECTOR3D PolygonShapeEditor::getFactors(VECTOR3D from, VECTOR3D to){
   VECTOR3D result(1,1,1);
   from-=mCenter;
   to-=mCenter;
   if(from.x!=0) result.x=to.x/from.x;
   if(from.y!=0) result.y=to.y/from.y;
   if(from.z!=0) result.z=to.z/from.z;
   return result;
}
float PolygonShapeEditor::radians(VECTOR3D v){
   //first set v to the normalized difference b-Viewpoint
   v=vecNormalize(v);
   //now v.x should be the cosinus
   float alpha = -acos(v.x);

   //we have to check for the signum of v.y
   if(v.y<0) alpha = 2*M_PI-alpha;

   return alpha;
}

float PolygonShapeEditor::getAngle(VECTOR3D from, VECTOR3D to){
   from-=mCenter;
   to-=mCenter;
   return radians(to)-radians(from);
}

void PolygonShapeEditor::fillMatrix(float* matrix){
   matrix[0]= 1;
   matrix[1]= 0;
   matrix[2]= 0;
   matrix[3]= 0;

   matrix[4]= 0;
   matrix[5]= 1;
   matrix[6]= 0;
   matrix[7]= 0;

   matrix[8]= 0;
   matrix[9]= 0;
   matrix[10]=1;
   matrix[11]=0;

   matrix[12]=0;
   matrix[13]=0;
   matrix[14]=0;
   matrix[15]=1;
}
void PolygonShapeEditor::translate(float* matrix,VECTOR3D t){
   fillMatrix(matrix);
   matrix[3]=t.x;
   matrix[7]=t.y;
   matrix[11]=t.z;
}
void PolygonShapeEditor::rotate(float* matrix,VECTOR3D c,float rad){
   fillMatrix(matrix);
   matrix[0]=cos(rad);
   matrix[1]=sin(rad);
   matrix[4]=-sin(rad);
   matrix[5]=cos(rad);
   matrix[3]=c.x - c.x*cos(rad) - c.y*sin(rad);
   matrix[7]=c.y + c.x*sin(rad) - c.y*cos(rad);
}
void PolygonShapeEditor::scale(float* matrix,VECTOR3D c,VECTOR3D s){
   fillMatrix(matrix);
   matrix[0]=s.x;
   matrix[5]=s.y;
   matrix[10]=s.z;
   matrix[3]=c.x - s.x*c.x;
   matrix[7]=c.y - s.y*c.y;
   matrix[11]=c.z - s.z*c.z;
}


