/*
 * Copyleft 2001 Lukas Degener, Ingmar Kanitscheider. All rights granted ;-)
 *----------------------
 * file: src/PolygonShape.h
 *----------------------
 * This file is part of the CountourWarp Effect based on the GLEffect Framework.
 * In this file you will find the declaration of the PolygonShape class
 * Implementation is in PolygonShape.cpp
 *
 * Comments are doxygen compatible
 */

/*! @file PolygonShape.h
 *
 *  @brief
 *  Declaration of Polygonshape class
 *  @author Ingmar Kanitscheider, Lukas Degener
 */


#ifndef POLYGONSHAPE_H
#define POLYGONSHAPE_H
// to reduce warning messages in Visual C++

#if _MSC_VER >= 1000
#pragma warning(disable:4786)
#endif

#include <iterator>
#include <vector>
#include <list>
#include <map>
#include <utility>

using namespace std;

#include "geomObj.h"


/*! @class PolygonShape
 *
 * @ingroup contourWarpClasses
 *
 * @brief Represents a shape consisting of polygons
 *
 * An instance of this class holds informations about a set of nodes, and their
 * connections among each other.
 *
 * @note I did a moreless complete rewrite after beeing completly pissed off by all the
 * fuzz about pointer/refferences to Nodes and their (in)consistense.
 *
 * A node consists of a VECTOR3D describing the nodes
 * position in 3-D space, along with an integer key to identify the node. Keys are
 * unique, meaning in one shape there can not exist more then one node with the same key.
 * Node to key bindings are guarantied to remain consistend as long as the instance lives.
 * A key is guarantied to remain valid, unless the corresponding node is deleted.
 *
 * An edge is internaly represented by a pair of keys, that identify the nodes, connected
 * by the edge. Nevertheless, you con check out edges of type Edge (including
 * position informations) for conveniance. @see PolygonShape::getEdge(int)
 *
 * The only way to add nodes/edges to a shape is by adding edges. Removal of edges or
 * nodes is both possible although it should be noted that a node is deleted automaticaly
 * if it isn't connected to any other point after the removal of an edge/node. Also, of
 * cause, edges are deleted if one of the nodes they connect is deleted. As a result,
 * you won't find invalid edges or non-connected nodes within a PolygonShape.
 *
 * All modifications to the node positions are done solely by this class. There is no way
 * of doing this from outside. PolygonShape comes with a somewhat high-level interface for
 * modifying node positions. it implements a selection model, which basicaly is a subset
 * of the shape's nodes. In this selection, together with the node's keys, their position
 * at the time of selecting is stored. Selected nodes may be transformed by applying a
 * a affine transformation to them @see PolygonShape::applyTransformation(float * , bool)
 *
 * To make undos easier, i have added a state history. it is simply a double linked list
 * of PolygonShape instances which can be used to save or load the complete state of a
 * shape. since we don't have to bother about pointer consitency anymore, this is
 * pretty easy. (YES! I'M INVINCIBLE!! ;-)))
 * @note IMPORTANT: For this trick to work, it is essential, that both operator= and
 * operator== aswell as the copy constructor ignore the member field mStates.
 * So it's not that i've forgotten something in this methods, i left it out on purpose!!
 * To make you understand why this is neccessary, just imagine what would happen if you
 * add it in there: Each time you save a PolygonShape, its mStates field is saved as well,
 * meaning each PolygonShape in this list is duplicated which again will duplicate
 * its mStates member and so on. This all would leave
 * you with over-exponential space requirements and you sure don't want that!
 */


class PolygonShape{
//! @name data structures
// @{
private:
   /*!
    * @brief The STL vector in which the edges are stored.
    *
    * The Vector does the (de-)allocation of all memory needed for our Edges
    * We let it initaly allocate mem for 3 elements
    */
	vector<pair<int,int> > mEdges;

   /*!
    * @brief The map containing the nodes
    */
   map<int,VECTOR3D> mNodes;

   /*!
    * @brief A list of keys of selected Nodes
    */
   map<int,VECTOR3D> mSelection;
   
   /*!
    * @brief The state history
    *
    * a list of PolygonShapes, each representing the state of this PolygonShape at
    * a certain point in time. this has to be a list of pointers to avoid recursive
	* data structure
    */
   list<PolygonShape*> mStates;
//! @}

//! @name state variables
// @{

   /*!
    * @brief The last node added to a strip.
    *
    * It is possible, that this node is allready
    * in this shape. In this case <code>mLastKey</code> is the key to it.
    */
   VECTOR3D mLastNode;

   /*!
    * @brief The key of the last node added
    *
    * Holds the key to the last node inserted into a strip, or
    * -1, if that node did not belong to the shape yet.
    */
   int mLastKey;

   /*!
    * @brief The next unused key
    *
    * The next key to be used when a node is added
    * to mNodes.
    * @note DO NOT EVEN THINK of doing something other than
    * incrementing to this key. OK, constructors and the like don't count.
    * But after that, the only assignment operator i want to see
    * behind <code>mNextKey</code> is '++' !
    * Did i make myself clear enough? i hope so.
    */
   int mNextKey;

   /*!
    * @brief Does the shape hold valid information to continue a srip?
    *
    * This flag should usualy be set, if it's possible to continue drawing a strip.
    * This means, <code>startStrip</code> has been called at least once.
    */
   bool mHaveStart;


public:
//! @}

//! @name Constructing/Destructing
// @{

   /**
    * @brief Default Constructor:
    *
    * Does nothing but create an empty Shape
    */
   PolygonShape();


   /**
    * @brief Copy Constructor:
    *
    * creates a new shape which is equal to the original.
    * The state of the original is completle reasambled, apart of the <code>mStates</code>
    * member. Please have a look at the notes above!
    */
   PolygonShape(const PolygonShape &original);


   /**
    * @brief Deconstructor
    *
    * Frees all memory occupied by the shape and all of its members
    */
    ~PolygonShape();

//! @}

//! @name overloaded operators
// @{


   /**
    * @brief assignement operator
    *
    * Completely reasamble the state of original in this shape, but ignoring the
    * <code>mStates</code> member. See node above!
    */
   PolygonShape &operator=(const PolygonShape &original);

   /**
    * @brief compares two PolygonShape instances for equality
    *
    * Not too much of a usefull function, since it does not check for graph
    * isomorphism (which is a realy nasty job, NP-complete, i think) but simply
    * tests all the member fields for equality.
    * To be honest, i don't give a fuck about how usefull this is. It's just
    * to comply with the requirements of a "nice" class i read somewhere. To meet this
    * requirements, all we have to do is make sure, that <code>a=b</code> makes
    * <code>a</code> equal to b in respect to this operator.
    * @see operator=(PolygpnShape &)
    */
   int operator==(PolygonShape &other);
//! @}

//! @name Non-Isomorph manipulations
// @{

   /**
    * @brief continues an edge strip
    *
    * Adds an edge from the last node (returned by getLastNode()) to <code>next</code>
    * If there is no last node (because the shape is empty) no edge is added
    * Instead startStrip(VECTOR3D) is called with <code>next</code> as parameter
    * @param next the next point to be included into the strip. A new node with this
    * position will be added to the strip, no matter if there is allready a node at this
    * position or not. If you want to include a existing node into the strip, use
    * continueStrip(int)
    * @returns the key of the added Node or -1 if the shape was empty.
    */
   int continueStrip(VECTOR3D next);

   /**
    * @brief continues an edge strip
    *
    * Adds an edge from the last node (returned by lastNode() to next
    * If there is no last node (because the shape is empty) no edge is added
    * Instead startStrip(int) is called with <code>next</code> as parameter
    * @param key the key of the next point in the strip.
    * @returns key if the node was added or -1 if key wasn't found.
    */
   int continueStrip(int key);


   /**
    * @brief starts an edge strip
    *
    * Tells the Shape that you are about to draw a new strip of Edges
    * starting from the node identified by <code>key</code>
    * @note Also note that the next call of getLastNode() will return <code>key</code>,
    * or -1 if <code>key</code> wasn't found
    * @param key the key of the node at which the strip should be started
    * @returns <code>key</code>, or -1 if <code>key<code> wasn't found
    */
   int startStrip(int key);

   /**
    * @brief starts an edge strip
    *
    * Tells the Shape that you are about to draw a new strip of Edges
    * @note <code>first</code> is  not appended to the shape's list of nodes
    * until you actualy call <code>continueStrip</code>.
    * Also note that the next call of lastnode will return -1;
    * @param first the position at which you want to start the strip. Once you actualy
    * create a line with a subsequent call to <code>continueStrip</code>, a new node
    * with position <code>first</code> will be added to the shape, no matter if there is
    * allready a node at this position or not. If you want to use a existing node to start
    * a strip, use startStrip(int).
    * @returns -1 by definition, since there is no node added yet and thus a key can
    * not be returned. Of cause one can argue if it would have been nicer to make this
    * method void. Well not that important anyway. ;-)
    *
    */
   int startStrip(VECTOR3D first);

   /**
    * @brief removes all edges connecting <code>a</code> with <code>b</code>
    *
    * @note if one of the points is not connected to any other point after the removal
    * of this edge, this point is removed.
    */
   void removeEdge(int a, int b);


   /**
    * @brief remove a node <code>key</code>
    *
    * removes a node from the shape by deleting all edges, that hold a reference to
    * <code>key</code>
    * @note be aware of the fact, that this might imply the removal of adjacent nodes.
    * @param key the key of the node you want to delete
    */
   void removeNode(int key);
//! @}

//! @name Isomorph mainpulations
// @{
   /**
    * @brief marks a node as selected.
    *
    * this is done by adding its key/position pair to the
    * selection. If <code>key</code> is not found no action is taken
    * @param key the key of the node you want to select
    */
   void select(int key);

   /**
    * @brief marks a node as  not selected.
    *
    * this is done by removing its key/position pair from the
    * selection. If <code>key</code> is not found in the selection, no action is taken
    * @param key the key of the node you want to unselect
    */
   void unselect(int key);

   /**
    * @brief clear the selection
    *
    * marks all nodes as not selected by removing all keys/position pairs
    * from the selection.
    */
   void clearSelection();

   /**
    * @brief select the whole shape
    *
    * marks all nodes as selected by adding them to the selection
    */
   void selectAll();

   /**
    * @brief selects all nodes in a rectangular region
    *
    * @param first the first corner of the rectangle, i.e. upper-left
    * @param second the second corner of the rectangle, i.e. lower-right
    */
   void selectRegion(VECTOR3D first, VECTOR3D second);

   /**
    * @brief selects all nodes in a circular region
    *
    * @param first the center of the circle/sphere
    * @param qradius the radius of the circle/sphere.
    *       this should be the maximal quadratic distance
    *       from center a point may have to become selected.
    */
   void selectRegion(VECTOR3D first, float qradius);

   /**
    * @brief applies an afine transformation to the selection
    *
    * You can use this method in two ways:
    *
    * -  to transform the selected nodes by using their current positions
    * -  to do the same thing in respect to the positions saved in the selection.
    * @note the second variant comes in very handy if you want to realize coninous
    * transformation while avoiding dead spots, where the transformation becomes
    * non-injective. To ship around this singularities, you can first update the selection
    * (using updateSelection()) and then do multible transformations all relative to the
    * positions in the selection. PolygonShapeEditor makes extensive use of this feature.
    * @param matrix should be a 4x4 matrix, stored line after line.
    * @param absolute if this flag is set, transformation accures relative to the positions
    * saved in the selection.
    */
    void applyTransformation(float * matrix, bool absolute=false);

   /**
    * @brief updates the selection
    *
    * Update the positions saved in the selection to the corresponding current positions
    * of the selected nodes. @see applyTransformation(float * , bool)
    */
   void updateSelection();

//! @}

//! @name information retrieval
// @{


   /**
    * @brief returns a vector of adjacent edges
    *
    * Use this method to get a vector filled with all edges that
    * refer to <code>key</key>.
    * @node the return type <code>vector<pair<int,int> ></code>
    * can mostly be used as any standard array type in C.
    * @param key the key of the node you want to get adjacent lines to.
    * @returns a vector storing values of type pair<int,int> which represents the
    * adjacent edges.
    */
   vector<pair<int,int> > adjacentEdges(int key);


   /**
    * @brief returns the Edge at <code>index</code>
    *
    * Get a pair of keys to nodes representing an edge
    * @param index the index of the edge
    * @returns a pair<int,int> containing the keys to the nodes the edge connects
    */
   pair<int,int> getPair(int index);

   /**
    * @brief returns the Edge at <code>index</code>
    *
    * @param index the index of the edge
    * @returns the Edge at index <code>index</code>
    */
   Edge getEdge(int index);

   /**
    * @brief get a node position
    *
    * @param key the key that identifies the node.
    * @returns the position of the node identified by <code>key</code>
    *             if key is not found, a dummy is returned (0,0,0)
    *             Not very sensible but what should i do.
    */
   VECTOR3D getNode(int key);


   /**
    * @brief get the index of an Edge
    *
    * @param edge a <code>pair<int,int></code> containing the keys to the nodes
    * that are connected by <code>edge</code>.
    * @returns the index of an edge equal to <code>edge</code> or -1
    * if no such edge exists.
    */
   int indexOf(pair<int,int> edge);

   /**
    * @brief get the index of an Edge
    * @see indexOf(pair<int,int>)
    */
   int indexOf(int a, int b);

   /**
    * @brief gets the last added edge
    * @returns a pair of key to the nodes connected by the edge that was last added
    * to the shape
    */
   pair<int,int> getLastEdge();

   /**
    * @brief returns the number of Edges in this shape
    */
   int getEdgeCount();

   /**
    * @brief Returns the number of Nodes in this Shape
    */
   int getNodeCount();

   /**
    * @brief get the last node added
    *
    * @returns the key of the last Node inserted into a strip
    * this maybe -1 in which case the node isn't checked in jet.
    * To retrieve the position anyway,use getLastNode()
    */
   int getLastKey();

   /**
    * @brief get the position of the last node
    * @returns the position of the last node inserted into a strip.
    * this node maybe already contained in the shape.
    * If this is true, you can get its key with getLastKey()
    */
   VECTOR3D getLastNode();

   /**
    * @brief find the node
    * @param point the position at which to look for a node
    * @returns the key of the node closest to the given position.
    */
   int findNearestNode(VECTOR3D point);



   /**
    * @brief checks if a node is selected
    * @param key the key of the node you want to check
    * @returns true if <code>key</code> is in the selection.
    */
   bool isSelected(int key);


   /**
    * @brief get all selected nodes
    *
    * @returns a <code>map<int,VECTOR3D></code> containing all key/position pairs
    * that are in the selection.
    * @note remember that the positions in the returned map are those stored in
    * the selection and will propably differ from the current position of the nodes.
    * @see applyTransformation(float * ,bool)
    */
   map<int,VECTOR3D> getSelection();

   /**
    * @brief get all nodes
    *
    * @returns a <code>map<int,VECTOR3D></code> containing the keys of all Nodes
    * and their current positions
    */
   map<int,VECTOR3D> getNodes();

//! @}

//! @name state history
// @{


   /**
    * @brief save the current state
    *
    * Saves the current state of this shape to the mStates list.
    * @returns the number of saved states after insertion.
    * @see revert()
    */
   int save();

   /**
    * @brief revert to the last saved state
    *
    * Reverts the state of this Shape to the last state saved in mStates
    * and removes it from the history
    * @returns the number of saved states after removal of the last one
    * or -1 if there is no state to revert to.
    */
   int revert();
//! @}
};

#endif


