/*
 * Copyleft 2001 Lukas Degener, Ingmar Kanitscheider. All rights granted {}-)
 *----------------------
 * file: src/PolygonShape.cpp
 *----------------------
 * This file is part of the CountourWarp Effect based on the GLEffect Framework.
 * In this file you will find the implementation of the PolygonShape class
 * Declaration is in PolygonShape.h
 */

//only precess this file if NEW_POLYGONSHAPE is defined
//#ifdef NEW_POLYGONSHAPE

// to reduce warning messages in Visual C++
#if _MSC_VER >= 1000
#pragma warning(disable:4786)
#endif // _MSC_VER >= 1000

#include <stdlib.h>
#include <float.h>
#include <math.h>

#ifdef _WIN32
#include <minmax.h>
#endif

//local includes
#include "PolygonShape.h"

//--------------------
// class PolygonShape
//--------------------
//***************************************************
//* For method documentation look in PolygonShape.h *
//***************************************************


PolygonShape::PolygonShape(){
   mLastKey=-1;
   mNextKey=0;
   mHaveStart=false;
}

PolygonShape::PolygonShape(const PolygonShape &original){
   //Indeed, since we don't have to bother about pointers anymore,
   //this should actualy be quiet simple:
   mNodes=original.mNodes;
   mEdges=original.mEdges;
   mSelection=original.mSelection;
   mLastNode=original.mLastNode;
   mLastKey=original.mLastKey;
   mNextKey=original.mNextKey;
}

PolygonShape::~PolygonShape(){
   //nothing left to do actualy...
   //Well ok, i think i can aswell clear my containers
   //but i don't think it's realy neccessary.
   mNodes.clear();
   mEdges.clear();
   mSelection.clear();
}

PolygonShape &PolygonShape::operator=(const PolygonShape &original){
   //Indeed, since we don't have to bother about pointers anymore,
   //this should actualy be quiet simple:
   mNodes=original.mNodes;
   mEdges=original.mEdges;
   mSelection=original.mSelection;
   mLastNode=original.mLastNode;
   mLastKey=original.mLastKey;
   mNextKey=original.mNextKey;
   return *this;
}

int PolygonShape::operator==(PolygonShape &other){
   //Ghn!
   return (mNodes==other.mNodes)&&(mEdges==other.mEdges)&&(mSelection==other.mSelection)
        &&(mLastNode==other.mLastNode)&&(mLastKey==other.mLastKey)&&(mNextKey==other.mNextKey);
}

int PolygonShape::continueStrip(VECTOR3D next){
   //if shape is empty, start a strip
   if(!mHaveStart){
      return startStrip(next);
   }
   //if the first point is not checked in yet, do it.
   if(mLastKey==-1){
      //cout<<"checking in old point at "<<mNextKey<<endl;
      mNodes[mNextKey]=mLastNode;
      mLastKey=mNextKey;
      mNextKey++;
   }
   //check in the new node
   //cout<<"checking in new point at "<<mNextKey<<endl;
   mNodes[mNextKey]=next;
   int second=mNextKey;
   mNextKey++;
   pair<int,int> p;
   p.first=mLastKey;
   p.second=second;
   mEdges.push_back(p);
   mLastKey=second;
   mLastNode=mNodes[second];
   return second;
}

int PolygonShape::continueStrip(int key){
   //if key is not found return -1
   if(mNodes.find(key)==mNodes.end()){
      return -1;
   }
   //if shape is empty, start a strip
   if(!mHaveStart){
      return startStrip(key);
   }

   //if the first point is not checked in yet, do it.
   if(mLastKey==-1){
      //cout<<"checking in old point at "<<mNextKey<<endl;
      mNodes[mNextKey]=mLastNode;
      mLastKey=mNextKey;
      mNextKey++;
   }
   //cout<<"continue to "<<key<<endl;
   mEdges.push_back(pair<int,int>(mLastKey,key));
   mLastKey=key;
   mLastNode=mNodes[key];
   return key;

}

int PolygonShape::startStrip(int key){

   //if key is not found return -1
   if(mNodes.find(key)==mNodes.end()){
      return -1;
   }
   //cout<<"start with key "<<key<<endl;
   mLastKey=key;
   mLastNode=mNodes[key];
   mHaveStart=true;
   return key;
}

int PolygonShape::startStrip(VECTOR3D first){
   //cout<<"start with point"<<endl;
   mLastNode=first;
   mLastKey=-1;
   mHaveStart=true;
   return -1;
}

void PolygonShape::removeEdge(int a, int b){
   //investigate our list of edges
   //remove all edges from point1 to point2
   //for both points count remaining refs.
   int refcnt1=0,refcnt2=0;
   bool deleted = false;

   //cout<<"about to erase all edges from "<<&point1<<" to "<<&point2<<endl;
   int i=getEdgeCount()-1;
   while(i>=0){
      pair<int,int> e = mEdges[i];
	  vector<pair<int,int> >::iterator it = mEdges.begin();
      int p1=e.first;
      int p2 = e.second;
      //cout<<"---------------------------------------"<<endl;
      //cout<<"edge "<<i<<"("<<&e<<"): from "<<p1<<" to "<<p2<<endl;
      if(p1==a && p2==b){
         //cout<<"MATCH! erasing edge "<<&e<<" from position "<<i<<endl;
         mEdges.erase(it+i);
         deleted=true;
      }
      else {
         if(p1==a||p2==a){
            //cout<<"Ref to first point found: "<<&point1<<endl;
            refcnt1++;
         }
         if(p1==b||p2==b){
            //cout<<"Ref to second point found: "<<&point2<<endl;
            refcnt2++;
         }
      }
      i--;
   }
   //if there are no remaining refs to a node, remove it
   //(the deleted flag should be true, if both points are part of the shape
   if(deleted){
      if(refcnt1==0) {
         //cout<<"No Refs left to first point. Deleting "<<&point1<<endl;
         mNodes.erase(a);
		 mSelection.erase(a);
      }
      if(refcnt2==0){
         //cout<<"No Refs to second point. Deleting "<<&point2<<endl;
         mNodes.erase(b);
		 mSelection.erase(b);
      }
   }
   if(mEdges.empty()) mHaveStart=false;
}

void PolygonShape::removeNode(int key){
   //we remove a node by calling removeEdge for each adjacent Edge
   vector<pair<int,int> > adj=adjacentEdges(key);
   vector<pair<int,int> >::iterator it = adj.begin();
   while(it!=adj.end()){
      removeEdge((*it).first,(*it).second);
      it++;
   }
   if(mEdges.empty()) mHaveStart=false;
}

vector<pair<int,int> > PolygonShape::adjacentEdges(int key){
   //examine each edge in mEdges
   //if point1 or point2 member refer to point, add edge to result
   vector<pair<int,int> > result;

   for (int i=0;i<getEdgeCount();i++){
      pair<int,int> e=getPair(i);
      int p1 = e.first;
      int p2 = e.second;
      if(p1==key||p2==key) result.push_back(e);
   }
   return result;
}

pair<int,int> PolygonShape::getPair(int index){
   return mEdges[index];
}
Edge PolygonShape::getEdge(int index){
   return Edge(mNodes[mEdges[index].first],mNodes[mEdges[index].second]);
}

VECTOR3D PolygonShape::getNode(int key){
   return mNodes[key];
}

pair<int,int> PolygonShape::getLastEdge(){
   return mEdges[mEdges.size()-1];
}

int PolygonShape::indexOf(pair<int,int> edge){
   //examine each edge. if points refer to same VECTO3Ds, return index
   for(int i=0;i<getEdgeCount();i++){
      pair<int,int> e = getPair(i);
      if(e==edge){
         return i;
      }
   }
   return -1;
}

int PolygonShape::indexOf(int a, int b){
   return indexOf(pair<int,int>(a,b));
}

int PolygonShape::getEdgeCount(){
   return mEdges.size();
}

int PolygonShape::getNodeCount(){
   return mNodes.size();
}

int PolygonShape::getLastKey(){
   return mLastKey;
}

VECTOR3D PolygonShape::getLastNode(){
   return mLastNode;
}

int PolygonShape::findNearestNode(VECTOR3D point){
   if(mNodes.empty()) return -1;
   //initialize min to the greatest possible float.
   float min = FLT_MAX;
   map<int,VECTOR3D>::iterator min_ptr = mNodes.end();
   //get an iterator over the nodes
   map<int,VECTOR3D>::iterator it = mNodes.begin();
   while(it!=mNodes.end()){
      float dist = vecSqrMagnitude((*it).second - point);
      if(dist<=min){
         min_ptr=it;
         min=dist;
      }
      it++;
   }


   return (*min_ptr).first;
}

void PolygonShape::select(int key){
   map<int,VECTOR3D>::iterator it=mNodes.find(key);
   if(it==mNodes.end()){
      return;
   }
   mSelection.insert(*it);
   //cout<<mSelection.size()<<" nodes selected."<<endl;
}

void PolygonShape::unselect(int key){
   if(mSelection.find(key)==mSelection.end()){
   //cout<<"unselect couldn't find"<<key<<endl;
      return;
   }
   //cout<<"unselecting"<<key<<endl;
   mSelection.erase(key);
   //cout<<mSelection.size()<<" nodes selected."<<endl;
}

bool PolygonShape::isSelected(int key){
   //cout<<"key "<<key<<" was found "<<mSelection.count(key)<<" times in Selection"<<endl;
   if (mSelection.count(key)>0) return true;
   return false;
}

void PolygonShape::clearSelection(){
   mSelection.clear();
   //cout<<mSelection.size()<<" nodes selected."<<endl;
}

void PolygonShape::selectAll(){
   mSelection=mNodes;
   //cout<<mSelection.size()<<" nodes selected."<<endl;
}
map<int,VECTOR3D> PolygonShape::getSelection(){
   return mSelection;
}

map<int,VECTOR3D> PolygonShape::getNodes(){
   return mNodes;
}

void PolygonShape::updateSelection(){
   map<int,VECTOR3D>::iterator it = mSelection.begin();
   while(it!=mSelection.end()){
      mSelection[(*it).first]=mNodes[(*it).first];
      it++;
   }
}

void PolygonShape::selectRegion(VECTOR3D first, VECTOR3D second){
   //set up maximum/minimum for coordinates:
   float maxX = max(first.x,second.x);
   float minX = min(first.x,second.x);
   float maxY = max(first.y,second.y);
   float minY = min(first.y,second.y);
   map<int,VECTOR3D>::iterator it = mNodes.begin();
   while(it!=mNodes.end()){
      if((*it).second.x >= minX
      && (*it).second.x <= maxX
      && (*it).second.y >= minY
      && (*it).second.y <= maxY){
         mSelection.insert(*it);
      }
      it++;
   }
   //cout<<mSelection.size()<<" nodes selected."<<endl;
}

void PolygonShape::selectRegion(VECTOR3D center, float qradius){
   map<int,VECTOR3D>::iterator it = mNodes.begin();
   while(it!=mNodes.end()){
      float distance=vecSqrMagnitude((*it).second-center);
      if(distance<=qradius){
         mSelection.insert(*it);
      }
      it++;
   }
   //cout<<mSelection.size()<<" nodes selected."<<endl;
}

void PolygonShape::applyTransformation(float * matrix, bool absolute){
   map<int,VECTOR3D>::iterator it = mSelection.begin();
   while(it!=mSelection.end()){
      VECTOR3D point;
      if(absolute){
         point=(*it).second;
      }
      else{
         point=mNodes[(*it).first];
      }
      mNodes[(*it).first]=atrans(point,matrix);
      it++;
   }
}

int PolygonShape::save(){
   PolygonShape* me = new PolygonShape(*this);
   mStates.push_back(me);
   return mStates.size();
}

int PolygonShape::revert(){
   if ( mStates.empty() )
	   return -1;
   PolygonShape* latest = mStates.back();
   *this = *latest;
   mStates.pop_back();
   delete latest;
   return mStates.size();
}


//#endif

