/*
 * Copyleft 2001 Lukas Degener, Ingmar Kanitscheider. All rights granted ;-)
 *----------------------
 * file: src/FloatEditor.h
 *----------------------
 * This file is part of the CountourWarp Effect based on the GLEffect Framework.
 * In this file you will find the declaration of the FloatEditor class
 * Implementation is in FloatEditor.cpp
 *
 * Comments are doxygen compatible
 */

/*! @file FloatEditor.h
 *
 *  @brief
 *  Declaration of FloatEditor class
 *  @author Ingmar Kanitscheider, Lukas Degener
 */

#ifndef FLOATEDITOR_H
#define FLOATEDITOR_H

#include <qvariant.h>
#include <qwidget.h>
#include <qslider.h>
#include <qlcdnumber.h>
#include <qtooltip.h>
#include "WarpAlgorithm.h"

//Huh? (msvc seems to need this?)
class QVBoxLayout;
class QHBoxLayout;
class QGridLayout;
class QLabel;
class QSpinBox;

/*! @class FloatEditor
 *
 * @ingroup contourWarpClasses
 *
 * @brief GUI widget for editing algorithm specific parameters
 *
 * An instance of this class represents a GUI widget specialy designed for editing
 * algorithm specific parameters of a WarpAlgorithm instance. As the name suggests, it was made to
 * edit parameters of type float.
 *
 * @note This class is moreless a quick hack, so don't expect too much. Anyway, it should
 * give an idea to anyone interested on how to use the paramInfo struct achieved from an
 * algorithm.
 *
 * A FloatEditor supports means of communicating with the rest of the gui through
 * a single public slot setValue(float) and a signal valueChanged(int,float)
 * Within this Project, it is used as a child widget of the AlgorithmControl widget.
 *
 * Apearance is a bit clumsy right now, if you want a more elaborated gui, you should
 * think about rewriting this class from scratch. At the moment, the following
 * features are implemented:
 *
 * -  a slider for setting the parameter
 * -  a LCD for visual feedback
 * -  bidirectional communication with the application
 * -  two icons representing the effect of very big and very small values.
 *    (paramInfo::minIcon, paramInfo::maxIcon)
 * -  a tooltip for more detailed description for the slider (paramInfo::descr)
 * -  tooltips for both icons (paramInfo::minDescr paramInfo::maxDescr)
 * -  reads default value (paramInfo::dflt)
 * -  correctly sets minimum/maximum boundries for the slider (paramInfo::min,
 *    paramInfo::max)
 *
 * @todo Implement additional slot for reseting value to default.
 */


class FloatEditor : public QWidget
{
    Q_OBJECT

public:
   /*!
    * @brief Creates a new FloatEditor widget
    *
    * Creates a FloatEditor and initializes it using parameter information found in
    * <code>info</info>
    *
    * @param paramId set this to something you like;-) . The FloatEditor will store this
    * value and include it as argument each time the valueChanged(int,float) signal is
    * emmited. You should use this to easiely identify the source of the signal.
    *
    * @param info meta information to describe the parameter edited by this FloatEditor
    * instance. Usualy it gets filled out by WarpAlgorithm::getParamInfo(int,paramInfo)
    *
    * @param parent A pointer to the parent widget
    * 
    * @param fl Window style flags
    *
    * Apart <code>info</code>, all arguments are passed through to the QWidget constructor
    *
    */
   FloatEditor( int paramId = 0,QWidget* parent = 0, paramInfo* info=0, WFlags fl = 0 );

   /*!
    * @brief deconstructor
    *
    * Does nothing ;-)
    */
   ~FloatEditor();

private:
   /*!
    * @brief pointer to the QSlider child widget
    */
   QSlider* slider;

   /*!
    * @brief pointer to the QLCDNumber child widget
    */
   QLCDNumber* lcd;

   /*!
    * @brief pointer to the widgets layout manager
    */
   QVBoxLayout* FloatEditorLayout;

   /*!
    * @brief the Editor's id
    * @see FloatEditor( int ,QWidget* , paramInfo* , WFlags )
    */
   int ID;


public slots:

   /*!
    * @brief set the Editors value
    * @param v the new value
    */
   void setValue(float v);

private slots:

   /*!
    * @brief should be called, when the slider value changed
    *
    * This is an internal slot, which is connected to the <code>valueChanged</code>
    * signal of slider. It's use is to update the lcd accordingly.
    * @param v the new value set by the slider.
    * @note unfortunatly, the QSlider class only supports int values. To get
    * floats anyway, the sliders actual range is scaled by 20. Consequently we have
    * to divide it by 20 within this class, to get the real value.
    * This value is subsequently passed on to the LCD and also emmited via the
    * valueChanged(int,float)
    */
   void sliderValueChanged(int v);

signals:
   /*!
    * @brief emitted on value change
    *
    * Hmm. i honestly don't know anymore what this method was good for.
    * Will have another look on the code. Stay tuned.
    * @todo find out what this method was used for and comment it accordingly,
    * or remove it if not used anymore.
    */
   void valueChanged(double v);

   /*!
    * @brief emitted on value change
    *
    * @param paramId the Id that was set for this editor instance
    * @param v the new value
    * @see FloatEditor( int ,QWidget* , paramInfo* , WFlags )
    */
   void valueChanged(int paramId, float v);

};

#endif // FLOATEDITOR_H

