/*
 * Copyleft 2001 Lukas Degener, Ingmar Kanitscheider. All rights granted ;-)
 *----------------------
 * file: src/ContourWarpEffect.h
 *----------------------
 * This file is part of the CountourWarp Effect based on the GLEffect Framework.
 * In this file you will find the declaration of the WarpAlgorithm* classes
 * Implementation is in WarpAlgorithm.cpp
 *
 * Comments are doxygen compatible
 */


/*****************************************************************************\
 *                              ContourWarp Effect
\*****************************************************************************/

/**
 *
 * @defgroup contourWarpClasses Contour Warp Effect classes
 *
 */

/*! @file ContourWarpEffect.h
 *
 *  @brief
 *	A ContourWarp effect class
 *   
 *  @author Lukas_and_Ingmar
 *     
 */

#ifndef CONTOURWARPEFFECT_H
#define CONTOURWARPEFFECT_H

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

// system library headers
#include <stdio.h>
#include <stdlib.h>

// Qt headers
#include <qlcdnumber.h>
#include <qslider.h>
#include <qlabel.h>
#include <qradiobutton.h>
#include <qvbuttongroup.h>
#include <qhbox.h>
#include <qcheckbox.h>
#include <qdialog.h>
#include <qtoolbutton.h>

// Local headers
#include "../main/GLEffect.h" 
#include "vector.h"
#include "PolygonShape.h"
#include "PolygonShapeEditor.h"
#include "AlgorithmControl.h"
#include "WarpAlgorithm.h"

//---------------------------------------------------------------------------
//  Forward references
//---------------------------------------------------------------------------

// controls classes

class QTimer;


//**************************************************************************
// ContourWarp Effect
//**************************************************************************

/*! @class ContourWarpEffect
 *
 * @ingroup contourWarpClasses
 * 
 * @brief The main class of our project
 *
 * Most of the GUI interaction of the user is implemented here as well
 * as the central class coordination.
 * The ContourWarpEffect class communicates with the following classes:
 * - CVertexGrid: A class for storing a square grid of vertices
 * - WarpAlgorithm: This class transforms the vertexGrid with the warp algorithm
 * - AlgorithmControl: This class manages a gui widget for algorithm control
 * - PolygonShapeEditor: This class manages user interaction on drawing of contoures
 * - PolygonShape: This class covers a contour.
 */
class ContourWarpEffect: public GLEffect
{
	Q_OBJECT

//---------------------------------------------------------------------------
//  Public Instance Methods
//---------------------------------------------------------------------------
public:

// --------------------------------------------------------------------------
//! @name  Creation, desctruction
// @{

	//! Primary class contructor
	ContourWarpEffect(GLFramework* parent);
	
	//! Destructor
	virtual ~ContourWarpEffect();
	
// --------------------------------------------------------------------------
// @} 

//! @name Public API functions
// @{


	//! Creates control widgets. 
	virtual void createControls(QWidget* parentWindow);

   //! Overwritten GLEffect::hide() method. Unsets forceMouseTracking.
   virtual void hide();
   //! Overwritten GLEffect::show() method. Sets forceMouseTracking.
   virtual void show();


// --------------------------------------------------------------------------
// @}

/** @name Rendering, animation and effect control functions */
	
// @{		  
		//! Resets the effect's parameters.
		virtual void reset();
		
		//! Effect-specific implementation of the rendering routine.
		virtual void render();

		//! overwritten GLEffect::update() method
		virtual void update();

		virtual void stop();
		virtual void play();
		virtual void pause();

// --------------------------------------------------------------------------
// @} 

//! @name Event handlers 
// @{ 
		
		//! Mouse move events handler.
		virtual void mouseMoveEvent(QMouseEvent*);
		//! Mouse click events handler.
		virtual void mousePressEvent(QMouseEvent*);
        //! Mouse release events handler.
		virtual void mouseReleaseEvent(QMouseEvent*);
        //! Mouse wheel events handler
        virtual void wheelEvent(QWheelEvent*);
		
// --------------------------------------------------------------------------
// @}


// --------------------------------------------------------------------------
//! @name	Signals:
// @{

	signals:

// --------------------------------------------------------------------------
// @}

//! @name	Slots:
// @{

	public slots:

	private slots:

		//! Timer slot. Calls applyWarp().
		void animate();
		//! Warp button slot. Enters/leaves warp mode. 
		void warpMode(bool enable);
		//! Called when warp recalculation must be forced.
		void warpChanged();
		//! Forces warp recalculation in warping mode.
		void shapeChanged();
		//! Vertex grid initialization. Force recaching.
		void algoChanged();
		//! Sets the vertex grid resolution
		void setGridResolution (int value);
		//! Decides whether to draw filled polygons, lines or points
		void setPolygonMode (int value);
		//! Called when entering/leaving differential mode
		void setDiffMode (bool value);
		//! Called when entering/leaving "frame as edges"-mode
		void setFrameAsEdges (bool value);
		//! Decides whether to draw the origin of scale and rotation transformations
		void setDrawOrigin(bool value);
		//! Decides whether to draw the contoures
		void setDrawEdges(bool value);
		//! Decides whether to draw little rects about the nodes of the shapes
		void setDrawNodes(bool value);
		//! Decides whether to draw the original shape during warp
		void setDrawOrigShape(bool value);
		//! Manages edit mode popup menu. Sets the appropriate edit cursor
		void editModeChanged(int mode);
		//! Decides whether to allow only to draw contoures within the image
		void contOnlyImage(bool value);


// --------------------------------------------------------------------------
// @}


//---------------------------------------------------------------------------
//  Private/Protected Instance Methods
//---------------------------------------------------------------------------

protected:

	//! Initializes GL states and parameters.
	virtual void initialize();


private:
	//! inline method which draws a square in radius r about a point
	void pointRect(VECTOR3D v, float r);
	
   //! updates the mouse* and mouse3D* mouse coords
   void updateMouse(QMouseEvent *qEv);

   //! returns the projection of the mouse coords on the image
   int mouseOnImage(VECTOR3D *output);

   //! renders a contour
   void RenderPolygonShape(PolygonShape *ps);

   //! transforms the vertex grid with the current warp algorithm
   void applyWarp();

//---------------------------------------------------------------------------
//  Private/Protected Instance Variables
//---------------------------------------------------------------------------

private:

	//! Disabled default constructor 
	ContourWarpEffect();
	//! Disabled assignment operator
	ContourWarpEffect& operator = (const ContourWarpEffect& source);

// --------------------------------------------------------------------------
//! @name Member controls
//! @{
	
   //! Dialog for setting warping parameters
   QDialog        *mWarpDialog;
   //! Button for edit mode change
   /*!
    * @see editModeChanged(int) slot.
    */
   QToolButton    *mEditButton;
   //! Button for entering/leaving warp mode.
   /*!
    * @see warpMode(bool) slot.
    */
   QPushButton    *mWarpButton;

   //! Wedge caching label in warp dialog.
   QLabel		  *mWCacheLabel;
   
   //! Box the label and the slider for the grid resolution is in.
   QHBox		  *mSliderBox;

// --------------------------------------------------------------------------
//! @}


//! @name Member Variables
//! @{
	//! warp calculation timer.
    /*!
	 * @see animate() slot.
	 */
	QTimer* effectTimer; 

	//! Texture object. Needed one time in initialize() 
	GLuint texture; 

	//! Pointer to a square grid of vertices the image lies on
	CVertexGrid  *vertexGrid; 
	//! Normalized top edge vector of the vertex grid.
	VECTOR3D	m_vTop;
	//! Normalized left edge vector of the vertex grid.
	VECTOR3D	m_vLeft;
	
	//! original contour line
	PolygonShape *origShape;
	//! transformed contour line
	PolygonShape *trafoShape;
	//! transformed contour line in the last frame. Used in differential mode.
	PolygonShape *trafoDiffShape;
	//! an Editor which changes the PolygonShapes in reaction of mouse events
	PolygonShapeEditor *PSEditor;

	//! A widget for algorithm control placed in the warp dialog.
	AlgorithmControl *algoControl;

	//! See warpMode(bool) slot.
	bool m_bIsWarping;
	//! See warpChanged() slot.
	bool m_bWarpChanged;
	//! See setDiffMode() slot.
	bool m_bDiffMode;
	//! See setFrameAsEdges(bool) slot.
	bool m_bFrameAsEdges;
	//! See setDrawEdges(bool) slot.
	bool m_bDrawEdges;
	//! See setDrawOrigin(bool) slot.
	bool m_bDrawOrigin;
	//! See setDrawNodes(bool) slot.
	bool m_bDrawNodes;
	//! See setDrawOrigShape(bool) slot.
	bool m_bDrawOrigShape;
	//! See contOnlyImage(bool) slot.
	bool m_bContOnlyImage;

	// the polygon mode in rendering
	GLenum polygonMode;
// --------------------------------------------------------------------------
//! @}

//---------------------------------------------------------------------------
//  Class methods and variables
//---------------------------------------------------------------------------

};

#endif


