
/*****************************************************************************\
 *                              ContourWarp Effect
\*****************************************************************************/

/*! @file ContourWarpEffect.cpp
 *
 *  @brief
 *	ContourWarpEffect effect class implementation
 *    
 *  @author Lukas Degener, Ingmar Kanitscheider
 *  
 *  
 */

//---------------------------------------------------------------------------
// Compiler switch for microsoft
//---------------------------------------------------------------------------

// to reduce warning messages in Visual C++
#if _MSC_VER >= 1000
#pragma warning(disable:4786)
#endif // _MSC_VER >= 1000

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

// for some calculations
#include <math.h>

// Qt headers
#include <qtimer.h>
#include <qvgroupbox.h>
#include <qhbox.h>
#include <qpushbutton.h>
#include <qlayout.h>
// QT3 puts this into an extra include
#ifdef QTTHREE
#include <qcursor.h>
#endif

// Local Headers
#include "geomObj.h"
#include "ContourWarpEffect.h"

//---------------------------------------------------------------------------
//  Some usefull macros
//---------------------------------------------------------------------------

#define SAFE_DELETE(x) {if (x) {delete(x);(x)=NULL;}}
#define SAFE_DELETE_ARRAY(x) {if (x) {delete[](x);(x)=NULL;}}

// usefull macro to call glVertex3f with argument const VECTOR3D;
// the parentheses in the macro definition are very important
// for calls such VECTO3F(v-w)
#define VECTO3F(v) (v).x,(v).y,(v).z



/*****************************************************************************\
 *                              ContourWarp Effect
\*****************************************************************************/

// **** Construction/desctruction ****

/**
 * @param parent Pointer to the parent window.
 *
 * @see GLEffect::GLEffect(), effectInfo
 **/

ContourWarpEffect::ContourWarpEffect(GLFramework* parent):GLEffect(parent)
{

	// initialize effect info data
	effectInfo->effectName = "ContourWarp Effect";
	effectInfo->fileDialogNames[0] = "Load Image(s)";
	effectInfo->needNewTextures = true;
	effectInfo->requiredImageLists = 1;
	effectInfo->shortDescription = "Warps images with contoures";
	effectInfo->version = "1.0";
	effectInfo->author = "Lukas and Ingmar";
	effectInfo->needRawTextures = false;

	// classes which will be initialized later are set to NULL
	algoControl = NULL;

	// create member classes
	vertexGrid = new CVertexGrid();
	PSEditor = new PolygonShapeEditor();
	origShape = new PolygonShape();
	trafoShape = new PolygonShape();
	trafoDiffShape = new PolygonShape();

	// connect the editor signal
	connect (PSEditor, SIGNAL(shapeChanged()), this, SLOT(shapeChanged()));

	// By default, the original contour line is edited
	PSEditor->setShape(origShape);

	// reset flags
	m_bIsWarping = false;
	m_bWarpChanged = false;
	m_bDiffMode = false;
	m_bFrameAsEdges = true;
	m_bContOnlyImage = false;

	// set normal polygon mode as default
	polygonMode = GL_FILL;

	// The vertex grid must be initialized before the controls!

	// initialize vertexGrid under the image
	vertexGrid->SetGridData (VECTOR3D (-1.2f, 1.2f, 0.0f),
							 VECTOR3D ( 1.2f, 1.2f, 0.0f),
							 VECTOR3D (-1.2f,-1.2f, 0.0f),
							 20, 20);

	/* This has to be included as the last line in your constructor.
	Don't forget it! */
	createControlPanel(false);
   	hideControls();

}

/**
 *
 *
 * @see GLEffect::~GLEffect()
 **/
ContourWarpEffect::~ContourWarpEffect()
{
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_DEPTH_TEST);

	// delete all non-Qt objects
	// the SAFE_DELETE macro is defined above
	SAFE_DELETE (vertexGrid);
	SAFE_DELETE (PSEditor);
	SAFE_DELETE (origShape);
	SAFE_DELETE (trafoShape);
	SAFE_DELETE (trafoDiffShape);
	SAFE_DELETE (algoControl);
}

// Initialization
/**
 * 
 *
 * @see GLEffect::init()
 **/

void ContourWarpEffect::initialize()
{
	// init effect params
	viewDistance = 3;
	nearClipPlane = 1;
	farClipPlane = 30;
	fovAngle = 60.0;

	currentDepthCoord = 0.0;

	glClearColor(0.0,0.0,0.0,1);
	glShadeModel(GL_SMOOTH);

	glLoadIdentity();

	// set texturing parameters
	glEnable(GL_DEPTH_TEST);

	glEnable(GL_TEXTURE_2D);

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	glGenTextures(1, &texture);
	glBindTexture(GL_TEXTURE_2D, texture);

	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);

	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

	// calculate once the normalized top and left edge vector of the vertex grid
	m_vTop = vecNormalize (vertexGrid->m_vTopRight - vertexGrid->m_vTopLeft);
	m_vLeft = vecNormalize (vertexGrid->m_vBottomLeft - vertexGrid->m_vTopLeft);

	// fetch image in order to view them after starting
	texImage1 = parentFrame->fetchImage(0,0);

	// we are in drawing mode after starting so set the cross cursor
	glView->setCursor(QCursor::QCursor(CrossCursor));
}

/**
 * Almost all gui widgets are initialized here except the parameter widgets and the
 * "frame as edges"-checkbox of the warp dialog which is initialized in AlgorithmControl
 * and the edit mode submenu which is initialized in PolygonShapeEditor.
 *
 * @see GLEffect::createControlPanel()
 **/
void ContourWarpEffect::createControls(QWidget* parentWindow)
{
  // Timer used to calculate warping
  effectTimer = new QTimer();
  connect(effectTimer, SIGNAL(timeout()), this, SLOT(animate()) );

  //create the controlpanel
  controlPanel=new QVGroupBox(parentWindow);

  //create a top-level modeless dialog for warping issues
   mWarpDialog=new QDialog(0,0,false, Qt::WStyle_Customize
                           | Qt::WStyle_NormalBorder | Qt::WStyle_Tool
                           | Qt::WStyle_StaysOnTop );
  //do some fancy layout stuff i don' quiet understand so i just took it
  //from Ibragims code
  QGridLayout * algoLayout = new QGridLayout(mWarpDialog,1,1,10);
  algoLayout->setSpacing(3);
  QVBox * algobox = new QVBox(mWarpDialog);
  algoLayout->addWidget(algobox,0,0);
  
  //Now, everything that goes into the warp dialog has algobox as its
  //parent. Everthing else use controlPanel.

  //Create algorithm params into algobox
  algoControl = new AlgorithmControl(algobox);
  // set the vertexGrid for the active WarpAlgorithm
  algoControl->getAlgorithm()->setVertexGrid(vertexGrid);

  // Connect signals of AlgorithmControl
  connect (algoControl, SIGNAL(frameAsEdgesChanged(bool)), this, SLOT(setFrameAsEdges(bool)));
  connect (algoControl, SIGNAL(warpChanged()), this, SLOT(warpChanged()));
  connect (algoControl, SIGNAL(algoChanged()), this, SLOT(algoChanged()));

  // additional widgets to be placed into the warp dialog
  
  // a diff mode checkbox
  QCheckBox * dmCheckBox = new QCheckBox ("Differentialmodus", algobox);
  connect (dmCheckBox, SIGNAL (toggled(bool)), this, SLOT(setDiffMode(bool)));
  QToolTip::add (dmCheckBox, "Im Differentialmodus wird das Warping\nin vielen kleinen Schritten berechnet");
  // a checkbox whether to show the original shape
  QCheckBox * origCheckBox = new QCheckBox ("Ausgangs Skelett sichtbar", algobox);
  origCheckBox->setChecked(true);
  m_bDrawOrigShape=true;
  connect (origCheckBox, SIGNAL (toggled(bool)), this, SLOT(setDrawOrigShape(bool)));
  // a wedge caching label
  mWCacheLabel = new QLabel (algobox, "Wedge Caching: deaktiviert");

  // widgets to be placed into the controlpanel 
  
  //create another popup which holds the widgets for display mode configuratioons
   QPopupMenu *settings = new QPopupMenu(controlPanel);
   // three button groups
   QVButtonGroup * pmRadioGroup = new QVButtonGroup ("Rendering");
   QVButtonGroup * pmCheckGroup = new QVButtonGroup ("Display");
   QVButtonGroup * pmCheckGroup2 = new QVButtonGroup ("Konturen");
   // the first button group "Rendering" contains three radio buttons
   QRadioButton * pmRadioPoint = new QRadioButton ("Nur Punkte", pmRadioGroup);
   QRadioButton * pmRadioLine = new QRadioButton ("Nur Linien", pmRadioGroup);
   QRadioButton * pmRadioFilled = new QRadioButton ("Normal", pmRadioGroup);
   pmRadioPoint->setAutoMask(true);
   pmRadioFilled->setChecked(true);
   connect (pmRadioGroup, SIGNAL(clicked(int)), this, SLOT(setPolygonMode(int)));
   // the second button group "Display" contains three check boxes
   QCheckBox * dispShowEdges = new QCheckBox ("Kanten sichtbar",pmCheckGroup);
   dispShowEdges->setChecked(true);
   m_bDrawEdges=true;
   connect (dispShowEdges, SIGNAL (toggled(bool)), this, SLOT(setDrawEdges(bool)));
   QCheckBox * dispShowNodes = new QCheckBox ("Knoten sichtbar",pmCheckGroup);
   dispShowNodes->setChecked(true);
   m_bDrawNodes=true;
   connect (dispShowNodes, SIGNAL (toggled(bool)), this, SLOT(setDrawNodes(bool)));
   QCheckBox * dispShowOrigin = new QCheckBox ("Ursprung sichtbar",pmCheckGroup);
   dispShowOrigin->setChecked(true);
   m_bDrawOrigin=true;
   connect (dispShowOrigin, SIGNAL (toggled(bool)), this, SLOT(setDrawOrigin(bool)));
   // the third button group "Konturen" contains only one check box
   QCheckBox * contImage = new QCheckBox ("Konturen auf Bild beschrnken",pmCheckGroup2);
   connect (contImage, SIGNAL(toggled(bool)), this, SLOT(contOnlyImage(bool)));
   // the last item of the popup menu is a resolution slider and a label placed in a
   // QHBox
   mSliderBox = new QHBox();
   QLabel * resLabel = new QLabel (mSliderBox, "reslabel");
   resLabel->setText ("Auflsung der Sttzpunkte:");
   QSlider * resSlider = new QSlider (Qt::Horizontal, mSliderBox, "resSlider");
   resSlider->setMinValue(10);
   resSlider->setMaxValue(100);
   resSlider->setValue(vertexGrid->m_numCols);
   QLCDNumber * resLCD = new QLCDNumber (mSliderBox, "resLCD");
   resLCD->display(vertexGrid->m_numCols);
   connect(resSlider, SIGNAL(valueChanged(int)), resLCD, SLOT(display(int)));
   connect(resSlider, SIGNAL(valueChanged(int)), this, SLOT(setGridResolution(int)));
   // add boxes to the menu
   settings->insertItem(pmRadioGroup);
   settings->insertSeparator();
   settings->insertItem(pmCheckGroup);
   settings->insertItem(pmCheckGroup2);
   settings->insertItem(mSliderBox);
   
   //get the Popup from the Editor:
   QPopupMenu *tools=PSEditor->getPopup();
   
   //create a HBox to put the buttons into
   QHBox * buttonbox = new QHBox(controlPanel);
   //create a Toolbutton, which pops up the menu and shows the active edit mode
   mEditButton = new QToolButton(buttonbox);
   mEditButton->setPopup(tools);
  //Set the initial icon. we will get it right from the popup
  //Warning! this will only work, if PSEditor is allready initialized at this point!!
   mEditButton->setIconSet(*(tools->iconSet(PSEditor->getEditMode())));
   mEditButton->setPopupDelay(1);
   mEditButton->setUsesTextLabel(false);
   QToolTip::add (mEditButton, "Editiermodus");
   
   //create another ToolButton which pops up the settings menu
  QToolButton *settingsButton = new QToolButton(buttonbox);
  QPixmap iSettings("icons/configure.png");
  settingsButton->setPixmap(iSettings);
  settingsButton->setPopup(settings);
  settingsButton->setPopupDelay(1);
  settingsButton->setUsesTextLabel(false);
  QToolTip::add (settingsButton, "Anzeige/Einstellungen");

  //add warpbutton to panel
  mWarpButton = new QPushButton(buttonbox);
  mWarpButton->setText("Warp!");
  mWarpButton->setToggleButton(true);
  mWarpButton->setOn(false);
  connect(mWarpButton, SIGNAL(toggled(bool)),this,SLOT(warpMode(bool)));
  QToolTip::add (mWarpButton, "Warp beginnen");

  //further Signal->Slot connections:
  connect (PSEditor, SIGNAL (editModeChanged(int)), this, SLOT(editModeChanged(int)));



}

// **** Animation and Rendering Control functions ****

/**
 * 
 * Deletes all contoures.
 *
 **/

void ContourWarpEffect::reset()
{
	origShape->selectAll();
	origShape->clearSelection();
	trafoShape->selectAll();
	trafoShape->clearSelection();
	PSEditor->setEditMode(MODE_DRAW);

	update();

}

/*!
 * 
 * Note for mathematicians: Draws a circle with radius r
 * regarding the infinity norm.;-)
 * @param w point in the middle of the square
 * @param r radius (half of the edge length) of the square
 * @note Must be called between glBegin(GL_QUADS) and glEnd()
 */
void ContourWarpEffect::pointRect(VECTOR3D w, float r)
{
	// top-left
	glVertex3f(VECTO3F(w+r*(-m_vTop-m_vLeft)));
	// bottom-left
	glVertex3f(VECTO3F(w+r*(-m_vTop+m_vLeft)));
	// bottom-right
	glVertex3f(VECTO3F(w+r*(+m_vTop+m_vLeft)));
	// top-right
	glVertex3f(VECTO3F(w+r*(+m_vTop-m_vLeft)));
}

/*!
 * Besides of drawing the contour, this function renders also
 * the stuff needed for shape editing as e.g. the little nodes squares,
 * the center of rotation and scale transformations, and rects and circles
 * needed whithin the selecting mode.
 *
 * @param ps pointer to an PolygonShape instance covering the contour
 * 
 * @note In warping mode, this method is called twice. Once for the
 * original shape and once for the transformed shape.
 */

void ContourWarpEffect::RenderPolygonShape(PolygonShape *ps)
{
	// render image frame
	if (m_bFrameAsEdges)
	{
		glDisable (GL_DEPTH_TEST);
		glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
		glBegin (GL_QUADS);
		// use the handy pointRect() inline method with the middle point and half
		// length of the vertex grid edge
		pointRect ((vertexGrid->m_vTopRight+vertexGrid->m_vBottomLeft)/2,
			vecMagnitude (vertexGrid->m_vTopRight-vertexGrid->m_vTopLeft)/2);
		glEnd();
		// restore GL states for the case if there are no edges to render below
		glPolygonMode (GL_FRONT_AND_BACK, polygonMode);
		glEnable (GL_DEPTH_TEST);
		
	}

	// if PSEditor is in rotation or scale mode draw center of transformation
	int mode = PSEditor->getEditMode();
	if (m_bDrawOrigin && 
		(mode == MODE_ROTATE || mode == MODE_SCALE))
	{
		glDisable (GL_DEPTH_TEST);
		VECTOR3D vCenter = PSEditor->getCenter();
		
		VECTOR3D vLeft = vCenter - 0.03f * m_vTop;
		VECTOR3D vRight = vCenter + 0.03f * m_vTop;
		VECTOR3D vTop = vCenter - 0.03f * m_vLeft;
		VECTOR3D vBottom = vCenter + 0.03f * m_vLeft;

		// draw a little cross around the center point
		glBegin (GL_LINES);
			glVertex3fv ((const float*)&vLeft);
			glVertex3fv ((const float*)&vRight);
			glVertex3fv ((const float*)&vTop);
			glVertex3fv ((const float*)&vBottom);
		glEnd();
		glEnable (GL_DEPTH_TEST);
	}
		
	// to reduce function calls store edge count in a variable.
	int EdgeCount = ps->getEdgeCount();

	// declare loop variable for VC/gcc compatibility outside
	int i;

	if ( EdgeCount > 0 || PSEditor->getEditorState()==STATE_DRAWING_STRIP)
	{
		// turn off z-buffer test while drawing lines
		glDisable(GL_DEPTH_TEST);

		// draw the lines of the contoures

		// a dummy edge for easier access
		Edge dummy;

		// begin rendering contoures
		glBegin(GL_LINES);
			for (i=0; (i<EdgeCount)&&m_bDrawEdges; i++)
			{
            	dummy = ps->getEdge(i);
				// for each edge connect the start and end point
				// with a line
				glVertex3fv((const float*)&dummy.point1);
				glVertex3fv((const float*)&dummy.point2);
			}
			if (PSEditor->getEditorState()==STATE_DRAWING_STRIP)
			{
				dummy = PSEditor->getEditEdge();
				// now, draw the edit edge similarly
				glVertex3fv((const float*)&dummy.point1);
				glVertex3fv((const float*)&dummy.point2);
			}
		// end rendering
		glEnd();

		// now, if the editor is in selecting state, draw the selection region

		// don't fill these regions
		glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);

		if (PSEditor->getEditorState()==STATE_SELECTING_BOX)
		{
			// the information about the selection box is archieved
			// through the edit edge
            dummy=PSEditor->getEditEdge();
            
			glBegin(GL_QUADS);

			// NOTE: This way of drawing the rectangle isn't
			// basis independent, i.e., it isn't correct if the vertex Grid
			// doesn't lie in the z=0 plane.
			// For real basis independency, use the dot product
			// between the edit edge and the m_vLeft/m_vTop vectors
			VECTOR3D a(dummy.point2.x,dummy.point1.y,0);
            VECTOR3D b(dummy.point1.x,dummy.point2.y,0);
			// draw the selection box
            glVertex3fv((const float*)&dummy.point1);
            glVertex3fv((const float*)&a);
            glVertex3fv((const float*)&dummy.point2);
            glVertex3fv((const float*)&b);

			glEnd();

		} else
		if (PSEditor->getEditorState()==STATE_SELECTING_CIRCLE)
		{
			// the information about the selection circle is archieved
			// through the edit edge
			dummy = PSEditor->getEditEdge();
		
			// save old matrix
			glPushMatrix();

			// move the first point at the origin
			glTranslatef (VECTO3F(dummy.point1));
			
			// if we wanted real basis independency, we would call 
			// glMultMatrix with m_vTop, -m_vLeft, -m_vTop x m_vLeft
			// as column vectors

			// Perhaps it costs a little bit time allocating memory
			// on each frame, but I didn't want make the quadric
			// a member of ContourWarpEffect
			GLUquadricObj *quadric = gluNewQuadric ();
			
			// we need double for the gluDisk() function
			double radius = (double)vecMagnitude(dummy.point2-dummy.point1);
			gluDisk (quadric, radius, radius,100,2);
			gluDeleteQuadric (quadric);

			// restore matrix
			glPopMatrix();
		}

		// now, draw small rects about the points of the contoures

        // do this only if m_bDrawNodes is true
        if(!m_bDrawNodes){
            // turn on z-buffer test
		    glEnable(GL_DEPTH_TEST);
            return;
		}
			  
		// the radius of the rectangle must fit in the circle
		// with radius of the sensitive area
		float r = PSEditor->getRadius() / sqrt(2.0);

		// start rendering rectangles
		glBegin(GL_QUADS);
		// get the list of nodes...
        map<int,VECTOR3D> nodes = ps->getNodes();
        map<int,VECTOR3D>::iterator it = nodes.begin();
		while (it!=nodes.end())
		{
			// draw the vertex-rect
			pointRect ((*it).second, r);
			it++;
		} 
		if (PSEditor->getEditorState()==STATE_DRAWING_STRIP)
		{
			dummy = PSEditor->getEditEdge();
			// draw the rectangle about the first vertex
			// of the edge about to be painted by the user
			pointRect (dummy.point1, r);
		}
        

		// end rendering
		glEnd();
		
		glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);


        //now render the selected nodes
        glBegin(GL_QUADS);
        nodes = ps->getSelection();
        it = nodes.begin();
			while (it!=nodes.end())
			{
				// draw the vertex-rect
                // don't take positions from the selection but from the shape!
				pointRect (ps->getNode((*it).first), r);
            it++;
			}
		// end rendering
		glEnd();
		// turn on z-buffer test
		glEnable(GL_DEPTH_TEST);
	}
}

/*!
 * @param value one of the OpenGL constants <code>GL_FILL</code>, <code>GL_LINE</code>
 * or <code>GL_POINT</code>. Default is <code>GL_FILL</code>.
 */
void ContourWarpEffect::setPolygonMode(int value)
{
	switch(value)
	{
		case 0: polygonMode = GL_POINT; break;
		case 1: polygonMode = GL_LINE; break;
		case 2: polygonMode = GL_FILL; break;
	}

	update();
}


/**
 * 
 * Renders the effect. 
 *
 **/

void ContourWarpEffect::render()
{
	// a loop variable used several times
	int i;

	// enable texturing
	glEnable(GL_TEXTURE_2D);
	
	// clear render buffer
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	// use green color
	glColor3f (0.0f, 1.0f, 0.0f);
	// hook up the current texture
	if(texImage1!=0)
	 glTexImage2D(GL_TEXTURE_2D,0,4,texImage1->width(), texImage1->height(),
				 0, GL_RGBA, GL_UNSIGNED_BYTE, texImage1->bits());
	
	glBindTexture(GL_TEXTURE_2D, texture);

	// here starts rendering of the vertex grid
	
	// set the polygon mode
	glPolygonMode (GL_FRONT_AND_BACK, polygonMode);
		
	// render the field in a strip of quads along columns
	for (i=0; i<vertexGrid->m_numCols-1; i++)
	{
		// start rendering
		glBegin (GL_QUAD_STRIP);
		
		// with the right-left ordering we get a strip
		// of quads with points oriented counterclockwise

		// these are the start points of a quad strip column
		// first the right
		glTexCoord2fv (vertexGrid->TexCoord(i+1,0));
		glVertex3fv ((const float*)&vertexGrid->Vector(i+1,0));
		// then the left
		glTexCoord2fv (vertexGrid->TexCoord(i+0,0));
		glVertex3fv ((const float*)&vertexGrid->Vector(i+0,0));
		for (int j=0; j<vertexGrid->m_numRows-1; j++)
		{
			// in each row set the points
			// first the right
			glTexCoord2fv (vertexGrid->TexCoord(i+1,j+1));
			glVertex3fv ((const float*)&vertexGrid->Vector(i+1,j+1));
			// then the left
			glTexCoord2fv (vertexGrid->TexCoord(i+0,j+1));
			glVertex3fv ((const float*)&vertexGrid->Vector(i+0,j+1));
		}

		// end rendering
		glEnd();
	}

	glDisable(GL_TEXTURE_2D);

	// draw polygon shape

	// use blue color
	glColor3f (0.0f, 0.0f, 1.0f);

	// draw original contour line
   // don't do this during warp, if m_bDrawOrigShape is false
   if(!m_bIsWarping||m_bDrawOrigShape) RenderPolygonShape(origShape);

	if (m_bIsWarping)
	{
		// use red color
		glColor3f (1.0f, 0.0f, 0.0f);

		// draw warped contour line only in warping mode
		RenderPolygonShape(trafoShape);
	}
	
}

/*!
 * Default: 20.
 */
void ContourWarpEffect::setGridResolution(int value)
{
	// The position of the vertex grid still remains
	// Set only the resolution
	vertexGrid->SetGridData (vertexGrid->m_vTopLeft, vertexGrid->m_vTopRight,
					vertexGrid->m_vBottomLeft, value, value);
	
	// caching is becoming invalid
	algoControl->getAlgorithm()->undoPrecalculation();

	// recalculate warping
	m_bWarpChanged = true;
}

/*!
 * In differential mode, the widgets for parameter control, the "frame as edges" checkbox
 * and the resolution slider are disabled in order to avoid unpredictable results.
 * 
 * Default: false
 *
 * @see warpMode(bool) slot for differential mode explanation
 */
void ContourWarpEffect::setDiffMode(bool value)
{
	m_bDiffMode = value;
	// if we enter/leave diff mode, disable/enable a lot of controls
	algoControl->enableControls(!m_bDiffMode);
	// disable grid resolution slider
	mSliderBox->setEnabled (!m_bDiffMode);
	if (m_bIsWarping)
	{
		if (m_bDiffMode)	
			// caching is now invalid
			algoControl->getAlgorithm()->undoPrecalculation();
	}
	
	// if we quit diff mode, force recalculation of the warp
	if (!m_bDiffMode)
		m_bWarpChanged = true;

	update();
}

/*!
 * Marks that the warp has to be recalculated.
 */
void ContourWarpEffect::warpChanged()
{
	if (m_bIsWarping)
	{
		// force warp recalculation
		m_bWarpChanged = true;
		// if we're not in diff mode, wedge caching is becoming invalid
		if (!m_bDiffMode)
			algoControl->getAlgorithm()->undoPrecalculation();
		
		update();
	}
}

void ContourWarpEffect::shapeChanged()
{
	if (m_bIsWarping)
		m_bWarpChanged = true;
}

void ContourWarpEffect::algoChanged()
{
	// The new algorithm needs a vertex grid pointer.
	algoControl->getAlgorithm()->setVertexGrid(vertexGrid);
	
	// In diff mode, cache is becoming invalid
	if (m_bIsWarping && !m_bDiffMode)
		algoControl->getAlgorithm()->undoPrecalculation();
}

/*!
 * If m_bFrameAsEdges is <code>true</code>, the frame of the image
 * will be added as a constant edge to the warp algorithm.
 * This way, the image border stays fixed during warp.
 *
 * Default: true
 */
void ContourWarpEffect::setFrameAsEdges(bool value)
{
	m_bFrameAsEdges = value;
}

/*!
 * In normal mode, the warp algorithm is called to calculate the warp based on the
 * transformation of the edges between the original shape and the transformed one.
 *
 * In differential mode the warp algorithm is called to calculate the warp between
 * the transformed edge in the last call and the actual transformed edge.
 * As a result, the wedges are newly calculated in differential mode which makes
 * overlaps rarer.
 */
void ContourWarpEffect::applyWarp()
{
	if (!m_bWarpChanged)
		return;
	
	// a pointer to the actual Warp Algorithm
	WarpAlgorithm *algo = algoControl->getAlgorithm();

	// build the normalized cross product of the edge vectors of the plane
	// to get the plane normal vector
	VECTOR3D vNormal = vecNormalize (vecCrossProduct (m_vLeft, m_vTop));

	if (m_bDiffMode)
		algo->transformGrid(trafoDiffShape, trafoShape, vNormal);
	else
	{
		vertexGrid->ResetVertices();

		// only here and in warpMode can Wedge Caching be done, because the vertex grid is
		// surely resetted.
		if (!algo->doesPrecalculation())
			algo->doPrecalculation(origShape);

		algo->transformGrid(origShape, trafoShape, vNormal);
	}

	// let me know if wedge caching is used
	if (algo->doesPrecalculation())
		mWCacheLabel->setText("Wedge Caching: Aktiviert");
	else
		mWCacheLabel->setText("Wedge Caching: Deaktiviert");
	
	*trafoDiffShape = *trafoShape;

	m_bWarpChanged = false;
}

void ContourWarpEffect::update()
{
	// Alternatively warping can be made each time the image is redrawed
	// If you want to use this, activate the following code and deactivate
	// the corresponding code in animate().
	// CAUTION: This can slow down the gui on high resolutions!
	/*if (m_bIsWarping)
		applyWarp();*/

	// actualize texImage1 can't be bad thing
	texImage1 = parentFrame->fetchNextImage(0);
    //if original shape is empty, disable warp
    //Don't do this during warp, since it will slow down things a lot.
    if(!m_bIsWarping)mWarpButton->setEnabled(origShape->getEdgeCount()>0);
	GLEffect::update();
}

void ContourWarpEffect::animate()
{
	// at the moment, the warping is timer-controled
	if (m_bIsWarping)
		applyWarp();
	
	update();

}


void ContourWarpEffect::mousePressEvent(QMouseEvent *qME)
{
   // update mouse vaiables
   updateMouse(qME);
   VECTOR3D mousePos;
   // project mouse coordinates on image plane
   if (mouseOnImage(&mousePos))
	    // send coordinate to the editor
		PSEditor->glPanelPressEvent(qME, mousePos);
   // force redrawing
   update();
}

void ContourWarpEffect::mouseReleaseEvent(QMouseEvent *qME)
{
   // update mouse vaiables
   updateMouse(qME);
   VECTOR3D mousePos;
   // project mouse coordinates on image plane
   if (mouseOnImage(&mousePos))
	    // send coordinate to the editor
		PSEditor->glPanelReleaseEvent(qME, mousePos);
   // force redrawing
   update();
}

void ContourWarpEffect::mouseMoveEvent(QMouseEvent *qME)
{
   // update mouse vaiables
   updateMouse(qME);
   VECTOR3D mousePos;
   // project mouse coordinates on image plane
   if (mouseOnImage(&mousePos))
	    // send coordinate to the editor
		PSEditor->glPanelMoveEvent(qME, mousePos);
   // force redrawing
   update();
}

void ContourWarpEffect::wheelEvent(QWheelEvent* e){
   // the edit mode can be changed with the wheel
   if (e->delta()<0) PSEditor->nextEditMode();
   else PSEditor->previousEditMode();
}

/*!
 * @note The two dimensional Y-coordinate of the mouse position
 * must be inverted regarding the viewport because of the different
 * orientation of the Y-axis in GL space and Qt space.
 * @see GLEffect::get3DMouseCoords (GLdouble,GLdouble,GLdouble,GLdouble)
 * @see mouseOnImage(VECTOR3D*)
 */
void ContourWarpEffect::updateMouse(QMouseEvent *mEv){
	// get mouse states
	mouseX = mEv->x();
	// invert the two dimensional Y-coordinate of the mouse position
	mouseY = glView->height()-mEv->y();
	mouseButtonPressed = mEv->state();

	// transform to 3d mouse coordinates
	get3DMouseCoords(currentDepthCoord, mouse3D_X, mouse3D_Y, mouse3D_Z);

	// optionally provide normalized 2d mouse coordinates

	// normalize coordinates to [-1...1]
	normMouseX = mouseX / (GLdouble(glView->width())/2);
	normMouseX -= 1.0;

	normMouseY = mouseY / (GLdouble(glView->height())/2);
	normMouseY -= 1.0;

}

/*!
 * cuts a ray through the unprojected mouse coords on the near
 * and far plane with the image plane.
 * if m_bContOnlyImage is true the nearest point on the image is
 * returned if the mouse position is outside the image.
 *
 * @param output pointer to a VECTOR3D struct to be filled with
 *        the projected 3D mouse coordinate on the image.
 * @returns 1 if the image is between near and far plane and the plane
 *          isn't parallel to the mouse ray.
 */
int ContourWarpEffect::mouseOnImage(VECTOR3D *output)
{

	// to avoid float-double compatibility problem...
	double mn_x, mn_y, mn_z;
	double mf_x, mf_y, mf_z;
	// find out 3d mouse coords on the near and far plane
	get3DMouseCoords (0.0, mn_x,mn_y, mn_z);
	get3DMouseCoords (1.0, mf_x,mf_y, mf_z);
	VECTOR3D mouse3dnear = VECTOR3D ((float)mn_x,(float)mn_y,(float)mn_z);
	VECTOR3D mouse3dfar  = VECTOR3D ((float)mf_x,(float)mf_y,(float)mf_z);
	
	// create ray from the 3d mouse coords on near and far plane.
	Edge mouseray = Edge(mouse3dnear, mouse3dfar);

	// the final 3d position on the image plane
	VECTOR3D mousePos;
	Edge VGTop = Edge (vertexGrid->m_vTopLeft,vertexGrid->m_vTopRight);
	Edge VGLeft = Edge (vertexGrid->m_vTopLeft,vertexGrid->m_vBottomLeft);

	// we just have to set the edge type, and IntersectEdgeParallelogram does the rest:
	// if we only want to draw in the image, this function projects the mouse position
	// on the frame of the image.
	if (!m_bContOnlyImage)
	{
		VGTop.type = line;
		VGLeft.type = line;
	}

	// Note to Lukas: The IntersectEdgeParallelogram function checks
	// if the user has clicked in the image.
	// If there is a bug in this function perhaps turning the
	// following types into line helps because then, the function
	// doesn't perform the click-in-image check.
	//VGTop.type = edge; VGLeft.type = edge;
	// call IntersectEdgeParallelogram helper function from geomObj.cpp
	// it finds the intersection point between a ray and (a parallelogram
	// or a plane).
	return (IntersectEdgeParallelogram (mouseray, VGLeft, VGTop, output));
 
}

/*!
 *
 * @see GLEffect::show() 
 */
void ContourWarpEffect::show(){
  
   GLEffect::show();
   mWarpDialog->show();
   forceMouseTracking(true);
}

/*!
 *
 * @see GLEffect::hide() 
 */
void ContourWarpEffect::hide(){
   forceMouseTracking(false);

   mWarpDialog->hide();
   GLEffect::hide();
}

/*!
 * Default: true
 */
void ContourWarpEffect::setDrawEdges(bool b){
   m_bDrawEdges=b;
   update();
}

/*!
 * Default: true
 */
void ContourWarpEffect::setDrawOrigin(bool b){
   m_bDrawOrigin=b;
   update();
}

/*!
 * Default: true
 */
void ContourWarpEffect::setDrawNodes(bool b){
   m_bDrawNodes=b;
   update();
}

/*!
 * Default: true
 */
void ContourWarpEffect::setDrawOrigShape(bool b){
   m_bDrawOrigShape=b;
   update();
}

/*!
 * Default: false
 */
void ContourWarpEffect::contOnlyImage(bool b)
{
	m_bContOnlyImage = b;
}

/*!
 * Default: MODE_DRAW
 */
void ContourWarpEffect::editModeChanged(int mode){
   // set the appropiate icon to the edit button
   QPopupMenu *popup = PSEditor->getPopup();
   mEditButton->setIconSet(*(popup->iconSet(PSEditor->getEditMode())));

   switch (mode)
   {
		// in drawing mode, use a cross cursor
		case MODE_DRAW:
			glView->setCursor(QCursor::QCursor(CrossCursor)); break;
		default:
			glView->setCursor(QCursor::QCursor(ArrowCursor));
   }

   update();
}

/*!
 * The following things does this method:
 * - When entering warp mode (<code>enable</code> is true):
 *		- shows warp dialog.
 *		- sets the image frame as constant edges for the warp algorithm.
 *		- tries to do wedge caching if not in differential mode.
 *		- lets the editor edit the transformed shape.
 *		- starts the timer.
 * - When leaving warp mode (<code>enable</code> is false):
 *		- hides warp dialog.
 *		- move vertices to their initial positions.
 *		- lets the editor edit the original shape.
 *		- deallocates memory used by wedge caching in the warp algorithm.
 *		- stops the timer.
 */

void ContourWarpEffect::warpMode(bool enable){
	// set warping flag
	m_bIsWarping = enable;

	// we need an algo instance below
	WarpAlgorithm *algo = algoControl->getAlgorithm();

	// tell the editor to enter/leave restricted mode
    PSEditor->setRestricted(enable);

	if (m_bIsWarping)
	{
		// show the warp dialog
		mWarpDialog->show();
		// set the warp as to be recalculated
		m_bWarpChanged = true;

		// set edge frame as constant edges
		VECTOR3D vBottomRight = vertexGrid->m_vBottomLeft
							+vertexGrid->m_vTopRight-vertexGrid->m_vTopLeft;
	
		Edge edgeBuf[4] = {Edge (vertexGrid->m_vTopLeft, vertexGrid->m_vTopRight),
						   Edge (vertexGrid->m_vTopRight, vBottomRight),
						   Edge (vBottomRight, vertexGrid->m_vBottomLeft),
						   Edge (vertexGrid->m_vBottomLeft, vertexGrid->m_vTopLeft)};
		algo->setConstantEdges (edgeBuf, 4);
		
		// initialize the warping shape with the original shape
		*trafoShape = *origShape;
		// initialize the "original shape" for differential mode
		// with the original shape
		*trafoDiffShape = *trafoShape;
				
		// wedge cache is bogus in diff mode
		if (!m_bDiffMode)
			algo->doPrecalculation(origShape);
		// the user now edit the warping shape
		PSEditor->setShape (trafoShape);
		// start animation to see the warp
		effectTimer->start(50, false);
	} else
	{
		// hide the warp dialgo
		mWarpDialog->hide();
		// the user edit the original shape again
		PSEditor->setShape (origShape);
		// reset the vertices to their initial position
		vertexGrid->ResetVertices();
		
		// deallocate memory
		algo->undoPrecalculation();

		// stop animation
		effectTimer->stop();
	}

	update();
}

void ContourWarpEffect::stop() {}
void ContourWarpEffect::play() {}
void ContourWarpEffect::pause() {}

