/*
 * Copyleft 2001 Lukas Degener, Ingmar Kanitscheider. All rights granted ;-)
 *----------------------
 * file: src/AlgorithmControl.h
 *----------------------
 * This file is part of the ContourWarp Effect based on the GLEffect Framework.
 * In this file you will find the declaration of the AlgorithmControl class
 * Implementation is in AlgorithmControl.cpp
 */

/*! @file AlgorithmControl.h
 *
 *  @brief
 *  Declaration of AlgorithmControl class
 *  @author Lukas Degener, Ingmar Kanitscheider
 */

#ifndef ALGORITHMCONTROL_H
#define ALGORITHMCONTROL_H
#include <qwidget.h>
#include <qcombobox.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qcheckbox.h>
#include "WarpAlgorithm.h"
#include "FloatEditor.h"

/*! @class AlgorithmControl
 *
 * @ingroup contourWarpClasses
 * 
 * @brief Algorithm parameter control widget
 * 
 * This class manages a gui widget, which lets the user choose a specific Algorithm and
 * adjust its parameters.
 *
 * @note: Not all widgets the user sees in the algorithm control panel are declared here.
 * The "differential mode" and the "View original shape" checkbox declares
 * ContourWarpEffect.
 */

class AlgorithmControl: public QWidget
{
   Q_OBJECT

public:
	  /*! 
	   * @brief Qt constructor
	   * 
	   * Initializes the algorithms, all child widgets and
	   * installs signal/slots connections
	   */
	  AlgorithmControl(QWidget* parent = 0, const char* name = 0, WFlags fl = 0 );

	  /*! 
	   * @brief Destructor
	   *
	   * Deallocates the algorithm instances.
	   */
     ~AlgorithmControl();

      /*!
 	   * @brief Gets the current algorithm
	   *
       * Returns the current WarpAlgorithm choosen by the user.
	   * The user chooses the warp algorithm with the combo box at top of the widget.
	   *
	   * @note By now, only one algorithm is implemented, so the combo box is always
	   * deactivated.
       */

	  WarpAlgorithm *getAlgorithm();

      /*!
	   * @brief Enables/Disables the widgets hosted by AlgorithmControl
	   * 
	   * This is e.g. used when the differential mode is enabled. Because it may cause
	   * unpredictable results if the parameters are changed at same time, these
	   * sliders are deactivated. The other widgets being deactivated are the Edge as Frame
	   * check box and the algorithm selection combo box:
	   *
	   * @param value <code>true</code> means enabling, <code>false</code> disabling
       */
	  void enableControls (bool value);

signals:

	  /*!
	   * @brief emitted when the frameAsEdge is toggled
	   *
	   * @note This signal is used by ContourWarpEffect for deciding 
	   * whether a small frame is drawed around the image.
	   *
	   * @param value This parameter corresponds to <code>toggled(bool)</code>
	   */
	  void frameAsEdgesChanged(bool value); 

	  /*!
	   * @brief emitted when the warping transformation must be recalculated
	   * 
	   * This occurs if the parameters or the "edge as frame"-checkbox is changed.
	   */
	  void warpChanged();

	  /*!
	   * @brief emitted when an algorithm is choosed by the user
	   */
	  void algoChanged();

private slots:
      /*!
       * @brief called when the user has choosen an algorithm
       */
      void setAlgorithm(int index);

      /*!
       * @brief called when the user has altered a parameter of type float
       */
      void setParam(int index, float value);

      /*!
       * @brief called when the user has altered a parameter of type int
       */
      void setParam(int index, int value);

	  /*!
       * @brief called when the user has altered a parameter of type bool
       */
      void setEdgesAsFrame(bool value);

private:
      /*!
       * @brief An array of pointer to classes derived from WarpAlgorithm
	   *
	   * This array is filled in the constructor.
       */
      WarpAlgorithm **mAlgos;

      /*!
       * @brief An array of pointers to the float editor widgets for the current algo
	   *
	   * This array is filled in the constructor.
       */
      FloatEditor **mEditors;

      /*!
       * @brief An array of pointers to the QVBox class for each algo
	   * 
	   * @note In each QVBox the parameter widgets for the corresponding algo
	   * are placed. Only the QVBox of the current algo is visible.
       */
      QVBox **mBoxes;
      /*!
       * @brief The index of the currently choosen algo
       */
      int mCurrentAlgo;

      /*!
       * @brief The number of algorithms.
       */
      int mAlgoCount;

      /*!
       * @brief This combo presents the list of algos to the user.
       */
      QComboBox *mCombo;

	  /*!
	   * @brief This QCheckBox handles the "edge as frame" flag
	   */
	  QCheckBox *mEAFCheckBox;

      /*!
       * @brief The layout of the components
       */
      QVBoxLayout *mLayout;
};
#endif

