/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file SplineLib.h
 *
 *  @brief
 *	Deklarationen von B-Spline-Auswertungsfunktionen.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef SplineLibH
#define SplineLibH

#include <stdlib.h>
#include "CVector.h"

//! Makro fr einfachen Arrayzugriff auf die Basismatrix
#define baseMatrix(x,y) BaseMatrix[(y)*(Order+2)+x]

//! Berechnet die B-Spline-Basisfunktion an der Stelle \b t.
void CalculateSplineBase(float t, int Order, int Index, float* KnotVector, float* BaseMatrix);

/*!
 * @class SplineLib
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Berechnet einen Punkt auf der B-Splinekurve an der Stelle \b t.
 *
 *  Der Templateparameter \b N gibt die Anzahl der Komponenten der
 *  Kontrolpunkte an, der Parameter \b T deren Typ.  Die Routine Wertet
 *  zunchst mittels \b CalculateSplineBase() die zur Berechnung des Punktes
 *  notwendigen Spline-Basisfunktionen aus.  Es sind genau \b Order-1 Stck.
 *  Danach werden die Kontrollpunkte, welche einen Einfluss ausben mit den
 *  Basisfunktionen gewichtet und summiert. Das Ergebnis wird dann von der
 *  Funktion zurckgeliefert.
 *
 *  \pre
 *  Der Parameter \b t muss aus dem Intervall [ \b KnotVector[Index] , \b
 *  KnotVector[Index] [ stammen.
 *  \pre
 *  Das Array \b ControlPoints mu \b numpts+1 Elemente aufweisen.
 *  \pre
 *  Der Knotenvektor \b KnotVector mu \b Order+ \b NumPts+1 Elemente
 *  aufweisen.
 *  \pre
 *  \b BaseMatrix muss ein Zeiger auf ein gengend groen Speicherbereich sein,
 *  der mindestens (\b Order+2) * (\b Order+2) \b float -Elemente fassen muss.
 *  \pre
 *  Der Templateparameter \b N sollte mindestens 1 sein.
 *  \pre
 *  Der Templateparameter \b T sollte einen numerischen zu \b float oder \b int
 *  kompatiblen Typ darstellen, oder zumindest einen Datenyp fr den alle
 *  gngigen arithmetischen Operatoren berladen worden sind.
 *
 */

template<int N, class T> CVector<N,T> CalculateSplinePoint(float t, int Order, int NumPts, int Index, CVector<N,T>* ControlPoints, float* KnotVector, float* BaseMatrix)
{

	CVector<N,T> resPoint;

  // Calculate point
  if (t<KnotVector[Order+NumPts])
  {

		// Calculate weight functions
	  CalculateSplineBase(t,Order,Index,KnotVector,BaseMatrix);

	  int j = 1;
		for (int i=Index-Order+1 ; i<=Index ; i++)
	  {

			resPoint+=baseMatrix(j,Order)*ControlPoints[i];
	    j++;

  	};

  }
  else
  {

  	resPoint = ControlPoints[NumPts];

  }

  return resPoint;

};

#endif

