/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file SplineLib.cpp
 *
 *  @brief
 *	Implementation der B-Spline-Auswertungsfunktionen.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#include <stdlib.h>
#include <memory.h>
#include <math.h>

#include "SplineLib.h"

/*! Die Routine \b CalculateSplineBase() berechnet die fr die Auswertung
 *  des Punktes an der Stelle \b t ntigen B-Spline-Basisfunktionen der
 *  Ordnung \b Order. Dabei macht sich die Routine zunutze, dass nur wenige
 *  der normalerweise zur Berechnung eines Punkts verwendeten 
 *  B-Spline-Basisfunktionen ungleich 0 sind.
 *  Es werden genau diese anhand eines Dreieckschemas iterativ berechnet.
 *  Zur ermittlung der Basisfunktionen werden auch nur die Teilterme berechnet, die
 *  ungleich 0 sind.
 *  Dadurch wird die normalerweise bei der Berechnung von B-Splines 
 *  auftretende Division-Durch-Null Problematik umgangen.
 *  
 *  \pre 
 *  Der Parameter \b t muss aus dem Intervall [ \b KnotVector[Index] , \b KnotVector[Index+1] [ stammen.
 *  \pre
 *  \b BaseMatrix muss ein Zeiger auf ein gengend groen Speicherbereich sein, der mindestens
 *  (\b Order+2) * (\b Order+2) \b float -Elemente fassen muss.
 *  
 */

// Calculates the B-Spline weight functions for a given intervall [t_index, t_index+1[
void CalculateSplineBase(float t, int Order, int Index, float* KnotVector, float* BaseMatrix)
{

	// Reset base matrix
  memset(BaseMatrix,0,sizeof(float)*(Order+2)*(Order+2));

	// Set first base value
  baseMatrix(Order,1) = 1.0f;

	// Calculate the further rows of the triangle scheme
  for (int k = 2 ; k<=Order ; k++)
  {
    for (int i = 0 ; i<k ; i++)
    {

      float t1,t2;

			// Check if the first part of the term has to be calculated
      if (fabsf(baseMatrix(Order-i,k-1)) > 1E-8)
      {
        t1 = ((t-KnotVector[Index-i]) * baseMatrix(Order-i,k-1))/(KnotVector[Index-i+k-1]-KnotVector[Index-i]);
      }
      else
      {
        t1=0;
      };

			// Check if the second part of the term has to be calculated
      if (fabsf(baseMatrix(Order-i+1,k-1)) > 1E-8)
      {
        t2 = ((KnotVector[Index+k-i]-t) * baseMatrix(Order-i+1,k-1))/(KnotVector[Index-i+k]-KnotVector[Index-i+1]);
      }
      else
      {
        t2=0;
      };

			// Sum terms
      baseMatrix(Order-i,k)=t1+t2;

    };

  };

};


