/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file QKeyFrameBar.h
 *
 *  @brief
 *	Deklarationen der Klasse \b QKeyFrameBar.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef wdg_QKeyFrameBarH
#define wdg_QKeyFrameBarH

#include <stdlib.h>
#include <qwidget.h>
#include <qcolor.h>
#include <qpainter.h>

#include "CKeyFrame.h"
#include "CFrameList.h"

/**
 * @class QKeyFrameBar
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Steuerelement zur Darstellung und Manipulation von Keyframes
 *
 *  Die Klasse \b QKeyFrameBar realisiert ein QT-Steuerelement auf der
 *  Basis von QWidget, welches zur Anzeige und Manipulation von
 *  Keyframes dient, die in einem \b CFrameList -Objekt gespeichert sind.
 *
 *  Das Steuerlement stellt eine Zeitleiste dar, in der die Keyframes
 *  als senkrechte farbige Striche eingetragen sind.
 *  Das Steuerelement besitzt einen Framecursor, der mit der Maus verschoben
 *  werden kann.
 *  Ebenfalls mit der Maus knnen existierende Keyframes entlang der Zeitachse
 *  bewegt werden.  Das Steuerelement stellt Funktionalitt bezglich des
 *  Sichtbereichs. Er kann gezoomt und gescrollt werden.
 *
 */
class QKeyFrameBar : public QWidget
{
	Q_OBJECT;

public:

  /** \enum enuFrameBarMode
   * 
   *  \brief Bearbeitungsmoduskonstanten. 
   *  
   *   Die Enumeration \b enuFrameBarMode enthlt Konstanten, die fr den gerade
   *  aktiven Bearbeitungsmodus des QKeyframeBars stehen.
   *
   *   
   */
   typedef enum 
   {	
	   mdNothing, /**< Nichts zu tun! */
	   mdMoveFrameCursor, /**< Der FrameCursor wird gerade bewegt. */
	   mdMoveFrame /**< Ein Keyframe wird gerade verschoben. */
   
   } enuFrameBarMode;

private:

	//! Dient zur Sicherung der alten FrameCursor-Position.
	float PositionBackup; 	
	
	//! Dient zur Sicherung der KeyFrame-Position whrend des Verschiebens.
	float TempFramePosition;		
	
	//! Dient zur Sicherung der ursprnglichen Position des verschobenen Keyframes.
	int		OldXFramePos;					

protected:

	//! Bestimmt, ob ein Interpolationssignal gesendet werden soll.
	bool SendInterpolationEvent;
	
	//! Speichert den aktuellen Bearbeitungsmodus des QKeyFrameBars.
	enuFrameBarMode Mode;				

	//! Bestimmt, ob Keyframes mit der Maus verschoben werden drfen.
	bool FrameMoving;						

	//! Bestimmt, ob das Frameraster gezeichnet werden soll.
	bool ShowRaster;						
	
	//! Bestimmt, ob der Framecursor gezeichnet werden soll.
	bool ShowCursor;						
	
	//! Bestimmt, ob die Frameliste gezeichnet werden soll.
	bool ShowFrames;						
	
	//! Bestimmt, ob der Framecursor ausgewhlt wurde.
	bool CursorSelected;				

	//! Startzeitindex fr die Darstellung des Sichtbereichs.
	float DisplayStart;					
	
	//! Endzeitindex fr die Darstellung des Sichtbereichs.
	float DisplayEnd;	
	
	//! Auflsung des zu zeichnenden Rasters.
	float RasterResolution;			

	//! Position des Framecursors.
	float CursorPosition;				
	
	//! Farbe des Cursors.
	QColor CursorColor;					

	//! Farbe des Framerasters.
	QColor RasterColor;				

	//! Hintergrundfarbe.
	QColor BackgroundColor;		

	//! Farbe der Frames.
	QColor FrameColor;	
	
	//! Farbe eines angewhlten Frames.
	QColor SelectedFrameColor;	

	//! Textfarbe.
	QColor TextColor;					  

	//! Zeiger auf das angewhlte Frame.
	CKeyFrame* SelectedFrame;		

	//! Referenz zu der darzustellenden KeyFrame-Liste.
	CFrameList* FrameList;			

	//! Gibt zurck, ob sich der Mauszeiger auf dem Framecursor befindet.
	bool MouseInCursor(int x, int y);

	//! Liefert das Frame zurck, welches sich unter dem Cursor befindet.
	CKeyFrame* FrameAtCursor(int x, int y);

	//! Zeichnet den Hintergrund.
	virtual void paintBackground(QPainter& p);

	//! Zeichnet das Frameraster.
	virtual void paintRaster(QPainter& p);

	//! Zeichnet ein Keyframe.
	virtual void paintFrame(QPainter &p, float TimeIndex, QColor& col);

	//! Zeichnet die Keyframe-Liste.
	virtual void paintFrameList(QPainter &p);

	//! Lscht den Cursor.
	virtual void eraseCursor(QPainter& p);
	
	//! Zeichnet den Cursor.
	virtual void paintCursor(QPainter& p, bool Selected);

	//! Event-Handler.
	virtual void paintEvent(QPaintEvent* Event);
	
	//! Event-Handler.
	virtual void mouseMoveEvent(QMouseEvent* Event);

	//! Event-Handler.
	virtual void mousePressEvent(QMouseEvent* Event);

	//! Event-Handler.
	virtual void mouseReleaseEvent(QMouseEvent* Event);
	
	//! Event-Handler.
	virtual void keyPressEvent(QKeyEvent* Event); 

public:

	//! Konstruktor.
	QKeyFrameBar( QWidget * parent=0, const char * name=0);
	
	//! Setzt die Referenz auf die darzustellende Keyframe-Liste.
	void setFrameList(CFrameList* frameList);

	//! Gibt die Referenz auf die darzustellende Keyframeliste zurck.
	CFrameList& frameList()
	{
		
		return *FrameList;

	};

	//! Gibt zurck, ob Frames verschoben werden drfen.
	bool frameMoving()
	{

		return FrameMoving;

	}

	//! Gibt zurck, ob das Frameraster gezeichnet werden soll.
	bool showRaster()
	{

		return ShowRaster;

	};

	//! Gibt zurck, ob der Framecursor gezeichnet werden soll.
	bool showCursor()
	{

		return ShowCursor;

	};

	//! Gibt zurck, ob die Frameliste gezeichnet werden soll.
	bool showFrames()
	{

		return ShowFrames;

	};

	//! Gibt die Position des Framecursors zurck.
	float cursorPosition()
	{

		return CursorPosition;

	};

	//! Gibt die Auflsung des Framerasters zurck.
	float rasterResolution()
	{

		return RasterResolution;

	};

	//! Gibt eine Referenz auf das angewhlte Frame zurck.
	CKeyFrame& selectedKeyFrame()
	{

		return *SelectedFrame;

	};

	//! Gibt die Farbe des Framecursors zurck.
	const QColor& cursorColor()
	{

		return CursorColor;

	};

	//! Gibt die Farbe des Framerasters zurck.
	const QColor& rasterColor()
	{

		return RasterColor;

	};

	//! Gibt die Hintergrundfarbe zurck.
	const QColor& backgroundColor()
	{

		return BackgroundColor;

	};

	//! Gibt die Framefarbe zurck.
	const QColor& frameColor()
	{

		return FrameColor;

	};

	//! Gibt die Farbe von angewhlten Frames zurck.
	const QColor& selectedFrameColor()
	{

		return SelectedFrameColor;

	};

	//! Gibt die Textfarbe zurck.
	const QColor& textColor()
	{

		return TextColor;

	};

	//! Gibt den Anfang des Sichtbereichs zurck.
	float displayStart()
	{

		return DisplayStart;

	};

	//! Gibt das Ende des Sichtbereichs zurck.
	float displayEnd()
	{

		return DisplayEnd;

	};

	//! Standarddestruktor.
	~QKeyFrameBar();

public slots:

	//! Setzt den KeyFrame-Verschiebemodus.
	void setFrameMoving(bool newValue);

	//! Setzt den Raster-Zeichenmodus.
	void setShowRaster(bool newValue);

	//! Setzt den Cursor-Zeichenmodus.
	void setShowCursor(bool newValue);

	//! Setzt den Keyframe-Zeichenmodus.
	void setShowFrames(bool newValue);
	
	//! Setzt die Framecursorposition.
	void setCursorPosition(float newValue);

	//! Setzt das angewhlte Keyframe.
	void setSelectedKeyFrame(CKeyFrame* selectedFrame);
	
	//! Setzt die Auflsung des Keyframerasters.
	void setRasterResolution(float newValue);
	
	//! Setzt die Farbe des Framecursors.
	void setCursorColor(const QColor& newColor);

	//! Setzt die Farbe des Framerasters.
	void setRasterColor(const QColor& newColor);

	//! Setzt die Hintergrundfarbe.
	void setBackgroundColor(const QColor& newColor);

	//! Setzt die Farbe der Keyframes.
	void setFrameColor(const QColor& newColor);

	//! Setzt die Farbe der angewhlten Keyframes.
	void setSelectedFrameColor(const QColor& newColor);

	//! Setzt die Textfarbe.
	void setTextColor(const QColor& newColor);

	//! Setzt den Anfang des Sichtbereichs.
	void setDisplayStart(float newStart);

	//! Setzt das Ende des Sichtbereichs.
	void setDisplayEnd(float newEnd);

	//! Setzt Anfanf und Ende des Sichtbereichs.
	void setDisplayRange(float dispStart, float dispEnd);

	//! Vergrert den Sichtbereich um den Delta-Wert (Herauszoomen).
	void growDisplayRange(float delta = 0.1f);

	//! Verkleinert den Sichtbereich um den Delte-Wert (Hineinzoomen).
	void shrinkDisplayRange(float delta = 0.1f);

	//! Zentriert den Sichtbereich an der gegebenen Position.
	void centerView(float Position);
	
	//! Zentriert den Sichtbereich auf die Framecursorposition.
	void centerView();
	
	//! Scrollt den Sichtbereich um den Deltawert.
	void scroll(float dx);

	//! Scrollt den Sichtbereich nach links.
	void scrollLeft();

	//! Scrollt den Sichtbereich nach rechts.
	void scrollRight();

	//! Zoomt den Sichtbereich hinein.
	void zoomIn();

	//! Zoomt den Sichtbereich hinaus.
	void zoomOut();

signals:

	//! Wird gesendet, wenn sich die Cursorposition gendert hat.
	void cursorPositionChanged(float);
	//! Wird gesendet, wenn der Cursor mit rechter Maustaste verschoben wird.
	void interpolateAt(float);
	//! Wird gesendet, wenn die rechte Maustaste losgelassen wurde.
	void finishInterpolate();

};

#endif
