/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file QKeyFrameBar.cpp
 *
 *  @brief
 *	Implementation der Klasse \b QKeyFrameBar.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#include <stdlib.h>
#include <math.h>
#include <qpainter.h>
#include <qpointarray.h>

#include "QKeyFrameBar.h"

/*! Der Konstruktor setzt alle Steuerelementeigenschaften auf Standardwerte.
 *  Der Parameter \b name gibt den Namen des Widgets an.
 *  Der Parameter \b parent gibt das Eltern-Widget an, in welches das QKeyframeBar-Widget eingefgt wird.
 */
QKeyFrameBar::QKeyFrameBar( QWidget * parent, const char * name) : QWidget(parent,name,0)
{
	
	// Interne Instanz-Variablen setzen
	SendInterpolationEvent=false;
	FrameList = NULL;
	SelectedFrame = NULL;
	
	DisplayStart = -1.;
	DisplayEnd = 1.0f;
	CursorPosition = 0.0f;
	RasterResolution = 0.25f;

	FrameMoving = true;

	ShowCursor = true;
	ShowFrames = true;
	ShowRaster = true;
	CursorSelected = false;

	// Standardfarben setzen
	BackgroundColor = white;
	CursorColor = blue;
	RasterColor = gray;
	FrameColor = red;
	SelectedFrameColor = yellow;

	// Bearbeitungsmodus setzen
	Mode = mdNothing;

	// Mousetracking fr das Widget aktivieren
	setMouseTracking(true);

	// Tastaturfocus fr das Widget erlauben
	setFocusPolicy(QWidget::StrongFocus);

};

/*! Die Methode \b setFrameList() setzt die Referenz
 *  auf die darzustellende Keyframeliste. Die
 *  Darstellung wird dabei aktualisiert.
 */
void QKeyFrameBar::setFrameList(CFrameList* frameList)
{

	// Prfen, ob die darzustellende Liste sich von der
	// momentanten unterscheidet
	if (FrameList!=frameList)
	{

		// Framelistenzeiger zuweisen
		FrameList = frameList;

		// Widget neu zeichnen
		repaint(false);

	};

};

void QKeyFrameBar::paintBackground(QPainter& p)
{

	// Hintergrund des Widgets zeichnen
	p.setPen(NoPen);
	p.setBrush(BackgroundColor);
	p.setBackgroundColor(BackgroundColor);
	p.fillRect(rect(),p.brush());

};

void QKeyFrameBar::eraseCursor(QPainter& p)
{

	// Prfen, ob Cursor im Sichtbereich
	if ((CursorPosition>=DisplayStart) && (CursorPosition<=DisplayEnd))
	{
		// Koordinaten berechnen
		int x = (int)(rect().width()*((CursorPosition-DisplayStart)/(DisplayEnd-DisplayStart)));

		// Punktarray erzeugen
		QPointArray Triangle(3);
		Triangle.setPoint(0,x-6,0);
		Triangle.setPoint(1,x+6,0);
		Triangle.setPoint(2,x,12);

		p.setBrush(BackgroundColor);
		p.setPen(BackgroundColor);

		// Schatten lschen
		p.translate(1,1);
		p.drawPolygon(Triangle);

		// Cursordreieck lschen
		p.translate(-1,-1);
		p.drawPolygon(Triangle);
		
	};

};

void QKeyFrameBar::paintCursor(QPainter& p, bool Selected)
{

	// Prfen, ob Cursor im Sichtbereich
	if ((CursorPosition>=DisplayStart) && (CursorPosition<=DisplayEnd))
	{
		// Koordinaten berechnen
		int x = (int) (rect().width()*((CursorPosition-DisplayStart)/(DisplayEnd-DisplayStart)));

		// Punktarray erzeugen
		QPointArray Triangle(3);
		Triangle.setPoint(0,x-6,0);
		Triangle.setPoint(1,x+6,0);
		Triangle.setPoint(2,x,12);

		// Schatten zeichnen
		p.translate(1,1);
		p.setBrush(darkGray);
		p.setPen(darkGray);
		p.drawPolygon(Triangle);
		//p.drawLine(x,0,x,rect().height());

		// Cursordreieck zeichnen
		p.translate(-1,-1);

		QColor tmpColor = CursorColor;
		int h,s,v;
		
		// Farbe anpassen
		if (Selected)
		{
			tmpColor.hsv(&h,&s,&v);
			s/=1.7;
			tmpColor.setHsv(h,s,v);
		};

		p.setBrush(tmpColor);
		p.setPen(black);
		//p.drawLine(x,0,x,rect().height());
		p.drawPolygon(Triangle);
		
		// Dreieckshighlight zeichnen
		Triangle.setPoint(0,x+5,1);
		Triangle.setPoint(1,x-5,1);
		Triangle.setPoint(2,x,10);

		// Farbe anpassen
		tmpColor.hsv(&h,&s,&v);
		s/=2;
		tmpColor.setHsv(h,s,v);

		p.setPen(tmpColor);
		p.drawPolyline(Triangle);

	};

};

void QKeyFrameBar::paintRaster(QPainter &p)
{

	// Startindex berechnen
	int sn = (int) (DisplayStart/RasterResolution);
	// Position und Displaygre berechnen
	float pos = sn * RasterResolution;
	// Position ausgleichen
	if (pos<DisplayStart) pos++;
	float sze = DisplayEnd-DisplayStart;

	p.setPen(RasterColor);
	QFont tmpFont("small",7);
	p.setFont(tmpFont);

	// Rasterlinien zeichnen
	while ((pos<=DisplayEnd) && (pos>=DisplayStart))
	{

		int x = (int) (width()*(pos-DisplayStart)/sze);
		p.drawLine(x,0,x,rect().height());
		p.drawText(x+1,rect().height()-1,QString().setNum(pos,'g',3));
		sn++;
		pos = sn*RasterResolution;

	};

};

void QKeyFrameBar::paintFrame(QPainter &p, float TimeIndex, QColor& col)
{

	// Prfen, ob sich der Frame innerhalb des Sichtbereichs befindet
	if ((TimeIndex>=DisplayStart) && (TimeIndex<=DisplayEnd))
	{

		// Position berechnen
		int x = (int) ((rect().width()*(TimeIndex-DisplayStart))/(DisplayEnd-DisplayStart));

		// Schatten zeichnen
		p.setPen(darkGray);
		p.drawLine(x+1,0,x+1,rect().height());

		// Frame zeichnen
		p.setPen(col);
		p.drawLine(x,0,x,rect().height()-2);

	};

};

void QKeyFrameBar::paintFrameList(QPainter &p)
{

	// Prfen, ob eine Frameliste zugewiesen wurde
	if (FrameList!=NULL)
	{

		// Erstes sich im Sichtbereich befindendes Frame suchen
		if (FrameList->count()>0)
		{
			
			FrameList->first();

			while ((FrameList->current()!=NULL) && (FrameList->current()->timeIndex()<DisplayStart))
			{

				FrameList->next();

			};

			// Frames zeichnen
			while ((FrameList->current()!=NULL) && (FrameList->current()->timeIndex()<=DisplayEnd))
			{

				if (FrameList->current() == SelectedFrame)
				paintFrame(p,FrameList->current()->timeIndex(),SelectedFrameColor);
				else
				paintFrame(p,FrameList->current()->timeIndex(),FrameColor);
				FrameList->next();

			};

		};

	};

};

void QKeyFrameBar::paintEvent( QPaintEvent* Event )
{
	
	// Painter-Objekt erzeugen
	QPainter p(this);

	// Hintergrund des Widgets zeichnen
	paintBackground(p);

	// Raster zeichnen
	if (ShowRaster)
	{

		paintRaster(p);

	};
	
	// Frames zeichnen
	if (ShowFrames)
	{

		paintFrameList(p);

	};

	// Cursor zeichnen
	if (ShowCursor)
	{

		paintCursor(p,CursorSelected);

	};

};

bool QKeyFrameBar::MouseInCursor(int x, int y)
{

	// Cursorposition relativ zum Widget berechnen
	int cx = (int) ((CursorPosition-DisplayStart)*rect().width()/(DisplayEnd-DisplayStart));

	// Prfen, ob sich der Mauscursor innerhalb des FrameCursor befindet, und
	// Ergebnis zurckgeben
	return ((x>cx-6) && (x<cx+6) && (y>0) && (y<12));

};

// Liefert das Frame zurck, welches sich unter dem Mauszeiger befindet
CKeyFrame* QKeyFrameBar::FrameAtCursor(int x, int y)
{

	// Prfen, ob eine Frameliste zugewiesen wurde
	if (FrameList!=NULL)
	{

		// Alle Frames durchlaufen
		if (FrameList->count()>0)
		{

			CKeyFrame* tmpFrame =	FrameList->first();
			while (tmpFrame!=NULL)
			{

				// Position berechnen
				int xp = (int) (rect().width()*(tmpFrame->timeIndex()-DisplayStart)/(DisplayEnd-DisplayStart));
				
				// Prfen, ob Frame unter dem Mauszeiger liegt.
				if ((x>=xp-1) && (x<=xp+1) && (y>=0) && (y<=rect().height()))
				{
					return tmpFrame;
				};

				tmpFrame = FrameList->next();

			};

			// Liste komplett durchlaufen, somit kein Frame unter dem Cursor
			return NULL;

		}
		else
		return NULL;

	}
	else
	return NULL;

};

/*! Der Event-Handler \b mouseMoveEvent() behandelt
 *  Mausbewegungen. 
 *
 *  Befindet sich das \b QKeyFrameBar im Bearbeitungsmodus
 *  \b mdMoveFrameCursor, so wird der Framecursor an die
 *  entsprechende Stelle verschoben. Falls das Senden von 
 *  Cursorsignalen aktiviert ist, so wird das Signal
 *  \b cursorPositionChanged() gesendet. Wird der Cursor
 *  mit der rechten Maustaste gezogen, so wird das Signal
 *  \b interpolateAt() gesendet.
 *  
 *  Befindet sich das \b QKeyFrameBar im Bearbeitungsmodus
 *  \b mdMoveFrame, so wird das angewhlte Keyframe an
 *  die entsprechende Stelle verschoben.
 * 
 */
void QKeyFrameBar::mouseMoveEvent( QMouseEvent* Event)
{

	switch (Mode)
	{
		
		// Keine Bearbeitung
		case mdNothing:
			{

				// Mauscursor zurcksetzen
				unsetCursor();

				// Prfen, ob der Framecursor angezeigt werden soll
				if (ShowCursor)
				{

					if (MouseInCursor(Event->x(),Event->y()))
					{

						// Cursor setzen
						setCursor(pointingHandCursor);

					}

				}

				// Prfen, ob die Frameliste angezeigt werden soll
				if (ShowFrames)
				{

					if (FrameAtCursor(Event->x(),Event->y())!=NULL)
					{

						// Cursor setzen
						setCursor(sizeHorCursor);

					}

				}

			}
		break;

		// Framecursor bewegen
		case mdMoveFrameCursor:
			{

				QPainter tmpPainter(this);
				
				// Cursor lschen
				eraseCursor(tmpPainter);

				// Neue Cursorposition berechnen
				CursorPosition = Event->x()*(DisplayEnd-DisplayStart)/rect().width()+DisplayStart;

				// Signal aussenden
				emit cursorPositionChanged(CursorPosition);
				if (SendInterpolationEvent)  
				{

					emit interpolateAt(CursorPosition);

				};

				// Raster neuzeichnen
				if (ShowRaster) paintRaster(tmpPainter);

				// Frames neuzeichnen
				if (ShowFrames) paintFrameList(tmpPainter);

				// Cursor neuzeichnen
				paintCursor(tmpPainter,CursorSelected);
			
			}
		break;

		// Frame bewegen
		case mdMoveFrame:
			{

				QPainter tmpPainter(this);

				// Frameposition berechnen
				TempFramePosition = ((float) Event->x()/rect().width())*(DisplayEnd-DisplayStart)+DisplayStart;

				// Geisterframe zeichen
				tmpPainter.setRasterOp(XorROP);
				tmpPainter.setPen(white);
				if (OldXFramePos!=-1) tmpPainter.drawLine(OldXFramePos,0,OldXFramePos,rect().height());
				tmpPainter.drawLine(Event->x(),0,Event->x(),rect().height());
				tmpPainter.setRasterOp(CopyROP);
				OldXFramePos = Event->x();

				// Cursor neuzeichnen
				paintCursor(tmpPainter,CursorSelected);

			}
		break;

	};

	// Alten Eventhandler aufrufen
	QWidget::mouseMoveEvent(Event);

};

/*! Der Event-Handler \b mousePressEvent() behandelt
 *  das Drcken von Maustasten. 
 *
 *  Wurde die Linkemaustaste gedrckt, so selektiert
 *  der Eventhandler je nachdem wo sich der Mauszeiger befindet
 *  entweder den Framecursor, oder ein Keyframe.
 *  Danach wird \b QKeyFrameBar in den jeweiligen Verschiebemodus
 *  versetzt.
 *  
 *  Wurde der Framecursor mit der rechten Maustaste angelickt,
 *  so wird er selektiert, \b QKeyFrameBar in den Verschiebemodus versetzt
 *  und zustzlich das Flag \b SendInterpolationEvent gesetzt, so
 *  da beim Verschieben des Cursors Interpolationssignale ausgelst werden.
 * 
 */
void QKeyFrameBar::mousePressEvent( QMouseEvent* Event)
{

	switch (Mode)
	{

		case mdNothing:
			{

				// Eine der Maustasten gedrckt?
				if ((Event->button()==LeftButton) || (Event->button()==RightButton))
				{
					
					SendInterpolationEvent = true;
					
					// Prfen, ob Framecursor angezeigt werden soll
					if (ShowCursor)
					{

						// Prfen, ob der Mauszeiger sich auf dem Framecursor befindet
						if (MouseInCursor(Event->x(),Event->y()))
						{
							
							// Framecursor anwhlen und zeichnen
							CursorSelected = true;
							QPainter p( this );
							paintCursor(p,CursorSelected);
							
							// Modus ndern
							Mode = mdMoveFrameCursor;
							grabMouse();

						}
						else
						{
							// Framecursor abwhlen und zeichnen
							CursorSelected = false;
							QPainter p(this);
							paintCursor(p,CursorSelected);
							releaseMouse();
						}

					}
	
					// Verschieben von Frames nur mit dem linken Button
					if (Event->button()==LeftButton)
					{

						SendInterpolationEvent = false;

						// Prfen, ob Frames angezeigt werden sollen
						if (ShowFrames)
						{
							
							// Prfen, ob der Mauszeiger sich auf einem Frame befindet
							CKeyFrame* tmpFrame = FrameAtCursor(Event->x(),Event->y());

							if (tmpFrame!=NULL)
							{

								QPainter tmpPainter(this);
								
								// Altes Frame abgewhlt zeichnen und abwhlen
								if (SelectedFrame!=NULL)
								{
									paintFrame(tmpPainter,SelectedFrame->timeIndex(),FrameColor);
								}
								SelectedFrame = tmpFrame;

								// Angewhltes Frame zeichnen
								paintFrame(tmpPainter,SelectedFrame->timeIndex(),SelectedFrameColor);

								// Cursor neuzeichnen
								if (ShowCursor) paintCursor(tmpPainter,CursorSelected);

								// Angewhltes Frame zum aktuellen Frame der Liste machen
								FrameList->current(SelectedFrame->timeIndex());
								
								TempFramePosition = SelectedFrame->timeIndex();

								// Prfen, ob das verschieben von Keyframes erlaubt ist

								if (FrameMoving)
								{
									// Modus ndern
									Mode = mdMoveFrame;
									grabMouse();

									OldXFramePos = -1;
								}
								else
								{
									Mode = mdNothing;
								};
	
							}
							else
							{

								QPainter tmpPainter(this);
								
								// Altes Frame abgewhlt zeichnen und abwhlen
								if (SelectedFrame!=NULL)
								{
									paintFrame(tmpPainter,SelectedFrame->timeIndex(),FrameColor);
									SelectedFrame = NULL;
								}

								// Cursor neuzeichnen
								if (ShowCursor) paintCursor(tmpPainter,CursorSelected);
								
								// Maus freigeben
								releaseMouse();

							};

						};

					};

				};

			}
		break;

	};

	// Alten Eventhandler aufrufen
	QWidget::mousePressEvent(Event);

};


/*! Der Event-Handler \b mouseReleaseEvent() behandelt
 *  das Loslassen von Maustasten. 
 *
 *  Grundstzlich wird \b QKeyFrameBar in den Modus 
 *  \b mdNothing versetzt.
 *   
 *  Wurde die rechte Maustaste losgelassen, so wird das
 *  Signal \b finishInterpolate() gesendet.
 *  
 *  Wurde ein Keyframe verschoben, so wird das Keyframe in der 
 *  Keyframeliste verschoben und die Ansicht aktualisiert.
 * 
 */
void QKeyFrameBar::mouseReleaseEvent( QMouseEvent* Event )
{

	switch (Mode)
	{
		case mdMoveFrameCursor:
			{

				if (SendInterpolationEvent)
				{

					emit finishInterpolate();

				}

			}
		break;

		case mdMoveFrame:
			{

				// Prfen, ob ein Frame angewhlt wurde
				if (SelectedFrame!=NULL)
				{

					// Frame verschieben
					FrameList->current(SelectedFrame->timeIndex());
					FrameList->shift(TempFramePosition-SelectedFrame->timeIndex());

					// Widget neuzeichnen
					repaint(false);

				}

			}
		break;

	}

	// Bearbeitungsmodus zurcksetzen
	Mode = mdNothing;
	
	// Maus freigeben						
	releaseMouse();	

	// Alten Eventhandler aufrufen
	QWidget::mouseReleaseEvent(Event);

};

/*! Der Event-Handler \b keyPressEvent behandlet
 *  das Drcken von Tasten. 
 *
 *  Wurden die horizontalen Cursortasten bettigt, so
 *  scrollt der Eventhandler die Ansicht entweder nach
 *  links, oder nach rechts.
 *
 *  Wurden die Tasten + oder - gedrckt, so zoomt
 *  der Eventhandler die Ansicht entsprechend.
 * 
 */
void QKeyFrameBar::keyPressEvent(QKeyEvent* Event)
{

	switch (Event->key())
	{
			
		case Key_Right:
			{
						
				if (CursorSelected) 
				{

					// Cursor nach rechts bewegen
					setCursorPosition(CursorPosition+0.05);

				}
				else
				{

					// Nach links scrollen
					scroll(0.1f);

				}

			}
		break;

		case Key_Left:
			{
				
				if (CursorSelected) 
				{

					// Cursor nach links bewegen
					setCursorPosition(CursorPosition-0.05);

				}
				else
				{

					// Nach rechts scrollen
					scroll(-0.1f);

				}
			}
		break;

		case Key_Plus:
			{

				// Sichtbereich verkleinern (hineinzoomen)
				shrinkDisplayRange();

			}
		break;

		case Key_Minus:
			{

				// Sichtbereich vergrern (herauszoomen)
				growDisplayRange();

			}
		break;

		default:
		// Alten Eventhandler aufrufen
		QWidget::keyPressEvent(Event);

	};

};

void QKeyFrameBar::setShowRaster(bool newValue)
{

	if (newValue!=ShowRaster)
	{

		ShowRaster=newValue;
		repaint(false);

	};

};

void QKeyFrameBar::setShowCursor(bool newValue)
{

	if (newValue!=ShowCursor)
	{

		Mode = mdNothing;
		ShowCursor=newValue;
		repaint(false);

	};

};

void QKeyFrameBar::setShowFrames(bool newValue)
{

	if (newValue!=ShowFrames)
	{

		ShowFrames=newValue;
		repaint(false);

	};

};

/*! Die Methode \b setCursorPosition() setzt den Framecursor
 *  an die ber \b newValue spezifizierte Position.
 *  Das Widget wird dabei neu gezeichnet.
 */
void QKeyFrameBar::setCursorPosition(float newValue)
{
		// Prfen, ob Wert sich gendert hat
		if ( fabsf(newValue - CursorPosition) < 1E-8 ) return;

		if (ShowCursor)
		{
				QPainter tmpPainter(this);
				
				// Cursor lschen
				eraseCursor(tmpPainter);

				// Neue Cursorposition berechnen
				CursorPosition = newValue;

				// Raster neuzeichnen
				if (ShowRaster) paintRaster(tmpPainter);

				// Frames neuzeichnen
				if (ShowFrames) paintFrameList(tmpPainter);

				// Cursor neuzeichnen
				paintCursor(tmpPainter,CursorSelected);

		}
		else
		{

			CursorPosition=newValue;

		}

		// Cursor-Signal senden
		emit cursorPositionChanged(CursorPosition);
		
};

/*! Die Methode \b setRasterResolution() setzt die Auflsung
 *  des Framerasters. Der Parameter \b newValue gibt an,
 *  in welchen Abstnden ein senkrechter Skalenstrich mit
 *  eine Zeitangabe gezeichnet werden soll.
 *  Das Steuerelement wird neu gezeichnet.
 */
void QKeyFrameBar::setRasterResolution(float newValue)
{

	// Prfen, ob der neue Wert sich vom alten unterscheidet
	if ( fabsf(newValue-RasterResolution) > 1E-8 )
	{

		RasterResolution = newValue;
		repaint(false);

	};

};
	
void QKeyFrameBar::setCursorColor(const QColor& newColor)
{

	if (newColor!=CursorColor)
	{

		CursorColor = newColor;
		if (ShowCursor) 
		{
			QPainter p(this);
			paintCursor(p,CursorSelected);
		}

	};

};

void QKeyFrameBar::setRasterColor(const QColor& newColor)
{

	if (newColor!=RasterColor)
	{

		QPainter tmpPainter(this);
		RasterColor = newColor;
		if (ShowRaster) paintRaster(tmpPainter);
		if (ShowFrames) paintFrameList(tmpPainter);
		if (ShowCursor) paintCursor(tmpPainter,CursorSelected);

	};

};

void QKeyFrameBar::setBackgroundColor(const QColor& newColor)
{

	if (newColor!=BackgroundColor)
	{
		BackgroundColor = newColor;
		repaint(false);
	};

};

void QKeyFrameBar::setFrameColor(const QColor& newColor)
{

	if (newColor!=FrameColor)
	{
		
		QPainter tmpPainter(this);
		FrameColor = newColor;
		if (ShowFrames) paintFrameList(tmpPainter);
		if (ShowCursor) paintCursor(tmpPainter,CursorSelected);

	};

};

void QKeyFrameBar::setSelectedFrameColor(const QColor& newColor)
{

	if (newColor!=SelectedFrameColor)
	{
		
		SelectedFrameColor = newColor;
		update();

	};

};

void QKeyFrameBar::setTextColor(const QColor& newColor)
{

	if (newColor!=TextColor)
	{
		
		TextColor = newColor;
		update();

	};


};

/*! Die Methode \b centerView(float position) richtet den Sichtbereich so aus, da
 *  die durch \b position festgelegte Stelle genau in der Mitte liegt.
 *  Die Gre des Sichtbereichs bleibt erhalten.
 *  Das Steuerelement wird neu gezeichnet.
 */
void QKeyFrameBar::centerView(float position)
{
	// Ansicht zentrieren
	float sze = (DisplayEnd-DisplayStart)/2.0f;
	DisplayStart = position-sze;
	DisplayEnd = position+sze;
	
	// Widget neu zeichnen
	repaint(false);

};

/*! Die Methode \b centerView() ruft die berladene Methode 
 *  \b centerView(float position), so da der Sichtbereich genau
 *  auf den Framecursor zentriert wird.
 */
void QKeyFrameBar::centerView()
{
	centerView(CursorPosition);
};

/*! Die Methode \b scroll() verschiebt den Sichtbereich um
 *  den durch \b dx angegebenen Wert. Die Sichtbereichsgre bleibt
 *  erhalten.
 *  Das Steuerelement wird neu gezeichnet.
 */
void QKeyFrameBar::scroll(float dx)
{
	
	// Prfen, ob Deltawert grer als Schwellwert
	if (fabsf(dx) > 1E-8)
	{
	
		// Ansichtsbereich verschieben
		DisplayStart+=dx;
		DisplayEnd+=dx;

		// Widget neu zeichnen
		repaint(false);
	}

};

void QKeyFrameBar::scrollLeft()
{
	scroll(-0.1f);
};

void QKeyFrameBar::scrollRight()
{
	scroll(0.1f);
};

void QKeyFrameBar::zoomIn()
{
	shrinkDisplayRange();
};

void QKeyFrameBar::zoomOut()
{	
	growDisplayRange();
};


void QKeyFrameBar::setDisplayStart(float newStart)
{
	
	if (fabsf(newStart-DisplayStart)>1E-8)
	{
		// Bereichsanfang setzen
		DisplayStart = newStart;

		// Widget neu zeichnen
		repaint(false);
	
	};

};

void QKeyFrameBar::setDisplayEnd(float newEnd)
{
	
	if (fabsf(newEnd-DisplayEnd)>1E-8)
	{
		
		// Bereichsende setzen
		DisplayEnd = newEnd;

		// Widget neu zeichnen
		repaint(false);

	};

};

/*! Die Methode \b setDisplayRange() legt das Zeitfenster fest.
 *  nur die Keyframes, die sich innerhalb des Sichtbereichs befinden
 *  werden dargestellt.
 *  Der Parameter \b dispStart gibt den Zeitindex an, der zum linken Rand
 *  von \b QKeyFrameBar korrespondiert.
 *  Der Parameter \b dispEnd gibt den Zeitindex an, der zum rechten Rand
 *  von \b QKeyFrameBar korrespondiert.
 *  
 *  \note
 *  Der Parameter \b dispEnd muss grer dem Parameter \b dispStart sein!
 */
void QKeyFrameBar::setDisplayRange(float dispStart, float dispEnd)
{
	
	if ((fabsf(dispStart-DisplayStart)>1E-8) || (fabsf(dispEnd-DisplayEnd))>1E-8)
	{
		
		// Sichtbereich setzen
		DisplayStart = dispStart;
		DisplayEnd = dispEnd;

		// Widget neu zeichnen
		repaint(false);

	};

};

/*! Die Methode \b growDisplayRange() erweitert den Sichtbereich,
 *  indem sie den Wert \b delta vom Anfang des Sichtbereichs 
 *  subtrahiert und zum Ende des Sichtbereichs aufaddiert.
 */
void QKeyFrameBar::growDisplayRange(float delta)
{

	// Prfen, ob Deltawert grer als Schwellwert
	if (fabsf(delta) > 1E-8)
	{
		
		// Sichtbereich erweitern
		DisplayStart-=delta;
		DisplayEnd+=delta;

		// Widget neu zeichnen
		repaint(false);

	};

};

/*! Die Methode \b shrinkDisplayRange() verkleinert den Sichtbereich,
 *  indem sie den Wert \b delta vom Ende des Sichtbereichs 
 *  subtrahiert und zum Anfang des Sichtbereichs aufaddiert.
 */
void QKeyFrameBar::shrinkDisplayRange(float delta)
{
	
	// Prfen, ob Deltawert grer als Schwellwert
	if (fabsf(delta) > 1E-8)
	{
		// Sichtbereich verkleinern
		DisplayStart+=delta;
		DisplayEnd-=delta;

		// Widget neu zeichnen
		repaint(false);

	};

};

/*! Mit der Methode \b setSelectedKeyFrame() lt sich eine Referenz
 *  auf ein Keyframe festlegen, welches als markiert gelten soll.
 *
 *  \note Das spezifizierte Keyframe muss eine Element der Keyframeliste sein,
 *  die dem \b QKeyframeBar zugrunde liegt.
 */
void QKeyFrameBar::setSelectedKeyFrame(CKeyFrame* selectedFrame)
{

	SelectedFrame = selectedFrame;

	if (SelectedFrame!=NULL)
	{

		// Ausgewhltes Frame zum aktuellen Frame der Liste machen
		FrameList->current(selectedFrame->timeIndex());
		// Cursor abwhlen
		CursorSelected = false;

	};

	update();

};

void QKeyFrameBar::setFrameMoving(bool newValue)
{
	if (newValue!=FrameMoving)
	{
		FrameMoving = newValue;
	};

};

/*! Der Standarddestruktor setzt die Framelist-Referenz auf NULL zurck.
 */
QKeyFrameBar::~QKeyFrameBar()
{

	// Interne Instanz-Variablen zurcksetzen
	FrameList = NULL;

};





