/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CVector.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CVector.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef cls_CVector
#define cls_CVector

#if     _MSC_VER > 1000
#pragma once
#endif

#include <stdlib.h>
#include <math.h>

/*!
 * @class CVector
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Einfache Vektorklasse.
 *
 *  Das Klassentemplate \b CVector implementiert einen \b N - dimensionalen
 *  Vektor mit Komponenten vom Datentyp \b T.
 *  Die ntigen arithmetischen Operatoren wurden berschrieben, so dass es
 *  mglich ist, alle Rechenoperationen durchzufhren, die auf Vektoren im
 *  mathematischen Sinne definiert sind.
 *
 *  \pre 
 *  Der Templateparameter \b N muss mindestens eins betragen. 
 *  Der durch den Templateparameter \b T spezifizierte Komponentendatentyp muss
 *  in irgendeiner Weise in den Typ \b int oder \b float konvertierbar sein.
 *  Ferner mssen alle gngigen arithmetischen Operatoren fr den Datentyp \b T
 *  berladen worden sein.
 * 
 */

template <int N, class T> class CVector
{

public:

	/*! \brief Komponentenarray.
	 *
	 *  Das Array speichert \b N Komponenten vom Datentyp \b T.
	 */
	T c[N];	
	
	/*! \brief Standardkonstruktor.
	 *
	 *  Der Standardkonstruktor initialisiert alle Komponenten des Vektors mit dem
	 *  Wert 0. Deshalb ist es, wie oben bereits beschrieben, notwendig dass der Datentyp
	 *  \b T zu (int) oder (float) konvertiert werden kann.
	 *
	 */
	CVector()
	{
		
		// Alle Komponenten auf Null setzen
		for (int i=0 ; i<N ; i++) c[i]=0;

	};

	/*! \brief Konvertierkonstruktor.
	 *
	 *  Dieser zustzliche Konstruktor erlaubt es ein CVektor-Objekt auf der Basis eines
	 *  \b N elementigen Arrays mit Elementen vom Typ \b T zu erzeugen. Dabei wird das Array
	 *  \b a[] in das Komponentenarray \b c[] kopiert.
	 *
	 */
	CVector(T a[N])
	{
		
		// Array-Komponenten den Vektor-Komponenten zuweisen
		for (int i=0 ; i<N ; i++) c[i]=a[i];

	};

	/*! \brief Kopierkonstruktor.
	 *
	 *  Der Kopierkonstruktor erzeugt ein CVektor-Objekt auf der Basis eines anderen CVector-Objekts.
	 *  Dabei werden die Werte der Komponenten des Vektors \b s dem neu erzeugten Vektor zugewiesen.
	 *  Es ist nicht zwangsweise erforderlich, dass der Vektor \b s und der neu erzeugte Vektor die gleiche
	 *  Anzahl an Komponenten aufweisen. Hat der Zielvektor weniger Komponenten so werden nur soviele 
	 *  wie mglich kopiert. Hat der Zielvektor mehr Komponenten, so werden die briggebliebenen mit 0
	 *  initialisiert.
	 *
	 */
	template <int K, class F> CVector(const CVector<K,F>& s)
	{	

		// Prfen, ob Quellvektor mehr Komponenten besitzt als Zielvektor
		if (K>N)
		{
			
			// N Komponenten kopieren
			for (int i=0 ; i<N ; i++) c[i]=s.c[i];

		}
		else
		{
			int i;

			// K Komponenten kopieren
			for (i=0 ; i<K ; i++) c[i]=s.c[i];

			// brige Komponenten mit Null initialisieren
			for (i=K ; i<N ; i++) c[i]=0;

		}

	};
	
	//! Liefert die Anzahl der Komponenten des Vektors zurck.
	int componentCount()
	{

		return N;

	};

	//! Liefert die Gre einer Vektorkomponente in Bytes zurck.
	int componentSize()
	{
	
		return sizeof(T);

	};

	/*! \brief Gibt die euklidische Lnge des Vektors zurck.
	 *
	 *  Die Routine berechnet die euklidische Lnge des Vektors, indem sie die Summe der
	 *  Komponentenquadrate bildet und daraus die Wurzel zieht.
	 *
	 */
	T length()
	{
		
		// Summe der Komponentenquadrate bilden
		T l = 0;
		for (int i =0 ; i<N ; i++) l+=c[i]*c[i];
		
		// Lnge zurckgeben
		return sqrt(l);

	};

	/*! \brief Normalisiert den Vektor.
	 *
	 *  Die Routine normalisiert den Vektor, so dass seine euklidsche Lnge anschlieend 1 ergibt.
	 *  Dabei wird zunchst die euklidische Lnge des Vektors berechnet und alle Komponenten
	 *  durch diese dividiert.
	 * 
	 *  \remarks
	 *  Vektoren der Lnge Null werden abgefangen.
	 *
	 */
	void normalize()
	{
		
		// Lnge bestimmen
		T l = this->length();

		// Prfen, ob NULL-Vektor vorliegt
		if (fabsf(l) > 1E-8) for (int i=0 ; i<N; i++) c[i]/=l;

	};

	/*! \brief "+="-Operator (Vektoraddition).
	 *
	 *  Der Vektor \b a wird auf den links vom "+="-Operator stehenden Vektor aufaddiert.
	 *  Hierbei wird eine komponentenweise Addition durchgefhrt.
	 * 
	 */
	CVector<N,T>& operator+=(const CVector<N,T>& a)
	{
	
		// Komponentenweise Addition durchfhren
		for (int i=0 ; i<N ; i++) c[i]+=a.c[i];

		// Ergebnis zurckgeben
		return *this;

	};

	/*! \brief "-="-Operator (Vektorsubtraktion).
	 *
	 *  Der Vektor \b a wird von dem links vom "-="-Operator stehenden Vektor abgezogen.
	 *  Hierbei wird eine komponentenweise Subtraktion durchgefhrt.
	 * 
	 */
	CVector<N,T>& operator-=(const CVector<N,T>& a)
	{

		// Komponentenweise Subtraktion durchfhren
		for (int i=0 ; i<N ; i++) c[i]-=a.c[i];

		// Ergebnis zurckgeben
		return *this;

	};

	/*! \brief "*="-Operator (Skalarmultiplikation).
	 *
	 *  Der links vom "*="-Operator stehende Vektor wird mit dem skalaren Wert \b f vom Typ \b T multipliziert.
	 *  Hierbei wird eine komponentenweise Multiplikation durchgefhrt.
	 * 
	 */
	CVector<N,T>& operator*=(T f)
	{
		
		// Komponentenweise Multiplikation durchfhren
		for (int i=0 ; i<N ; i++) c[i]*=f;

		// Ergebnis zurckgeben
		return *this;

	};

	/*! \brief "/="-Operator (Skalardivision).
	 *
	 *  Der links vom "/="-Operator stehende Vektor wird durch den skalaren Wert \b f vom Typ \b T dividiert.
	 *  Hierbei wird eine komponentenweise Division durchgefhrt.
	 * 
	 */
	CVector<N,T>& operator/=(T f)
	{
	
		// Komponentenweise Division durchfhren
		for (int i=0 ; i<N ; i++) c[i]/=f;

		// Ergebnis zurckgeben
		return *this;

	};

	/*! \brief "%="-Operator (Vektorkreuzprodukt).
	 *
	 *  Es wird das Vektorprodukt zwischen dem links vom "%="-Operator stehenden Vektor und dem
	 *  Vektor \b b berechnet. Das Ergebnis wird wieder dem links vom "%="-Operator stehenden Vektor abgelegt.
	 * 
	 *  \remarks
	 *  Dieser Operator ist nur auf 3-dimensionale Vektoren anwendbar.
	 *
	 */

	CVector<3,T>& operator%=(const CVector<3,T>& b)
	{

		// Kreuzprodukt berechnen
		T cx = c[1]*b.c[2]-c[2]*b.c[1];
		T cy = c[2]*b.c[0]-c[0]*b.c[2];
		T cz = c[0]*b.c[1]-c[1]*b.c[0];

		// Errechnete Komponenten speichern
		c[0] =cx;
		c[1] =cy;
		c[2] =cz;

		// Ergebnis zurckgeben
		return *this;

	};

};

// *** DEKLARATIONSTEIL ******************************************************

/*! \brief "+"-Operator (Vektoraddition).
 *
 *  Der Operator liefert den Summenvektor der beiden Vektoren \b a und \b b zurck.
 *  Es wird eine komponentenweise Addition durchgefhrt.
 *
 *  \remarks
 *  Die Vektoraddition wird mittels des "+="-Operators und eines temporren Vektors
 *  durchgefhrt.
 *
 */
template <int N, class T> CVector<N,T> operator+ (const CVector<N,T> &a, 
																									const CVector<N,T> &b)
{
	
	// Addition ber den += Operator und einen temporren 
	// Vektor durchfhren
	return CVector<N,T>(a)+=b;

};

/*! \brief "-"-Operator (Vektorsubtraktion).
 *
 *  Der Operator liefert den Differenzvektor zwischen den beiden Vektoren \b a und \b b zurck.
 *  Es wird eine komponentenweise Subtraktion durchgefhrt.
 * 
 *  \remarks
 *  Die Vektorsubtraktion wird mittels des "-="-Operators und eines temporren Vektors
 *  durchgefhrt.
 *
 */
template <int N, class T> CVector<N,T>  operator- (const CVector<N,T> &a, 
																									 const CVector<N,T> &b)
{

	// Subtraktion ber den -= Operator und einen 
	// temporren Vektor durchfhren
	return CVector<N,T>(a)-=b;

};

/*! \brief "*"-Operator (Skalarmultiplikation).
 *
 *  Der Operator liefert den mit \b f skalierten Vektor \b a zurck.
 *  Es wird eine komponentenweise Multiplikation durchgefhrt.
 *
 *  \remarks
 *  Die Skalarmultiplikation wird mittels des "*="-Operators und eines temporren Vektors
 *  durchgefhrt.
 * 
 */
template <int N, class T> CVector<N,T> operator* (T f, const CVector<N,T> &a)
{
	
	// Multiplikation ber den *= und einen 
	// temporren Vektor durchfhren
	return CVector<N,T>(a)*=f;

};

/*! \brief "*"-Operator (Skalarprodukt).
 *
 *  Der Operator berechnet das Standardskalarprodukt der beiden Vektoren \b a und \b b.
 *  Dabei werden die Komponenten der Vektoren paarweise miteinander multipliziert 
 *  und danach die Produkte aufsummiert.
 * 
 */
template <int N, class T> T operator* (const CVector<N,T> &a,const CVector<N,T> &b)
{
	
	// Wert des Standard-Skalarprodukts auf 0 setzen
	T val = 0;
	
	// Komponenten miteinander multiplizieren und
	// in <val> aufsummieren
	for (int i=0 ; i<N ; i++) val+=a.c[i]*b.c[i];

	// Ergebnis zurckgeben
	return val;

};

/*! \brief "/"-Operator (Skalardivision).
 *
 *  Der Operator liefert den mit \b f skalierten Vektor \b a zurck.
 * 
 *  \remarks
 *  Die Skalardivision wird mittels des "/="-Operators und eines temporren Vektors
 *  durchgefhrt.
 *
 */
template <int N, class T> CVector<N,T> operator/ (const CVector<N,T> &a, T f)
{
  
	// Komponentenweise Division ber /= Operator und einen
	// temporren Vektor durchfhren
	return CVector<N,T>(a)/=f;

};

/*! \brief "%="-Operator (Vektorkreuzprodukt).
 *
 *  Der Operator liefert das Vektorkreuzprodukt der beiden Vektoren \b a und \b b zurck.
 * 
 *  \remarks
 *  Die Berechnung des Kreuzprodukts wird mittels des "%="-Operators und eines temporren Vektors
 *  durchgefhrt.
 *
 */
template <class T> CVector<3, T> operator%(const CVector<3, T> &a, const CVector<3, T> &b)
{
	
	// Vektorkreuzprodukt mit Hilfe von %= und einem
	// temporren Vektor berechnen
	return CVector<3,T>(a)%=b;

};

/*! \brief "=="-Operator (Vergleich).
 *
 *  Der Operator prft ob die Vektoren \b a und \b b gleich sind. Er liefert nur dann TRUE zurck, wenn
 *  alle Komponenten der Vektoren paarweise bereinstimmen.
 *
 *  \note Jede Komponente wird mittels == verglichen, dies kann
 *        bei Fliekommadatentypen zu Problemen fhren!
 *
 */
template <int N, class T> bool operator==(const CVector<N,T> &a, const CVector<N,T> &b)
{
	bool res = true;
	for (int i = 0 ; i<N ; i++)
	{
		res = res && (a.c[i]==b.c[i]);
	};
	return res;
};

/*! \brief "!="-Operator (Vergleich).
 *
 *  Der Operator prft ob die Vektoren \b a und \b b ungleich sind.
 *
 *  \remarks
 *  Es wird zuerst ein Vergleich mittels "==" durchgefhrt. Das Ergebnis wird dann negiert
 *  und zurckgegeben.
 *
 */
template <int N, class T> bool operator!=(const CVector<N,T> &a, const CVector<N,T> &b)
{
	return !(a==b);
};

#endif


