/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CReferenceList.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CReferenceList.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef cls_CReferenceListH
#define cls_CReferenceListH

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

#include <stdlib.h>
#include "CDynamicArray.h"

/*! 
 *
 * @class CReferenceList
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Dynamische Liste von Referenzen. 
 *
 *  Das Klassentemplate \b CReferenceList realisiert auf der Grundlage der
 *  Klasse \b CDynamicArray eine dynamische Liste von Referenzen.
 *  Es knnen Referenzen eingefgt und entfernt werden.
 *  Beim Einfgen von Referenzen wird geprft, ob sich 
 *  diese bereits in der Liste befinden. Ist dies
 *  der Fall so werden diese nicht nochmals in die Liste
 *  eingefgt.
 *
 *  Der Templateparameter \b T legt den Datentyp fest, auf den die 
 *  aufzunehmenden Referenzen verweisen.
 *
 *  \remarks
 *  Die Klasse wird vom BSplineWrapper-Effekt dazu verwendet 
 *  Referenzen auf die vom Benutzer markierten Kontrollpunkte 
 *  zu speichern.
 *
 *  \note 
 *  Aus dem Datentyp \b T wird automatisch ein Zeigerdatentyp 
 *  erstellt. Setzt man z.B. fr \b T den Typ \b int ein, so 
 *  speichert die Liste tatschlich Elemente vom Typ \b int*, 
 *  d.h. Zeiger auf \b int -Elemente.
 *
 *  \note
 *  Anders als z.B. bei der \b CReferenceGrid -Klasse wird hier
 *  der Speicher, welcher durch die Objekte auf die 
 *  die eingefgten Referenzen verweisen, beim Entfernen oder
 *  Lschen der Liste absichtlich \b nicht freigegeben.
 *
 */

template <class T> class CReferenceList
{

protected:

	//! Speichert Referenzen auf Elemente vom Datentyp \b T.
	CDynamicArray<T*> ListData;
	
public:

	/*! \brief Fgt eine Referenz auf ein Element vom Datentyp \b T in die Liste ein.
	 *
	 *  \remarks 
	 *  Die neue Referenz wird an das Ende der Liste eingefgt.
	 *
	 */
	void insert(T* element)
	{

		// Elementreferenz in der Liste suchen
		for (int i = 0 ; i < ListData.getSize() ; i++)
		{	

			// Elementreferenz wurde gefunden, Einfgevorgang abbrechen.
			if (ListData[i] == element) return;

		};

		// Elementreferenz in die Liste einfgen, da sie nicht gefunden wurde.
		ListData.resize(ListData.getSize()+1);
		ListData[ListData.getSize()-1]=element;

	};

	/*! \brief Entfernt die Referenz auf das durch \b element spezifizierte Element aus der Liste.
	 *
	 *  \note
	 *  Die Methode \b remove() ndert die Reihenfolge der Elemente innerhalb der Liste.
	 *  Um nicht alle Elemente umkopieren zu mssen, wird das letzte Element der Liste
	 *  an die Stelle des Elements, welches entfernt werden soll gesetzt. Danach wird die
	 *  Gre der Liste um eins vermindert.
	 *
	 *  \note
	 *  Der Speicher den das Objekt belegt, auf welches die entfernte Referenz 
	 *  verwiesen hat, wird \b nicht freigegeben.
	 *
	 */
	void remove(T* element)
	{

		// Elementreferenz in der Liste suchen
		for (int i = 0; i<ListData.getSize() ; i++)
		{
			
			// Prfen, ob Referenz bereinstimmt
			if (ListData[i]==element)
			{

				// Letztes Element an die Stelle <i> in der Liste verschieben
				ListData[i] = ListData[ListData.getSize()-1];
				
				// Liste verkleinern
				ListData.resize(ListData.getSize()-1);
				
				// Routine verlassen
				return;

			}

		}

	};

	/*! \brief Entfernt die an der Stelle \b index in der Liste gespeicherte Referenz.
	 *
	 *  \remarks
	 *  Wird ein Index auerhalb des gltigen Bereichs spezifiziert, so 
	 *  kehrt die Routine umgehend zurck, ohne ein Element zu entfernen.
	 *
	 *  \note
	 *  Die Methode \b remove() ndert die Reihenfolge der Elemente innerhalb der Liste.
	 *  Um nicht alle Elemente umkopieren zu mssen, wird das letzte Element der Liste
	 *  an die Stelle des Elements, welches entfernt werden soll gesetzt. Danach wird die
	 *  Gre der Liste um eins vermindert.
	 *
	 *  \note
	 *  Der Speicher den das Objekt belegt, auf welches die entfernte Referenz 
	 *  verwiesen hat, wird \b nicht freigegeben.
	 */
	void remove(int index)
	{
	
		// Prfen, ob Index im gltigen Bereich
		if ((index>=0) && (index<ListData.getSize()))
		{
			
			// Letztes Element an die Stelle <index> in der Liste kopieren
			ListData[index] = ListData[ListData.getSize()-1];
			
			// Liste verkleinern
			ListData.resize(ListData.getSize()-1);

		};

	};

	/*! \brief Lscht den Inhalt der gesamten Liste.
	 *
	 *  \note
	 *  Der Speicher den die Objekte belegen, auf welche die gelschten Referenzen 
	 *  verwiesen haben, wird \b nicht freigegeben.
	 */
	void clear()
	{
		// Listengre auf 0 setzen
		ListData.resize(0);
	};

	//! Gibt die Anzahl der in der Liste gespeicherten Referenzen.
	int count()
	{
		return ListData.getSize();
	};

	/*! \brief "[]"-Operator (Indizierter Zugriff).
	 *
	 *  Gibt die Referenz auf ein Element vom Datentyp \b T an der Position \b Index innerhalb der
	 *  Liste zurck.
	 */

	T* operator[](int Index)
	{
		return ListData[Index];
	};

};

#endif

