/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CReferenceGrid.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CReferenceGrid.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef cls_CReferenceGridH
#define cls_CReferenceGridH

#if     _MSC_VER > 1000
#pragma once
#endif

#include <stdlib.h>
#include <stdexcept>
using std::invalid_argument;

#include "CGrid.h"

/*!
 *
 * @class CReferenceGrid
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Von \b CGrid abgeleitete Klasse, die auf die Speicherung 
 *  von dynamisch alloziierten Daten spezialisiert ist.
 *
 *  Das von \b CGrid abgeleitete Klassentemplate \b CReferenceGrid
 *  bernimmt das gesamte Verhalten von \b CGrid, ist jedoch speziell
 *  fr die Speicherung von dynamisch alloziierten Elementen ausgelegt.
 *
 *  Bei einer Grennderung, oder beim Aufruf der Methode \b clear() 
 *  wird der fr ein Element zuvor dynamisch alloziierte Speicher
 *  mittels \b delete() freigegeben.
 *
 *  Der Templateparameter \b T legt den Datentyp fr die
 *  Gridelemente fest.
 *
 *  \note 
 *  Aus dem Datentyp \b T wird automatisch ein Zeigerdatentyp 
 *  erstellt. Setzt man z.B. fr \b T den Typ \b int ein, so 
 *  speichert das Grid tatschlich Elemente vom Typ \b int*, 
 *  d.h. Zeiger auf \b int -Elemente.
 */

template <class T> class CReferenceGrid : public CGrid<T*>
{
	
protected:

	//! Gibt den fr ein Element dynamisch alloziierten Speicher wieder frei.
  virtual void disposeElement(int intX,int intY);

public:

	//! Standardkonstruktor.
	CReferenceGrid();
	
	//! Zusatzkonstruktor.
	CReferenceGrid(int intWidth, int intHeight);
	
	//! Kopierkonstruktor.
	CReferenceGrid(CReferenceGrid<T>& objGrid);

	//! Standarddestruktor.
	virtual ~CReferenceGrid();

};

/*!
 *  Der Standardkonstruktor erzeugt ein NULL-Grid, d.h. ein
 *  Grid mit Null Zeilen und Null Spalten.
 *  
 *  \remarks
 *  Das Flag \b DisposeData wird auf den Wert \b true gesetzt.
 *  Dies hat zur Folge, da das zuvor in \b CGrid festgelegte
 *  Verhalten, fr jedes freigewordene Element bei
 *  Lsch- oder Grennderungsoperationen die Methode 
 *  \b disposeElement() aufzurufen, aktiviert wird.
 *
 */
template <class T> CReferenceGrid<T>::CReferenceGrid() : CGrid<T*>()
{

	// Instanzvariablen initialisieren
	DisposeData = true;		     // Freigeben von Elementen aktivieren 

};

/*!
 *  Der zustzliche Konstruktor erzeugt ein Grid mit den 
 *  Ausmaen \b intWidth x \b intHeight.
 *  
 *  \remarks
 *  Das Flag \b DisposeData wird auf den Wert \b true gesetzt.
 *  Dies hat zur Folge, da das zuvor in \b CGrid festgelegte
 *  Verhalten, fr jedes freigewordene Element bei
 *  Lsch- oder Grennderungsoperationen die Methode 
 *  \b disposeElement() aufzurufen, aktiviert wird.
 *
 */
template <class T> CReferenceGrid<T>::CReferenceGrid(int intWidth, int intHeight) : CGrid<T*>(intWidth,intHeight)
{

	// Instanzvariablen initialisieren
	DisposeData = true;		     // Freigeben von Elementen aktivieren 

};


/*!
 *  Der Kopierkonstruktor erzeugt ein Grid auf der
 *  Basis eines anderen \b CReferenceGrid -Objekts. Dabei wird
 *  die Gre des Zielgrids an die Gre des Quellgrids angepat.
 *  
 *  \remarks
 *  Das Flag \b DisposeData wird auf den Wert \b true gesetzt.
 *  Dies hat zur Folge, da das zuvor in \b CGrid festgelegte
 *  Verhalten, fr jedes freigewordene Element bei
 *  Lsch- oder Grennderungsoperationen die Methode 
 *  \b disposeElement() aufzurufen, aktiviert wird.
 *
 *  \note
 *  Es wird kein Deepcopy durchgefhrt, was zur Folge hat,
 *  dass die in den Elementen gespeicherten Referenzen in beiden
 *  Grids anschlieend auf dieselben Daten zeigen.
 */
template <class T> CReferenceGrid<T>::CReferenceGrid(CReferenceGrid<T>& objGrid) : CGrid<T*>(objGrid)
{

	// Instanzvariablen initialisieren
	DisposeData = true;		     // Freigeben von Elementen aktivieren 

};

/*!
 *  Die Methode \b disposeElement() wird von der \b CReferenceGrid -Klasse
 *  reimplementiert. Die Methode prft zunchst ob der Zeiger
 *  an der Position \b intX, \b intY ungleich NULL ist. Ist dies
 *  der Fall, so wird mittels \b delete() der fr das Element
 *  dynamisch alloziierte Speicher wieder freigegeben. Danach
 *  wird der Zeiger auf NULL zurckgesetzt.
 *
 *  \remarks
 *  Die Methoden \b resize() und \b clear() sind bereits
 *  in der Klasse \b CGrid so implementiert, dass diese
 *  fr jedes freigewordene Element die Methode \b disposeElement()
 *  aufrufen, wenn das Flag \b DisposeData auf \b true gesetzt ist.
 *  Jedoch wird das Flag in der \b CGrid -Klasse vom Konstruktor auf \b false
 *  gesetzt und die Methode \b disposeElement() ist so implementiert, dass
 *  sie keine Wirkung zeigt.
 */
template <class T> void CReferenceGrid<T>::disposeElement(int intX,int intY)
{
	
	// Objekt lschen, falls nicht NULL
	if (get(intX,intY)!=NULL) 
	{
		 delete get(intX,intY);
		 // Zeiger zurcksetzen
		 set(intX,intY,NULL);
	};

};

/*!
 *  Der Standarddestruktor lscht das Gitter mittels
 *  \b clear() wodurch der fr jedes Element dynamisch
 *  alloziierte Speicher freigegeben wird.
 *  Danach wird der Speicherbereich des Grids fr die 
 *  Elementzeiger selbst freigegeben.
 *  
 */
template <class T> CReferenceGrid<T>::~CReferenceGrid()
{
	
	// Speicher der Datenelemente durch Aufruf von clear() freigeben
	clear();
	// Freigabe von Speicher fr Datenelemente deaktivieren
	DisposeData = false; 

};

#endif

