/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CMeshBuilder.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CMeshBuilder.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef cls_CMeshBuilderH
#define cls_CMeshBuilderH

#if     _MSC_VER > 1000
#pragma once
#endif

#include <stdlib.h>

#include "CGrid.h"
#include "CVector.h"

/*!
 *
 * @class CMeshBuilder
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Dient zur Erzeugung von regulren Vertexgittern.
 *
 *  Mit dem Klassentemplate \b CMeshBuilder knnen regulre Vertexgitter
 *  erzeugt werden. Der Datentyp der Gitterelemente ist hierbei die \b CVector
 *  -Klasse. Die Anzahl der Vektorkomponenten wird ber den Templateparameter
 *  \b N festgelegt und deren Datentyp ber den Templateparameter \b T.
 *
 *  ber die Klassenmember wird mathematisch eine Ebene im Raum beschrieben.
 *  Auf dieser Ebene werden dann Gitterpunkte in quidistanten Abstnden
 *  generiert und in einem \b CGrid -Objekt abgelegt.
 *  
 *  \pre
 *  Der Templateparameter \b N muss mindestens zwei betragen.  Der ber den
 *  Templateparameter \b T fesgelegte Komponentendatentyp muss ein zu \b float
 *  oder \b int kompatibler Datentyp sein.  Ferner mssen fr den Datentypen \b
 *  T berladungen fr alle arithmetischen Operatoren existieren.
 *
 */
template<int N, class T> class CMeshBuilder
{

public:

	//! Speichert den u-Spannvektor der Gitterebene.
	CVector<N,T> u;	     

	//! Speichert den v-Spannvektor der Gitterebene.
	CVector<N,T> v;      
	
	//! Speichert den Ursprungsvektor der Gitterebene.
	CVector<N,T> origin; 

	/*! \brief Speichert die Breite des Gitters in Globalkoordinaten.
   *
	 *  Der Member \b width gibt nicht die Anzahl der Spalten des zu erzeugenden \b CGrid -Objekts, sondern
	 *  die horizontale Ausdehnung des Gitters im globalen Koordinatensystem.
	 *
	 */
	T width;             

	/*! \brief Speichert die Hhe des Gitters in Globalkoordinaten.
   *
	 *  Der Member \b height gibt nicht die Anzahl der Zeilen des zu erzeugenden \b CGrid -Objekts, sondern
	 *  die vertikale Ausdehnung des Gitters im globalen Koordinatensystem.
   *
	 */
	T height;            

	/*! \brief Speichert die horizontale Auflsung des Gitters.
   *
	 *  Die horizontale Auflsung bezeichnet hierbei die Anzahl der
	 *  Spalten des zu erzeugenden \b CGrid -Objekts.
	 *
	 */
	int horResolution;  
	
	/*! \brief Speichert die vertikale Auflsung des Gitters.
   *
	 *  Die vertikale Auflsung bezeichnet hierbei die Anzahl der
	 *  Zeilen des zu erzeugenden \b CGrid -Objekts.
	 *
	 */
	int verResolution;   


	//! Standardkonstruktor.
	CMeshBuilder();
	
	//! Erzeugt ein \b CGrid -Objekt.
	CGrid< CVector<N,T> >* build();

};

/*!
 *  Der Standardkonstruktor initialisiert die Klassenmember wie folgt:
 *
 *  -# Der Nullvektor stellt den Ursprung \b origin der Gitterebene dar.
 *  -# Dem Spannvektor \b u wird der erste Standardbasisvektor zugewiesen.
 *  -# Dem Spannvektor \b v wird der zweite Standardbasisvektor zugewiesen.
 *  -# Die Gitterausdehnung innerhalb des globalen Koordinatensystems ist 1.
 *  -# Die Auflsung des Gitters ist 2 x 2.
 */
template<int N, class T> CMeshBuilder<N,T>::CMeshBuilder()
{
	
  // u und v initialisieren
	u.c[0]=1.0;
	v.c[1]=1.0;
	
	// Auflsung mind. 2
	horResolution = 2;
	verResolution = 2;

	// Einheitsmae
	width  = 1.0;
	height = 1.0;

};

/*! 
 *  Die Methode \b build() erzeugt ein Gitter, welches auf der durch \b origin, \b v und \b u>
 *  parametrisierten Ebene liegt, und gibt einen Zeiger auf das erzeugte Objekt zurck. 
 *  Die Vektoren \b v und \b u werden zuvor normiert, und danach mit \b width und \b height w
 *  whrend der Berechnung der Gitterpunkte skaliert.
 *
 *  \note 
 *  Die Gitterauflsung muss mind. 2 x 2 betragen, da sonst 
 *  ein NULL-Zeiger zurckgegeben wird!
 *
 *  \note
 *  Der Speicher fr das zurckgegebene \b CGrid -Objekt wird dynamisch alloziiert
 *  und muss spter mittels \b delete() wieder freigegeben werden.
 *
 */
template<int N, class T> CGrid< CVector<N,T> >* CMeshBuilder<N,T>::build()
{

	// Prfen, ob die Auflsung grer 2 x 2 ist
	if ((horResolution>1)  && (verResolution>1))
	{

		// Ergebnisgitter erzeugen
		CGrid< CVector<N,T> >* outGrid = new CGrid< CVector<N,T> >(horResolution,verResolution);

		// Temporren Vertex erzeugen
		CVector<N,T> outVertex;

		// Lngen der Vektoren <u> und <v> bestimmen
		float ulength = sqrt(u*u);
		float vlength = sqrt(v*v);

		// Gitter aufbauen
		for (int y = 0 ; y<verResolution; y++)
		{

			for (int x = 0 ; x<horResolution; x++)
			{

				outVertex = origin + ((T)(x)*width*u)/(ulength*(horResolution-1)) + ((T)(y)*height*v)/(vlength*(verResolution-1));
				outGrid->set(x,y,outVertex);

			}

		}

		// Gitter zurckgeben
		return outGrid;

	}
	else
	{

		return NULL;

	}

};

#endif
