/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CMatrix.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CMatrix.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef cls_CMatrixH
#define cls_CMatrixH

#if     _MSC_VER > 1000
#pragma once
#endif

#include <stdlib.h>
#include <memory.h>
#include <stdexcept>
using std::invalid_argument;

#include "CVector.h"

/*!
 *
 * @class CMatrix
 *
 * @ingroup bsplineWarpClasses
 *
 * \brief Einfache Matrixklasse.
 *
 *  Das Klassentemplate \b CMatrix implementiert eine quadratische \b N x \b N
 *  Matrix mit Komponenten vom Datentyp \b T.
 *  Die ntigen arithmetischen Operatoren wurden berschrieben, so dass es
 *  mglich ist, alle Rechenoperationen durchzufhren, die auf Matrizen im
 *  mathematischen Sinne definiert sind.
 *
 *  \pre 
 *  Der Templateparameter \b N muss mindestens eins betragen.  Der durch den
 *  Templateparameter \b T spezifizierte Komponentendatentyp muss in
 *  irgendeiner Weise in den Typ \b int oder \b float konvertierbar sein.
 *  Ferner mssen alle gngigen arithmetischen Operatoren fr den Datentyp \b T
 *  berladen worden sein.
 * 
 */

template <int N, class T> class CMatrix
{
	
public:

	/*! \brief Komponentenarray.
	 *
	 *  Das zweidimensionale Array speichert \b N x \b N Komponenten vom Datentyp \b T.
	 */	
	T c[N][N]; 


	/*! \brief Standardkonstruktor.
	 *
	 *  Der Standardkonstruktor erzeugt eine Null-Matrix, d.h. alle Elemente
	 *  werden mit 0 initialisiert. 
	 *
	 */
	CMatrix()
	{
	
		// Matrixkooefizienten zurcksetzen
		clear();

	};
	
	/*! \brief Konvertierkonstruktor.
	 *
	 *  Der Konvertierkonstruktor erzegut eine Matrix auf der Basis eines
	 *  zweidimensionalen Arrays mit den Ausmaen \b N x \b N, welches Elemente
	 *  vom Datentyp \b T beinhlatet. Dabei wird das Array \t d[] mittels \b memmove() in
	 *  das Komponentenarray kopiert.
	 *
	 */

	CMatrix(T d[N][N])
	{

		// Kooeffizienten kopieren
		memmove(*c,*data,sizeof(d));

	};

	/*! \brief Kopierkonstruktor.
	 *
	 *  Der Kopierkonstruktor erzeugt ein neues \b CMatrix - Objekt auf der basis
	 *  eines anderen \b CMatrix - Objekts, welches die gleichen Mae aufweisen muss, aber
	 *  einen anderen Elementtyp \b F aufweisen kann, der zu \b T kompatibel ist. 
	 *  Die Elementdaten werden deswegen komponentenweise kopiert.
	 *
	 */
	template <class F> CMatrix(const CMatrix<N,F>& s)
	{
		
		for (int x=0 ; x<N ; x++)
		{

			for (int y=0 ; y<N; y++)
			{

				c[y][x]=s.c[y][x];

			}

		};

	};

	//! Macht die Matrix zur Einheitsmatrix.
	void makeIdentity();
	
	//! Macht die Matrix zur Null-Matrix.
	void clear();
	
	//! Transponiert die Matrix.
	void transpose();

	//! Gibt die Anzahl der Spalten der Matrix zurck.
	int columns()
	{
		return M;
	};

	//! Gibt die Anzahl der Zeilen der Matrix zurck.
	int rows()
	{
		return N;
	};

	//! Gibt die Gre einer Matrixkomponente in Bytes zurck.
	int componentSize()
	{
		return sizeof(T);
	};

	/*! \brief "+="-Operator (Matrixaddition).
	 *
	 *  Addiert die Matrix \b a zur der sich links vom "+="-Operator
	 *  befindenden Matrix.
	 *  Dabei wird eine komponentenweise Addition durchgefhrt.
	 *
	 */
	CMatrix<N,T>& operator+= (const CMatrix<N,T>& a)
	{

		// Koeffizienten komponenteweise addieren		
		for (int x=0 ; x<N; x++)
		{

			for (int y=0 ; y<N; y++)
			{

				c[y][x]+=a.c[y][x];

			};

		};

		return *this;

	};

	/*! \brief "-="-Operator (Matrixaddition).
	 *
	 *  Subtrahiert die Matrix \b a von der sich links vom "+="-Operator
	 *  befindenden Matrix.
	 *  Dabei wird eine komponentenweise Subtraktion durchgefhrt.
	 *
	 */
	CMatrix<N,T>& operator-= (const CMatrix<N,T>& a)
	{

		// Koeffizienten komponenteweise addieren
		for (int x=0 ; x<N; x++)
		{

			for (int y=0 ; y<N; y++)
			{

				c[y][x]-=a.c[y][x];

			};

		};

		return *this;

	};

	/*! \brief "+="-Operator (Skalarmultiplikation).
	 *
	 *  Multipliziert die sich links vom "+="-Operator
	 *  befindende Matrix mit dem Skalaren Wert \b f .
	 *  Dabei wird eine komponentenweise Multiplikation durchgefhrt.
	 *
	 */
	CMatrix<N,T>& operator*= (const T f)
	{

		// Koeffizienten komponentenweise mit Skalar multiplizieren
		for (int x=0 ; x<N ; x++)
		{

			for (int y=0 ; y <N ; y++)
			{

				c[y][x]*=f;

			};

		};

		return *this;

	};

	/*! \brief "+="-Operator (Skalardivision).
	 *
	 *  Dividiert die sich links vom "+="-Operator
	 *  befindende Matrix durch den Skalaren Wert \b f .
	 *  Dabei wird eine komponentenweise Division durchgefhrt.
	 *
	 */
	CMatrix<N,T>& operator/= (const T f)
	{

		// Koeffizienten komponentenweise durch Skalar dividieren		
		for (int x=0 ; x<N ; x++)
		{

			for (int y=0 ; y <N ; y++)
			{

				c[y][x]/=f;

			};

		};

		return *this;

	};

};

/*! Die Routine initialisiert die Diagonalelemente der Matrix mit dem Wert 1, alle
 *  brigen mit dem Wert 0.
 */
template <int N, class T> void CMatrix<N,T>::makeIdentity()
{

	// Matrixkooefizienten zurcksetzen
	clear();

	// Diagonalelemente auf 1 setzen
	for (int i=0 ; i<N ; i++) c[i][i]=1;

};

/*! Die Routine initialisiert alle Matrixkomponenten mit dem Wert 0.
 */
template <int N, class T> void CMatrix<N,T>::clear()
{

	// Kooeffizienten zurcksetzen
	for (int x = 0 ; x<N; x++)
	{
		
		for (int y = 0; y<N ; y++)
		{
			c[y][x] = 0;
		};

	};
	
};

/*! Dabei werden die Elemente c[i][j] und c[j][i] miteinander vertauscht.
 */
template <int N, class T> void CMatrix<N,T>::transpose()
{

	T tempdata[N][N];

	// Temporre transponierte matrix bilden;
	for (int x = 0 ; x<N ; x++)
	{
		
		for (int y = 0; y<N ; y++)
		{
			tempdata[x][y] = c[y][x];
		};

	};

	// Transponierte Kooeffizienten in die Matrix zurckschreiben
	for (int x = 0 ; x<N ; x++)
	{
		
		for (int y = 0; y<N ; y++)
		{
			c[y][x] = tempdata[y][x];
		};

	};

};

/*! \brief "+"-Operator (Matrixaddition).
 *
 *  Berechnet die Summe der Matrizen \b a und \b b und gibt diese zurck.
 *  
 *  \remarks
 *  Die Addition wird mit Hilfe einer temporren Matrix und des "+="-Operators durchgefhrt.
 *
 */
template <int N, class T> CMatrix<N,T> operator+ (const CMatrix<N,T>& a, const CMatrix<N,T>& b)
{
	
	return CMatrix<N,T>(a)+=b;

};

/*! \brief "-"-Operator (Matrixsubtraktion).
 *
 *  Berechnet die Differenz der Matrizen \b a und \b b und gibt diese zurck.
 *  
 *  \remarks
 *  Die Subtraktion wird mit Hilfe einer temporren Matrix und des "-"-Operators durchgefhrt.
 *
 */
template <int N, class T> CMatrix<N,T> operator- (const CMatrix<N,T>& a, const CMatrix<N,T>& b)
{
	
	return CMatrix<N,T>(a)-=b;

};

/*! \brief "*"-Operator (Skalarmultiplikation).
 *
 *  Skaliert die Matrix \b a mit dem skalaren Wert \b f und gibt diese zurck.
 *  
 *  \remarks
 *  Die Multiplikation wird mit Hilfe einer temporren Matrix und des "*="-Operators durchgefhrt.
 *
 */
template <int N, class T> CMatrix<N,T> operator* (const T f, const CMatrix<N,T>& a)
{
	
	return CMatrix<N,T>(a)*=f;

};

/*! \brief "/"-Operator (Skalarmultiplikation).
 *
 *  Skaliert die Matrix \b a mit dem skalaren Wert \b f und gibt diese zurck.
 *  
 *  \remarks
 *  Die Multiplikation wird mit Hilfe einer temporren Matrix und des "/="-Operators durchgefhrt.
 *
 */
template <int N, class T> CMatrix<N,T> operator/ (const CMatrix<N,T>& a,const T f)
{
	
	return CMatrix<N,T>(a)/=f;

};

/*! \brief "/"-Operator (Matrixmultiplikation).
 *
 *  Berechnet das Matrixprodukt der beiden Matrizen \b a und \b b und gibt dieses zurck.
 *
 */
template <int N, class T> CMatrix<N,T> operator* (const CMatrix<N,T>& a, const CMatrix<N,T>& b)
{

	CMatrix<N,T> tmp;

	for (int x = 0 ; x<N ; x++)
	{

		for (int y = 0 ; y<N ; y++)
		{
	
			tmp.c[y][x] = 0;

			for (int i = 0 ; i<N ; i++)
			{

				tmp.c[y][x]+=a.c[i][y] * b.c[i][x];

			};

		};

	};

	// Ergebnis zurckliefern
	return tmp;

};

/*! \brief "/"-Operator (Vektormultiplikation).
 *
 *  Berechnet das Vektorprodukt der Matrix \b a und des Vektors \b p und liefert den Ergebnisvektor zurck.
 *
 *  \pre
 *  Der Vektor muss \b N Komponenten aufweisen.
 *
 */
template <int N, class T> CVector<N,T> operator* (const CMatrix<N,T>& a, const CVector<N,T>& p)
{

	CVector<N,T> tmp;

	for (int i= 0 ; i < N ; i++)
	{

		tmp.c[i]=0;

		for (int y = 0 ; y <N ; y++)
		{
		
			tmp.c[i]+=p.c[y]*a.c[y][i];

		}

	};

	// Ergebnis zurckliefern
	return tmp;

};

#endif

