/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CLinearDeformer.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CLinearDeformer.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef cls_CLinearDeformerH
#define cls_CLinearDeformerH

#if     _MSC_VER > 1000
#pragma once
#endif

#include <stdlib.h>

#include "CControlGridDeformer.h"
#include "CReferenceGrid.h"
#include "CDynamicArray.h"

/*!
 * @class CLinearDeformer
 *
 * @ingroup bsplineWarpClasses
 *
 * \brief Klassentemplate, welches einen linearen Deformationsalgorithmuis
 * implementiert, der auf Vertexgittern operiert.
 *
 *  Das Klassentemplate \b CLinearDeformer implementiert einen linearen
 *  Deformationsalgorithmus, der Vertexgitter mit Hilfe eines Kontrollgitters
 *  verzerrt. 
 *  Die Klasse erlaubt die Spezifikation eines Quell-, eines Ziel- und eines
 *  Kontrolgitters.  Die Elemente der Vertexgrids sind vom Typ \b CVector. Die
 *  Anzahl der Komponenten wird vom Template auf 2 festgesetzt.  
 *  Der Templateparameter \b T gibt den Datentyp der Vektorkomponenten an.
 *
 *  \remarks
 *  Das als Quell- und Zielgitter kann ein und das selbe Gitter spezifiziert
 *  werden, da der Deformationsalgorithmus eine Kopie der Gitterkoordinaten
 *  anlegt, und danach auf die Originalkoordinaten des Quellgitters nicht mehr
 *  zugreift.
 *
 *  \note Die Ausmae der zugewiesenen Gitter mssen bereinstimmen, da
 *  ansonsten eine Exception vom Typ \b invalid_argument ausgelst wird.
 *  Ausgenommen hiervon ist das Kontrollgitter.
 *
 *  \note Die Mae der Grids sollten mindestens 2 x 2 betragen.
 *
 *  \note Nachdem alle drei Gitter spezifiziert wurden, hat ein verndern der
 *  Positionen der Quellgittervertices keine Auswirkungen auf das Ergebnis.  Um
 *  den gewnschten Effekt zu erhalten, mssen die Gitter nochmals neu
 *  zugewiesen werden. 
 *
 *  \pre
 *  Fr den Datentyp \b T mssen alle gngigen arithmetischen Operatoren
 *  berladen sein.  Ferner sollte der \b T zu \b float oder zu \b int
 *  kompatibel sein.
 *
 */
template <class T> class CLinearDeformer : public CControlGridDeformer<2,T>
{

protected:

	//! Speichert die Zugehrigkeit der Quellgittervertices zu den einzelnen Zellen des Kontrollgitters.
	CReferenceGrid< CDynamicArray< int > > VertexAssignmentMap;
	
	//! Speichert die Koordinaten der Quellgittervertices relativ zu ihren Kontrollgitterzellen.
	CGrid< CVector<2,T> > VertexCellCoordinates;
	
	//! Erzeugt die Vertex-Zelle-Zugehrigkeitsliste.
	void BuildVertexAssignmentMap();

public:

	//! Weist das Quellgitter zu.
	virtual void setSourceGrid(CGrid< CVector<2,T> >* objSourceGrid);

	//! Weist das Kontrollgitter zu.
	virtual void setControlGrid(CGrid< CVector<2,T> >* objControlGrid);

	//! Deformiert das Quellgitter und kopiert das Ergebnis ins Zielgitter.
	virtual void deform();

	//! Standarddestruktor.
	virtual ~CLinearDeformer();

};

/*! Die Methode \b setSourceGrid() weist dem Deformationsalgorithmus
 *  ein Quellgitter zu. 
 *
 *  \remarks
 *  Falls bereits ein Kontrollgitter zugewiesen wurde, so wird
 *  mittels \b BuildVertexAssignmentMap() die Vertex-Zelle-Zugehrigkeitstabelle
 *  erzeugt.
 *
 *  \pre Ein evtl. zuvor zugewiesenes Kontrollgitter mu regelmig sein.
 *
 */
template <class T> void CLinearDeformer<T>::setSourceGrid(CGrid< CVector<2,T> >* objSourceGrid)
{

	// Quellgitter zuweisen
	CControlGridDeformer<2,T>::setSourceGrid(objSourceGrid);
	
	// Prfen, ob bereits ein Kontrollgitter zugewiesen wurde
	if (ControlGrid!=NULL)
	{

		// Vertexzugehrigkeitstabelle erzeugen
		BuildVertexAssignmentMap();

	};

};

/*! Die Methode \b setControlGrid() weist dem Deformationsalgorithmus
 *  ein Kontrollgitter zu. 
 *
 *  \remarks
 *  Falls bereits ein Quellgitter zugewiesen wurde, so wird
 *  mittels \b BuildVertexAssignmentMap() die Vertex-Zelle-Zugehrigkeitstabelle
 *  erzeugt.
 *
 *  \pre Das zugewiesene Kontrollgitter mu regelmig sein.
 *
 */
template <class T> void CLinearDeformer<T>::setControlGrid(CGrid< CVector<2,T> >* objControlGrid)
{
	
	// Kontrollgitter zuweisen
	CControlGridDeformer<2,T>::setControlGrid(objControlGrid);

	// Prfen, ob bereits ein Quellgitter zugewiesen wurde
	if (SourceGrid!=NULL)
	{

		// Vertexzugehrigkeitstabelle erzeugen
		BuildVertexAssignmentMap();

	};

};

/*! Die Methode \b BuildVertexAssignmentMap() erzeugt eine
 *  Vertex-Zelle-Zugehrigkeitstabell. Dabei wird fr jede
 *  Kontrollgitterzelle eine Liste der Vertices angelegt, die
 *  sich in ihr befinden. 
 *  Ferner werden die Koordinaten der Quellgittervertices
 *  relativ zu ihrer Zelle berechnet und ebenfalls in einem
 *  zustzlichen Array gespeichert.
 *
 *  \pre 
 *  Das Kontrollgitter mu regelmig sein, d.h. aus rechteckigen
 *  Zellen bestehen, damit der Algorithmus korrekt funktioniert.
 *
 */
template <class T> void CLinearDeformer<T>::BuildVertexAssignmentMap()
{

	// Prfen, ob beide Kontroll- und Quellgitter zugewiesen wurden
	if ((SourceGrid!=NULL) && (ControlGrid!=NULL))
	{

		// Gre des Zellkoordinatengitters anpassen
		VertexCellCoordinates.resize(SourceGrid->width(),SourceGrid->height());
		
		// Gre der Zugehrigkeitsmap anpassen
		VertexAssignmentMap.clear();
		VertexAssignmentMap.resize(ControlGrid->width()-1, ControlGrid->height()-1);

		// Temporres Array erzeugen, um die Zugehrigkeit eines Vertex
		// zu einer Kontrollgitterzelle zu speichern.
		CDynamicArray<int> AssignmentArray(SourceGrid->width()*SourceGrid->height());

		// Zugehrigkeitn der Quellgittervertices zu den Kontrollgitterzellen
		// ermitteln und Koordinaten relativ zur jeweiligen Zelle berechnen
		CVector<2,T>* ptrVertex = (CVector<2,T>*) SourceGrid->getElements();
		CVector<2,T>* ptrCoord  = (CVector<2,T>*) VertexCellCoordinates.getElements();

		for (int i=0 ; i<SourceGrid->width()*SourceGrid->height(); i++)
		{
			
			for (int x = 0; x<ControlGrid->width()-1 ; x++)
			{
			
				for (int y = 0; y<ControlGrid->height()-1; y++)
				{
				
					// Prfen, ob sich der Vertex in der Zelle befindet
					if (
							(ptrVertex[i].c[0]>=ControlGrid->get(x,y).c[0]) &&
							(ptrVertex[i].c[0]<=ControlGrid->get(x+1,y).c[0]) &&
							(ptrVertex[i].c[1]>=ControlGrid->get(x,y).c[1]) &&
							(ptrVertex[i].c[1]<=ControlGrid->get(x,y+1).c[1])
						 )
					{

						// Zugehrigkeit zur Zelle speichern und Koordinaten
						// relativ zur Zelle berechnen
						AssignmentArray[i] = y*(ControlGrid->width()-1)+x;
						ptrCoord[i].c[0] = (ptrVertex[i].c[0]-ControlGrid->get(x,y).c[0])/(ControlGrid->get(x+1,y).c[0]-ControlGrid->get(x,y).c[0]);
						ptrCoord[i].c[1] = (ptrVertex[i].c[1]-ControlGrid->get(x,y).c[1])/(ControlGrid->get(x,y+1).c[1]-ControlGrid->get(x,y).c[1]);

					};

				};
			
			};

		};

		// Schleifenzhler
		int x;

		// Fr jede Zelle Anzahl der zugehrigen Vertices errechnen
		for (x = 0; x<ControlGrid->width()-1 ; x++)
		{

			for (int y = 0; y<ControlGrid->height()-1; y++)
			{
				
				int Count = 0;
				int Index = y*(ControlGrid->width()-1)+x;
				
				for (int i=0; i<AssignmentArray.getSize(); i++)
				{

					if (AssignmentArray[i]==Index) Count++;

				};

				// Speicher fr Zugehrigkeitstabelle der Zelle reservieren,
				// falls Anzahl > 0 

				if (Count>0)
				{
					VertexAssignmentMap.set(x,y,new CDynamicArray< int >(Count) );
				}
				else
				{
				
					VertexAssignmentMap.set(x,y,NULL);

				};

			};

		};

		// Fr jede Zelle Referenzen auf die zugehrigen Vertices
		// speichern
		for (x = 0; x<ControlGrid->width()-1 ; x++)
		{
			for (int y = 0; y<ControlGrid->height()-1; y++)
			{

				if (VertexAssignmentMap.get(x,y)!=NULL)
				{

					int Count = 0;
					int Index = y*(ControlGrid->width()-1)+x;
				
					for (int i=0; i<AssignmentArray.getSize(); i++)
					{
						if (AssignmentArray[i]==Index) 
						{

							VertexAssignmentMap.get(x,y)->set(Count,i);
							Count++;

						};

					};

				};

			};

		};

	};

};

/*! Die Methode \b Deform() implementiert den eigentlichen Deformationsalgorithmus.
 *  Dabei wird jede Zelle des Kontrollgitters durchlaufen. Die Eckpunkte einer
 *  Zelle bilden ein lineares Koordinatensystem. Die Basisvektoren dieses Koordinatensystems
 *  werden vom Algorithmus fr die Zelle berechnet. 
 *  Danach wird jeder Quellgittervertex, der zu der Zelle gehrt, 
 *  in dieses Kooridnatensystem transformiert. Durch Addition des Zellenortsvektors
 *  werden die lokalen Koordinaten wieder in die globalen umgerechnet.
 *  Die Ergebnisse der Transformation werden in dem Zielgitter gespeichert.
 *
 *  \remarks
 *  Da zur Berechnung nicht die Eigentlichen Vertexkoordinaten des Quellgitters
 *  verwendet werden, sondern die zuvor berechneten Relativkoordianten,
 *  kann als Zielgitter das gleiche Gitter zugewiesen werden, wie das Quellgitter.
 *
 *  \pre
 *  Es wurden zuvor alle drei Gitter zugewiesen. Das Kontrollgitter
 *  war whrend des Zuweisungsvorgangs noch regelmig.
 * 
 */
template <class T> void CLinearDeformer<T>::deform()
{

	// Prfen, ob alle Gitter zugewiesen wurden
	if ((ControlGrid!=NULL) && (SourceGrid!=NULL) && (DestinationGrid!=NULL))
	{

		// Deformation der Vertices in jeder Kontrollgitterzelle berechnen
		for (int y = 0; y< VertexAssignmentMap.height() ; y++)
		{
			
			for (int x=0 ; x< VertexAssignmentMap.width(); x++)
			{

				// Prfen, ob Vertices in Zelle vorhanden
				if (VertexAssignmentMap.get(x,y)!=NULL)
				{

					if (VertexAssignmentMap.get(x,y)->getSize()>0)
					{
						
						int Count = VertexAssignmentMap.get(x,y)->getSize();
						int* ptrVertexIndices = (int*) VertexAssignmentMap.get(x,y)->get();
						CVector<2,T>* srcVertices = (CVector<2,T>*) VertexCellCoordinates.getElements();
						CVector<2,T>* dstVertices = (CVector<2,T>*) DestinationGrid->getElements();

						// Vektoren des Koordinatensystems berechnen
						CVector<2,T> u1 = ControlGrid->get(x,y+1)-ControlGrid->get(x,y);
						CVector<2,T> u2 = ControlGrid->get(x+1,y+1)-ControlGrid->get(x+1,y);
						CVector<2,T> v  = ControlGrid->get(x+1,y)-ControlGrid->get(x,y);
						CVector<2,T> h  = u2-u1;
						CVector<2,T> o  = ControlGrid->get(x,y);

						// Punkte deformieren
						for (int i=0 ; i<Count ; i++)
						{
							dstVertices[*ptrVertexIndices] = o + srcVertices[*ptrVertexIndices].c[1] * u1 + srcVertices[*ptrVertexIndices].c[0] * (v + srcVertices[*ptrVertexIndices].c[1] * h);
							ptrVertexIndices++;
						};

					};

				};

			};

		};

	};

};

/*! Der Standarddestruktor gibt den Speicher, den die Vertexzugehrigkeitstabelle
 *  belegt, wieder frei.
 *
 *
 *
 */
template <class T> CLinearDeformer<T>::~CLinearDeformer()
{

	// Elemente lschen
	VertexAssignmentMap.clear();
	VertexCellCoordinates.clear();
	
	// Speicher freigeben, durch Verkleinern auf 0 x 0
	VertexAssignmentMap.resize(0,0);
	VertexCellCoordinates.resize(0,0);

};

#endif



