/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CGridPainter.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CGridPainter.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef cls_CGridPainterH
#define cls_CGridPainterH

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

#include <stdlib.h>
#include <qgl.h>

#include "CAbstractGrid.h"
#include "CGrid.h"

// Anzahl der verschiedenen Gitter
#define cstGITTERANZAHL 5

/*!
 * @class CGridPainter
 *
 * @ingroup bsplineWarpClasses
 *
 * \brief Klasse zum Zeichnen von in \b CGrid -Objekten abgelegten
 * Vertexgittern mittels OpenGL.
 *
 *  Die Klasse \b CGridPainter erlaubt es Vertexgitter mit Hilfe von OpenGL zu
 *  zeichnen.  Dabei wird das \b CGrid -Objekt dazu verwendet, die Vertexdaten
 *  zu speichern.  Das Zeichnen erfolgt ber die Vertexarray-Extension von
 *  OpenGL.  Die Klasse erlaubt es das Vertexgitter auf verschiedene Arten zu
 *  zeichnen.
 *
 *  Fr alle Array-Typen von OpenGL (\e Vertex-, \e Color-, \e Normal-, \e
 *  TextureCoord- und \e Edgeflag -Array) knnen \b CGrid -Objekte spezifiziert
 *  werden.  Lediglich auf das Farbindex-Array wurde verzichtet.
 *
 *  \note
 *  Alle verwendeten Grids mssen dieselben Ausmae aufweisen, d.h. dieselbe
 *  Anzahl an Spalten und dieselbe Anzahl and Zeilen.
 */


class CGridPainter
{

protected:
	
	//! Indexarray fr das Zeichnen mittels \b glDrawElements(). 
	GLuint* IndexArray; 
	
	//! Speichert einen Zeiger auf das \b CGrid -Objekt, welches die Vertex-Koordinaten enhlt.
	CAbstractGrid* VertexGrid; 
	
	//! Speichert einen Zeiger auf das \b CGrid -Objekt, welches die Normalen-Koordinaten enthlt.
	CAbstractGrid* NormalGrid; 
	
	//! Speichert einen Zeiger auf das \b CGrid -Objekt, welches die Farbinformationen enthlt.
	CAbstractGrid* ColorGrid;  
	
	//! Speichert einen Zeiger auf das \b CGrid -Objekt, welches die Texturkoordinaten enthlt.
	CAbstractGrid* TexelGrid;  
	
	//! Speichert einen Zeiger auf das \b CGrid -Objekt, welches die welches die EdgeFlag-Informationen enthlt.
	CAbstractGrid* EdgeGrid;   

	//! Speichert Zeiger auf die \b CGrid- Objektzeiger.
	CAbstractGrid** Grids[cstGITTERANZAHL]; 

	//! Speichert die Anzahl der Elemente eines Vertex.
	int VertexElements;	
	
	//! Speichert die Anzahl der Elemente einer Farbe.
	int ColorElements;  
	
	//! Speichert die Anzahl der Elemente eines Texels.
	int TexelElements;	

	//! Speichert den OpenGL-Datentyp einer Vertex-Komponente.
	GLenum VertexType; 

	//! Speichert den OpenGL-Datentyp einer Normalen-Komponente.
	GLenum NormalType; 

	//! Speichert den OpenGL-Datentyp einer Farb-Komponente.
	GLenum ColorType;  

	//! Speichert den OpenGL-Datentyp einer Texel-Komponente.
	GLenum TexelType;  

	//! Prft die bereinstimmung der Gitterausmae.
	bool CheckGridMeasurements(CAbstractGrid* objGrid);

	//! Erzeugt ein Index-Array zum Zeichnen mittels \b glDrawElements().
	virtual void BuildIndexArray(int BaseIndex, int Size);

	//! Initialisiert die OpenGL-Arrays.
	void SetupGLArrays();

public:

	/*! \enum enuPaintMode
	 * 
	 *  \brief Zeichenmoduskonstanten.
	 *
	 *  Die Enumeration \b enuPaintMode enthlt alle Konstanten, die den
	 *  Zeichenmodus der \b CGridPainter -Klasse steuern.
	 *
	 */

	/*! \var enuPaintMode pmSolid
	 *  Das Vertexgitter wird als eine Ansammlung von Trianglestrips gezeichnet. Die Dreiecke werden gefllt.
	 */

	/*! \var enuPaintMode pmWireframe
	 *  Das Vertexgitter wird als eine Ansammlung von vertikalen und horizontalen Linien gezeichnet. 
	 *  Die Linienbreite kann ber \b lineWidth spezifiziert werden.
	 */
 
	/*! \var enuPaintMode pmPoints
	 *  Das Vertexgitter wird als eine Ansammlung von Punkten gezeichnet. 
	 *  Die Punktgre kann ber \b pointSize spezifiziert werden.
	 */
	typedef enum {pmSolid,pmWireframe,pmPoints} enuPaintMode;

	//! Speichert den aktuellen Zeichenmodus.
	enuPaintMode paintMode; 

	//! Speichert die Punktgre.
	GLfloat pointSize; 

	//! Speichert die Linienbreite.
	GLfloat	lineWidth; 

	//! Speichert ein Flag, welches angibt, ob das VertexGrid benutzt werden soll.
	bool useVertexGrid; 

	//! Speichert ein Flag, welches angibt, ob das NormalGrid benutzt werden soll.
	bool useNormalGrid; 

	//! Speichert ein Flag, welches angibt, ob das ColorGrid benutzt werden soll.
	bool useColorGrid;  

	//! Speichert ein Flag, welches angibt, ob das TexelGrid benutzt werden soll.
	bool useTexelGrid;  

	//! Speichert ein Flag, welches angibt, ob das EdgeGrid benutzt werden soll.
	bool useEdgeGrid;   

	//! Standardkonstruktor.
	CGridPainter();
	
	//! Liefert den Zeiger auf das Vertex-Gitter zurck.
	const CAbstractGrid* getVertexGrid()
	{
		return VertexGrid;
	};

	//! Liefert den Zeiger auf das Normalen-Gitter zurck.
	const CAbstractGrid* getNormalGrid()
	{
		return NormalGrid;
	};


	//! Liefert den Zeiger auf das Color-Gitter zurck.
	const CAbstractGrid* getColorGrid()
	{
		return ColorGrid;
	};

	//! Liefert den Zeiger auf das Texel-Gitter zurck.
  const CAbstractGrid* getTexelGrid()
	{
		return TexelGrid;
	};

	//! Liefert den Zeiger auf das EdgeFlag-Gitter zurck.
	const CAbstractGrid* getEdgeGrid()
	{
		return EdgeGrid;
	};

	//! Setzt den Zeiger auf das Vertex-Gitter.
	void setVertexGrid(CAbstractGrid* objVertexGrid, int intVertexElements = 0, GLenum enuVertexType = 0);

	//! Setzt den Zeiger auf das Normalen-Gitter.
	void setNormalGrid(CAbstractGrid* objNormalGrid, GLenum enuNormalType = 0);

	//! Setzt den Zeiger auf das Farb-Gitter.
	void setColorGrid(CAbstractGrid* objColorGrid, int intColorElements = 0, GLenum enuColorType= 0);

	//! Setzt den Zeiger auf das Texturkoordinaten-Gitter.
	void setTexelGrid(CAbstractGrid* objTexelGrid, int intTexelElements = 0, GLenum enuTexelType = 0);

	//! Setzt den Zeiger auf das EdgeFlag-Gitter.
	void setEdgeGrid(CAbstractGrid* objEdgeGrid);

	//! Rendert das gesamte Vertexgrid.
	virtual void render();
	
	//! Rendert eine Zelle des Vertexgrids.
	virtual void render(int intX, int intY);
	
	//! Rendert einen rechteckigen Ausschnitt des Vertexgrids.
	virtual void render(int intLeft, int intTop, 
		                  int intWidth, int intHeight);


	//! Standarddestruktor.
	virtual ~CGridPainter();

};
 
#endif


