/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CGridPainter.cpp
 *
 *  @brief
 *	Implementation der Klasse \b CGridPainter.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#include <stdlib.h>

#include "CGridPainter.h"
#include "CGrid.h"

#include <stdexcept>
using std::invalid_argument;

#include <qgl.h>

/*!
 *  Der Standardkonstruktor initialisiert alle Gridzeiger mit NULL.
 *  Ferner werden alle Arrays deaktiviert. Die Linien- und Punktbreite
 *  wird auf 1.0 gesetzt.
 *
 */

CGridPainter::CGridPainter()
{

	// Gridzeiger initialisieren
	IndexArray = NULL;
	VertexGrid = NULL;
	NormalGrid = NULL;
	ColorGrid = NULL;
	TexelGrid = NULL;
	EdgeGrid = NULL;

	// Typinformationen initialisieren
	VertexElements = 0;
	ColorElements = 0;
	TexelElements = 0;

	VertexType = 0;
	NormalType = 0;
	ColorType = 0;
	TexelType = 0;

	// Alle Arrays deaktivieren
	useVertexGrid = false;
	useNormalGrid = false;
	useColorGrid = false;
	useTexelGrid = false;
	useEdgeGrid = false;

	// Linien- und Punktbreite setzen
	lineWidth = 1.0;
	pointSize = 1.0;

	// Grid-Zeiger setzen
	Grids[0] = &VertexGrid;
	Grids[1] = &NormalGrid;
	Grids[2] = &ColorGrid;
	Grids[3] = &TexelGrid;
	Grids[4] = &EdgeGrid;

};

/*!
 *  Die Funktion \b CheckGridMeasurements() berprft, ob 
 *  die Ausmae des Grids \b objGrid mit den Ausmaen
 *  der bisher zugewiesenen \b CGrid -Objekte bereinstimmen.
 *  Ist dies nicht der Fall, so gibt die Funktion \b false zurck.
 *  Stimmen die Ausmae berein, oder wurden bis jetzt keine
 *  anderen \b CGrid -Objekte spezifiziert, so gibt die Funktion
 *  \b true zurck.
 *
 */
bool CGridPainter::CheckGridMeasurements(CAbstractGrid* objGrid)
{

	// Prfen, ob Gitter-Gren bereinstimmen
	if (objGrid!=NULL)
	{

		for (int i=0 ; i<cstGITTERANZAHL; i++)
		{

			if (*Grids[i]!=NULL)
			{
				
				// Ausmae vergleichen
				if (((*Grids[i])->width()!=objGrid->width()) ||
					 ((*Grids[i])->height()!=objGrid->height()))
				{

					return false;

				}

			}

		}

	}
	
	// Alles OK!
	return true;

};

/*! Die Routine \b BuildIndexArray() erzeugt ein Index-Array, so 
 *  dass der Aufruf von \b glDrawElementes() eine Vertexgitterzeile zeichnet. 
 *
 *  Der Parameter \b BaseIndex
 *  gibt den Startindex an. Insgesamt werden \b size * 2 Indizes
 *  erzeugt.
 *
 *  \remarks
 *  Die lineare Reihenfolge der Daten innerhalb eines \b CGrid -Objekts stimmt nicht
 *  mit der von OpenGL vorgeschriebenen Definitionsreihenfolge fr Quad- und Dreiecksprimitive.
 *  Deswegen muss zustzlich ein Indexarray erzeugt werden, welches die
 *  richtige Reihenfolge herstellt.
 *
 */
void CGridPainter::BuildIndexArray(int BaseIndex, int size)
{
	
	// Speicher fr Index-Array reservieren

	IndexArray = (GLuint*) realloc(IndexArray, 2*sizeof(GLuint)*size);
	
	// Index-Array fllen
	int offset = VertexGrid->width();

	for (int x = 0 ; x<size ; x++)
	{

		IndexArray[x<<1] = BaseIndex+x; // Index aktuelle Zeile
		IndexArray[(x<<1)+1] = BaseIndex+x+offset; // Index nchste Zeile

	};

};

/*! ber den Parameter \b intVertexElements wird festgelegt, aus wievielen Komponenten
 *  ein Vertex besteht. Der Parameter \b enuVertexType gibt den OpenGL-Datentyp einer Komponente an.
 *  
 *  \note
 *  Die Ausmae des zuzuweisenden Gitters mssen mit den Ausmaen der
 *  bereits festgelegten Gitter bereinstimmen. Ist dies nicht der Fall,
 *  so wird eine Exception vom Typ \b invalid_argument ausgelst.
 *
 */
void CGridPainter::setVertexGrid(CAbstractGrid* objVertexGrid, int intVertexElements, GLenum enuVertexType)
{

	// Gitterausmae berprfen
	if (CheckGridMeasurements(objVertexGrid))
	{

		// Zeiger und Typeninformationen speichern
		VertexGrid = objVertexGrid;
		VertexElements = intVertexElements;
		VertexType = enuVertexType;

	}
	else
	{
	
		// Exception auslsen
		throw invalid_argument("Die Ausmae des neuen Vertex-Gitters passen nicht zu den Ausmaen der bereits vorhandenen Gitter!");

	}

};

/*! 
 *  Der Parameter \b enuNormalType gibt den OpenGL-Datentyp einer Normalenkomponente an.
 *   
 *  \note
 *  Es wird davon ausgegangen, dass ein Normalenvektor aus \b drei Kompontenten besteht.
 *
 *  \note
 *  Die Ausmae des zuzuweisenden Gitters mssen mit den Ausmaen der
 *  bereits festgelegten Gitter bereinstimmen. Ist dies nicht der Fall,
 *  so wird eine Exception vom Typ \b invalid_argument ausgelst.
 *
 */
void CGridPainter::setNormalGrid(CAbstractGrid* objNormalGrid, GLenum enuNormalType)
{

	// Gitterausmae berprfen
	if (CheckGridMeasurements(objNormalGrid))
	{

		// Zeiger und Typeninformationen speichern
		NormalGrid = objNormalGrid;
		NormalType = enuNormalType;

	}
	else
	{

		// Exception auslsen
		throw invalid_argument("Die Ausmae des neuen Normalen-Gitters passen nicht zu den Ausmaen der bereits vorhandenen Gitter!");

	}

};

/*! ber den Parameter \b intColorElements wird festgelegt, aus wievielen Komponenten
 *  eine Farbe besteht. Der Parameter \b enuColorType gibt den OpenGL-Datentyp einer Farbkomponente an.
 *  
 *  \note
 *  Die Ausmae des zuzuweisenden Gitters mssen mit den Ausmaen der
 *  bereits festgelegten Gitter bereinstimmen. Ist dies nicht der Fall,
 *  so wird eine Exception vom Typ \b invalid_argument ausgelst.
 *
 */
void CGridPainter::setColorGrid(CAbstractGrid* objColorGrid, int intColorElements,GLenum enuColorType)
{

	// Gitterausmae berprfen
	if (CheckGridMeasurements(objColorGrid))
	{

		// Zeiger und Typeninformationen speichern
		ColorGrid = objColorGrid;
		ColorElements = intColorElements;
		ColorType = enuColorType;

	}
	else
	{

		// Exception auslsen
		throw invalid_argument("Die Ausmae des neuen Color-Gitters passen nicht zu den Ausmaen der bereits vorhandenen Gitter!");

	}

};

/*! ber den Parameter \b intTexelElements wird festgelegt, aus wievielen Komponenten
 *  eine Texturkoordinate besteht. Der Parameter \b enuTexelType gibt den OpenGL-Datentyp einer Texelkoordinatenkomponente an.
 *  
 *  \note
 *  Die Ausmae des zuzuweisenden Gitters mssen mit den Ausmaen der
 *  bereits festgelegten Gitter bereinstimmen. Ist dies nicht der Fall,
 *  so wird eine Exception vom Typ \b invalid_argument ausgelst.
 *
 */
void CGridPainter::setTexelGrid(CAbstractGrid* objTexelGrid, int intTexelElements,GLenum enuTexelType)
{
	// Gitterausmae berprfen

	if (CheckGridMeasurements(objTexelGrid))
	{
		TexelGrid = objTexelGrid;
		TexelElements = intTexelElements;
		TexelType = enuTexelType;
	}
	else
	{
		// Exception auslsen
		throw invalid_argument("Die Ausmae des neuen Texel-Gitters passen nicht zu den Ausmaen der bereits vorhandenen Gitter!");
	}

};

/*  \note 
 *  Der Datentyp eines Edgeflag-Gitterelements muss \b GLbool sein.
 *
 *  \note
 *  Die Ausmae des zuzuweisenden Gitters mssen mit den Ausmaen der
 *  bereits festgelegten Gitter bereinstimmen. Ist dies nicht der Fall,
 *  so wird eine Exception vom Typ \b invalid_argument ausgelst.
 *
 */
void CGridPainter::setEdgeGrid(CAbstractGrid* objEdgeGrid)
{

	// Gitterausmae berprfen
	if (CheckGridMeasurements(objEdgeGrid))
	{

		EdgeGrid = objEdgeGrid;

	}
	else
	{

		// Exception auslsen
		throw invalid_argument("Die Ausmae des neuen EdgeFlag-Gitters passen nicht zu den Ausmaen der bereits vorhandenen Gitter!");

	}

};

/*! Die Routine \b SetupGLArrays initialisert die OpenGL-Vertexarray-Zeiger.
 *  Dabei wird ber die \b useXXXGrid Flags ermittelt, welche Arrays benutzt werden sollen.
 *  Danach werden die Zeiger auf die Griddaten an OpenGL bergeben und das entsprechende
 *  Array in OpenGL aktiviert, allerdings nur dann, wenn auch tatschlich 
 *  ein Grid fr ein aktiviertes Array vorliegt. Liegt kein Grid
 *  vor, so wird das entsprechende Array in OpenGL deaktiviert.
 *
 */
void CGridPainter::SetupGLArrays()
{
		// Vertexarray initialisieren und aktivieren
		if (useVertexGrid) 
		{

			// Prfen, ob ein Vertex-Grid spezifiziert wurde
			if (VertexGrid!=NULL)
			{

				 // Zeiger auf Griddaten an OpenGL bergeben und Array in OpenGL aktivieren
				 glVertexPointer(VertexElements,VertexType,0,VertexGrid->getElements());
				 glEnableClientState(GL_VERTEX_ARRAY);

			}
			else
			{

				// Array in OpenGL deaktivieren
				glDisableClientState(GL_VERTEX_ARRAY);

			}

		}
		else
		{

			// Array in OpenGL deaktivieren
			glDisableClientState(GL_VERTEX_ARRAY);

		}

		// Normalenarray initialisieren und aktivieren
		if (useNormalGrid) 
		{

			// Prfen, ob ein Normal-Grid spezifiziert wurde
			if (NormalGrid!=NULL)
			{

				 // Zeiger auf Griddaten an OpenGL bergeben und Array in OpenGL aktivieren
				 glNormalPointer(NormalType,0,NormalGrid->getElements());
				 glEnableClientState(GL_NORMAL_ARRAY);

			}
			else
			{

				// Array in OpenGL deaktivieren
				glDisableClientState(GL_NORMAL_ARRAY);

			}

		}
		else
		{
			
			// Array in OpenGL deaktivieren
			glDisableClientState(GL_NORMAL_ARRAY);

		}

		// Farbarray initialisieren und aktivieren
		if (useColorGrid) 
		{

			// Prfen, ob ein Color-Grid spezifiziert wurde
			if (ColorGrid!=NULL)
			{

				 // Zeiger auf Griddaten an OpenGL bergeben und Array in OpenGL aktivieren
				 glColorPointer(ColorElements,ColorType,0,ColorGrid->getElements());
				 glEnableClientState(GL_COLOR_ARRAY);

			}
			else
			{
				
				// Array in OpenGL deaktivieren
				glDisableClientState(GL_COLOR_ARRAY);

			}

		}
		else
		{
				// Array in OpenGL deaktivieren
				glDisableClientState(GL_COLOR_ARRAY);
		}

		// Texturkoordinatenarrays initialisieren und aktivieren
		if (useTexelGrid) 
		{

			// Prfen, ob ein Texel-Grid spezifiziert wurde
			if (TexelGrid!=NULL)
			{

				 // Zeiger auf Griddaten an OpenGL bergeben und Array in OpenGL aktivieren
				 glTexCoordPointer(TexelElements,TexelType,0,TexelGrid->getElements());
				 glEnableClientState(GL_TEXTURE_COORD_ARRAY);

			}
			else
			{

				// Array in OpenGL deaktivieren
				glDisableClientState(GL_TEXTURE_COORD_ARRAY);

			}

		}
		else
		{

			// Array in OpenGL deaktivieren
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);

		}

		// Edgeflagarray initialisieren und aktivieren
		if (useEdgeGrid) 
		{

			// Prfen, ob ein EdgeFlag-Grid spezifiziert wurde
			if (EdgeGrid!=NULL)
			{

				 // Zeiger auf Griddaten an OpenGL bergeben und Array in OpenGL aktivieren
				 glEdgeFlagPointer(0,EdgeGrid->getElements());
				 glEnableClientState(GL_EDGE_FLAG_ARRAY);

			}
			else
			{
			
  			// Array in OpenGL deaktivieren
				glDisableClientState(GL_EDGE_FLAG_ARRAY);

			}

		}
		else
		{
			
			// Array in OpenGL deaktivieren
			glDisableClientState(GL_EDGE_FLAG_ARRAY);

		};
	
};

/*! 
 *  Die Routine \b render() zeichnet das gesamte Vertexgitter unter
 *  Verwednung aller ber die \b useXXXGrid -Flags aktivierten Grids,
 *  fr die auch tatschlich ein Zeiger auf ein \bCGrid -Objekt angegeben wurde.
 *  Die Art der Darstellung wird ber die Klassenmember \b paintMode, \b pointSize und \b lineWidth gesteuert.
 *
 *  \note
 *  Es wird nur dann etwas gezeichnet, wenn ein Zeiger auf ein Vertex-Grid
 *  zuvor spezifiziert, und das Vertexgrid mittels \b useVertexGrid aktiviert wurde.
 *  
 */
void CGridPainter::render()
{

	// Prfen, ob ein Vertex-Grid angegeben wurde
	if ((VertexGrid!=NULL) && (useVertexGrid))
	{

		int y=0; // Zeilenzhler

		// Vertex-Arrays initialisieren
		SetupGLArrays();

		// Zeichenmodus abfragen
	  switch(paintMode)
		{
		
			// Solid-Modus, Dreiecke werden gefllt
			case pmSolid:
			{

				// OpenGL-Polygonmodus setzen
				glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);

				// Quadstrips zeichnen
				for (y=0; y<VertexGrid->height()-1; y++)
				{

					BuildIndexArray(y*VertexGrid->width(),VertexGrid->width());
					glDrawElements(GL_QUAD_STRIP,VertexGrid->width()*2,GL_UNSIGNED_INT,IndexArray);

				};

			}
			break;

			// Wireframe-Modus, es werden nur die Gitterzellenrnder gezeichnet
			case pmWireframe:
			{
			
				// Linenbreite und OpenGL-Polygonmodus setzen
				glLineWidth(lineWidth);
				glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);

				// Linien zeichnen
				for (y=0; y<VertexGrid->height()-1; y++)
				{
					BuildIndexArray(y*VertexGrid->width(),VertexGrid->width());
					glDrawElements(GL_QUAD_STRIP,VertexGrid->width()*2,GL_UNSIGNED_INT,IndexArray);
				};

			}
			break;

			// Punkt-Modus, es werden nur die Punkte der Gitterzellen gezeichnet
			case pmPoints:
			{

				// Punktgre setzen und Punkte zeichnen
				glPointSize(pointSize);
				glDrawArrays(GL_POINTS,0,VertexGrid->width()*VertexGrid->height());

			}
			break;

		};

	};

};

/*! Die Methode \b render(int intX, int intY) zeichnet nur eine einzige Gitterzelle,
 *  welche ber die Paramter \b intX und \b intY spezifiziert wurde.
 *  Es werden nur die ber \b useXXXGrid -Flags aktivierten Grids,
 *  fr die auch tatschlich ein Zeiger auf ein \bCGrid -Objekt angegeben wurde, verwendet.
 *  Die Art der Darstellung wird ber die Klassenmember \b paintMode, \b pointSize und \b lineWidth gesteuert.
 *
 *  \note
 *  Es wird nur dann etwas gezeichnet, wenn ein Zeiger auf ein Vertex-Grid
 *  zuvor spezifiziert, und das Vertexgrid mittels \b useVertexGrid aktiviert wurde.
 */
void CGridPainter::render(int intX, int intY)
{

	// Prfen, ob ein Vertex-Grid angegeben wurde
	if ((VertexGrid!=NULL) && (useVertexGrid))
	{

		// Vertex-Arrays initialisieren
		SetupGLArrays();

		// Zeichenmodus abfragen
		switch (paintMode)
		{

			// Solid-Modus, Dreiecke werden gefllt
			case pmSolid:
			{

				// OpenGL-Polygonmodus setzen
				glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);

				// Trianglestrip zeichnen
				BuildIndexArray(intY*VertexGrid->width()+intX,2);
				glDrawElements(GL_QUAD_STRIP,4,GL_UNSIGNED_INT,IndexArray);

			}
			break;

			// Wireframe-Modus, Es werden nur die Gitterzellenrnder gezeichnet
			case pmWireframe:
			{

				// Linienbreite und OpenGL-Polygonmodus setzen
				glLineWidth(lineWidth);
				glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);

				// Quadstrip zeichnen
				BuildIndexArray(intY*VertexGrid->width()+intX,2);
				glDrawElements(GL_QUAD_STRIP,4,GL_UNSIGNED_INT,IndexArray);

			}
			break;

			// Punkte-Modus, es werden nur die Eckpunkte der Gitterzellen gezeichnet
			case pmPoints:
			{

				// Punktgre setzen
				glPointSize(pointSize);

				// Punkte zeichnen
				BuildIndexArray(intY*VertexGrid->width()+intX,2);
				glDrawElements(GL_POINTS,4,GL_UNSIGNED_INT,IndexArray);

			}
			break;

		};

	};

};

/*! Die Methode \b render(int intLeft, int intTop, int intWidth, int intHeight) zeichnet einen rechteckigen Auschnitt des
 *  Gitters. Die Parameter \b intLeft und \b intTop geben die Gitterzelle
 *  an, die die linke obere Ecke des Auschnitts darstellt.
 *  Die Parameter \b intWidth und \b intHeight legen die Breite und
 *  die Hhe des Auschnitts fest.
 *  Es werden nur die ber \b useXXXGrid -Flags aktivierten Grids,
 *  fr die auch tatschlich ein Zeiger auf ein \bCGrid -Objekt angegeben wurde, verwendet.
 *  Die Art der Darstellung wird ber die Klassenmember \b paintMode, \b pointSize und \b lineWidth gesteuert.
 *
 *  \note
 *  Es wird nur dann etwas gezeichnet, wenn ein Zeiger auf ein Vertex-Grid
 *  zuvor spezifiziert, und das Vertexgrid mittels \b useVertexGrid aktiviert wurde.
 */
void CGridPainter::render(int intLeft, int intTop, 
		                      int intWidth, int intHeight)
{

	// Prfen, ob ein Vertex-Grid angegeben wurde
	if ((VertexGrid!=NULL) && (useVertexGrid))
	{

		int y=0; // Zeilenzhler

		// Vertex-Arrays initialisieren
		SetupGLArrays();

		// Zeichenmodus abfragen
		switch (paintMode)
		{

			// Solid-Modus, Dreiecke werden gefllt
			case pmSolid:
			{

				// OpenGL-Polygonmodus setzen
				glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);

				// Quadstrips zeichnen
				for (y=intTop; y<intTop+intHeight; y++)
				{

					BuildIndexArray(y*VertexGrid->width()+intLeft,(intWidth+1));
					glDrawElements(GL_QUAD_STRIP,(intWidth+1)*2,GL_UNSIGNED_INT,IndexArray);

				}
			
			}
			break;

			// Wireframe-Modus, Es werden nur die Gitterzellenrnder gezeichnet
			case pmWireframe:
			{	
			
				// Linienbreite und OpenGL-Polygonmodus setzen
				glLineWidth(lineWidth);
				glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);

				// Quadstrips zeichnen
				for (y=intTop; y<intTop+intHeight; y++)
				{

					BuildIndexArray(y*VertexGrid->width()+intLeft,(intWidth+1));
					glDrawElements(GL_QUAD_STRIP,(intWidth+1)*2,GL_UNSIGNED_INT,IndexArray);

				}
			
			}
			break;

			// Punkte-Modus, es werden nur die Eckpunkte der Gitterzellen gezeichnet
			case pmPoints:
			{
				// Punktgre setzen
				glPointSize(pointSize);

				// Punkte zeichnen
				for (y=intTop; y<intTop+intHeight; y++)
				{

					BuildIndexArray(y*VertexGrid->width()+intLeft,(intWidth+1));
					glDrawElements(GL_POINTS,(intWidth+1)*2,GL_UNSIGNED_INT,IndexArray);

				}
			
			}
			break;

		}

	};

};

// Standard-Destruktor

CGridPainter::~CGridPainter()
{
	
	// Index-Array freigeben
	realloc(IndexArray,0);

};

