/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CGridInterpolator.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CGridInterpolator.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef cls_CGridInterpolatorH
#define cls_CGridInterpolatorH

#if     _MSC_VER > 1000
#pragma once
#endif

#include "CInterpolator.h"
#include "CGrid.h"

#include <stdlib.h>
#include <stdexcept>
using std::invalid_argument;

// ****************************************************************
//
// Klasse: CGridInterpolator
//
// Zweck : Klassetemplate zur Interpolation zwischen zwei 
//			   Grid-Objekten mit dem Elementtyp <T>.
//				 Es ist mglich ein Quell-, ein Ziel- und ein 
//				 Ergebnisgitter zuzuweisen.
//
// Voraussetzungen: 
//				 Alle drei Gitter mssen dieselben Ausmae haben, da
//         sonst bei der Zuweisung eine INVALID_ARGUMENT-Exception
//				 ausgelst wird. 
//				 Fr den Gitterelementtyp <T> mssen die Operatoren +,-,* 
//				 und berschrieben sein.
//
// Autor: Christoph Brzozowski
//
// ****************************************************************

/*!
 * @class CGridInterpolator
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Linearer Interpolationsalgorithmus fr \b CGrid -Objekte.
 *
 *  Das Klassentemplate \b CGridInterpolator realisiert einen linearen
 *  Interpolationsalgorithmus fr CGrid -Objekte.
 *  Der Templateparameter \b T legt hierbei den Datentype der Gridelemente
 *  fest.
 *  Es knnen ein Quell- ein Ziel- und ein Ergebnisgrid spezifiziert werden.
 * 
 *  \note
 *  Alle drei Gitter mssen einen identischen Elementtyp haben und gleiche
 *  Ausmae aufweisen.
 *
 *  \pre
 *  Fr den Elementtyp \b T mssen mssen alle gngigen arithmetischen
 *  Operatoren berladen worden sein. Insbesondere muss eine berladung des
 *  "*"-Operator fr \b float und \b T existieren.
 *  
 */

template <class T> class CGridInterpolator : public CInterpolator
{

protected:

	//! Speichert einen Zeiger auf das Quellgitter.
  CGrid<T>* SourceGrid;			
	//! Speichert einen Zeiger auf das Zielgitter.
	CGrid<T>* DestinationGrid; 
	//! Speichert einen Zeiger auf das Ergebnisgitter.
	CGrid<T>* ResultGrid;		  

public:

	//! Standardkonstruktor.
	CGridInterpolator();

	//! Weist der Klasse ein Quellgitter zu.
	void setSourceGrid(CGrid<T>* sourceGrid);

	//! Weist der Klasse ein Zielgitter zu.
	void setDestinationGrid(CGrid<T>* destinationGrid);

	//! Weist der Klasse ein Ergebnisgitter zu.
	void setResultGrid(CGrid<T>* resultGrid);

	//! Gibt eine Referenz auf das Quellgitter zurck.
	CGrid<T>& sourceGrid()
	{

		return SourceGrid;

	};

	//! Gibt eine Referenz auf das Zielgitter zurck.
	CGrid<T>& destinationGrid()
	{

		return DestinationGrid;

	};
	
	//! Gibt eine Referenz auf das Ergebnisgitter zurck.
	CGrid<T>& resultGrid()
	{

		return ResultGrid;

	}

	//! Interpoliert linear zwischen dem Quell- und dem Zielgitter.
	virtual void interpolate(float TimeIndex);

	//! Standarddestruktor.
	virtual ~CGridInterpolator();

};

/*!
 *  Der Standardkonstruktor initialisiert alle Elementzeiger
 *  mit NULL.
 */
template <class T> CGridInterpolator<T>::CGridInterpolator()
{

	// Zeiger initialisieren
	SourceGrid = NULL;
	DestinationGrid = NULL;
	ResultGrid = NULL;

};

/*!
 *  Die Methode \b setSourceGrid() weist dem Interpolationsalgorithmus ein
 *  Quellgitter zu.
 * 
 *  \note
 *  Wurde bereits ein Ergebnis- oder ein Zielgitter zugewiesen,
 *  so werden die Ausmae verglichen. Stimmen sie nicht berein, so wird eine
 *  Exception vom Typ \b INVALID_ARGUMENT ausgelst.
 */ 
template <class T> void CGridInterpolator<T>::setSourceGrid(CGrid<T>* sourceGrid)
{

	// Prfen, ob Quellgitter angegeben wurde
	if (sourceGrid!=NULL)
	{

		// Prfen, ob bereits ein Zielgitter zugewiesen wurde und
		// Ausmae vergleichen
		if (DestinationGrid!=NULL)
		{

			if ((sourceGrid->width()!=DestinationGrid->width()) || 
				  (sourceGrid->height()!=DestinationGrid->height()))
			{

				// Exception auslsen
				throw new invalid_argument("Die Ausmae des Quell-, Ziel- und Ergebnisgitters mssen bereinstimmen!");  

			};

		}
		else
		// Prfen, ob bereits ein Ergebnisgitter zugewiesen wurde und
		// Ausmae vergleichen
		if (ResultGrid!=NULL)
		{

			if ((sourceGrid->width()!=ResultGrid->width()) || 
				  (sourceGrid->height()!=ResultGrid->height()))
			{

				// Exception auslsen
				throw new invalid_argument("Die Ausmae des Quell-, Ziel- und Ergebnisgitters mssen bereinstimmen!");  

			};

		};

	};
	
	// Quellgitter zuweisen
	SourceGrid = sourceGrid;

};

/*!
 *  Die Methode \b setDestinationGrid() weist dem Interpolationsalgorithmus ein
 *  Zielgitter zu.
 * 
 *  \note
 *  Wurde bereits ein Ergebnis- oder ein Quellgitter zugewiesen,
 *  so werden die Ausmae verglichen. Stimmen sie nicht berein, so wird eine
 *  Exception vom Typ \b INVALID_ARGUMENT ausgelst.
 */ 
template <class T> void CGridInterpolator<T>::setDestinationGrid(CGrid<T>* destinationGrid)
{

	// Prfen, ob Zielgitter angegeben wurde
	if (destinationGrid!=NULL)
	{

		// Prfen, ob bereits ein Quellgitter zugewiesen wurde und
		// Ausmae vergleichen
		if (SourceGrid!=NULL)
		{

			if ((destinationGrid->width()!=SourceGrid->width()) || 
				  (destinationGrid->height()!=SourceGrid->height()))
			{

				// Exception auslsen
				throw new invalid_argument("Die Ausmae des Quell-, Ziel- und Ergebnisgitters mssen bereinstimmen!");  

			};

		}
		else
		// Prfen, ob bereits ein Ergebnisgitter zugewiesen wurde und
		// Ausmae vergleichen
		if (ResultGrid!=NULL)
		{

			if ((destinationGrid->width()!=ResultGrid->width()) || 
				  (destinationGrid->height()!=ResultGrid->height()))
			{

				// Exception auslsen
				throw new invalid_argument("Die Ausmae des Quell-, Ziel- und Ergebnisgitters mssen bereinstimmen!");  

			};

		};

	};
	
	// Zielgitter zuweisen
  DestinationGrid = destinationGrid;

};

/*!
 *  Die Methode \b setResultGrid() weist dem Interpolationsalgorithmus ein
 *  Ergebnisgitter zu. Das Ergebnisgitter speichert nach einem
 *  Aufruf der Methode \b interpolate() deren Ergebnis.
 * 
 *  \note
 *  Wurde bereits ein Quell- oder ein Zielgitter zugewiesen,
 *  so werden die Ausmae verglichen. Stimmen sie nicht berein, so wird eine
 *  Exception vom Typ \b INVALID_ARGUMENT ausgelst.
 */ 
template <class T> void CGridInterpolator<T>::setResultGrid(CGrid<T>* resultGrid)
{

	// Prfen, ob Ergebnisgitter angegeben wurde
	if (resultGrid!=NULL)
	{
		// Prfen, ob bereits ein Quellgitter zugewiesen wurde und
		// Ausmae vergleichen
		if (SourceGrid!=NULL)
		{

			if ((resultGrid->width()!=SourceGrid->width()) || 
				  (resultGrid->height()!=SourceGrid->height()))
			{

				// Exception auslsen
				throw new invalid_argument("Die Ausmae des Quell-, Ziel- und Ergebnisgitters mssen bereinstimmen!");  

			};

		}
		else
		// Prfen, ob bereits ein Zielgitter zugewiesen wurde und
		// Ausmae vergleichen
		if (DestinationGrid!=NULL)
		{

			if ((resultGrid->width()!=DestinationGrid->width()) || 
				  (resultGrid->height()!=DestinationGrid->height()))
			{

				// Exception auslsen
				throw new invalid_argument("Die Ausmae des Quell-, Ziel- und Ergebnisgitters mssen bereinstimmen!");  

			};

		};

	};
	
	// Ergebnisgitter zuweisen
  ResultGrid = resultGrid;

};

/*!
 *  Die Methode \b interpolate() interpoliert linear zwischen
 *  dem Quellgitter und dem Zielgitter. Dabei wird jedes Element
 *  des Quellgitters mit 1 - \b TimeIndex multipliziert, jedes
 *  Element des Zielgitters mit \b TimeIndex. Danach werden
 *  die skalierten Zielgitterelemente zu den skalierten 
 *  Quellgitterelementen addiert. Das Ergebnis wird im 
 *  Ergebnisgitter gespeichert.
 * 
 *  Ist \b Timeindex < 0, so wird das Quellgitter in das
 *  Ergebnisgitter kopiert.
 *  Ist \b Timeindex > 1, so wird das Zielgitter in das
 *  Ergebnisgitter kopiert.
 *
 *  \note
 *  Es wird nur dann interpoliert, wenn alle drei Gitter
 *  zugewiesen wurden. Ansonsten ist der Zustand des
 *  Ergebnisgitters undefiniert.
 *
 */
template <class T> void CGridInterpolator<T>::interpolate(float TimeIndex)
{

	// Prfen, ob alle Gitter zugewiesen wurden
	if ((SourceGrid!=NULL) && (DestinationGrid!=NULL) && (ResultGrid!=NULL))
	{

		// Zeiger auf die Gitterdaten holen
		T* srcElement = (T*) SourceGrid->getElements();
		T* dstElement = (T*) DestinationGrid->getElements();
		T* resElement = (T*) ResultGrid->getElements();

		// Prfen, ob Zeitindex<0
		if (TimeIndex<0)
		{

			// Quellgitter in das Ergebnisgitter kopieren
			for (int i=0 ; i<ResultGrid->width()*ResultGrid->height() ; i++)
			{

				// Element kopieren
				*resElement=*srcElement;
				srcElement++;
				resElement++;

			};

		}
		else
		// Prfen, ob Zeitindex >1.0
		if (TimeIndex>1.0f)
		{

			// Zielgitter in das Ergebnisgitter kopieren
			for (int i=0 ; i<ResultGrid->width()*ResultGrid->height() ; i++)
			{

				// Element kopieren
				*resElement=*dstElement;
				dstElement++;
				resElement++;

			};

		}
		else
		{

			// Gewichtetes Mittel berechnen
			for (int i=0 ; i<ResultGrid->width()*ResultGrid->height() ; i++)
			{
				
				*resElement=*srcElement+TimeIndex*(*dstElement - *srcElement);
				srcElement++;
				dstElement++;
				resElement++;

			};

		};

	};

};

/*! 
 *  Der Standarddestruktor setzt alle Elementzeiger auf 
 *  NULL zurck.
 *
 */
template <class T> CGridInterpolator<T>::~CGridInterpolator()
{

	// Gitterreferenzen auflsen
	SourceGrid = NULL;
	DestinationGrid = NULL;
	ResultGrid = NULL;

};

#endif

