/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CGridDeformer.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CGridDeformer.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#ifndef cls_CGridDeformerH
#define cls_CGridDeformerH

#if     _MSC_VER > 1000
#pragma once
#endif

#include <stdlib.h>
#include <stdexcept>
using std::invalid_argument;

#include "CDeformer.h"
#include "CGrid.h"
#include "CVector.h"

/*!
 * @class CGridDeformer
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Abstrakte Basisklasse fr alle abgeleiteten Deformationsalgorhtimen,
 * die auf \b CGrid -Objekten operieren.
 *
 *  Die abstrakte Basisklasse \b CGridDeformer definiert eine Schnittstelle,
 *  die allen Deformationsalgorithmen gemeinist, welche auf Vertexgittern
 *  operieren.  Zur Speicherung der Vertexdaten werden \b CGrid -Objekte
 *  verwendet, deren Elemente \b CVector -Objekte sind.
 *  Der Templateparameter \b N legt die Anzahl der Komponenten des Vektors
 *  fest, der Templateparmatter \b T ihren Datentyp.
 *
 *  \note
 *  Die Methode \b deform() bleibt in dieser Klasse weiterhin unimplementiert.
 */

template <int N, class T> class CGridDeformer : public CDeformer
{

protected:

	//! Speichert einen Zeiger auf das Quellgitterobjekt.
	CGrid< CVector<N,T> >* SourceGrid;	    
	
	//! Speichert einen Zeiger auf das Zielgitterobjekt.
	CGrid< CVector<N,T> >* DestinationGrid; 

public:

	//! Standardkonstruktor.
	CGridDeformer();

	//! Setzt den Zeiger auf das Quellgitter.
	virtual void setSourceGrid(CGrid< CVector<N,T> >* objSourceGrid);

	//! Setzt den Zeiger auf das Zielgitter.
	virtual void setDestinationGrid(CGrid< CVector<N,T> >* objDestinationGrid);

	//! Gibt eine Referenz auf das Quellgitter zurck.
	inline CGrid< CVector<N,T> >& sourceGrid();

	//! Gibt eine Referenz auf das Zielgitter zurck.
	inline CGrid< CVector<N,T> >& destinationGrid();

};

/*!
 *  Der Standardkonstruktor initialisert beide Gitterzeiger
 *  mit NULL.
 *
 */
template < int N, class T > CGridDeformer< N,T >::CGridDeformer()
{

	// Gitterzeiger initialisieren
	SourceGrid = NULL;
	DestinationGrid = NULL;

};

/*! 
 *  Die Methode \b setSourceGrid() setzt den Zeiger auf das Quellgitter, d.h.
 *  auf das Gitter welches spter deformiert werden soll.
 *
 *  \note
 *  Falls das Zielgitter bereits gesetzt wurde, wird geprft, 
 *  ob die Ausmae der beiden \b CGrid -Objekte bereinstimmen. 
 *  Ist dies nicht der Fall, so wird eine Exception vom Typ \b invalid_argument ausgelst!
 *
 */
template <int N, class T> void CGridDeformer<N,T>::setSourceGrid(CGrid< CVector<N,T> >* objSourceGrid)
{

	if (objSourceGrid !=NULL)
	{

		// Prfen, ob Zielgitter bereits gesetzt
		if (DestinationGrid !=NULL)
		{

			// Prfen, ob Ausmae bereinstimmen
			if ((objSourceGrid->width()!=DestinationGrid->width()) ||
				  (objSourceGrid->height()!=DestinationGrid->height()))
			{
				
				// Exception auslsen
				throw invalid_argument("Die Ausmae des Quell- und Zielgitters mssen bereinstimmen!");

			}

		}

	}
	
	// Quellgitterreferenz zuweisen
	SourceGrid=objSourceGrid;

};

/*! 
 *  Die Methode \b setDestinationGrid() setzt den Zeiger auf das Zielgitter, d.h.
 *  auf das Gitter welches spter das Ergebnis der Deformation aufnehmen soll.
 *
 *  \note
 *  Falls das Quellgitter bereits gesetzt wurde, wird geprft, 
 *  ob die Ausmae der beiden \b CGrid -Objekte bereinstimmen. 
 *  Ist dies nicht der Fall, so wird eine Exception vom Typ \b invalid_argument ausgelst!
 *
 */
template <int N, class T> void CGridDeformer<N,T>::setDestinationGrid( CGrid< CVector<N,T> >* objDestinationGrid)
{

	if (objDestinationGrid !=NULL)
	{

		// Prfen, ob Zielgitter bereits gesetzt
		if (SourceGrid !=NULL)
		{
		
			// Prfen, ob Ausmae bereinstimmen
			if ((objDestinationGrid->width()!=SourceGrid->width()) ||
				  (objDestinationGrid->height()!=SourceGrid->height()))
			{

				// Exception auslsen
				throw invalid_argument("Die Ausmae des Quell- und Zielgitters mssen bereinstimmen!");

			}

		}

	}

	// Zielgitterreferenz zuweisen
	DestinationGrid=objDestinationGrid;

};

template <int N, class T> inline CGrid< CVector<N,T> >& CGridDeformer<N,T>::sourceGrid()
{
	return SourceGrid; 
};

template <int N, class T> inline CGrid< CVector<N,T> >& CGridDeformer<N,T>::destinationGrid()
{
	return DestinationGrid; 
};

#endif


